/*

Copyright 1993 by the University of Edinburgh, Department of Computer Science

Permission to use, copy, modify, distribute, and sell this software and its
documentation for any purpose is hereby granted without fee, provided that
the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation, and that the name of the University of Edinburgh not be used
in advertising or publicity pertaining to distribution of the software
without specific, written prior permission.  The University of Edinburgh
makes no representations about the suitability of this software for any
purpose.  It is provided "as is" without express or implied warranty.

*/

/* testCard -- display some vaguely useful patterns */

/* Comments to gdmr@dcs.ed.ac.uk */

#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/Xaw/Label.h>

static char *instructions =
	"Use button 1 to select a pattern\nUse button 2 to select a colour";

static String fallback_resources[] = {
	"*card.translations: #augment \
!<Btn1Down>: XawPositionSimpleMenu(patternMenu) MenuPopup(patternMenu) \\n\
!<Btn2Down>: XawPositionSimpleMenu(colourMenu) MenuPopup(colourMenu)",
	"*SmeBSB.Foreground: White",
	"*Font: -*-*-*-*-*-*-*-240-*-*-*-*-iso8859-1",
	NULL,
};

static void setVertical();
static void setHorizontal();
static void setCross();
static void setCircles();
static void setColour();
static void setGreyscale();
static void setColourbars();
static void quitProgram();

static void addMenuLine();
static void addPatternMenu();

static XtAppContext appContext;
static Widget shell;
static Widget cardShell;
static Widget card;
static Widget patternMenu;
static Widget colourMenu;

static Colormap colormap;

static Pixel white;
static Pixel black;
static Pixel fore;
static Pixel back;

static void setupColours();
static void setupGreys();
static unsigned long colours[8];
static unsigned long greys[8];

static Pixmap pattern;
static int patternWidth;
static int patternHeight;

static GC drawGC;
static GC clearGC;
static GC barsGC;

main(argc, argv)
int argc;
char **argv;
{	Arg args[8];
	int n;
	XGCValues gcValues;
	Cardinal depth;

	n = 0;
	XtSetArg(args[n], XtNwidth, 1);			n++;
	XtSetArg(args[n], XtNheight, 1);		n++;
	XtSetArg(args[n], XtNmappedWhenManaged, False);	n++;
	shell = XtAppInitialize(&appContext, "TestCard",
			NULL, 0, &argc, argv,
			fallback_resources, args, n);
	XtRealizeWidget(shell);

	patternWidth = WidthOfScreen(XtScreen(shell)),
	patternHeight = HeightOfScreen(XtScreen(shell)),

	n = 0;
	XtSetArg(args[n], XtNdepth, &depth);		n++;
	XtSetArg(args[n], XtNcolormap, &colormap);	n++;
	XtGetValues(shell, args, n);

	white = WhitePixelOfScreen(XtScreen(shell));
	black = BlackPixelOfScreen(XtScreen(shell));

	cardShell = XtCreatePopupShell("cardShell", overrideShellWidgetClass,
			shell, NULL, 0);

	n = 0;
	XtSetArg(args[n], XtNforeground, white);	n++;
	XtSetArg(args[n], XtNbackground, black);	n++;
	XtSetArg(args[n], XtNlabel, instructions);	n++;
	XtSetArg(args[n], XtNinternalHeight, 0);	n++;
	XtSetArg(args[n], XtNinternalWidth, 0);		n++;
	XtSetArg(args[n], XtNwidth, patternWidth);	n++;
	XtSetArg(args[n], XtNheight, patternHeight);	n++;
	card = XtCreateManagedWidget("card", labelWidgetClass,
			cardShell, args, n);

	n = 0;
	XtSetArg(args[n], XtNbackground, black);	n++;
	XtSetArg(args[n], XtNforeground, white);	n++;
	XtSetArg(args[n], XtNlabel, "Patterns");	n++;
	XtSetArg(args[n], XtNborderWidth, 2);		n++;
	XtSetArg(args[n], XtNborderColor, white);	n++;
	patternMenu = XtCreatePopupShell("patternMenu", simpleMenuWidgetClass,
			card, args, n);

	addMenuLine(patternMenu, 2);

	addPatternMenu("Vertical x2",  setVertical,  2);
	addPatternMenu("Vertical x3",  setVertical,  3);
	addPatternMenu("Vertical x5",  setVertical,  5);
	addPatternMenu("Vertical x10", setVertical, 10);
	addPatternMenu("Vertical x20", setVertical, 20);
	addPatternMenu("Vertical x50", setVertical, 50);

	addMenuLine(patternMenu, 1);

	addPatternMenu("Horizontal x2",  setHorizontal,  2);
	addPatternMenu("Horizontal x3",  setHorizontal,  3);
	addPatternMenu("Horizontal x5",  setHorizontal,  5);
	addPatternMenu("Horizontal x10", setHorizontal, 10);
	addPatternMenu("Horizontal x20", setHorizontal, 20);
	addPatternMenu("Horizontal x50", setHorizontal, 50);

	addMenuLine(patternMenu, 1);

	addPatternMenu("Cross x10", setCross, 10);
	addPatternMenu("Cross x20", setCross, 20);
	addPatternMenu("Cross x50", setCross, 50);

	addMenuLine(patternMenu, 1);
	addPatternMenu("Circles x10", setCircles, 10);
	addPatternMenu("Circles x20", setCircles, 20);
	addPatternMenu("Circles x50", setCircles, 50);

	n = 0;
	if (depth > 1) {
		addMenuLine(patternMenu, 1);
		XtSetArg(args[n], XtNlabel, "Colours");		n++;
	}
	else {
		XtSetArg(args[n], XtNlabel,
			"Sorry, no other colours available");	n++;
	}
	XtSetArg(args[n], XtNforeground, white);		n++;
	XtSetArg(args[n], XtNbackground, black);		n++;
	XtSetArg(args[n], XtNborderWidth, 2);			n++;
	XtSetArg(args[n], XtNborderColor, white);		n++;
	colourMenu = XtCreatePopupShell("colourMenu", simpleMenuWidgetClass,
			card, args, n);
	if (depth > 1) {
		addMenuLine(colourMenu, 2);
		setupColours();
		addPatternMenu("Colour bars", setColourbars, 0);
		setupGreys();
		addPatternMenu("Greyscale", setGreyscale, 0);
	}

	addMenuLine(patternMenu, 2);

	addPatternMenu("QUIT", quitProgram, 0);

	XtRealizeWidget(cardShell);

	if (depth == 1) {
		fore = 1;
		back = 0;
	}
	else {
		fore = white;
		back = black;
	}
	gcValues.foreground = fore;
	gcValues.background = back;
	gcValues.function = GXcopy;
	drawGC = XCreateGC(XtDisplay(card), XtWindow(card),
			GCForeground | GCBackground | GCFunction, &gcValues);

	gcValues.foreground = back;
	gcValues.background = fore;
	clearGC = XCreateGC(XtDisplay(card), XtWindow(card),
			GCForeground | GCBackground | GCFunction, &gcValues);

	barsGC = XCreateGC(XtDisplay(card), XtWindow(card),
			GCFunction, &gcValues);

	pattern = XCreatePixmap(XtDisplay(card), XtWindow(card),
			patternWidth, patternHeight,
			DefaultDepthOfScreen(XtScreen(card)));

	XtPopup(cardShell, XtGrabNone);
	XtAppMainLoop(appContext);
	/*NOTREACHED*/
}

static void setupColours()
{	int i;
	XColor xcolor;
	Widget w;
	Arg args[8];
	int n;
	static char *names[] =
		{ "", "Red", "Green", "Yellow", "Blue", "Magenta", "Cyan" };

	colours[0] = black;

	for (i = 1; i < 7; i++) {
		xcolor.red   = i & 1 ? 65535 : 0;
		xcolor.green = i & 2 ? 65535 : 0;
		xcolor.blue  = i & 4 ? 65535 : 0;
		if (XAllocColor(XtDisplay(card), colormap, &xcolor)) {
			colours[i] = xcolor.pixel;
			n = 0;
			XtSetArg(args[n], XtNlabel, names[i]);		n++;
			XtSetArg(args[n], XtNforeground, xcolor.pixel);	n++;
			w = XtCreateManagedWidget("colourMenuEntry",
				smeBSBObjectClass, colourMenu,
				args, n);
			XtAddCallback(w, XtNcallback,
				setColour, (XtPointer) xcolor.pixel);
		}
		else {
			(void) fprintf(stderr,
					"XAllocColor() failed: colour %d\n", i);
			colours[i] = black;
		}
	}

	colours[7] = white;
	n = 0;
	XtSetArg(args[n], XtNlabel, "White");		n++;
	XtSetArg(args[n], XtNforeground, white);	n++;
	w = XtCreateManagedWidget("colourMenuEntry", smeBSBObjectClass,
			colourMenu, args, n);
	XtAddCallback(w, XtNcallback, setColour, (XtPointer) white);
}

static void setupGreys()
{	XColor xcolor;
	int i;

	greys[0] = black;

	for (i = 1; i < 7; i++) {
		xcolor.red = xcolor.green = xcolor.blue = 8192 * i;
		if (XAllocColor(XtDisplay(card), colormap, &xcolor)) {
			greys[i] = xcolor.pixel;
		}
		else {
			(void) fprintf(stderr,
					"XAllocColor() failed: grey %d\n", i);
			greys[i] = black;
		}
	}

	greys[7] = white;
}

static void addMenuLine(menu, width)
Widget menu;
int width;
{	Widget w;
	Arg args[8];
	int n;

	n = 0;
	XtSetArg(args[n], XtNforeground, white);	n++;
	XtSetArg(args[n], XtNlineWidth, width);		n++;
	w = XtCreateManagedWidget("menuLine", smeLineObjectClass,
			menu, args, n);
}

static void addPatternMenu(menuLabel, menuCallback, menuData)
char *menuLabel;
XtCallbackProc menuCallback;
int menuData;
{	Widget w;
	Arg args[8];
	int n;

	n = 0;
	XtSetArg(args[n], XtNlabel, menuLabel);		n++;
	XtSetArg(args[n], XtNforeground, white);	n++;
	w = XtCreateManagedWidget("patternMenuEntry", smeBSBObjectClass,
			patternMenu, args, n);
	XtAddCallback(w, XtNcallback, menuCallback, (XtPointer) menuData);
}

static void updateCard()
{	static int beenHere = 0;
	Arg args[8];
	int n;

	if (beenHere) {
		XClearArea(XtDisplay(card), XtWindow(card),
				0, 0, 0, 0, True);
	}
	else {
		n = 0;
		XtSetArg(args[n], XtNbitmap, pattern);	n++;
		XtSetValues(card, args, n);
		beenHere = 1;
	}
}

static void clearPattern()
{	XFillRectangle(XtDisplay(card), pattern, clearGC,
			0, 0, patternWidth, patternHeight);
}

static void drawVertical(spacing)
int spacing;
{	int i;

	for (i = spacing - 1; i < patternWidth; i += spacing) {
		XDrawLine(XtDisplay(card), pattern, drawGC,
			i, 0, i, patternHeight);
	}
}

static void drawHorizontal(spacing)
int spacing;
{	int i;

	for (i = spacing; i < patternHeight; i += spacing) {
		XDrawLine(XtDisplay(card), pattern, drawGC,
			0, i, patternWidth, i);
	}
}

static void setVertical(w, client, call)
Widget w;
XtPointer client, call;
{	clearPattern();
	drawVertical((int) client);
	updateCard();
}

static void setHorizontal(w, client, call)
Widget w;
XtPointer client, call;
{	clearPattern();
	drawHorizontal((int) client);
	updateCard();
}

static void setCross(w, client, call)
Widget w;
XtPointer client, call;
{	clearPattern();
	drawVertical((int) client);
	drawHorizontal((int) client);
	updateCard();
}

static void setCircles(w, client, call)
Widget w;
XtPointer client, call;
{	int cx, cy, radius, i, n;

	clearPattern();

	n = patternWidth / (int) client * 2;  /* XXX */
	cx = patternWidth / 2;
	cy = patternHeight / 2;

	for (i = 1; i < n; i++) {
		radius = ((int) client) * i;
		XDrawArc(XtDisplay(card), pattern, drawGC,
			cx - radius, cy - radius,
			2 * radius, 2 * radius,
			0, 360*64);
	}

	updateCard();
}

static void setColour(w, client, call)
Widget w;
XtPointer client, call;
{	XSetForeground(XtDisplay(card), drawGC, (unsigned long) client);
}

static void makeBars(colours)
unsigned long *colours;
{	int i;

	clearPattern();
	for (i = 0; i < 8; i++) {
		XSetForeground(XtDisplay(card), barsGC, *colours++);
		XFillRectangle(XtDisplay(card), pattern, barsGC,
				patternWidth * i / 8, 0,
				patternWidth / 8 + 1, patternHeight);
	}
	updateCard();
}

static void setGreyscale(w, client, call)
Widget w;
XtPointer client, call;
{	makeBars(greys);
}

static void setColourbars(w, client, call)
Widget w;
XtPointer client, call;
{	makeBars(colours);
}

static void quitProgram(w, client, call)
Widget w;
XtPointer client, call;
{	exit(0);
}
