/* game.c:
 *	Externally used routines here:
 * ChangeMode()		-- english<->kanji<->kana
 * CountKanji()		-- returns number of active kanji, given select parms
 * SetupGuess()		-- called each "round" to set things up
 * UseThisKanji()	-- Returns Boolean if we are allowed to use this kanji
 * cheatcallback()	-- callback for cheat button
 * englishcallback()	-- callback for english buttons
 * kanjicallback()	-- callback for kanji buttons
 * picktruevalue()	-- sets up which kanji is the right value
 * pickkanji()		-- picks four kanji to guess
 * printenglish()	-- sets english labels active/inactive, etc
 * printkanji()		-- sets kanji labels active/inactive, etc
 * setstatus(char *)	-- sets status bar
 * guessvalue(int)	-- given the pick of the user, will say if was right
 * Guessvalue()		-- accelerator wrapper for guessvalue()
 */
 

#include <stdio.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>
#include <Intrinsic.h>
#include <StringDefs.h>
#include <Xaw/Command.h>

#include "defs.h"
#include "externs.h"


int values[NUMBEROFCHOICES];
int truevalue;

int Kanji2English=GUESSKANJI;

Boolean doBell,showinorder,showEnglish;
Boolean useUsefile;
Boolean switchKanaEnglish;

extern Display *display;
extern Window mainwindow;

int englishwidth=7;
int englishheight=11;

/* setstatus:
 *  sets label for main status bar widget.
 *	( the long one at the bottom)
 */
void setstatus(char *s)
{
	XtVaSetValues(statusline,
		      XtNlabel,s,
		      NULL);
}

/*
*UseThisKanji()
*	Returns Boolean value on whether this is a "valid" kanji for us,
*	based on usefile markings,
*	current grade set,
*	and whether it has english/kana translation
*/
Boolean UseThisKanji(int kanjinum)
{
	struct translationstruct *kanji;

	/*usefile cancels it? */
	if(useUsefile && (useKanji[kanjinum] == 0)) return False;
	
	kanji = translations[kanjinum];
	/* nonexistant kanji? */
	if(kanji == NULL) return False;


	/* check against frequency limits */
	if(kanji->frequency==0)
		if(lowfrequency != 0) return False;

	   /* "1" is highest frequency!! */
	if(kanji->frequency < highfrequency) return False;
	if(lowfrequency >0){
		if(kanji->frequency >lowfrequency)
			return False;
	}

	/* does the appropriate form to display exist? */
	if( switchKanaEnglish || showEnglish){
		if(kanji->english == NULL)
			return False;
		if(kanji->english[0] == '\0')
			return False;
	}
	if( switchKanaEnglish || (!showEnglish)) {
		/* we're supposed to be showing kana.. are there any? */
		if(kanji->pronunciation == NULL)
			return False;
	}

	/* only thing left is to check grade level */
	switch(kanji->grade_level){
		case 1: case 2:
		case 3: case 4:
		case 5: case 6:
			if(gradelevelflags & (1 <<kanji->grade_level) ){
				return True;
			} else {
				return False;				
			}
		default:
			if(gradelevelflags & (1 <<7) ){
				return True;
			} else {
				return False;
			}
	}
}


/* CountKanji
 *This routine gets called a hell of a lot:
 *	When we change grade level, and
 *	when we change kana/english display.
 *	 (the secnd being because kanjidic does not always have
 *	 english and/or kana listings
 */
void CountKanji(){
	int counter;
	numberofkanji=0;
	for(counter=lowest;counter <=highest;counter++){
		if(UseThisKanji(counter)){
			numberofkanji++;
		}
	}
}

/* pickkanji:
 *   picks an acceptably random kanji char.
 *	NOTE THAT THIS DEPENDS ON "numberofkanji" being correct!
 */

int pickkanji(){
	int rand_kanji,count;
	

	rand_kanji = lrand48()%numberofkanji;

	for(count=lowest;count<=highest;count++) {
		if(UseThisKanji(count)) {
		      rand_kanji--;
		      if(rand_kanji <0)
			  return count;
		}
	}
	
	fprintf(stderr,"Internal error: picked kanji out of range\n");
	fprintf(stderr,"random pick: %d\n",rand_kanji);
	fprintf(stderr,"number of kanji: %d\n",numberofkanji);
	exit(0);

}
/*picktruevalue()
 *	Similar to pickkanji(),
 *	except that this has special handling for
 *	picking the "truevalue" (for going in order! )
 *	Note: this will either loop forever, or return same value
 *	   if there aren't enough valid kanji to use.
 *	   This should never happen, but....
 */
int picktruevalue(){
	static int lastpicked=0;/* we can't "really" auto-initialize this...
				 * set to zero so we can set it
				 * properly lower down
				 */
	
	if(!showinorder)
		return (pickkanji());
	if(lastpicked==0)
		lastpicked = lowest;
	else
		lastpicked++;
	while(!UseThisKanji(lastpicked)){
		lastpicked++;

		if(lastpicked >highest)
			lastpicked = lowest;
	}
	return lastpicked;
}
/*guessvalue(int)
 *	passed number by EITHER kanji button or english button,
 *	we don't care.
 *	Set status bar to correct/incorrect. beeps if incorrect
 *	returns True if correct, False otherwise.
 */
void guessvalue(int guess){
	if(guess == truevalue){
		setstatus("Correct!");
		SetupGuess();
		return;
	} 
	Beep();
	setstatus("Incorrect.");
	if(logfile != NULL){
		fprintf(logfile,"%x -> %x\n",values[truevalue],values[guess]);
	}
}
/* Guessvalue:
 *	accelerator function so we can guess with 1,2,3,4 keys
 */
void Guessvalue(Widget w,XEvent *event,String *params,Cardinal *num_parags){
	int value;
	value = atoi(*params);
	guessvalue(value);
}

/* printkanji:
 *	updates all the kanji buttons.
 */

void printkanji(){
	int i,knum;
	XChar2b onecharstring[2];

	for(i=0;i<NUMBEROFCHOICES;i++){
		UnreverseButton(kanjiWidget[i]);
	}
	onecharstring[1].byte1 = onecharstring[1].byte2 = 0;

	if(Kanji2English == GUESSKANJI){
		knum = values[truevalue];
		onecharstring[0].byte1 = (knum & 0xff00)>>8;
		onecharstring[0].byte2 = (knum & 0x00ff);

		XtVaSetValues(kanjiWidget[0],XtNlabel,onecharstring,NULL);
		for(i=1;i<NUMBEROFCHOICES;i++){
			XtVaSetValues(kanjiWidget[i],XtNlabel,"  ",NULL);
		}
		return;
	}
	/* else */
	for(i=0;i<NUMBEROFCHOICES;i++) {
		knum = values[i];

		onecharstring[0].byte1 = (knum & 0xff00)>>8;
		onecharstring[0].byte2 = (knum & 0x00ff);


		XtVaSetValues(kanjiWidget[i],
			      XtNlabel,onecharstring,
			      XtNsensitive,True,
			      NULL);

	}
}

/* SetWidgetLabel:
 *	Convenience function to
 *	set label of four lines of english/kana.
 *	Will change fonts, dependant on "showEnglish"
 */
void SetWidgetLabel(Widget widget,int Tnum)
{

	if(showEnglish == True){
		XtVaSetValues(widget,
			     XtNencoding,XawTextEncoding8bit,
			     XtNfont,fixedfont,
			     XtNlabel,translations[Tnum]->english,
			     NULL);
	} else {
		XtVaSetValues(widget,
			     XtNencoding,XawTextEncodingChar2b,
			     XtNfont,smallkfont,
			     NULL);
		XtVaSetValues(widget,
			      XtNlabel,translations[Tnum]->pronunciation,
			      NULL);
	}
}
/* printenglish()
 *  similar to printkanji()
 *  updates all "english" labels...
 *  EXCEPT: sometimes we want then to print kana! :-) so we use
 *  SetWidgetLabel() to do the correct type
 */
void printenglish(){
	int i;

	for(i=0;i<NUMBEROFCHOICES;i++){
		UnreverseButton(englishWidget[i]);
	}
	/* we must be guessing which character for one meaning */
	if(Kanji2English == GUESSMEANING){
		SetWidgetLabel(englishWidget[0],values[truevalue]);
		for(i=1;i<NUMBEROFCHOICES;i++){
			XtVaSetValues(englishWidget[i],
			  XtNlabel,"  ",XtNsensitive,False,NULL);
		}
		return;
	}
	/* ELSE */
	/* we have all FOUR  active for picking*/
	for(i=0;i<NUMBEROFCHOICES;i++){
		XtVaSetValues(englishWidget[i],XtNsensitive,True,NULL);
		SetWidgetLabel(englishWidget[i],values[i]);
	}
}



/* SetupGuess
 *		"MAIN LOOP" OF PROGRAM
 ******************************************************
 *
 *	sets up question...
 *	{ "what does this symbol? choose one of the four choices"}
 *	
 *	then returns, presumably falling back to XtAppMainLoop()
 *
 *	Kanji2English == GUESSKANJI : give kanji, and user guesses the meaning
 *					int English
 *	Kanji2English == GUESSENGLISH : give english, and user guesses which
 *					kanji fits best. 
 */
void SetupGuess()
{
	Boolean doloop;
	int valuecount;
	static int savedgrade = ALLGRADES;

	truevalue = lrand48() %NUMBEROFCHOICES;
	for(valuecount=0;valuecount<NUMBEROFCHOICES;valuecount++){
		if(valuecount == truevalue)
			values[valuecount] = picktruevalue();
		else
			values[valuecount] = pickkanji();
	}

	/* now weed out duplicates..
	 * This is messy.
	 *  We compare everything to everything else, and change
	 *   the "current" value if there is a conflict...
	 *    UNLESS current value is the "truevalue".
	 *  We then do a full compare again if there was a conflict
	 */
	do{
		int startcount=0;
		doloop = False;
		do {
			if(startcount == truevalue)
				continue;
			for(valuecount=0;valuecount<NUMBEROFCHOICES;valuecount++){
				if(valuecount == startcount)
					continue;
				if(values[startcount]== values[valuecount] ){
					doloop = True;
					values[startcount] = pickkanji();
				}
			}
		} while(++startcount<NUMBEROFCHOICES);

	} while(doloop== True);

	DescribeCurrent(values[truevalue]);

	printkanji();	/* update labels */
	printenglish(); /* likewise      */
	printgrades();	/* stupid buttons needs updating manually,
			 * EVERY durn time. arrg */

	/* This should really be here, but it works best here... */
	switch(Kanji2English){
		case GUESSKANJI:
			XtVaSetValues(kanjimode,XtNsensitive,False,NULL);
			XtVaSetValues(englishmode,XtNsensitive,True,NULL);
			break;
		case GUESSMEANING:
			XtVaSetValues(kanjimode,XtNsensitive,True,NULL);
			XtVaSetValues(englishmode,XtNsensitive,False,NULL);
			break;
	}
}


/* englishcallback:
 *	Handles clicking on "english" widgets.
 *	Only really neccessary if we are guessing the meaning of a
 *	kanji character
 */
void englishcallback(Widget w,XtPointer data,XtPointer calldata)
{
	if(Kanji2English!= GUESSKANJI)
		return;
	guessvalue((int) data);
}

/*
 *handles top four kanji buttons at top
 */
void kanjicallback(Widget w,XtPointer data,XtPointer calldata)
{
	if(Kanji2English != GUESSMEANING)
		return;
	guessvalue((int) data);
}

void cheatcallback(Widget w,XtPointer data,XtPointer calldata){
	if(Kanji2English == GUESSKANJI){
		ReverseButton(englishWidget[truevalue]);
	} else {
		ReverseButton(kanjiWidget[truevalue]);
	}
}

/* changemode:
 *   change from guessing kanji to guessing english, or vica verse,
 *	by clicking on the appropriate button
 *
 *   Also handles changing from english <->kana display,
 *   calling CountKanji(), because we have to .
 */
void ChangeMode(Widget w,XtPointer data,XtPointer calldata)
{
	/* whichbutton is
	 * EITHER
	 *    4kanji<->4english button,
	 * OR
	 *    kana<->english
	 */
	int whichbutton = (int) data;
#ifdef DEBUG
	printf("button mode change: %d\n",whichbutton);
#endif
	switch(whichbutton){
		case GUESSMEANING:
		case GUESSKANJI:
			if(Kanji2English == whichbutton)
				return;
			Kanji2English = whichbutton;
			SetupGuess();
			break;
		case TOGGLEKANA:
#ifdef DEBUG
			puts("We toggle kana/english now");
#endif
			showEnglish = !showEnglish;
			if(!switchKanaEnglish){
				CountKanji();
				SetupGuess();
			} else {
				printenglish();
			}
			
			break;
	}
}

