#ifndef ccfilter_h
#define ccfilter_h
/*
** ccfilter.h
**
**	This header defines the functions related to the Character
**	Content Filter. Some these are defined in ccfilter.c and
**	some must be defined by the application using the ccfilter.
**
**	Character Content Filter interprets the character stream
**	according to various standards (ISO 2022, CCITT T.51)
**
** Copyright 1992, 1993 by Markku Savela and
**	Technical Research Centre of Finland
*/
#include <X11/Xew/Basic.h>

typedef struct ccf_ContextRec *ccf_Context;

typedef  enum
    {
	ccf_G0 = 0,
	ccf_G1 = 1,
	ccf_G2 = 2,
	ccf_G3 = 3,
	ccf_CC = 4
    } ccf_Gs;

typedef enum
    {
	ccf_C0 = 0,
	ccf_C1 = 1
    } ccf_Cs;

_XFUNCPROTOBEGIN
/*
** ccf_Gn
**	ccf_Gn funtion is called to produce the graphic characters
**	extracted from the stream. The function is supplied with the
**	number of *characters* and a pointer to string of bytes.
**	Each character is represented by one or more bytes in the
**	string, depending on what kind of character set is designated
**	into the Gn set. (Application must remember this from
**	ccf_Designate_G function call). ccf_CC functions like ccf_Gn
**	functions, but is called when filter is working in passthrough
**	mode (Complete Code, coding system different from ISO 2022).
**
**	n is always > 0, when these are called.
*/
typedef void (*ccf_Gn)(
#if NeedFunctionPrototypes
		       void *,	/* client data */
		       ccf_Gs,	/* Identify G0, G1, G2, G3 or CC */
		       char *,	/* String of data bytes */
		       int	/* Number of *characters* in the string */
#endif
		       );

/*
** ccf_Cn
**	ccf_Cn is called for codes belonging to the control sets C0 and C1.
**	(control codes that are used in code extensions are handled
**	internally within the ccfilter).
**
**	The argument c is always within range 0..31.
*/
typedef void (*ccf_Cn)(
#if NeedFunctionPrototypes
		       void *,	/* client data */
		       ccf_Cs,	/* Identify C0 or C1 */
		       int	/* Control code: always in range 0..31 */
#endif
		       );
/*
** ccf_ESC
**	ccf_ESC is called for unrecognized escape sequences.
**	m is number of intermediate bytes, and I points to
**	the first intermediate (if m > 0).
*/
typedef void (*ccf_ESC)(
#if NeedFunctionPrototypes
			void *,	/* client data */
			char *,	/* Intermediate bytes pointer */
			int,	/* Number of intermediate bytes */
			int	/* Final Character (F) */
#endif
);

/*
** ccf_CSI
**	ccf_CSI is called for all CSI sequences.
**
**	n = number of parameters ( >= 0)
**	p = pointer to integers representing the parameter values,
**	    (value 0 indicates a request for default value).
**	    (parameter characters are '0'..'?')
**	I = Intermediate character(s) (' '..'/')
**	F = Final character ('@'..'~').
**	P = Private indicator ('<', '=', '>' or '?')
*/
typedef void (*ccf_CSI)(
#if NeedFunctionPrototypes
			void *,	/* client data */
			int,	/* Private Indicator, if non-zero */
			int *,	/* Parameter values array (p) */
			int,	/* Number of parameters >= 0 (n) */
			char *,	/* Intermediate bytes pointer */
			int,	/* Number of intermediate bytes */
			int	/* Final Character (F) */
#endif
			);
/*
** ccf_DG
**	Designate and invoke a character set into Gn. The function
**	must return the number of bytes per character in this set.
**	Return value 0 is an error and the designation of the Gn
**	doesn't change.
**
**	Gx = ccf_CC, for complete set. After this ccf_Filter will
**		be in a "pass through" state until ccf_Reset is called.
**		All characters are produced through ccf_CC function.
**
**	S  = zero for 94 character set, and non-zero for 96 character set.
**	M  = FALSE(=0), if single byte set, TRUE(=1), if multibyte set
**	F  = Final character, identifying the registered set.
*/
typedef int (*ccf_DG)(
#if NeedFunctionPrototypes
		      void *,	/* client data */
		      ccf_Gs,	/* Gx */
		      int,	/* Zero = 94-set, and non-Zero = 96-set */
		      int,	/* M Single/Multibyte indicator */
		      int	/* F Final Character */
#endif
		      );
/*
** ccf_DC
**	Designate and invoke control characters. Return Non-Zero,
**	if error.
**
**	Cx = 0, for C0
**	   = 1, for C1
**	F  = Final character, identifying the registered set.
*/
typedef int (*ccf_DC)(
#if NeedFunctionPrototypes
		      void *,	/* client data */
		      ccf_Cs,	/* Cx */
		      int	/* F Final Byte */
#endif
		      );
/*
** ccf_Feed
**	Feed in a portion of a byte stream. Note, that escape
**	sequences and multibyte characters can extend over portions
**	(e.g. there is no requirement that an escape sequence must
**	be contained within single portion as a whole). Returns on
**	success. (No error returns defined at the moment.)
*/
int ccf_Feed(
#if NeedFunctionPrototypes
	       ccf_Context,	/* Context to feed (from ccf_Reset) */
	       char *,		/* Pointer to bytes */
	       int		/* Number of bytes to feed */
#endif
);

/*
** ccf_Open
**	Open the filter stream. Returns a pointer to a filter context block
**	that must be given as parameter to ccf_Filter.
*/
ccf_Context ccf_Open(
#if NeedFunctionPrototypes
		      void *,	/* client data (for callbacks) */
		      ccf_Gn,	/* Craphic characters callback */
		      ccf_Cn,	/* Control characters callback */
		      ccf_ESC,	/* ESC Sequence callback */
		      ccf_CSI,	/* CSI Sequence callback */
		      ccf_DG,	/* Designate Graphic callback */
		      ccf_DC	/* Designate Control callback */
#endif
		      );
/*
** ccf_Close
**	Close ccf_Context (release data associated with the context)
*/
void ccf_Close(
#if NeedFunctionPrototypes
	       ccf_Context
#endif
);
_XFUNCPROTOEND

#endif

