/*
** TextLayout
**	Layout character content lines
**
** Copyright 1992, 1993 by Markku Savela and
**	Technical Research Centre of Finland
*/
#if SYSV_INCLUDES
#	include <memory.h>	/* memset() */
#endif
#if ANSI_INCLUDES
#	include <stdlib.h>
#endif
#include <string.h>
#include <stdio.h>

#include <X11/Xlib.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xew/TextP.h>
#include "TextImport.h"
#include "TextLayout.h"
#include "TextFont.h"

/*
** IsLayoutContent
**	macro returns True, if the Snip is generated by the layout process
*/
#define IsLayoutContent(s) \
	(((s)->mode.bits & Content_MASK) == Content_FORMATTED)

/*
**	This module has only one externally callable function
**	XeTextLayout, which lays out the content of
**	single widget. Layout process needs to keep up quite a
**	lot of state information. This state is kept in static
**	variables local to this module to lessen the need to
**	pass many between local functions here.
*/
typedef struct SequenceState
    {
	short maxAscent;	/* Max character backward extent */
	short maxDescent;	/* Max character forward extent */
	XeTextTag tag;		/* Current tag at this point */
	Dimension wNow;		/* Total width up to this point */
	int spaces;		/* Number adjustable spaces *inside* line */
    } SequenceState;

/*
** ExpectEntry
**	stores the original Snip layout information if it was laid out
**	before this layout process. This is needed only because of the need
**	to generate good expose area and terminate layout process as
**	soon as possible after no changes start to happen to the line
**	boxes (This is a huge ugly hack of jugling x,y values in Snips
**	and I don't like it a bit.. --msa)
*/
typedef struct ExpectEntry
    {
	Snip *s;
	Position x, y;
	Dimension xWidth;
	short ascent, descent;
    } ExpectEntry;

static int wArea;		/* Width of the positioning area */
static int lineLength;		/* Current available line length */
static int lineOffset;		/* Offset of from the line Home */
static Dimension lineSpacing;	/* Current line spacing */
static short backwardExtent;	/* Line box backward extent */
static short forwardExtent;	/* Line box forward extent */
static short previousExtent;	/* Forward extent of the previous line box */
static int yShift;		/* Font baseline shift from line baseline */
static int fontAscent;		/* Ascent of the last font seen */
static int fontDescent;		/* Descent of the last font seen */

static SequenceState current;	/* Current sequence state */
static SequenceState lastSpace;	/* State at last space */
static Snip *firstSnip;		/* First Snip of the sequence */
static Snip *beginLine;		/* First Snip of the line box */

typedef struct FontCacheEntry
    {
	unsigned long fontkey;	/* fn, posture, weight, size */
	char *charset;		/* Character Set Id */
	XFontStruct *font;	/* Associated font */
    } FontCacheEntry;

static FontCacheEntry font_cache[97]; /* Size should be a *PRIME* */
static int cache_hit = 0;	/* just for temp testing */
static int cache_miss = 0;
static int cache_full = 0;	/* This should stay ZERO!! */
static int cache_reset = 0;	/* Non-zero, if cache should be cleared */
static Widget cache_widget;	/* Widget for which cache is computed. This
				   is temporary hack and should not be left
				   into production version, as widget pointer
				   being same does not guarantee that the
				   widget is really the same. --msa
				*/


/*
** XeTextSnipExtents
**	Compute dimensions of a text Snip directly into the
**	Snip structure (fill in fields xWidth, ascent and descent)
**	Also, fill in the actual font information to be used.
**	Note: not yet fully implemented, doesn't handle right to
**	left writing direction among other things...
*/
static void XeTextSnipExtents(w, t)
XeTextWidget w;
Snip *t;
    {
	XCharStruct XE;
	int dir;
	SnipData *h = t->head;

	if (h == NULL)
	    {
		t->xWidth = 0;
		t->ascent = 0;
		t->descent = 0;
		return;
	    }
	if (h->font == NULL)
	    {
		/*
		** Font struct still undefined, load it
		*/
		int fn = Font_NUMBER(t->mode.bits);
		XeFontList fId;
		int hit, i;
		Dimension pSize;
		unsigned long fontkey;

		if (fn >= w->text.num_fonts)
			fn = 0; /* Try for default font */
		if (w->text.fonts == NULL || w->text.num_fonts == 0)
		    {
			pSize = w->text.line_spacing; /* or lineSpacing? */
			fId = 0;
		    }
		else
		    {
			pSize = w->text.fonts[fn].font_size;
			fId = w->text.fonts[fn].font_list;
		    }
		fontkey = t->mode.bits &
			(Weight_MASK | Italicized_MASK | Font_MASK);
		hit = (fontkey ^ (unsigned long)h->character_set) %
			XtNumber(font_cache);
		for (i = hit ;; )
		    {
			if (font_cache[i].font == NULL)
			    {
				cache_miss++;
				break;
			    }
			else if (font_cache[i].fontkey == fontkey &&
				 font_cache[i].charset == h->character_set)
			    {
				cache_hit++;
				break;
			    }
			i += 1;
			if (i == XtNumber(font_cache))
				i = 0;
			if (i == hit)
			    {
				/*
				** Nasty thing.. should really not happen.
				** Just take over the first hash entry. Not
				** really right solution, but we assume for
				** now that we really don't get here...
				*/
				font_cache[i].font = NULL;
				cache_full++;
				cache_reset = TRUE;
			    }
		    }
		if (font_cache[i].font == NULL)
		    {
			font_cache[i].fontkey = fontkey;
			font_cache[i].charset = h->character_set;
			font_cache[i].font = XeFindFont
				(w, h->character_set, fId,
				 t->mode.bits & Weight_MASK,
				 t->mode.bits & Italicized_MASK, pSize);
			
		    }
		h->font = font_cache[i].font;
	    }
	if (h->bytes == 2)
		XTextExtents16((XFontStruct *)h->font,
			       (XChar2b *)t->data, t->length,
			       &dir, &fontAscent, &fontDescent, &XE);
	else
		XTextExtents((XFontStruct *)h->font, t->data, t->length,
			     &dir, &fontAscent, &fontDescent, &XE);
	t->xWidth = XE.width;
	/*
	** Using Font ascent/descent gives better result for now..
	*/
	t->ascent = fontAscent;
	t->descent = fontDescent;
    }

/*
** XeTextSnipWidth
**	return the width of the text string.
*/
Dimension XeTextSnipWidth(w, t, s, n)
XeTextWidget w;
Snip *t;
char *s;
int n;
    {
	SnipData *h = t->head;

	if (h == NULL)
		return 0;
	if (h->font == NULL)
		XeTextSnipExtents(w, t); /* Force loading of Font */
	return h->bytes == 2 ?
		XTextWidth16(h->font, (XChar2b *)s, n) :
			XTextWidth(h->font, s, n);
    }


/*
** FindTabulationStop
**	Find and return a pointer the tabulation stop definition.
**	Returns NULL, if not defined for the reference.
**
**	The tabulation position is selected by the reference, which
**	may result positioning over existing text. The tabulated text
**	sections can occur in any order regardless of their final
**	placements on the line.
**
*/
#if NeedFunctionPrototypes
static XeTabStop *FindTabulationStop(XeTextWidget, int);
#endif
static XeTabStop *FindTabulationStop(w, reference)
XeTextWidget w;
int reference;
    {
	char ident[sizeof(int)*3+1];
	XeTabStop *t = w->text.line_layout_table;

	if (t == NULL)
		return NULL;
	memset(ident,0,sizeof(ident));
	sprintf(ident,"%d", (int)reference);
	for ( ; t->tabulation_reference != NULL; t++)
	    {
		if (strcmp(ident,t->tabulation_reference) == 0)
			return t;
	    }
	return NULL;
    }

/*
** HorizontalTab
**	The tabulation position is selected by searching forward from
**	the current position. LineLayoutTable is used, if present,
**	otherwise tab stops are assumed to be evenly spaced at every
**	eight character position (width of the character position is
**	defined as the width of the SPACE in current font).
*/
#if NeedFunctionPrototypes
static int HorizontalTab(XeTextWidget, int);
#endif
static int HorizontalTab(w, x)
XeTextWidget w;
int x;
    {
	XeTabStop *t = w->text.line_layout_table;
	int tabWidth, position;
	Snip *r;

	if (t != NULL)
	    {
		/* Horizontal Tabulation, try using lineLayoutTable */

		XeTabStop *s = NULL;
		int d = 0, e;

		for (; t->tabulation_reference != NULL; t++)
		    {
			e = t->tabulation_position - x;
			if (e > 0 && e < d)
			    {
				d = e;
				s = t;
			    }
		    }
		if (s != NULL)
			return s->tabulation_position; /* Found one */
	    }
	/*
	** Horizontal Tabulation, no lineLayoutTable, or current position
	** is past all defined positions there. Assume evenly spaced
	** tabulation positions and choose the next suitable.
	*/
	/* Use the first font used in the widget. This may not be
	   actually right, some widget wide specific default font should
	   be used. Fix later --msa
	*/
	tabWidth = 0;
	for (r = w->text.first; r != NULL; r = r->next)
		if (r->head != NULL)
		    {
			tabWidth = XeTextSnipWidth(w, r, "    ", 1) * 8;
			break;
		    }
	if (tabWidth == 0)
		tabWidth = 8 * 6; /* Just put something in there... --msa */
	x -= w->text.indentation;
	position = ((x + tabWidth) / tabWidth) * tabWidth;
	return position + w->text.indentation;
    }

/*
** ComputeWidth
**	Compute the width of the current Snip
*/
static void ComputeWidth(w, t)
XeTextWidget w;
Snip *t;
    {
	static int lastFontHeight; /* Last Baseline font height */

	int tmp;

	if (lastFontHeight <= 0)
		lastFontHeight = lineSpacing;
	switch (t->mode.bits & PartialLine_MASK)
	    {
	    case PartialLine_NONE:
		yShift = 0;
		break;
	    case PartialLine_UP:
		yShift = -(lastFontHeight / 3);
		break;
	    case PartialLine_DOWN:
		yShift = lastFontHeight / 3;
		break;
	    }
	t->y += yShift;
	XeTextSnipExtents(w, t);
	current.wNow += t->xWidth;
	tmp = t->ascent - yShift;
	if (tmp > current.maxAscent)
		current.maxAscent = tmp;
	tmp = t->descent + yShift;
	if (tmp > current.maxDescent)
		current.maxDescent = tmp;
    }

/*
** Position_XXXX functions will do the appropriate changes into the
** Snip chain to achieve the positioning of the content.
**
** Each will return a pointer to the last Snip that actually is included
** into this sequence. Usually it will be the supplied parameter 'last',
** except for the TEXT.
*/

/*
** CloseLineBox
**	Do final touches to the line box placement. Compute the amount of
**	vertical adjustment for the baseline (relative to the baseline of
**	the previous line) and update all snips within line.
**
**	Returns the vertical movement that was required to move from the
**	previous baseline to the baseline of this (closed) linebox.
*/
#if NeedFunctionPrototypes
static int CloseLineBox(XeTextWidget, Snip *);
#endif
static int CloseLineBox(w, end)
XeTextWidget w;
Snip *end;
    {
	int y_adj = previousExtent;
	Snip *t;

	/*
	** Proportional spacing is used only if the line has some content
	** in it. Otherwise fixed lineSpacing is assumed.
	*/
	if (w->text.proportional && (forwardExtent || backwardExtent))
	    {
		previousExtent = forwardExtent;
		y_adj += backwardExtent;
	    }
	else
	    {
		previousExtent = lineSpacing / 3;
		y_adj += lineSpacing - previousExtent;
	    }
	if (y_adj)
		for (t = beginLine;;)
		    {
			t->y += y_adj;
			if (t == end)
				break;
			t = t->next;
		    }
	return y_adj;
    }

/*
** WrapUpSequence
**	Do the finishing touches for the positioning functions:
**	search for endseq etc.
*/
#if NeedFunctionPrototypes
static Snip *WrapUpSequence(XeTextWidget, SequenceState *, Snip *, int);
#endif
static Snip *WrapUpSequence(w, s, end, x_adj)
XeTextWidget w;
SequenceState *s;
Snip *end;
int x_adj;
    {
	Snip *t;

	if (s->maxAscent > backwardExtent)
		backwardExtent = s->maxAscent;
	if (s->maxDescent > forwardExtent)
		forwardExtent = s->maxDescent;
	/*
	** Adjust final location of the snips in *sequence*
	*/
	if (x_adj)
		for (t = firstSnip; ; t = t->next)
		    {
			t->x += x_adj;
			if (t == end)
				break;
		    }
	return end;
    }

/*
** Position_TEXT
**	Position a TEXT sequence.
**
**	last	the last Snip actually taking part in positioning.
**
**	end	the actual Snip ending the sequence (this differs
**		from last, if line end contains spaces).
*/
static Snip *Position_TEXT(w, text, last, end)
XeTextWidget w;
SequenceState *text;
Snip *last, *end;
    {
	int offset = lineOffset;
	int wExtra = lineLength - text->wNow;
	XeAlignment alignment;
	/*
	** Force START aligment, if no real lineLength or if the sequence
	** is actually ended with Horizontal TAB. Use the alignment given
	** by the resource, unless the Snip specifies it explicitly by
	** Justify_MODE.
	*/
	if (lineLength <= 0 || (end->tab && end->tabref == Snip_NEXTTABSTOP))
		alignment = XeAlignment_START;
	else if (end->quad)
		alignment = end->quad;
	else if (end->mode.bits & Justify_MASK)
		alignment = Justify_MODE(end->mode.bits);
	else
		alignment = w->text.alignment;
	switch (alignment)
	    {
	    default:
	    case XeAlignment_START:
		break;
	    case XeAlignment_END:
		offset += wExtra;
		break;
	    case XeAlignment_CENTER:
		offset += wExtra / 2;
		break;
	    case XeAlignment_JUSTIFIED:
		/*
		** For the justifying to be applicable, the line
		** must include adjustable spaces, there must be
		** positive amount of loose space, and this must
		** not be the "hard line end" or last line of the
		** block.
		*/
		if (text->spaces && wExtra > 0 && IsLayoutContent(end) &&
		    end->next != NULL)
		    {
			Snip *t;
			Dimension wSpace = wExtra / text->spaces;
			Dimension adjust = 0;

			text->wNow += wExtra;
			wExtra -= text->spaces * wSpace;
			for (t = firstSnip; t != last; t = t->next)
			    {
				if (t->space)
				    {
					adjust += wSpace;
					if (wExtra)
					    {
						adjust += 1;
						wExtra -= 1;
					    }
				    }
				t->x += adjust;
			    }
			for (;; t = t->next)
			    {
				t->x += adjust;
				if (t == end)
					break;
			    }
		    }
		break;
	    }
	return WrapUpSequence(w, text, end, offset);
    }

/*
** Position_ITEM
**	Position ITEM identifier
*/
static Snip *Position_ITEM(w, item, last)
XeTextWidget w;
SequenceState *item;
Snip *last;
    {
	int offset;
	/*
	** Compute the offset of the item from the line home position,
	** taking alignment into account.
	*/
	if (w->text.itemization->identifier_alignment == XeAlignment_START)
		offset = w->text.itemization->identifier_start_offset;
	else
		offset = w->text.itemization->identifier_end_offset -
			item->wNow;
	return WrapUpSequence(w, item, last, offset);
    }

/*
** LookFor
**	Search for characters from a bounded string. Return
**	a pointer to the first character found or NULL, if none.
**	(comparable with strpbrk).
*/
static char *LookFor(d, s, n)
char *d, *s; int n;
    {
	unsigned char c;
	register unsigned char *p;
	unsigned char *q = (unsigned char *)s + n;

	if (d == NULL || s == NULL || n <= 0)
		return NULL; /* This really means erroneus call.. */
	for ( ; c = *d; d++)
		for (p = (unsigned char *)s; p < q; ++p)
			if (*p == c)
				return (char *)p;
	return NULL;
    }

/*
** Position_STAB
**	Position STAB sequence
*/
static Snip *Position_STAB(w, stab, last, tabref)
XeTextWidget w;
SequenceState *stab;
Snip *last;
int tabref;
    {
	static XeTabStop none;

	int offset;
	XeTabStop *ts = FindTabulationStop(w, tabref);
	Snip *t;

	if (ts == NULL)
		/* Error? Couldn't find the definition for the tab stop!!
		** Just default to All Zeroes Tabulation_Stop for now...
	        */
		ts = &none;
	/*
	** As the basic reference point in everything else is the line
	** home position, but tab stops are defined relative to the
	** positioning area, IND must be subtracted...
	*/
	offset = ts->tabulation_position - w->text.indentation;
	/*
	** Compute offset from start of the positioning area to
	** the beginning of the positioned text.
	*/
	switch (ts->alignment)
	    {
	    case XeAlignment_END:
		offset -= stab->wNow;
		break;
	    case XeAlignment_CENTER:
		offset -=  stab->wNow / 2;
		break;
	    case XeAlignment_AROUND:
		if (ts->alignment_character_string == NULL)
		    {
			/*
			** Aligned around specified without giving the
			** characters, an error, but treat as if aligned
			** around character was not found, assume 'end_aligned'
			*/
			offset -= stab->wNow;
			break;
		    }
		for (t = firstSnip; ; )
		    {
			char *s = NULL;

			/*
			** Temporary kludge: Do not search alignment chars
			** from snips that are multibyte (not right now
			** either). Should verify that the aligment char
			** is from same character set as the Snip and search
			** only in that case... --msa
			*/
			if (t->head && t->head->bytes == 1)
				s = LookFor(ts->alignment_character_string,
					    t->data, t->length);
			if (s != NULL)
			    {
				/*
				** Found the character around which
				** the text should be aligned. Align point
				** will be the PP of the align character.
				*/
				offset -= t->x - firstSnip->x +
					XeTextSnipWidth(w, t, t->data,
							s - t->data);
				break;
			    }
			if (t == last)
			    {
				/*
				** Aligned around was specified, but none
				** of the alignment characters occurred
				** within tabulated text. Default to
				** end-aligned in that case.
				*/
				offset -= stab->wNow;
				break;
			    }
			t = t->next;
		    }
		break;
	    case XeAlignment_START:
	    default:
		break;
	    }
	return WrapUpSequence(w, stab, last, offset);
    }

/*
** SplitLine
**	Split the line from the specified Snip (s) and return a pointer
**	to a Snip that will belong to the next line.
*/
static Snip *SplitLine(w, text, last)
XeTextWidget w;
SequenceState *text;
Snip *last;
    {
	Snip *t = last;
	Snip *s;
	int x, y;

	x = t->x;
	y = t->y;
	/*
	** Eat up all extra spaces to the end of the line
	*/
	for ( ; ; )
	    {
		if (t->space)
		    {
			t->xWidth = 0;
			t->x = x;
			t->y = y;
		    }
		if (t->endseq)
			break;
		s = t->next;
		if (s == NULL || !s->space)
			break;
		t = s;
	    }
	/*
	** If the last element doesn't end the sequence, then generate
	** the line ending sequence here.
	*/
	if (!t->endseq)
	    {
		s = _XeInsertSnip(&t->next);
		s->endseq = Snip_EndLine;
		s->mode = t->mode;
		s->mode.bits &= ~Content_MASK;
		s->mode.bits |= Content_FORMATTED;
		s->layout = TRUE;
		s->valid = TRUE;
		s->x = x;
		s->y = y;
		t = s;
	    }
	return Position_TEXT(w, text, last, t);
    }

/*
** UpdateExposeArea
**	Extend expose area from the Snip
*/
#if NeedFunctionPrototypes
static void UpdateExposeArea(XRectangle *, int, int, int, int);
#endif
static void UpdateExposeArea(r, x, y, width, height)
XRectangle *r;
int x, y, width, height;
    {
	if (width == 0 || height == 0)
		return;
	if (r->width == 0)
	    {
		r->x = x;
		r->y = y;
		r->width = width;
		r->height = height;
	    }
	else
	    {
		if (x < r->x)
		    {
			r->width += r->x - x;
			r->x = x;
		    }
		if (x + width > r->x + r->width)
			r->width = x + width - r->x;
		if (y < r->y)
		    {
			r->height += r->y - y;
			r->y = y;
		    }
		if (y + height > r->y + r->height)
			r->height = y + height - r->y;
	    }
    }

/*
** UpdateExpose
**	Compare Snips in the line box to the saved coordinates of the
**	previous Snips. Update expose region to cover all changes.
**
**	Return 'mode', which has the following interpretation
**
**	mode=-1	searching for valid=FALSE position,
**	mode=1	have found valid=FALSE, now searching for valid=TRUE,
**	mode=0	have found valid=TRUE, layout process can break now.
**	mode=2	do not search for anything. Layout will not terminate
**		until end of content.
*/
#define LOOKFOR_MODE (-1)
#define LOOKFOR_NONE 2

#if NeedFunctionPrototypes
static int UpdateExpose(XRectangle *,ExpectEntry *,ExpectEntry *,Snip *,int);
#endif
static int UpdateExpose(expose, start, end, last, mode)
XRectangle *expose;
ExpectEntry *start, *end;
Snip *last;
int mode;
    {
	int changes = 0;
	Snip *s;

	if (!expose)
	    {
		for ( ;start < end && (s = start->s) != last; start++)
		    {
			if (IsEditableContent(s->mode.bits))
			    {
				if (!s->valid ||
				    start->x != s->x || start->y != s->y)
					changes++;
				if (s->valid)
				    {
					if (mode == 1)
						mode = 0;
				    }
				else if (mode == -1)
					mode = 1;
			    }
			s->layout = s->valid = TRUE;
		    }
	    }
	else for ( ;start < end && (s = start->s) != last; start++)
	    {
		if (IsEditableContent(s->mode.bits))
		    {
			if (!s->valid || start->x != s->x || start->y != s->y)
			    {
				changes++;
				UpdateExposeArea(expose, s->x,
						 s->y - s->ascent,
						 s->xWidth,
						 s->ascent + s->descent);
				if (s->layout)
					UpdateExposeArea
						(expose, start->x,
						 start->y - start->ascent,
						 start->xWidth,
						 start->ascent+start->descent);
			    }
			if (s->valid)
			    {
				if (mode == 1)
					mode = 0;
			    }
			else if (mode == -1)
				mode = 1;
		    }
		else if (s->layout)
			UpdateExposeArea(expose, s->x, s->y - s->ascent,
					 s->xWidth, s->ascent + s->descent);
		s->layout = s->valid = TRUE;
	    }
	
	/*
	** Restore the original (x,y) for the snips that were processed,
	** but didn't get included into the linebox.
	*/
	while (start < end)
	    {
		start->s->x = start->x;
		start->s->y = start->y;
		start->s->xWidth = start->xWidth;
		start->s->ascent = start->ascent;
		start->s->descent = start->descent;
		start++;
	    }
	return mode ? mode : changes != 0;
    }

/*
** _XeTextLayout
**	Layout widget character content into widget *internal*
**	"Processable Formatted Form". This function can be called
**	any number of times.
**
**	from = 	NULL, redo the layout fully, ignore all existing
**		old layout information.
**
**	from != NULL, start layout process from the first possible
**		linebox before this point (Snip).
**
**	If expose is non-NULL, the pointed XRectangle will be expanded
**	to contain the area of the layout (usefull only if from != NULL).
**	The rectangle must be properly initialize before call
**	(expose->width = 0 is sufficient to make area initially empty).
*/
int _XeTextLayout(w, from, expose_region)
XeTextWidget w;
Snip *from;
Region expose_region;
    {
	typedef enum
	    {
		Sequence_TEXT,
		Sequence_ITEM,
		Sequence_STAB
	    } LayoutSequence;

	/*
	** lay, addLayout
	**	if addLayout == non-zero, then we are processing application
	**	inserted layout content before the Snip pointed by 'lay'.
	**	(note! lay may also be NULL, if inserting occurs after
	**	the last editable Snip). When addLayout == 0, 'lay' is
	**	NULL.
	** expect_default, expect, expect_point, expect_end
	**	are used in line building to detect when layout hits a point
	**	that doesn't change. Fixed array expect_default is used and
	**	it should be sufficient for most cases. However, the code
	**	is prepared to allocate a larger array dynamically on demand.
	*/
	Snip *s, *p, **h, *lay;
	XeTextTag tag;
	int i, x, y, yHighMark, xHighMark;
	ExpectEntry expect_default[200];
	ExpectEntry *expect, *expect_point = &expect_default[0];
	ExpectEntry *expect_end = &expect_default[XtNumber(expect_default)];
	int layout = w->text.format;
	int lookfor, tabref, firstSeq, addLayout, itemization;
	LayoutSequence sequence;
	XRectangle expose_rect, *expose;

	/*
	** Keep track of the bounding box of exposed area only if
	** expose_region is requested by the caller. (Init box to empty)
	*/
	expose_rect.width = 0;
	expose = expose_region ? &expose_rect : NULL;
	/*
	** Initialize the global state
	*/
	if (cache_widget != (Widget)w || cache_reset)
	    {
		for (i = 0; i < XtNumber(font_cache); i++)
			font_cache[i].font = NULL;
		cache_hit = 0;
		cache_miss = 0;
		cache_reset = 0;
		cache_widget = (Widget)w;
	    }
	wArea = w->text.column_width ? w->text.column_width : w->core.width;
	if (wArea <= 0)
		layout = False;
	lineSpacing = w->text.line_spacing;
	firstSeq = True;
	if (lineSpacing == 0)
	    {
		/*
		** If lineSpacing == 0, then the default lineSpacing
		** will be the font height of the primary (default) font
		** using the default character set. Unfortunately, one
		** needs to do some complex stuff to get it...
		*/
		Snip *list = NULL;
		s = _XeInsertSnip(&list);
		s->head = (SnipData *)XtCalloc(1, XtOffsetOf(SnipData, s[0]));
		s->data = &s->head->s[0];
		s->head->bytes = 1;
		s->head->refs = 1;
		s->head->font = NULL;
		s->head->character_set = XeTextCharsetDefault();
		XeTextSnipExtents(w, s);
		/* ..assuming fontAscent and fontDescent are put into Snip */
		lineSpacing = s->descent + s->ascent;
		_XeDeleteSnip(&list);
	    }
	yHighMark = previousExtent = 0;
	/*
	** Initialize layout breaking parameters
	*/
	y = 0;
	x = w->text.indentation;
	h = &w->text.first;
	tag = 0;
	if (from == NULL)
	    {
		w->text.x = x;
		w->text.y = y;
		xHighMark = 0;
		lookfor = LOOKFOR_NONE;
	    }
	else
	    {
		int gotcha, editable, base_y, max_y, min_y;
		/*
		** Skip over already laid out material, start layout from the
		** beginning of the previous line from the line has been
		** modified.
		*/
		for (s = from; ;s = s->next)
			if (s == NULL)
				return 0; /* Nothing to layout! */
			else if (!s->valid)
				break;
		/*
		** Search backwards for the line end marker from
		** which a new layout process can be started. It will
		** be the first line end having one of the following
		** conditions:
		** - line end is an editable hard line end, or
		** - line end is followed by valid layout which includes
		**   an editable space, or
		** - line end is followed by a full valid line which
		**   includes at least on valid editable component.
		*/
		for (gotcha = FALSE, editable = 0;;)
		    {
			if (s->back == &w->text.first)
				break;
			s = PreviousSnip(s);
			if (!s->valid)
			    {
				gotcha = FALSE;
				editable = 0;
				continue;
			    }
			if (IsEditableContent(s->mode.bits))
			    {
				editable += 1;
				tag = s->mode.tag;
				if (s->space)
					gotcha = TRUE;
				else if (HasEndLine(s))
					break;
			    }
			else if (HasEndLine(s))
				if (gotcha && editable)
					break;
				else
				    {
					gotcha = TRUE;
					editable = 0;
				    }
		    }
		/*
		** Now 's' points to the Snip (HasEndLine) *AFTER* which
		** the new layout will start. If the proportional spacing has
		** has been requested, we have to "guess" the forwardExtent
		** of the preceding line is, and set previousExtent from it.
		** If the line is has only white space or no proportional
		** spacing, use formula "lineSpacing / 3".
		** Additionally, have to find out the initial value for the
		** previous 'tag'. But, if the previos line is result of the
		** layout, assume tag is valid and processed already.
		*/
		p = s;
		gotcha = !IsEditableContent(p->mode.bits);
		base_y = max_y = min_y = p->y;
		do
		    {
			if (!gotcha && IsEditableContent(p->mode.bits))
			    {
				tag = p->mode.tag;
				gotcha = TRUE;
			    }
			if (p->y + p->descent > max_y)
				max_y = p->y + p->descent;
			if (p->y - p->ascent < min_y)
				min_y = p->y - p->ascent;
			if (p->back == &w->text.first)
				break;
			p = PreviousSnip(p);
		    } while (!HasEndLine(p));
		if (w->text.proportional && (max_y - min_y) > 0)
			previousExtent = max_y - base_y;
		else
			previousExtent = lineSpacing / 3;
		/*
	        ** Did we get the 'tag'? If not, search backwards...
		*/
		if (!gotcha)
		    {
			while (!IsEditableContent(p->mode.bits) &&
			       p->back != &w->text.first)
				p = PreviousSnip(p);
			tag = p->mode.tag;
		    }
		if (s->back == &w->text.first)
		    {
			tag = 0;
			previousExtent = 0;
		    }
		else
		    {
			yHighMark = y = s->y;
			firstSeq = HasEndParagraph(s);
			h = &s->next;
		    }
		xHighMark = w->text.w;
		lookfor = LOOKFOR_MODE;
	    }
	lineOffset = firstSeq ? w->text.first_line_offset : 0;
	lineLength = wArea - w->text.indentation - lineOffset;
	p = beginLine = lay = NULL;
	itemization = addLayout = FALSE;
	while (*h)
	    {
		static SequenceState initial = {0,0,0,0};

		Snip *space = NULL;	/* Set Non-NULL, if last space found */

		current = initial;	/* Initialize 'current' state */
		current.tag = tag;
		firstSnip = *h;
		if (beginLine == NULL)
		    {
			beginLine = firstSnip;
			forwardExtent = 0;
			backwardExtent = 0;
			expect = expect_point;
			if (firstSeq && w->text.itemization &&
			    w->text.itemization->identifier_alignment !=
			    XeAlignment_NONE)
				sequence = Sequence_ITEM;
			else
				sequence = Sequence_TEXT;
		    }
		x = w->text.x;
		yShift = 0;
		do
		    {
			/*
			** Locate the next Snip to process. Delete all
			** Snips generated by the previous layout process.
			** If the content ends without a proper endseq
			** Snip, then add one on behalf of the layout process.
			*/
			while (1)
			    {
				if ((s = *h) == NULL)
				    {
					s = _XeInsertSnip(h);
					s->endseq = Snip_End;
					s->mode.bits &= ~Content_MASK;
					s->mode.bits |= Content_FORMATTED;
					if (firstSnip == NULL)
						firstSnip = s;
					if (beginLine == NULL)
						beginLine = s;
					break;
				    }
				else if (addLayout)
					break;
				else if (IsLayoutContent(s))
				    {
					if (s == firstSnip)
						p = firstSnip = s->next;
					if (s == beginLine)
						beginLine = s->next;
					if (expose && s->layout && s->xWidth)
						UpdateExposeArea
							(expose,
							 s->x,
							 s->y - s->ascent,
							 s->xWidth,
							 s->ascent+s->descent);
					_XeDeleteSnip(h);
				    }
				else if (tag != s->mode.tag &&
					 w->text.layout_callbacks)
				    {
					XeTextLayoutCallbackData data;

					data.reason = XeCR_LAYOUT_CHANGE_TAG;
					data.old = tag;
					data.new = s->mode.tag;
					data.context =
						_XeMakeLayoutContext(w, h);
					data.feed = _XeTextFeedContent;
					lay = s;
					addLayout = TRUE;
					current.tag = tag = s->mode.tag;
					XtCallCallbackList
						((Widget)w,
						 w->text.layout_callbacks,
						 (XtPointer *)&data);
					_XeTextEndContent
						(data.context, (Region)0);
					if (s == firstSnip)
						firstSnip = *h;
					if (s == beginLine)
						beginLine = *h;
					s = *h;
					break;
				    }
				else
				    {
					current.tag = tag = s->mode.tag;
					break;
				    }
			    }
			if (w->text.layout_callbacks && s == beginLine)
			    {
				XeTextLayoutCallbackData data;

				data.reason = XeCR_LAYOUT_BEGIN_LINE;
				data.old = tag;
				data.context = _XeMakeLayoutContext(w, h);
				data.feed = _XeTextFeedContent;
				if (!addLayout)
				    {
					lay = s;
					addLayout = TRUE;
				    }
				data.new = lay ? lay->mode.tag : 0;
				XtCallCallbackList
					((Widget)w, w->text.layout_callbacks,
					 (XtPointer *)&data);
				_XeTextEndContent(data.context, (Region)0);
				s = beginLine = firstSnip = *h;
			    }
			if (lay == s)
			    {
				/* Layout Insertion terminated */
				addLayout = FALSE;
				lay = NULL;
				tag = current.tag = s->mode.tag;
			    }
			if (expect == expect_end)
			    {
				ExpectEntry *e = expect_point;
				int old = expect_end - expect_point;
				int new = old + XtNumber(expect_default);
				if (expect_point == expect_default)
				    {
					expect_point = (ExpectEntry *)
						XtMalloc(sizeof(ExpectEntry) *
							 new);
					memcpy((void *)expect_point, (void *)e,
					       sizeof(ExpectEntry) * old);
				    }
				else
					expect_point = (ExpectEntry *)
						XtRealloc((char *)expect_point,
							  sizeof(ExpectEntry) *
							  new);
				expect = &expect_point[old];
				expect_end = &expect_point[new];
			    }
			expect->s = s;
			expect->x = s->x;
			expect->y = s->y;
			expect->xWidth = s->xWidth;
			expect->ascent = s->ascent;
			expect->descent = s->descent;
			expect++;
			s->x = x;
			s->y = y;
			/*
			** All snips before the current one (s) will
			** fit into the line. If the current one
			** represents a valid breaking point, remember
			** this point by saving the line state into
			** 'lastSpace' *before* adding any effects
			** from this snip. (used by TEXT sequence only)
			*/
			if (s->space)
			    {
				lastSpace = current;
				space = s;
				current.spaces += 1;
			    }
			ComputeWidth(w, s);
			/*
			** Do the line splitting only if new layout is
			** requested.
			*/
			switch (sequence)
			    {
			    default:
				if (layout && current.wNow>lineLength && space)
				    {
					s = SplitLine(w, &lastSpace, space);
					tag = lastSpace.tag;
				    }
				else if (s->endseq)
					s = Position_TEXT(w,&current, s, s);
				break;
			    case Sequence_ITEM:
				if (s->endseq)
				    {
					s = Position_ITEM(w, &current, s);
					itemization = TRUE;
				    }
				break;
			    case Sequence_STAB:
				if (s->endseq)
					s = Position_STAB(w,&current,s,tabref);
				break;
			    }
			y = s->y - yShift;
			x = s->x + s->xWidth;
			h = &s->next;
			p = s;
		    } while (!s->endseq);
		if (HasEndLine(s))
		    {
			y += CloseLineBox(w, s);
			lookfor = UpdateExpose(expose, expect_point, expect,
					       s->next, lookfor);
			if (!lookfor)
				goto clean_up;
			beginLine = NULL;
		    }
		if (y > yHighMark)
			yHighMark = y;
		if (x > xHighMark)
			xHighMark = x;
		lineOffset = x - w->text.indentation;
		sequence = Sequence_TEXT;
		if (s->tab)
			if (s->tabref == Snip_NEXTTABSTOP)
				lineOffset = HorizontalTab(w, x)
					- w->text.indentation;
			else
			    {
				sequence = Sequence_STAB;
				tabref = s->tabref;
			    }
		firstSeq = HasEndParagraph(s);
		if (firstSeq || itemization)
		    {
			lineOffset = w->text.first_line_offset;
			itemization = FALSE;
		    }
		else if (HasEndLine(s))
			lineOffset = 0;
		lineLength = wArea - lineOffset - w->text.indentation;
	    }
	if (p && HasEndLine(p))
	    {
		/*
		** The content terminated by End of Line. This means that
		** the content includes one *EMPTY* line at end. (This is
		** ODA character content interpretation).
		*/
		y += previousExtent + lineSpacing;
	    }
	else if (beginLine)
	    {
		y += CloseLineBox(w, p);
		(void)UpdateExpose(expose, expect_point, expect, p->next, 0);
		y += previousExtent;
	    }
	else
		y += lineSpacing; /* Empty Content! */
	if (y > yHighMark)
		yHighMark = y;
	w->text.h = yHighMark;
	w->text.w = xHighMark;
    clean_up:
	if (expose && expose->width > 0)
		XUnionRectWithRegion(expose, expose_region, expose_region);
	if (expect_point != expect_default)
		XtFree((char *)expect_point);
	return 0;
    }
