/*
** TextImport.c
**
**	Convert text content from interchange format into internal
**	representation.
*/
#if SYSV_INCLUDES
#	include <memory.h>	/* memset() et al. */
#endif
#if ANSI_INCLUDES
#	include <stdlib.h>
#endif

#include <string.h>
#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xew/BasicP.h>
#include <X11/Xew/TextP.h>
#include "ccfilter.h"
#include "TextImport.h"

/*
** Incomplete Gx mapping implementation, not final!!! --msa
*/
typedef struct GxEntry
    {
	unsigned char *tt;	/* Conversion Table */
	char *charset;		/* Display Character Set Identifier */
	int (*move)(		/* Move data to Snip function */
#if NeedFunctionPrototypes
		    XeTextInsertContext,
		    int,unsigned char *,unsigned char *,char *
#endif
		);
    } GxEntry;

/*
** XeTextInsertContext
**	The Context indentifies an insert position in the text. There can
**	be multiple context's open simultaneously (but only one for each
**	widget at time).
**
** *WARNING*
**	Utmost care must be taken that the 'last' pointer in Context
**	always points to *EDITABLE* Snip or is NULL. (This is because
**	non-Editable format elements may get deleted without warning
**	from the Snip chains).
*/
typedef struct XeTextInsertContextRec
    {
	SnipMode mode;		/* Current Mode */
	SnipMode lock;		/* Locked Modes (mask bits) */
	int locked;		/* True if modes locked (InitialState) */
	Snip **list;		/* Current Snip List Head Pointer */
	Snip *last;		/* Current Last Snip */
	int pendingAccent;	/* Special for ISO 6937 conversion */
	int warnings;		/* Non-zero, if warnings issued */
	XeTextWidget w;		/* Text Widget being processed */
	Snip *first;		/* First snip referencing current data */
	int size;		/* Total available bytes in data */
	int used;		/* Used bytes in data */
	int fontinfo;		/* Font information of the current data */
	SnipData *head;		/* Current data being collected */
	ccf_Context ccf;	/* Text filtering context */
	GxEntry csmap[4];	/* Character set mapping state */
	Region expose;		/* Bounding region of changed area. This
				** only covers the changes into snips that
				** already had layout=TRUE and which were
				** changed. It cannot cover *new* inserted
				** content. Empty, until width is non-zero.
				*/
    } XeTextInsertContextRec, Context;

/*
** IgnoreFormattedContent
**	is TRUE, when inserting Formatted content and not specially
**	allowed by locking into it (only done in _XeMakeLayoutContext)
*/
#define IgnoreFormattedContent(cx) \
	(((cx)->mode.bits & Content_MASK) == Content_FORMATTED && \
	 !((cx)->lock.bits & Content_MASK))

/*
** Some standard Character Set Registry & Encodings
*/
static char iso8859_1[] = "iso8859-1";
static char iso8859_2[] = "iso8859-2";
static char iso8859_3[] = "iso8859-3";
static char iso8859_4[] = "iso8859-4";
static char iso8859_5[] = "iso8859-5";
static char iso8859_6[] = "iso8859-6";
static char iso8859_7[] = "iso8859-7";
static char iso8859_8[] = "iso8859-8";
static char iso8859_9[] = "iso8859-9";
static char jisx0201_1976_0[] = "jisx0201.1976-0";
static char gb2312_1980_0[] = "gb2312-1980-0";
static char jisx0208_1983_0[] = "jisx0208.1983-0";
static char ksc5601_1987_0[] = "ksc5601.1987-0";

/*
** ISO 2022 Code Switching export support
** ======================================
**	NOTE 1:	This section does not really belong into TextImport.
**		It is located here temporarily, until a new separate
**		character set module (TextCharset) gets created.
**
**	NOTE 2:	This support is also temporary kludge to get something
**		out. It blatantly assumes that the ISO 8859-1 set is the
**		default.
*/
typedef struct CodeSwitch 
    {
	XeCharsetIdentifier charset;
	XeCharsetDesignation select[4];
    } CodeSwitch;

/*
** NOTE:The following is supposed to describe exactly what is
**	actually stored into Snip data, and *NOT* what is the
**	actual desired exported ISO 2022 configuration.
*/
static CodeSwitch iso2022_select[] = 
    {
	/*			GL (7BIT)	GR (8BIT)		*/

	{iso8859_1,		{0x42, 		0x41|XeCharset_96,	0},},
	{iso8859_2,		{0x42, 		0x42|XeCharset_96,	0},},
	{iso8859_3,		{0x42, 		0x43|XeCharset_96,	0},},
	{iso8859_4,		{0x42, 		0x44|XeCharset_96,	0},},
	{iso8859_5,		{0x42, 		0x4c|XeCharset_96,	0},},
	{iso8859_6,		{0x42,		0x47|XeCharset_96,	0},},
	{iso8859_7,		{0x42, 		0x46|XeCharset_96,	0},},
	{iso8859_8,		{0x42, 		0x48|XeCharset_96,	0},},
	{iso8859_9,		{0x42, 		0x4d|XeCharset_96,	0},},
	{jisx0201_1976_0,	{0x4a,					0},},
	{gb2312_1980_0,		{0x41|XeCharset_M,			0},},
	{jisx0208_1983_0,	{0x42|XeCharset_M,			0},},
	{ksc5601_1987_0,	{0x43|XeCharset_M,			0},},
	{NULL, 			{					0},},
    };

XeCharsetDesignation *XeTextCharsetSelect(charset)
XeCharsetIdentifier charset;
    {
	CodeSwitch *p;

	for (p = iso2022_select; p->charset; p++)
		if (p->charset == charset)
			break;
	return &p->select[0];
    }

/*
** XeTextCharsetDefault
**	Return some default charset identifier.
*/
XeCharsetIdentifier XeTextCharsetDefault()
    {
	return iso8859_1;
    }

/*
** END OF ISO 2022 Export Support code
*/

/*
** NOTE:
**	This "hard coded" code conversion is intended to be a temporary
**	solution. The right way is to provide some means of application
**	to specify the conversion tables. That solution needs some
**	design (conversion must be able to handle those accents).
*/
#define NOTA 0	/* Not available */
#define NONS 1	/* Non Spacing character */
#define NBSB 2	/* No-Break Space */
/*
** T.51 supplementary set to ISO 8859/1 conversion table
*/
static unsigned char T51supplementary_ISO8859_1[128] =
    {
    /*   0/8   1/9   2/a   3/b   4/c   5/d   6/e   7/f */
	NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA,	/* never used */
	NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA,	/* never used */
	NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA,	/* never used */
	NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA,	/* never used */
	0xa0, 0xa1, 0xa2, 0xa3, NOTA, 0xa5, NOTA, 0xa7,	/* 20..27 */
	0xa4, 0x27, 0x22, 0xab, NOTA, NOTA, NOTA, NOTA,	/* 28..2f */
	0xb0, 0xb1, 0xb2, 0xb3, 0xd7, 0xb5, 0xb6, 0xb7,	/* 30..37 */
	0xf7, 0x27, 0x22, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,	/* 38..3f */
	NOTA, NONS, NONS, NONS, NONS, NONS, NONS, NONS,	/* 40..47 */
        NONS, NOTA, NONS, NONS, NONS, NONS, NONS, NONS,	/* 48..4f */
	NOTA, 0xb9, 0xae, 0xa9, NOTA, NOTA, 0xac, 0xa6,	/* 50..57 */
        NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA,	/* 58..5f */
	NOTA, 0xc6, 0xd0, 0xaa, NOTA, NOTA, NOTA, NOTA,	/* 60..67 */
	NOTA, 0xd8, NOTA, 0xba, 0xde, NOTA, NOTA, NOTA,	/* 68..6f */
	NOTA, 0xe6, NOTA, 0xf0, NOTA, NOTA, NOTA, NOTA,	/* 70..77 */
	NOTA, 0xf8, NOTA, 0xdf, 0xfe, NOTA, NOTA, 0xad,	/* 78..7f */
    };

#if 0
/*
** NULL Table, just provided for easy template when creating new tables
*/
static unsigned char Null_Conversion[128] =
    {
    /*   0/8   1/9   2/a   3/b   4/c   5/d   6/e   7/f */
	NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA,	/* never used */
	NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA,	/* never used */
	NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA,	/* never used */
	NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA,	/* never used */
	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,	/* 20..27 */
	0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,	/* 28..2f */
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,	/* 30..37 */
	0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,	/* 38..3f */
	0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,	/* 40..47 */
        0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,	/* 48..4f */
	0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,	/* 50..57 */
        0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,	/* 58..5f */
	0x60, 0xc1, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,	/* 60..67 */
	0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,	/* 68..6f */
	0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,	/* 70..77 */
	0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,	/* 78..7f */
    };
#endif

/*
** ISO 646, Swedish version SEN 850200 to ISO 8859-1 convertion
*/
static unsigned char ISO646_SE_to_ISO8859_1[128] =
    {
    /*   0/8   1/9   2/a   3/b   4/c   5/d   6/e   7/f */
	NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA,	/* never used */
	NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA,	/* never used */
	NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA,	/* never used */
	NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA,	/* never used */
	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,	/* 20..27 */
	0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,	/* 28..2f */
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,	/* 30..37 */
	0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,	/* 38..3f */
	0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,	/* 40..47 */
        0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,	/* 48..4f */
	0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,	/* 50..57 */
        0x58, 0x59, 0x5a, 0xc4, 0xd6, 0xc5, 0x5e, 0x5f,	/* 58..5f */
	0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,	/* 60..67 */
	0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,	/* 68..6f */
	0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,	/* 70..77 */
	0x78, 0x79, 0x7a, 0xe4, 0xf6, 0xe5, 0x7e, 0x7f,	/* 78..7f */
    };

/*
** ISO 646, Swedish version for names to ISO 8859-1 convertion
*/
static unsigned char ISO646_SE_NAMES_to_ISO8859_1[128] =
    {
    /*   0/8   1/9   2/a   3/b   4/c   5/d   6/e   7/f */
	NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA,	/* never used */
	NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA,	/* never used */
	NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA,	/* never used */
	NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA, NOTA,	/* never used */
	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,	/* 20..27 */
	0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,	/* 28..2f */
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,	/* 30..37 */
	0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,	/* 38..3f */
	0xc9, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,	/* 40..47 */
        0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,	/* 48..4f */
	0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,	/* 50..57 */
        0x58, 0x59, 0x5a, 0xc4, 0xd6, 0xc5, 0xdc, 0x5f,	/* 58..5f */
	0xe9, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,	/* 60..67 */
	0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,	/* 68..6f */
	0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,	/* 70..77 */
	0x78, 0x79, 0x7a, 0xe4, 0xf6, 0xe5, 0xfc, 0x7f,	/* 78..7f */
    };

/*
** Map T.51 Non-spacing character combined with the following code
** into ISO 8859-1 codes. This table consists of pairs of integers,
** the first representing the combined code of diatrical mark and the
** following letter, the latter is the corresponding ISO 8859-1 code.
*/
#define COMBINE(x,y)	(((int)x)*256+(y))

typedef enum
    {
	T51_Grave = 0x41,
	T51_Acute = 0x42,
	T51_Circumflex = 0x43,
	T51_Tilde = 0x44,
	T51_Macron = 0x45,
	T51_Breve = 0x46,
	T51_Dot = 0x47,
	T51_Umlaut = 0x48,
	T51_Ring = 0x4a,
	T51_Cedilla = 0x4b,
	T51_DoubleAcute = 0x4d,
	T51_Ogonek = 0x4e,
	T51_Caron = 0x4f
    } T51_Diatrical;

static int T51diatrical_ISO8859_1[] =
    {
	COMBINE(T51_Grave,	0x41), 0xc0,	/* A */
	COMBINE(T51_Acute,	0x41), 0xc1,
	COMBINE(T51_Circumflex,	0x41), 0xc2,
	COMBINE(T51_Tilde,	0x41), 0xc3,
	COMBINE(T51_Umlaut,	0x41), 0xc4,
	COMBINE(T51_Ring,	0x41), 0xc5,

	COMBINE(T51_Cedilla,	0x43), 0xc7,	/* C */

	COMBINE(T51_Grave,	0x45), 0xc8,	/* E */
	COMBINE(T51_Acute,	0x45), 0xc9,
	COMBINE(T51_Circumflex,	0x45), 0xca,
	COMBINE(T51_Umlaut,	0x45), 0xcb,

	COMBINE(T51_Grave,	0x49), 0xcc,	/* I */
	COMBINE(T51_Acute,	0x49), 0xcd,
	COMBINE(T51_Circumflex,	0x49), 0xce,
	COMBINE(T51_Umlaut,	0x49), 0xcf,

	COMBINE(T51_Tilde,	0x4e), 0xd1,	/* N */

	COMBINE(T51_Grave,	0x4f), 0xd2,	/* O */
	COMBINE(T51_Acute,	0x4f), 0xd3,
	COMBINE(T51_Circumflex,	0x4f), 0xd4,
	COMBINE(T51_Tilde,	0x4f), 0xd5,
	COMBINE(T51_Umlaut,	0x4f), 0xd6,

	COMBINE(T51_Grave,	0x55), 0xd9,	/* U */
	COMBINE(T51_Acute,	0x55), 0xda,
	COMBINE(T51_Circumflex,	0x55), 0xdb,
	COMBINE(T51_Umlaut,	0x55), 0xdc,

	COMBINE(T51_Acute,	0x59), 0xdd,	/* Y */
	
	COMBINE(T51_Grave,	0x61), 0xe0,	/* a */
	COMBINE(T51_Acute,	0x61), 0xe1,
	COMBINE(T51_Circumflex,	0x61), 0xe2,
	COMBINE(T51_Tilde,	0x61), 0xe3,
	COMBINE(T51_Umlaut,	0x61), 0xe4,
	COMBINE(T51_Ring,	0x61), 0xe5,

	COMBINE(T51_Cedilla,	0x63), 0xe7,	/* c */

	COMBINE(T51_Grave,	0x65), 0xe8,	/* e */
	COMBINE(T51_Acute,	0x65), 0xe9,
	COMBINE(T51_Circumflex,	0x65), 0xea,
	COMBINE(T51_Umlaut,	0x65), 0xeb,

	COMBINE(T51_Grave,	0x69), 0xec,	/* i */
	COMBINE(T51_Acute,	0x69), 0xed,
	COMBINE(T51_Circumflex,	0x69), 0xee,
	COMBINE(T51_Umlaut,	0x69), 0xef,

	COMBINE(T51_Tilde,	0x6e), 0xf1,	/* n */

	COMBINE(T51_Grave,	0x6f), 0xf2,	/* o */
	COMBINE(T51_Acute,	0x6f), 0xf3,
	COMBINE(T51_Circumflex,	0x6f), 0xf4,
	COMBINE(T51_Tilde,	0x6f), 0xf5,
	COMBINE(T51_Umlaut,	0x6f), 0xf6,

	COMBINE(T51_Grave,	0x75), 0xf9,	/* u */
	COMBINE(T51_Acute,	0x75), 0xfa,
	COMBINE(T51_Circumflex,	0x75), 0xfb,
	COMBINE(T51_Umlaut,	0x75), 0xfc,

	COMBINE(T51_Acute,	0x79), 0xfd,	/* y */
	COMBINE(T51_Umlaut,	0x79), 0xff,

	COMBINE(T51_Cedilla,	0x20), 0xb8,	/* SPACE */
	COMBINE(T51_Grave,	0x20), 0x60,
	COMBINE(T51_Acute,	0x20), 0xb4,
	COMBINE(T51_Macron,	0x20), 0xaf,
	COMBINE(T51_Umlaut,	0x20), 0xa8,
	COMBINE(T51_Circumflex,	0x20), 0x5e,
	COMBINE(T51_Tilde,	0x20), 0x7e,

	0,0
    };

/*
** DerefDataHeader
**	Subtract references to the data header. Returns NULL, if the
**	header has been released and the header pointer otherwise.
**
**	Can be called with NULL pointer, returns NULL.
*/
#if NeedFunctionPrototypes
static SnipData *DerefDataHeader(SnipData *);
#endif
static SnipData *DerefDataHeader(head)
SnipData *head;
    {
	if (head == NULL)
		return NULL;
	else if (head->refs == 0)
		XtError("TextImport: SnipData refs would be negative!");
	else if (--head->refs == 0)
		XtFree((void *)head);
	else
		return head;
	return NULL;
    }

/*
** MallocContext, FreeContext
**	The layout process may need temporary insertion contexts quite
**	frequently. On the other hand it is probably that there never
**	is too many simultaneous active insertion contexts open. Thus,
**	context blocks are never really released, but are cached and
**	reused instead.
*/
typedef struct EmptyContext
    {
	struct EmptyContext *next;
    } EmptyContext;

static EmptyContext *context_list;	/* Empty context blocks */

#if NeedFunctionPrototypes
static void FreeContext(Context *);
#endif
static void FreeContext(cx)
Context *cx;
    {
	DerefDataHeader(cx->head);
	ccf_Close(cx->ccf);
	if (cx->expose)
		XDestroyRegion(cx->expose);
	/*
	** Cache the block for future use
	*/
	((EmptyContext *)cx)->next = context_list;
	context_list = (EmptyContext *)cx;
    }

#if NeedFunctionPrototypes
static Context *MallocContext(void);
#endif
static Context *MallocContext()
    {
	static Context initial_context;
	Context *cx;

	if (context_list)
	    {
		cx = (Context *)context_list;
		context_list = context_list->next;
	    }
	else
		cx = (Context *)XtMalloc(sizeof(Context));
	*cx = initial_context;
	return cx;
    }

/*
** _XeInsertSnip
**	Insert a new empty Snip before the Snip pointed by 'h'.
**
**	Return a pointer to the newly created Snip.
*/
Snip *_XeInsertSnip(h)
Snip **h;
    {
	register Snip *t;

	if ((t = (Snip *)XtCalloc(1, sizeof(Snip))) == NULL)
		return NULL; /* FATAL ERROR! */
	if ((t->next = *h) != NULL)
		(*h)->back = &t->next;
	*h = t;
	t->back = h;
	t->head = NULL;
	t->data = NULL;
	return t;
    }
/*
** _XeDeleteSnip
**	Delete Snip pointed by 'h'
*/
void _XeDeleteSnip(h)
Snip **h;
    {
	register Snip *s = *h;

	if (s != NULL)
	    {
		if ((*h = s->next) != NULL)
			s->next->back = h;
		DerefDataHeader(s->head);
		XtFree((void *)s);
	    }
    }
/*
** AppendSnip
**	Add a new Snip to the Block of the current context. This function
**	is internal to this module and used only in the initial content
**	streams transformation.
*/
#if NeedFunctionPrototypes
static Snip *AppendSnip(Context *);
#endif
static Snip *AppendSnip(cx)
Context *cx;
    {
	register Snip *t;

	t = _XeInsertSnip(cx->last ? &cx->last->next : cx->list);
	if (t != NULL)
		cx->last = t;
	t->mode = cx->mode;
	return t;
    }

/*
** FlushAppend
**	Release extra buffer space
*/
#define HEAD_MIN_SIZE 512 /* Minimum initial data allocation */
#define HEAD_MIN_FUZZ 32  /* Don't bother to shrink if less than this free */

#if NeedFunctionPrototypes
static void FlushAppend(Context *);
#endif
static void FlushAppend(cx)
Context *cx;
    {
	SnipData *head, *h;
	int size = XtOffsetOf(SnipData, s[0]) + cx->used;
	Snip *s = cx->first;

	h = DerefDataHeader(cx->head);
	if (s && h &&
	    cx->used + HEAD_MIN_FUZZ <  cx->size &&
	    (head = (SnipData *)XtRealloc((char *)h, size)) != h)
	    {
		/*
		** This branch should not be taken usually. It means
		** that realloc moved the block, even if we just
		** truncated the allocation! (This happens if you run purify,
		** also seems to occur with Apollo sr10.2 --msa)
		*/
		for (; s; s = s->next)
			if (s->head == NULL)
				continue;
			else if (s->head == h)
			    {
				int i;
				i = (int)(s->data - &h->s[0]);
				s->data = &head->s[i];
				s->head = head;
			    }
			else
				break;
	    }
	cx->first = NULL;
	cx->used = 0;
	cx->size = 0;
	cx->head = NULL;
    }

/*
** BeginAppend
**	First check if a sequence of characters can be appended to the
**	current run of data being collected into context block.
**	This can be done, if
**		- context data has sufficient available space and
**		- character set is same and
**		- font selection is the same.
**	If not, then flush out the current collected data and start
**	a new empty collection (which must be sufficiently large for
**	the offered sequence).
**
**	Second, check if a new Snip block needs to be allocated or if
**	the current one can be used (If flush was required, a new Snip
**	is always needed).
*/
#define FONTINFO(m) ((m) & (Weight_MASK | Italicized_MASK | Font_MASK))

#if NeedFunctionPrototypes
static Snip *BeginAppend(Context *, int, char *, int);
#endif
static Snip *BeginAppend(cx, n, cset, bytes)
Context *cx;	/* Insert/Edit context */
int n;		/* Number of *CHARACTERS* to insert */
char *cset;	/* Character identifier */
int bytes;	/* Number of bytes in one *CHARACTER* */
    {
	Snip *t;
	int fontinfo = FONTINFO(cx->mode.bits);

	n *= bytes;	/* Convert n into number of bytes */
	if (cx->used + n > cx->size ||
	    cx->head->character_set != cset ||
	    fontinfo != cx->fontinfo)
	    {
		FlushAppend(cx);
		if (cx->size < n)
		    {
			/* Need to reallocate larger head block */
			DerefDataHeader(cx->head);
			cx->size = n > HEAD_MIN_SIZE ? n : HEAD_MIN_SIZE;
			cx->head = (SnipData *)
				XtCalloc(1, XtOffsetOf(SnipData, s[0])
					 + cx->size);
			cx->head->refs = 1; /* cx ref is counted ! */
		    }	
		cx->fontinfo = fontinfo;
		cx->head->bytes = bytes;
		cx->head->font = NULL;
		cx->head->character_set = cset;
		t = cx->first = AppendSnip(cx);
		t->head = cx->head;
		t->data = &cx->head->s[cx->used];
		cx->head->refs += 1;
	    }
	else if ((t = cx->last) == NULL ||
		 t->head != cx->head ||
		 t->space || t->endseq || t->brk ||
		 t->mode.bits != cx->mode.bits ||
		 t->mode.tag != cx->mode.tag)
	    {
		t = AppendSnip(cx);
		t->head = cx->head;
		t->data = &cx->head->s[cx->used];
		cx->head->refs += 1;
	    }
	cx->used += n;
	t->valid = False;
	return t;
    }
#undef FONTINFO

/*
** Convert_to_ISO8859_1
**	Converts codes into ISO 8859-1 codes using convertion table.
**
**	*NOTE*	This function is primarily for converting ISO 6937
**		Supplementary set and especially the floating accent
**		kludge works only for that (the T51* tables are written
**		for that set only).
*/
#if NeedFunctionPrototypes
static int Convert_to_ISO8859_1
	(Context *, int, unsigned char *, unsigned char *, char *);
#endif

static int Convert_to_ISO8859_1(cx, n, s, tt, cset)
Context *cx;
int n;
unsigned char *s;
unsigned char *tt;
char *cset;
    {
	int i, *q;
	Snip *t;
	SnipData *head;
	register unsigned char *p;
	register int c;

	if (n == 0)
		return 0;
	if (cx == NULL || (t = BeginAppend(cx, n, cset, 1)) == NULL)
		return 1;
	/*
	** Copy string with simplistic conversion into SnipData structure.
	** This assumes converted string doesn't need more space than the
	** original.
	*/
	head = t->head;
	p = (unsigned char *)t->data + (head->bytes * t->length);
	for ( ; n > 0; ++s, --n )
	    {
		c = *s & 0x7f;
		if (tt[c] == 0x20)
		    {
			if (!t->space && (t->endseq || t->length > 0))
			    {
				t = AppendSnip(cx);
				t->data = (char *)p;
				t->head = head;
				head->refs += 1;
			    }
			t->space = TRUE;
			t->valid = FALSE;
			t->length += 1;
			*p++ = *s;
			continue;
		    }
		else if (t->space)
		    {
			t = AppendSnip(cx);
			t->data = (char *)p;
			t->head = head;
			head->refs += 1;
		    }
		if (tt[c] == NONS)
		    {
			cx->pendingAccent = c;
			continue;
		    }
		if (cx->pendingAccent != 0)
		    {
			i = COMBINE(cx->pendingAccent, tt[c]);
			for (q = T51diatrical_ISO8859_1; *q; ++q)
				if (*q++ == i)
				    {
					*p++ = *q;
					t->length += 1;
					break;
				    }
			cx->pendingAccent = 0;
			continue;
		    }
		*p++ = tt[c];
		t->length += 1;
	    }
	return 0;
    }

/*
** Copy2_GL
**	Copy string of two byte characters into SnipData.
**	Conversion table is ignored. Each character is a pair of
**	two 7 bit bytes (Multibyte character in GL).
*/
#if NeedFunctionPrototypes
static int Copy2_GL
	(Context *, int, unsigned char *, unsigned char *, char *);
#endif

static int Copy2_GL(cx, n, s, tt, cset)
Context *cx;
int n;
unsigned char *s;
unsigned char *tt;
char *cset;
    {

	Snip *t;
	SnipData *head;
	register unsigned char *p;

	if (n == 0)
		return 0;
	if (cx == NULL || (t = BeginAppend(cx, n, cset, 2)) == NULL)
		return 1;
	head = t->head;
	p = (unsigned char *)t->data + (head->bytes * t->length);
	cx->pendingAccent = 0; /* No support for this here! --msa */
	for ( ; n > 0; n -= 1)
	    {
		*p++ = (*s++) & 0x7f;
		*p++ = (*s++) & 0x7f;
		t->length += 1;
	    }
	return 0;
    }
/*
** Copy1_GL
**	Copy string of one byte characters into SnipData.
**	Conversion table is ignored. Each character is a one
**	byte of 7 bits (single byte character in GL).
**
**	Separate contigous 0x20 runs into snips with "space"
**	property as true.
*/
#if NeedFunctionPrototypes
static int Copy1_GL
	(Context *, int, unsigned char *, unsigned char *, char *);
#endif
static int Copy1_GL(cx, n, s, tt, cset)
Context *cx;
int n;
unsigned char *s;
unsigned char *tt;
char *cset;
    {
	Snip *t;
	SnipData *head;
	register unsigned char c, *p;

	if (n == 0)
		return 0;
	if (cx == NULL || (t = BeginAppend(cx, n, cset, 1)) == NULL)
		return 1;
	head = t->head;
	p = (unsigned char *)t->data + (head->bytes * t->length);
	/*
	** This special kludge/hook is required to map ISO 6937 floating
	** accents into plain ISO 8859-1 characters. Not a pretty sight,
	** there must be a neater way... --msa
	** If accent combination is not found, the accent is ignored.
	*/
	if (cset == iso8859_1 && /* Should probably test for all iso8859's
				  ..and in real word should expect floating
				  accents from one character set combined
				  with character from another.. ugh... --msa */
	    cx->pendingAccent != 0)
	    {
		int *q, i;

		c = *s & 0x7f;
		i = COMBINE(cx->pendingAccent, c);
		for (q = T51diatrical_ISO8859_1; *q; ++q)
			if (*q++ == i)
			    {
				*p++ = *q;
				t->length += 1;
				n -= 1;
				s += 1;
				break;
			    }
	    }
	cx->pendingAccent = 0;
	for ( ; n > 0; ++s, --n )
	    {
		c = *s & 0x7f;
		if (c == 0x20)
		    {
			if (!t->space && (t->endseq || t->length > 0))
			    {
				t = AppendSnip(cx);
				t->data = (char *)p;
				t->head = head;
				head->refs += 1;
			    }
			t->valid = FALSE;
			t->space = TRUE;
		    }
		else if (t->space)
		    {
			t = AppendSnip(cx);
			t->data = (char *)p;

			t->head = head;
			head->refs += 1;
		    }
		*p++ = c;
		t->length += 1;
	    }
	return 0;
    }
/*
** Copy1_GR
**      Copy string of one byte characters into SnipData.
**      Conversion table is ignored. In each byte unconditionally
**	set the 8th bit (Single byte characters from GR).
*/
#if NeedFunctionPrototypes
static int Copy1_GR
	(Context *, int, unsigned char *, unsigned char *, char *);
#endif
static int Copy1_GR(cx, n, s, tt, cset)
Context *cx;
int n;
unsigned char *s;
unsigned char *tt;
char *cset;
    {
        Snip *t;
        SnipData *head;
        register unsigned char *p;

        if (n == 0)
                return 0;
	if (cx == NULL || (t = BeginAppend(cx, n, cset, 1)) == NULL)
		return 1;
	head = t->head;
	p = (unsigned char *)t->data + (head->bytes * t->length);
	cx->pendingAccent = 0; /* No support for this here! --msa */
        for ( ; n > 0; n -= 1 )
            {
                *p++ = (*s++) | 0x80;
                t->length += 1;
            }
        return 0;
    }

/*
** ccfilter callback functions
*/
typedef enum
    {
	BS  = 8,	/* Backspace */
	HT  = 9,	/* Horizontal tabulation */
	LF  = 10,	/* Line Feed */
	FF  = 12,	/* Form Feed */
	CR  = 13,	/* Carriage Return */
	SUB = 26,	/* Substitute */
	RS  = 0x1E	/* Record Separator, used as Paragraph end (not ODA) */
    } ODA_C0;

typedef enum
    {
	BPH = 2,
	NBH = 3,
	PLD = 11,
	PLU = 12,
	SOS = 24,
	ST  = 28
    } ODA_C1;

/*
** ODA_CSI
**	Final character in a simple CSI sequence
*/
typedef enum
    {
	HPB = 0x6A, /* Character Position Backward */
	HPR = 0x61, /* Character Position Relative */
	PTX = 0x5C, /* Parallel Texts */
	SGR = 0x6D, /* Select Graphic Rendition */
	SRS = 0x5B, /* Start Reverse String */
	VPB = 0x6B, /* Line Position Backward */
	VPR = 0x65  /* Line Position Relative */
    } ODA_CSI;

/*
** ODA_CSI_SPACE
**	Final characters in a CSI sequence with one intermediate
**	SPACE.
*/
typedef enum
    {
	GCC = 0x5F, /* Graphic Character Composition */
	IGS = 0x4D, /* Identify Graphic Subrepertoire */
	JFY = 0x46, /* Justify */
	QUAD= 0x48, /* Quad */
	SCS = 0x67, /* Set Character Spacing */
	SHS = 0x4B, /* Select Character Spacing */
	SACS= 0x5C, /* Set Additional Character Spacing */
	SLS = 0x68, /* Set Line Spacing */
	SRCS= 0x66, /* Set Reduced Character Spacing */
	SSW = 0x5B, /* Set SPACE Width */
	STAB= 0x5E, /* Selective Tabulation */
	SVS = 0x4C  /* Select Line Spacing */
    } ODA_CSI_SPACE;

static void init_GxMap(cx)
Context *cx;
    {
	/* Initial state, pure hack! --msa */

	cx->csmap[0].tt = NULL;
	cx->csmap[1].tt = NULL;
	cx->csmap[2].tt = NULL;
	cx->csmap[3].tt = NULL;
	cx->csmap[0].charset = iso8859_1;
	cx->csmap[1].charset = iso8859_1;
	cx->csmap[2].charset = iso8859_1;
	cx->csmap[3].charset = iso8859_1;
	cx->csmap[0].move = Copy1_GL;
	cx->csmap[1].move = Copy1_GL;
	cx->csmap[2].move = Copy1_GR;
	cx->csmap[3].move = Copy1_GR;
    }

static void Gn_Feed(client_data, Gn, s, n)
void *client_data;
ccf_Gs Gn;
char *s;
int n;
    {
	Context *cx = (Context *)client_data;

	if (IgnoreFormattedContent(cx))
		return;	/* Ignore Formatted content for now */
	if ((int)Gn < XtNumber(cx->csmap))
		(*cx->csmap[(int)Gn].move)
			(cx, n, (unsigned char *)s,
			 cx->csmap[(int)Gn].tt,
			 cx->csmap[(int)Gn].charset);
	else
	    {
		if (cx->warnings)
			return;
		cx->warnings += 1;
		XeWidgetWarningMsg
			((Widget)cx->w, "noteTextImportnonISO2022",
			 "Coding outside ISO 2022 not implemented",
			 (String *)NULL, 0);
	    }
    }

static void Do_C0(cx, c)
Context *cx;
int c;
    {
	register Snip *t;

	if (IgnoreFormattedContent(cx))
		return;	/* Completely Ignore Formatted content for now! */
	switch (c)
	    {
	    case HT:
		/*
		** HT is not allowed in ODA, but to support traditional
		** text files, convert this into a special STAB
		** sequence. Each HT will end up into differenct
		** sequence.
		*/
		t = AppendSnip(cx);
		t->tab = True;
		t->tabref = Snip_NEXTTABSTOP;
		t->endseq = Snip_End;
		break;	
	    case CR:
		/*
		** In processable format a CR can only appear as an
		** ITEM terminator or immediately preceding an LF.
		*/
		t = cx->last;
		if (t == NULL || t->endseq ||
		    ((t->mode.bits ^ cx->mode.bits) & Content_MASK))
			t = AppendSnip(cx);
		else
			t->valid = False;
		t->endseq = Snip_End;
		break;
	    case FF:
		/* Treat FF as LF for now */
	    case LF:
		/*
		** Each LF will terminate the current line. If line is
		** empty or the previous character already terminated
		** the line, an empty sequence will be created. An LF
		** will reset current sequence to TEXT.
		*/
		t = cx->last;
		if (t == NULL || HasEndLine(t) || t->space ||
		    ((t->mode.bits ^ cx->mode.bits) & Content_MASK))
			t = AppendSnip(cx);
		else
			t->valid = False;
		t->endseq = Snip_EndLine;
		break;
	    case RS:
		/*
		** RS is used experimentally to mark end of paragraph.
		** (Some other coding may get used in future.)
		*/
		t = cx->last;
		if (t == NULL || HasEndLine(t) || t->space ||
		    ((t->mode.bits ^ cx->mode.bits) & Content_MASK))
			t = AppendSnip(cx);
		else
			t->valid = False;
		t->endseq = Snip_EndParagraph;
		break;
	    default:
		break;
	    }
    }

static void Do_C1(cx, c)
Context *cx;
int c;
    {
	if (IgnoreFormattedContent(cx))
	    {
		if (c == ST && !(cx->lock.bits & Content_MASK))
		    {
			cx->mode.bits &= ~Content_MASK;
			cx->mode.bits |= Content_PROCESSABLE;
		    }
		return;	/* Completely Ignore Formatted content for now! */
	    }
	switch (c)
	    {
	    case PLU:
		/*
		** A PLU can only follow PLD, or it can start Partial
		** line up from neutral state (NONE). If preceded by
		** PLU, this one is ignored (PLU state does not change).
		*/
		if (!cx->locked)
			cx->lock.bits |= PartialLine_MASK;
		else if (cx->lock.bits & PartialLine_MASK)
			break;	/* Locked Mode, do not change */
		if (cx->mode.bits & PartialLine_DOWN)
			cx->mode.bits &= ~PartialLine_DOWN;
		else
			cx->mode.bits |= PartialLine_UP;
		break;
	    case PLD:
		/*
		** a PLD can only follow PLU, or it can start Partial
		** line down from neutral state (NONE). If preceded by
		** PLD, this one is ignored (PLD state does not change).
		*/
		if (!cx->locked)
			cx->lock.bits |= PartialLine_MASK;
		else if (cx->lock.bits & PartialLine_MASK)
			break;	/* Locked Mode, do not change */
		if (cx->mode.bits & PartialLine_UP)
			cx->mode.bits &= ~PartialLine_UP;
		else
			cx->mode.bits |= PartialLine_DOWN;
		break;
	    case BPH:
		/*
		** BPH is sensible only if there is a Snip to attach
		** (ignored otherwise)
		*/
		if (cx->last != NULL)
		    {
			cx->last->brk = Break_BPH;
			cx->last->valid = False;
		    }
		break;
	    case NBH:
		/*
		** NBH is sensible only if there is a Snip to attach
		** (ignored otherwise)
		*/
		if (cx->last != NULL)
		    {
			cx->last->brk = Break_NBH;
			cx->last->valid = False;
		    }
		break;
	    case SOS:
		if (!(cx->lock.bits & Content_MASK))
		    {
			cx->mode.bits &= ~Content_MASK;
			cx->mode.bits |= Content_FORMATTED;
		    }
		break;
	    default:
		break;
	    }
    }

static void Cn_Feed(client_data, Cn, c)
void *client_data;
ccf_Cs Cn;
int c;
    {
	if (client_data == NULL)
		return; /* FATAL ERROR */
	else if (Cn == ccf_C0)
		Do_C0((Context *)client_data, c);
	else if (Cn == ccf_C1)
		Do_C1((Context *)client_data, c);
    }

static void ESC_Feed(client_data, I, n, F)
void *client_data;
char *I;
int n, F;
    {
	/* All Ignored */
    }

/*
** SelectGraphicRendition
**	Process single parameter value of SGR or "graphic rendition".
**
**	(ISO 8613-6  11.1.8 Table 4)
*/
#define RENDITION_MASK (Weight_MASK | Italicized_MASK | Underline_MASK | \
			Blinking_MASK | ImageInversion_MASK | \
			CrossedOut_MASK | Font_MASK | \
			Foreground_MASK | Background_MASK | \
			Framing_MASK | Overlined_MASK)

static void SelectGraphicRendition(cx, r)
Context *cx;
int r;
    {

	unsigned int mask, value;

	switch (r)
	    {
	    case 0: /* reset to default rendition */
		/* Unfortunately, mode contains flags that make it not
		   possible to just do mode = init. This is indication of
		   incorrect grouping.. should be fixed --msa */
		mask = RENDITION_MASK;
		if (cx->locked)
			mask &= ~cx->lock.bits;
		value = 0;
		break;
	    case 1: /* bold or increased density */
		mask = Weight_MASK;
		value = Weight_BOLD;
		break;
	    case 2: /* faint or decreased density */
		mask = Weight_MASK;
		value = Weight_FAINT;
		break;
	    case 3: /* italicized */
		mask = value = Italicized_MASK;
		break;
	    case 4: /* underlined */
		mask = Underline_MASK;
		value = Underline_SINGLE;
		break;
	    case 5: /* slowly blinking */
		mask = Blinking_MASK;
		value = Blinking_SLOWLY;
		break;
	    case 6: /* rapidly blinking */
		mask = Blinking_MASK;
		value = Blinking_RAPIDLY;
		break;
	    case 7: /* negative image */
		mask = value = ImageInversion_MASK;
		break;
	    case 9: /* crossed-out */
		mask = value = CrossedOut_MASK;
		break;
	    case 10: /* primary (default) font */
	    case 11: /* first alternative font */
	    case 12: /* second alternative font */
	    case 13: /* third alternative font */
	    case 14: /* fourth alternative font */
	    case 15: /* fifth alternative font */
	    case 16: /* sixth alternative font */
	    case 17: /* seventh alternative font */
	    case 18: /* eigth alternative font */
	    case 19: /* ninth alternative font */
		mask = Font_MASK;
		value = Font_VALUE(r - 10);
		break;
	    case 21: /* doubly underlined */
		mask = Underline_MASK;
		value = Underline_DOUBLE;
		break;
	    case 22: /* normal intensity (neither bold nor faint) */
		mask = Weight_MASK;
		value = 0;
		break;
	    case 23: /* not italicized */
		mask = Italicized_MASK;
		value = 0;
		break;
	    case 24: /* not underlined (neither singly nor doubly) */
		mask = Underline_MASK;
		value = 0;
		break;
	    case 25: /* steady (not blinking) */
		mask = Blinking_MASK;
		value = 0;
		break;
	    case 26: /* variable spacing */
		mask = value = 0;
		break;
	    case 27: /* positive image */
		mask = ImageInversion_MASK;
		value = 0;
		break;
	    case 29: /* not crossed-out */
		mask = CrossedOut_MASK;
		value = 0;
		break;
	    case 30: /* ISO 6429, black display */
	    case 31: /* ISO 6429, red display */
	    case 32: /* ISO 6429, green display */
	    case 33: /* ISO 6429, yellow display */
	    case 34: /* ISO 6429, blue display */
	    case 35: /* ISO 6429, magenta display */
	    case 36: /* ISO 6429, cyan display */
	    case 37: /* ISO 6429, white display */
		mask = Foreground_MASK;
		value = Foreground_VALUE(r - 29);
		break;
	    case 39: /* ISO 6429, default display color */
		mask = Foreground_MASK;
		value = 0;
		break;
	    case 40: /* ISO 6429, black background */
	    case 41: /* ISO 6429, red background */
	    case 42: /* ISO 6429, green background */
	    case 43: /* ISO 6429, yellow background */
	    case 44: /* ISO 6429, blue background */
	    case 45: /* ISO 6429, magenta background */
	    case 46: /* ISO 6429, cyan background */
	    case 47: /* ISO 6429, white background */
		mask = Background_MASK;
		value = Background_VALUE(r - 39);
		break;	
	    case 49: /* ISO 6429, default background color */
		mask = Background_MASK;
		value = 0;
		break;
	    case 50: /* not variable spacing */
		mask = value = 0;
		break;
	    case 51: /* ISO 6429, framed */
		mask = Framing_MASK;
		value = Framing_FRAMED;
		break;
	    case 52: /* ISO 6429, encircled */
		mask = Framing_MASK;
		value = Framing_ENCIRCLED;
		break;
	    case 53: /* ISO 6429, overlined */
		mask = value = Overlined_MASK;
		break;
	    case 54: /* ISO 6429, not framed, not encircled */
		mask = Framing_MASK;
		value = 0;
		break;
	    case 55: /* ISO 6429, not overlined */
		mask = Overlined_MASK;
		value = 0;
		break;
	    default:
		mask = value = 0;
		break;
	    }
	if (!cx->locked)
		cx->lock.bits |= mask;
	else if (cx->lock.bits & mask)
		return;	/* Locked mode, do not change. Ignore control */
	cx->mode .bits &= ~mask;
	cx->mode.bits |= value;
    }



static void Do_CSI(cx, n, p, F)
Context *cx;
int n, *p, F;
    {
	switch (F)
	    {
	    case VPB:
		/* Not implemented */
		break;
	    case VPR:
		/* Not implemented */
		break;
	    case SGR:
		if (n == 0)
			SelectGraphicRendition(cx, 0);
		else while (--n >= 0)
			SelectGraphicRendition(cx, *p++);
		break;
	    default:
		break;
	    }
    }

static void Do_CSI1(cx, n, p, F, I)
Context *cx;
int n, *p, F, I;
    {
	register Snip *t;
	XeAlignment align;

	/*
	** In ODA, only intermediate used is SPACE--ignore all others.
	*/
	if (I != 0x20)
		return;
	switch (F)
	    {
	    case JFY: /* ISO 6429 : 1988 */
		if (!cx->locked)
			cx->lock.bits |= Justify_MASK;
		else if (cx->lock.bits & Justify_MASK)
			break;	/* Locked Mode, do not change */
		cx->mode.bits &= ~Justify_MASK;
		align = XeAlignment_NONE;
		while (--n >= 0)
			switch (*p++)
			    {
			    default:
			    case 0: /* no justification (use NONE default) */
			    case 1: /* word fill (ignored) */
			    case 4: /* hyphenation (ignored) */
			    case 8: /* italian hyphenation (ignored) */
				break;
			    case 2: /* word space */
			    case 3: /* letter space (assume word space) */
				align = XeAlignment_JUSTIFIED;
				break;
			    case 5: /* flush to line home position */
				align = XeAlignment_START;
				break;
			    case 6: /* centre between home and limit */
				align = XeAlignment_CENTER;
				break;
			    case 7: /* flush to line limit position */
				align = XeAlignment_END;
				break;
			    }
		cx->mode.bits |= Justify_VALUE(align);
		break;
	    case QUAD: /* ISO 6429 : 1988 */
		/*
		** Multiple QUAD on same spot just overwrite the previous
		** value.
		*/
		t = cx->last;
		if (t == NULL || !t->endseq)
		    {
			t = AppendSnip(cx);
			t->endseq = Snip_End;
		    }
		switch (n > 0 ? *p : 0)
		    {
		    default:
		    case 0: /* Start */
		    case 1: /* Start with leader fill */
			align = XeAlignment_START;
			break;
		    case 2: /* Center */
		    case 3: /* Center with leader fill */
			align = XeAlignment_CENTER;
			break;
		    case 4: /* End */
		    case 5: /* End with leader fill */
			align = XeAlignment_END;
			break;
		    case 6: /* Justify */
			align = XeAlignment_JUSTIFIED;
			break;
		    }
		t->quad = align;
		break;
	    case SLS:
		/* Fix later --msa */
		break;
	    case SVS:
		/* Fix later --msa */
		break;
	    case STAB:
		t = cx->last;
		if (t && !t->endseq)
			t->valid = False;
		else
			t = AppendSnip(cx);
		t->endseq = Snip_End;
		t->tab = True;
		t->tabref = *p;
		break;
	    default:
		break;
	    }
    }

static void CSI_Feed(client_data, P, p, n, I, nI, F)
void *client_data;
int P;
int *p;
int n;
char *I;
int nI;
int F;
    {
	Context *cx = (Context *)client_data;

	if (P)
		return;	/* No use for Private sequences now */
	if (cx == NULL)
		return;
	if (IgnoreFormattedContent(cx))
		return;	/* Completely Ignore Formatted content for now! */
	if (nI == 0)
		Do_CSI(cx, n, p, F);
	else if (nI == 1)
		Do_CSI1(cx, n, p, F, *I & 0xFF);
    }

/*
** 94-Character Graphic Character Sets (single byte)
*/
static int SingleByteSet94(F, g)
int F; GxEntry *g;
    {
	/*
	** Note: The following switch is mostly informative, the
	** only really correct mapping supported is from Reg. No.6
	*/
	switch (F)
	    {
	    case 0x40:	/* 2:IRV of ISO 646: 1983 */
	    case 0x41:	/* 4:ISO 646, British Version BSI 4730 */
	    case 0x42:	/* 6:ISO 646, USA Version X3.4 - 1968 */
		/*
		** In addition to USA version, let those two "near USA"
		** versions drop into this same set by default...
		*/
		g->tt = NULL;
		g->charset = iso8859_1;
		g->move = Copy1_GL;
		return 1;

	    case 0x43:	/* 8-1:NATS, Primary Set for Finland and Sweden */
	    case 0x45:	/* 9-1:NATS, Primary Set for Denmark and Norway */
	    case 0x60:	/* 60:ISO 646, Norwegian version NS 4551 */
	    case 0x47:	/* 10:ISO 646, Swedish version SEN 850200 */
		/*
		** In addition to Swedish version, let Finnish, Danish
		** and Norwegian fall into here too. This is right for
		** Finnish, but Norwegian and Danish should be handled
		** by different table.
		*/
		g->tt = ISO646_SE_to_ISO8859_1;
		g->charset = iso8859_1;
		g->move = Convert_to_ISO8859_1;
		return 1;
	    case 0x48:	/* 11:ISO 646, Swedish version for names */
		g->tt = ISO646_SE_NAMES_to_ISO8859_1;
		g->charset = iso8859_1;
		g->move = Convert_to_ISO8859_1;
		return 1;
	    case 0x59:	/* 15:ISO 646, Version for Italian, ECMA (Olivetti) */
	    case 0x4c:	/* 16:ISO 646, Version for Portuguese, -"- */
	    case 0x5a:	/* 17:ISO 646, Version for Spanish, -"- */
	    case 0x4b:	/* 21:ISO 646, German Version DIN 66083 */
	    case 0x52:	/* 25:ISO 646, French Version, NF Z 62010-1973 */
	    case 0x66:	/* 69:ISO 646, French Version NF Z 62010-1982 */
	    case 0x67:	/* 84:ISO 646, Version for Portuguese, ECMA (IBM) */
	    case 0x68:	/* 85:ISO 646, Version for the Spanish Languages,
			   ECMA (IBM) */
	    case 0x69:	/* 86:ISO 646, Hungarian Version, Hungarian Standard
			   7795/3 */
		/*
		** This is not correct. Each of the national variants of
		** should be run through a simple conversion that substitutes
		** the few varying characters in 646 with the correct
		** national characters from the 8859-1 set. Not very
		** difficult. Will do later --msa
		** This is only right for the USA version X3.4 - 1968.
		*/
		g->tt = NULL;
		g->charset = iso8859_1;
		g->move = Copy1_GL;
		return 1;

	    case 0x4a:	/* 14:ISO 646, Japanese version for Roman Chars */
		g->tt = NULL;
		g->charset = jisx0201_1976_0;
		g->move = Copy1_GL;
		return 1;
	    case 0x49:	/* 13:Katakana Character Set JIS C6220-1969 */
		g->tt = NULL;
		g->charset = jisx0201_1976_0;
		g->move = Copy1_GR;
		return 1;
	    case 0x44:	/* 8-2:NATS, Secondary Set for Finland and Sweden */
	    case 0x46:	/* 9-2:NATS, Secondary Set for Denmark and Norway */
	    case 0x5b:	/* 18:Character Set for Greek, -"- */
	    case 0x5c:	/* 19:Latin-Greek Character Set, -"- */
	    case 0x55:	/* 27:Latin-Greek Character Set, ECMA */
	    case 0x58:	/* 31:Greek Character Set for Bibliography,ISO 5428 */
	    case 0x4e:	/* 37:Basic Cyrillic Character Set, ISO 5427 */
	    case 0x4f:	/* 38:Extended Graphic Character Set Bibliography,
			   DIN 31624 */
	    case 0x4d:	/* 39:Character Set for African Languages,
			   DIN 31625 and ISO 6438 */
	    case 0x56:	/* 47:Character Set for Viewdata and Teletext (UK) */
	    case 0x57:	/* 49:INIS, Sub-set of the IRV */
	    case 0x5d:	/* 50:INIS, Non-standard Extension of Reg. 49 */
	    case 0x5e:	/* 51:INIS, Cyrillic Extension of Reg. 49 */
	    case 0x50:	/* 53:Extended Graphic Character Set for Bibliography
			   ISO 5416-1981 */
	    case 0x51:	/* 54:Extension of the Cyrillic Character Set of
			   Reg 37, ISO 5427-1981 */
	    case 0x53:	/* 55:Greek Character Set for Bibliography,
			   ISO 5428-1980 */
	    case 0x54:	/* 57:Coded Character Set for Information Interchange,
			   Chinese Standard GB 1988-80 */
	    case 0x5f:	/* 59:Arabic Character Set CODAR-U IERA (Morocco) */
	    case 0x61:	/* 61:Norwegian Character Set, Version 2, NS 4551 */
	    case 0x65:	/* 68:APL Character Set, Canadian APL Working Group */
	    case 0x62:	/* 70:Supplementary Set for Videotext, CCITT */
	    case 0x63:	/* 71:Second Supplementary Set for Videotex (mosaic),
			   CCITT */
	    case 0x64:	/* 72:Third Supplementary Set for Videotex (mosaic),
			   CCITT */
	    case 0x6a:	/* 88:Greek Character Set, ELOT,
			   Hellenic Organization for Standardization */
	    case 0x6b:	/* 89:7-bit Arabic Code for Inromation Interchange,
			   Arab Standard ASMO-449, ISO 9036 */
	    case 0x6c:	/* 90:Supplementary Set for Use with
			   Registration No.2 */
	    case 0x6d:	/* 91:Japanese OCR-A graphic set JIS C6229-1984 */
	    case 0x6e:	/* 92:Japanese OCR-B graphic set JIS C6229-1984 */
	    case 0x6f:	/* 93:Japanese OCR-B, Additional Graphic Set,
			   JIS C6229-1984 */
	    case 0x70:	/* 94:Japanese Basic Hand-printed Graphic Set for
			   OCR JIS C6229-1984 */
	    case 0x71:	/* 95:Japanese Additional Handprinted Graphic
			   Character Set for OCR JIS C6229-1984 */
	    case 0x72:	/* 96:Katakana hand-printed Graphic Character Set for
			   OCR JIS C6229-1984 */
	    case 0x73:	/* 98:E13B Graphic Character Set Japanese National
			   Committee for ISO/TC97/SC2 */
	    case 0x74:	/* 99:Supplementary Set of Graphic Characters for
			   Viedeotext and Teletext ANSI and CSA */
	    case 0x75:	/* 102:Teletex Primary Set of Graphic Characters,
			   CCITT T.61 */
	    case 0x76:	/* 103:Teletex Supplementary Set of Graphic characters
			   CCITT T.61 */
	    case 0x77:	/* 121:Alternate Primary Graphic Set No.1,
			   CSA Standard Z 243.4-1985 */
	    case 0x78:	/* 122:Alternate Primary Graphic Set No.2,
			   CSA Standard Z 243.4-1985 */
	    case 0x7c:	/* 128:Supplementary Set of graphic Characters for
			   CCITT T.101, Data Syntax III */
	    case 0x79:	/* 137:Mosaic-1 Set of Data Syntax I of CCITT T.101 */
	    case 0x7a:	/* 141:Serbocroatian and Slovenian Latin Alphabet */
	    case 0x7b:	/* 146:Serbocroatian Cyrillic Alphabet */
	    case 0x7d:	/* 147:Macedonian Cyrillic Aplhabet */
	    default:
		break;
	    }
	/*
	** As a default, use iso 8859-1 left side for all unsupported.
	*/
	g->tt = NULL;
	g->charset = iso8859_1;
	g->move = Copy1_GL;
	return 1;
    }


/*
** 96-Character Graphic Character Sets (single byte)
*/
static int SingleByteSet96(F, g)
int F; GxEntry *g;
    {
	/*
	** Note: The following switch is mostly informative, the
	** only really correct mapping supported is from Reg. No. 156
	** and from Reg. No. 100 (Null Mapping).
	*/
	switch (F)
	    {
	    case 0x41:	/* 100: Right-hand Part of Latin Alphabet No.1,
			   ISO 8859/1, ECMA-94 */
		g->tt = NULL;
		g->charset = iso8859_1;
		g->move = Copy1_GR;
		return 1;

	    case 0x42:	/* 101: Right-hand Part of Latin Alphabet No.2,
			   ISO 8859/2, ECMA-94 */
		g->tt = NULL;
		g->charset = iso8859_2;
		g->move = Copy1_GR;
		return 1;
	    case 0x43:	/* 109: Right-hand Part of Latin Alphabet No.3,
			   ISO 8859/3, ECMA-94 */
		g->tt = NULL;
		g->charset = iso8859_3;
		g->move = Copy1_GR;
		return 1;
	    case 0x44:	/* 110: Right-hand Part of Latin Alphabet No.4,
			   ISO 8859/4, ECMA-94 */
		g->tt = NULL;
		g->charset = iso8859_4;
		g->move = Copy1_GR;
		return 1;
	    case 0x46:	/* 126: Right-hand part of the Latin/Greek Alphabet,
			   Standard ELOT 928, ECMA-118, ISO  DIS 8859/7 */
		g->tt = NULL;
		g->charset = iso8859_7;
		g->move = Copy1_GR;
		return 1;
	    case 0x47:	/* 127: Right-hand part of the Latin/Arabic Alphabet,
			   ECMA-114, ISO 8859/6 */
		g->tt = NULL;
		g->charset = iso8859_6;
		g->move = Copy1_GR;
		return 1;
	    case 0x48:	/* 138: Latin/Hebrew Alphabet, Standard ECMA-121 */
		g->tt = NULL;
		g->charset = iso8859_8;
		g->move = Copy1_GR;
		return 1;
	    case 0x4c:	/* 144: Cyrillic Part of the Latin/Cyrillic Alphabet */
		g->tt = NULL;
		g->charset = iso8859_5;
		g->move = Copy1_GR;
		return 1;
	    case 0x4d:	/* 148: Right-hand part fo Latin Alphabet No.5 */
		g->tt = NULL;
		g->charset = iso8859_9;
		g->move = Copy1_GR;
		return 1;
	    case 0x52:	/* 156: Supplementary Set of ISO/IEC 6937 */
		g->tt = T51supplementary_ISO8859_1; 
		g->charset = iso8859_1;
		g->move = Convert_to_ISO8859_1;
		return 1;
/***** Unmapped ones ******/
	    case 0x7d:	/* 129: Supplementary Set of Mosaic Characters for
			   CCITT T.101, Data Syntax III */
	    case 0x49:	/* 139: Right-hand part for Czechoslovak Standard
			   CSN 369103 */
	    case 0x4a:	/* 142: Supplementary Set of Latin Alphabetic and
			   non-alphabetic Graphic characters */
	    case 0x4b:	/* 143: Technical Set */
	    case 0x4e:	/* 152: Residual Characters from ISO 6937-2: 1983 */
	    case 0x4f:	/* 153: Basic Cyrillic Character Set for 8-bit codes */
	    case 0x50:	/* 154: Supplementary set for Latin Alphabets No.1,
			   No.2 and No.5 */
	    case 0x51:	/* 155: Basic Box Drawings Set */
	    case 0x40:	/* 111: Right-hand Part of the Latin/Cyrillic Alphabet
			   ECMA-113 (Version of June 1986) */
	    case 0x45:	/* 123: General Purpose Supplementary Graphic Set
			   CSA Standard Z 243..4-1985 */
	    default:
		break;
	    }
	g->tt = NULL;
	g->charset = iso8859_1;
	g->move = Copy1_GL;
	return 1;
    }
/*
** 94-Character Graphic Character Sets (multibyte)
*/
static int MultiByteSet94(F, g)
int F; GxEntry *g;
    {
	switch (F)
	    {
	    case 0x41:	/* 58:Chinese Character Set,
			   Chinese Standard GB 2312-80 */
		g->tt = NULL;
		g->charset = gb2312_1980_0;
		g->move = Copy2_GL;
		return 2;
	    case 0x40:	/* 42:Japanese Character Set JISC C 6226-1978 */
			/* Fall through to next, not really right? --msa */
	    case 0x42:	/* 87:Japanese Character Set JIS C 6226-1983 */
		g->tt = NULL;
		g->charset = jisx0208_1983_0;
		g->move = Copy2_GL;
		return 2;
	    case 0x43:	/* 149:Korean Graphic Character Set,
			   Korean Standard KSC 5601-1987 */
		g->tt = NULL;
		g->charset = ksc5601_1987_0;
		g->move = Copy2_GL;
		return 2;
	    default:
		return 2;
	    }
	return 2;
    }
	
/*
** 96-Character Graphic Character Sets (multibyte)
*/
static int MultiByteSet96(F, g)
int F; GxEntry *g;
    {
	return 2;
    }

static int Designate_G(client_data, Gn, s96, M, F)
void *client_data;
ccf_Gs Gn;
int s96, M, F;
    {
	Context *cx = (Context *)client_data;
	GxEntry *g;

	if (Gn == ccf_CC)
		return 1;	/* Escaped out of ISO 2022 */
	if ((unsigned)Gn >= XtNumber(cx->csmap))
		return 0;
	g = &cx->csmap[(int)Gn];
	if (s96)
		return M ? MultiByteSet96(F, g) : SingleByteSet96(F, g);
	else
		return M ? MultiByteSet94(F, g) : SingleByteSet94(F, g);
    }

static int Designate_C(client_data, Cn, F)
void *client_data;
ccf_Cs Cn;
int F;
    {
	return 0;
    }

/*
** _XeTextBeginContent
**	Start processing a new unit of character content. Returns a
**	pointer to a new "context" block, and this context is implicitly
**	set after this call.
*/
XeTextInsertContext _XeTextBeginContent(w)
XeTextWidget w;
    {
	register Context *cx = MallocContext();

	if (cx == NULL)
		return NULL;
	cx->w = w;
	cx->last = NULL;
	cx->list = &w->text.first;
	/*
	** Delete previous internal structures, if any present
	*/
	while (w->text.first)
		_XeDeleteSnip(&w->text.first);
	init_GxMap(cx);
	cx->ccf = ccf_Open(cx, Gn_Feed, Cn_Feed, ESC_Feed, CSI_Feed,
			   Designate_G, Designate_C);
	if (w->text.initial_state != NULL)
		_XeTextFeedContent(cx, w->text.initial_state, 
				  strlen(w->text.initial_state));
	if (w->text.graphic_rendition != NULL)
	    {
		_XeTextFeedContent(cx, "\233", 1);
		_XeTextFeedContent(cx, w->text.graphic_rendition,
				   strlen(w->text.graphic_rendition));
		_XeTextFeedContent(cx, "\155", 1); /* ASCII "m" */
	    }
	cx->locked = TRUE;
	/*
	** Using the resource graphicRendition as initial state may not
	** be quite correct, and might be needed to be made an option --msa
	*/
	return (void *)cx;
    }

/*
** AdjustLastPointer
**	Adjust last pointer backward to the first editable Snip
**	starting backwards from the given Snip (t).
*/
#if NeedFunctionPrototypes
static void AdjustLastPointer(Context *, Snip *);
#endif
static void AdjustLastPointer(cx, r)
Context *cx;
Snip *r;
    {
	if (r)
	    {
		r->valid = False;
		do
		    {
			if (r->back == cx->list)
			    {
				r = NULL;
				break;
			    }
			r = PreviousSnip(r);
			r->valid = False;
		    } while (!IsEditableContent(r->mode.bits));
	    }
	cx->last = r;
    }

/*
** UpdateExposeArea
**	Extend expose area from the Snip
*/
#if NeedFunctionPrototypes
static void UpdateExposeArea(Context *, int, int, int, int);
#endif
static void UpdateExposeArea(cx, x, y, width, height)
Context *cx;
int x, y, width, height;
    {
	XRectangle rect;

	if (width == 0 || height == 0)
		return;
	if (!cx->expose)
		cx->expose = XCreateRegion();
	rect.x = x;
	rect.y = y;
	rect.width = width;
	rect.height = height;
	XUnionRectWithRegion(&rect, cx->expose, cx->expose);
    }


/*
** _XeTextInsertContent
**	Start inserting character content after the specified snip. Returns
**	pointer to a new "context" block, and this context is implicitly
**	set after this call.
**
**	*NOTE*	after should point to a Snip representing original text,
**		not anything generated by layout.
**
**	New insertion point is opened in Snip specified by the position
**	parameter and length of the Snip:
**
**	position > length,	insert totally after the snip (e.g, if
**				the snip is ending a line, the new data
**				will be inserted after the line end).
**	position == length,	insert at end of snip, but before the
**				possible end of line mark.
**	position < length,	insert *before* the character specified
**				by the position (0==first character and
**				<length-1>==last character).
**
**	'mode' parameter controls the initial mode, before applying the
**	value from the initialState resource:
**
**	mode == 0,	initial mode for the snip is EMPTY (all zeroes).
**	mode > 0,	if the insert point at end of the Snip, the mode
**			will be taken from the next editable Snip after
**			this Snip. (If insert point is inside the snip,
**			the mode is taken from the snip).
**	mode < 0,	Use the mode from insert point Snip (after).
*/
XeTextInsertContext _XeTextInsertContent(w, after, position, mode)
XeTextWidget w;	/* Text Widget that is being modified */
Snip *after;	/* Snip in into/after which insertion occurs. */
int position;	/* Character position within Snip */
int mode;	/* Flag to control initial mode selection */
    {
	static SnipMode init_mode;

	register Context *cx;
	Snip *t;
	int n;
	char *s;

	if (after == NULL)
		return _XeTextBeginContent(w); /* Bad call, easy way out.. */
	if ((cx = MallocContext()) == NULL)
		return NULL;
	if (position < 0)
	    {
		position = 0; /* Erroneus call.. just kludge over it */
		XeWidgetWarningMsg((Widget)w, "bugTextImportnegPosition",
				   "Negative position offset (BUG)",
				   (String *)NULL, 0);
	    }
	if (!IsEditableContent(after->mode.bits))
	    {
		XeWidgetWarningMsg((Widget)w, "bugTextImportInsertFormatted",
				   "Inserting at formatted content (BUG)",
				   (String *)NULL, 0);
		return NULL;
	    }
	cx->w = w;
	cx->list = &w->text.first;
	cx->last = after;
	cx->mode = after->mode;
	/*
	** Split content of snip After into two parts
	*/
	n = after->length - position;
	s = after->data; /* Needs to be saved, can be changed below */
	if (n > 0 || (n == 0 && HasEndPosition(after)))
	    {
		/*
		** Inserting within current Snip. The snip needs to be split.
		** This is achieved by changing the 'after' to reflect only
		** the tail part and creating a new empty snip in front of it.
		** After which the the new data will be inserted.
		*/
		AdjustLastPointer(cx, after);
		if (after->layout)
		    {
			UpdateExposeArea(cx, after->x,
					 after->y - after->ascent,
					 after->xWidth,
					 after->ascent + after->descent);
			after->layout = after->valid = FALSE;
		    }
		if (after->data && n > 0)
		    {
			after->data += after->head->bytes * position;
			after->length = n;
		    }
		else
		    {
			after->data = NULL;
			after->length = 0;
		    }
		/*
		** If there was data in 'after' *before* the insert point,
		** it now must be saved away.
		*/
		if (position > 0 && s &&
		    (t = BeginAppend(cx, position,
				     after->head->character_set,
				     after->head->bytes)) != NULL)
		    {
			memcpy((void *)t->data,(void *)s,
			       position * after->head->bytes);
			t->length += position;
		    }
	    }
	else if (mode < 0)
	    {
		/*
		** Inserting totally after the current Snip and request is
		** to use mode from following Snip. Do that.
		*/
		t = after;
		while ((t = t->next) != NULL)
			if (IsEditableContent(t->mode.bits))
			    {
				cx->mode = t->mode;
				break;
			    }
	    }
	if (mode == 0)
		cx->mode = init_mode;
	init_GxMap(cx);
	cx->ccf = ccf_Open(cx, Gn_Feed, Cn_Feed, ESC_Feed, CSI_Feed,
			   Designate_G, Designate_C);
	if (w->text.initial_state != NULL)
		_XeTextFeedContent(cx, w->text.initial_state, 
				  strlen(w->text.initial_state));
	cx->locked = TRUE;
	return cx;
    }
/*
** _XeMakeLayoutContext
**	Setup an insert context for inserting layout content.
*/
XeTextInsertContext _XeMakeLayoutContext(w, head)
XeTextWidget w;	/* Text Widget in question */
Snip **head;	/* Insertion point */
    {
	register Context *cx;
	if ((cx = MallocContext()) == NULL)
		return NULL;
	cx->last = NULL;
	cx->list = head;
	cx->w = w;
	init_GxMap(cx);
	cx->ccf = ccf_Open(cx, Gn_Feed, Cn_Feed, ESC_Feed, CSI_Feed,
			   Designate_G, Designate_C);
	cx->mode.bits = Content_FORMATTED;
	cx->lock.bits = Content_MASK;
	cx->locked = TRUE;
	return cx;
    }

/*
** SplitSnip
**	Split Snip t into two at offset, and return the pointer to the
**	*NEW* snip, which represents the fisrt part of the split.
**
**	*** No checks is made. Function assumes the offset <= length.
*/
#if NeedFunctionPrototypes
static Snip *SplitSnip(Context *, Snip *, int);
#endif
static Snip *SplitSnip(cx, t, offset)
Context *cx;
Snip *t;
int offset;
    {
	register Snip *r;
	/*
	** If the Snip has layout information, update expose area
	*/
	if (t->layout)
	    {
		UpdateExposeArea(cx, t->x, t->y - t->ascent,
				 t->xWidth, t->ascent + t->descent);
		t->layout = t->valid = FALSE;
	    }
	/*
	** The new snip is inserted before t, and the leading
	** data characters are moved to this snip.
	*/
	r = _XeInsertSnip(t->back);
	r->mode = t->mode;
	r->length = offset;
	if ((r->data = t->data) != NULL)
	    {
		t->data += (t->head->bytes * offset);
		r->head = t->head;
		r->head->refs += 1;
	    }
	t->length -= offset;
	return r;
    }

/*
** _XeTextDeleteContent
**	Cut abs(amount) number of *characters* from the current insertion
**	point. If amount > 0, then the characters are cut forward from the
**	insertion point. If amount < 0, then the characters are cut backward
**	from the instert position.
**
**	The function returns a pointer to the cut content that has
**	been deleted from the main chain. This function does not release
**	the space!
*/
Snip *_XeTextDeleteContent(context, amount)
XeTextInsertContext context;
long amount;
    {
	Snip *t, *r;
	Snip **h;
	int offset, vlength;
	Context *cx = (Context *)context;

	if (cx == NULL)
		return NULL;
	h = cx->last ? &cx->last->next : cx->list;
	if (amount > 0)
	    {
		/*
		** Delete/Cut forward. This operation never affects the
		** cx->last pointer or the snip pointed by it (other
		** than possibly changing the following Snip after it).
		*/
		for (t = *h; ; t = t->next)
		    {
			if (t == NULL)
			    {
				/* return all from *h to end, cut chain at h */
				if ((t = *h) != NULL)
				    {
					t->back = NULL;
					*h = NULL;
				    }	
				goto update_expose;
			    }
			if (IsEditableContent(t->mode.bits))
			    {
				amount -= (vlength = VirtualLength(t));
				if (amount < 0)
					break;
			    }
		    }
		offset = amount + vlength;
		if (offset > 0 && offset < vlength)
			(void)SplitSnip(cx, t, offset);
		/*
		** Return all from *h up to t, excluding t. t is always
		** non-NULL at this point.
		*/
		r = *h;
		if (r == t)
		    {
			XeWidgetWarningMsg((Widget)cx->w,
					   "bugTextImportEmptyFwdDelete",
					   "Empty Fwd Delete (BUG)",
					   (String *)NULL, 0);
			return NULL;
		    }
		*t->back = NULL;	/* Terminate chain (r) */
		r->back = NULL;
		*h = t;			/* Link chain (t) to h */
		t->back = h;
		/*
		** Need to clear the valid flag of the next snip to FALSE
		** so that the layout process knows from where to start
		** processing in partial layout. (Just deleting snips
		** does not necessarily leave any mark on the chain).
		*/
		t->valid = FALSE;	/* Hint for layout process */
		t = r;
	    }
	else if (amount < 0)
	    {
		/*
		** Delete/Cut backward. This operation will always affect
		** cx->last.
		*/
		if ((t = cx->last) == NULL)
			return NULL; /* Can't delete backwards from begin */
		if (t->next)
			t->next->valid = FALSE; /* Hint for layout process */
		for (;; t = PreviousSnip(t))
		    {
			if (IsEditableContent(t->mode.bits))
			    {
				amount += (vlength = VirtualLength(t));
				if (amount >= 0)
					break;
			    }
			else if (t->back == cx->list)
			    {
				/*
				** Reached beginning of the chain. Return
				** all from beginning up to *h.
				*/
				FlushAppend(cx);
				if ((*t->back = *h) != NULL)
					(*h)->back = t->back;
				t->back = NULL;
				*h = NULL;
				cx->last = NULL;
				goto update_expose;
			    }
		    }
		offset = amount;
		/*
		** The following is a special kludge to deal the deletion
		** of the last character of super/subscripts --msa
		*/
		if (offset > 0 && offset == t->length &&
		    HasEndPosition(t) && !t->endseq)
			offset -= 1;
		if (offset > 0 && offset < vlength)
			(void)SplitSnip(cx, t, offset);
		/* return all from t to *h */
		FlushAppend(cx);
		AdjustLastPointer(cx, t);
		if ((*t->back = *h) != NULL)
			(*h)->back = t->back;
		*h = NULL;
		t->back = NULL;
	    }
	else
		t = NULL;
    update_expose:
	for (r = t; r; r = r->next)
		if (r->layout)
		    {
			UpdateExposeArea(cx, r->x, r->y - r->ascent,
					 r->xWidth, r->ascent + r->descent);
			r->layout = r->valid = FALSE;
		    }
	return t;
    }

/*
** _XeTextInsertPoint
**	Return the pointer to the last Snip of the current context.
**	Additionally, return the current application tag value.
*/
Snip *_XeTextInsertPoint(cx, expose)
XeTextInsertContext cx;
Region expose;
    {
	if (expose && cx->expose)
	    {
		/* I do hope the XUnionRegion works with dest being
		   one of the sources too!!! --msa */
		XUnionRegion(cx->expose, expose, expose);
		XDestroyRegion(cx->expose);
		cx->expose = 0;
	    }
	return cx->last;
    }

/*
** _XeTextFeedContent
*/
int _XeTextFeedContent(cx, s, n)
XeTextInsertContext cx;
char *s;
long n;
    {
	if (cx == NULL)
		return 0;
	else
		return ccf_Feed(cx->ccf, s, n);
    }

/*
** _XeTextEndContent
**	"Close" a block of character content from further input. Return
**	the pointer to the last Snip of the context.
*/
Snip *_XeTextEndContent(cx, expose)
XeTextInsertContext cx;
Region expose;
    {
	Snip *last = cx->last;
	if (expose && cx->expose)
		/* I do hope the XUnionRegion works with dest being
		   one of the sources too!!! --msa */
		XUnionRegion(cx->expose, expose, expose);
	FlushAppend(cx);
	if (last && !last->endseq && last->length == 0)
	    {
		/*
		** Avoid leaving empty snips, delete this. Empty Snip
		** should be restricted to transient existance while
		** insert context is open.
		*/
		Snip **h = last->back;
		AdjustLastPointer(cx, last);
		last = cx->last;
		_XeDeleteSnip(h);
	    }
	FreeContext(cx);
	return last;
    }

/*
** _XeTextSetTag
**	set the application tag value in current insert context to
**	the specified value. Return the old value.
*/
XeTextTag _XeTextSetTag(cx, tag)
XeTextInsertContext cx;
XeTextTag tag;
    {
	XeTextTag old = cx->mode.tag;
	cx->mode.tag = tag;
	return old;
    }
/*
** _XeTextGetTag
**	Get the application tag value from current insert context.
*/
XeTextTag _XeTextGetTag(cx)
XeTextInsertContext cx;
    {
	return cx->mode.tag;
    }

