/*
** Copyright 1993 by Markku Savela and
**	Technical Research Centre of Finland
*/

/*
** I have used Athena Text Widget as a source for useful code snippets
** in many places.  --msa
*/
/*
Copyright 1989 by the Massachusetts Institute of Technology,
Cambridge, Massachusetts.

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the names of Digital or MIT not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.  

DIGITAL DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
DIGITAL BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSpEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

******************************************************************/


#include <stdio.h>
#if SYSV_INCLUDES
#	include <memory.h>

extern int atoi( /* No header for this in SYSV */
#if NeedFunctionPrototypes
		char *
#endif
		);

#endif
#if ANSI_INCLUDES
#	include <stddef.h>
#	include <stdlib.h>
#endif
#include <ctype.h>
#include <string.h>
#if ANSI_INCLUDES || SYSV_INCLUDES
#	include <limits.h>
#else
/*
** No limits.h, invent missing constants as needed here
*/
#ifndef LONG_MAX
#define LONG_MAX (~((unsigned long)0L)>1)
#endif
#endif
#include <X11/Xlib.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>
#include <X11/Xmu/Atoms.h>
#include <X11/Xmu/CharSet.h>
#include <X11/Xmu/StdSel.h>
#include <X11/Xew/XewInit.h>
#include <X11/Xew/TextEdP.h>
#include "TextLayout.h"
#include "TextImport.h"
#include "TextExport.h"

/*
** This is a slightly modified copy of the corresponding file from
** the Athena Widget Set.
*/
static char translations1[] =
"\
Ctrl<Key>A:	beginning-of-line() \n\
Ctrl<Key>B:	backward-character() \n\
Ctrl<Key>D:	delete-next-character() \n\
Ctrl<Key>E:	end-of-line() \n\
Ctrl<Key>F:	forward-character() \n\
Ctrl<Key>G:     multiply(Reset) \n\
Ctrl<Key>H:	delete-previous-character() \n\
Ctrl<Key>K:	kill-to-end-of-line() \n\
Ctrl<Key>L:	redraw-display() \n\
Ctrl<Key>M:	newline() \n\
Ctrl<Key>N:	next-line() \n\
Ctrl<Key>O:	newline-and-backup() \n\
Ctrl<Key>P:	previous-line() \n\
Ctrl<Key>T:     transpose-characters() \n\
Ctrl<Key>U:	multiply(4) \n\
Ctrl<Key>W:	kill-selection() \n\
Ctrl<Key>Y:	insert-selection(CUT_BUFFER1) \n\
", translations2[] = "\
Meta<Key>B:	backward-word() \n\
Meta<Key>F:	forward-word() \n\
Meta<Key>K:	kill-to-end-of-paragraph() \n\
Meta<Key>Y:	insert-selection(PRIMARY, CUT_BUFFER0) \n\
:Meta<Key>d:	delete-next-word() \n\
:Meta<Key>D:	kill-word() \n\
:Meta<Key>h:	delete-previous-word() \n\
:Meta<Key>H:	backward-kill-word() \n\
:Meta<Key>\\<:	beginning-of-file() \n\
:Meta<Key>\\>:	end-of-file() \n\
:Meta<Key>]:	forward-paragraph() \n\
:Meta<Key>[:	backward-paragraph() \n\
~Shift Meta<Key>Delete:		delete-previous-word() \n\
 Shift Meta<Key>Delete:		backward-kill-word() \n\
~Shift Meta<Key>BackSpace:	delete-previous-word() \n\
 Shift Meta<Key>BackSpace:	backward-kill-word() \n\
", translations3[] = "\
<Key>Right:	forward-character() \n\
<Key>Left:	backward-character() \n\
<Key>Down:	next-line() \n\
<Key>Up:	previous-line() \n\
<Key>Delete:	delete-previous-character() \n\
<Key>BackSpace:	delete-previous-character() \n\
<Key>Return:	newline() \n\
<Key>:		insert-char() \n\
<FocusIn>:	focus-in() \n\
<FocusOut>:	focus-out() \n\
<Btn1Down>:	select-start() \n\
<Btn1Motion>:	extend-adjust() \n\
<Btn1Up>:	extend-end(PRIMARY, CUT_BUFFER0) \n\
<Btn2Down>:	insert-selection(PRIMARY, CUT_BUFFER0) \n\
<Btn3Down>:	extend-start() \n\
<Btn3Motion>:	extend-adjust() \n\
<Btn3Up>:	extend-end(PRIMARY, CUT_BUFFER0) \
";

/*
** The default actions follow mostly the Athena Text Widget model
*/
static void
	MoveForwardChar(), MoveBackwardChar(), MoveForwardWord(),
	MoveBackwardWord(), MoveForwardParagraph(), MoveBackwardParagraph(),
	MoveToLineStart(), MoveToLineEnd(), MoveNextLine(),MovePreviousLine(),
	MoveBeginningOfFile(), MoveEndOfFile(), DeleteForwardChar(),
	DeleteBackwardChar(), DeleteForwardWord(), DeleteBackwardWord(),
	DeleteCurrentSelection(), KillForwardWord(), KillBackwardWord(),
	KillCurrentSelection(), KillToEndOfLine(), KillToEndOfParagraph(),
	InsertNewLineAndBackup(), InsertNewLine(),
	SelectWord(), SelectAll(), SelectStart(), SelectAdjust(), SelectEnd(),
	ExtendStart(), ExtendAdjust(), ExtendEnd(),
	InsertSelection(), RedrawDisplay(),
	InsertChar(), InsertString(), TextFocusIn(), TextFocusOut(),
	DisplayCaret(), Multiply(), TransposeCharacters(),
	NoOp();

static XtActionsRec actions[] =
    {
	/* motion bindings */
	{"forward-character",	MoveForwardChar},
	{"backward-character",	MoveBackwardChar},
	{"forward-word", 	MoveForwardWord},
	{"backward-word", 	MoveBackwardWord},
	{"forward-paragraph", 	MoveForwardParagraph},
	{"backward-paragraph", 	MoveBackwardParagraph},
	{"beginning-of-line", 	MoveToLineStart},
	{"end-of-line", 	MoveToLineEnd},
	{"next-line", 		MoveNextLine},
	{"previous-line", 	MovePreviousLine},
	{"beginning-of-file", 	MoveBeginningOfFile},
	{"end-of-file", 	MoveEndOfFile},
	/* delete bindings */
	{"delete-next-character", DeleteForwardChar},
	{"delete-previous-character", DeleteBackwardChar},
	{"delete-next-word", 	DeleteForwardWord},
	{"delete-previous-word", DeleteBackwardWord},
	{"delete-selection",	DeleteCurrentSelection},
	/* kill bindings */
	{"kill-word", 		KillForwardWord},
	{"backward-kill-word", 	KillBackwardWord},
	{"kill-selection", 	KillCurrentSelection},
	{"kill-to-end-of-line", KillToEndOfLine},
	{"kill-to-end-of-paragraph", KillToEndOfParagraph},
	/* new line stuff */
	{"newline-and-backup", 	InsertNewLineAndBackup},
	{"newline", 		InsertNewLine},
	/* Selection stuff */
	{"select-word", 	SelectWord},
	{"select-all", 		SelectAll},
	{"select-start", 	SelectStart},
	{"select-adjust", 	SelectAdjust},
	{"select-end", 		SelectEnd},
	{"extend-start", 	ExtendStart},
	{"extend-adjust", 	ExtendAdjust},
	{"extend-end", 		ExtendEnd},
	{"insert-selection",	InsertSelection},
	/* Miscellaneous */
	{"redraw-display", 	RedrawDisplay},
	{"insert-char", 	InsertChar},
	{"insert-string",	InsertString},
	{"focus-in", 	 	TextFocusIn},
	{"focus-out", 		TextFocusOut},
	{"display-caret",	DisplayCaret},
	{"multiply",		Multiply},
	{"transpose-characters",TransposeCharacters},
	{"no-op",               NoOp},
    };

#define offset(field) XtOffsetOf(XeTextEdRec, texted.field)

static XtResource resources[] =
    {
	{XtNallowEdit, XtCAllowEdit, XtRBoolean, sizeof(Boolean),
	     offset(allow_edit), XtRImmediate, (XtPointer)True},
	{XtNcursorPosition, XtCCursorPosition, XtRLong, sizeof(long),
	     offset(cursor_position), XtRImmediate, (XtPointer)0},
	{XtNdisplayCaret, XtCOutput, XtRBoolean, sizeof(Boolean),
	     offset(display_caret), XtRImmediate, (XtPointer)True},
	{XtNmodifyCallback, XtCModifyCallback, XtRCallback, sizeof(XtPointer),
		 offset(modify_callbacks), XtRCallback, (XtPointer)NULL},
    };
#undef offset

static void ClassPartInitialize(), ClassInitialize();
static void Initialize(), Destroy(), Redisplay();
static Boolean SetValues();

#define superclass	(&xeTextClassRec)

XeTextEdClassRec xeTextEdClassRec = {
  { /* core fields */
    /* superclass		*/	(WidgetClass) superclass,
    /* class_name		*/	"XeTextEd",
    /* widget_size		*/	sizeof(XeTextEdRec),
    /* class_initialize		*/	ClassInitialize,
    /* class_part_initialize	*/	ClassPartInitialize,
    /* class_inited		*/	FALSE,
    /* initialize		*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	XtInheritRealize,
    /* actions			*/	actions,
    /* num_actions		*/	XtNumber(actions),
    /* resources		*/	resources,
    /* num_resources		*/	XtNumber(resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	XtExposeCompressMultiple,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	XtInheritResize,
    /* expose			*/	Redisplay,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	NULL,
    /* query_geometry		*/	XtInheritQueryGeometry,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,
    /* extension		*/	NULL
  },
  { /* XeBasic fields */
    /* not used			*/	0
  },
  { /* XeText fields */
    /* not used			*/	0
  },
  { /* XeTextEd fields */
    /* not used			*/	0
  }
};

WidgetClass xeTextEdWidgetClass = (WidgetClass)&xeTextEdClassRec;

static void ClassInitialize()
    {
	int len1 = strlen(translations1);
	int len2 = strlen(translations2);
	int len3 = strlen(translations3);
	char *buf = XtMalloc ((unsigned)(len1 + len2 + len3 + 1));
	char *cp = buf;

	XewInitializeWidgetSet();

	(void) strcpy (cp, translations1); cp += len1;
	(void) strcpy (cp, translations2); cp += len2;
	(void) strcpy (cp, translations3);
	xeTextEdWidgetClass->core_class.tm_table = buf;
    }

static void ClassPartInitialize(class)
WidgetClass class;
    {
    }

#define insertCursor_width 6
#define insertCursor_height 3
static char insertCursor_bits[] = {0x0c, 0x1e, 0x33};

static Pixmap CreateInsertCursor(s)
Screen *s;
    {
	return (XCreateBitmapFromData
		(DisplayOfScreen(s), RootWindowOfScreen(s), insertCursor_bits,
		 insertCursor_width, insertCursor_height));
    }

static void GetCursorBounds(w, rect)
XeTextEdWidget w;
XRectangle * rect;
    {
	rect->width = (unsigned short) insertCursor_width;
	rect->height = (unsigned short) insertCursor_height;
	rect->x = w->texted.cursor_x - (short)(rect->width / 2);
	rect->y = w->texted.cursor_y - (short)rect->height;
    }

static void InsertCursor(w)
XeTextEdWidget w;
    {
	XRectangle rect;

	GetCursorBounds(w, &rect);
	XCopyPlane(XtDisplay(w), w->texted.cursor, XtWindow(w),
		   w->text.mygcXOR, 0, 0, (unsigned int)rect.width,
		   (unsigned int)rect.height,
		   (int) rect.x, (int) rect.y, 1);
	}

static void Initialize (request, new)
Widget request, new;
    {
	XeTextEdWidget w = (XeTextEdWidget)new;

	w->texted.cursor_position = 0;
	w->texted.mult = 1;
	w->texted.s.range[0] = 0; /* Start offset of selection */
	w->texted.s.range[1] = 0; /* End offset of selection */
	w->texted.s.nrect = 0;
	w->texted.s.selections = NULL;
	w->texted.s.atom_count = 0;
	w->texted.s.array_size = 0;
	w->texted.cursor = CreateInsertCursor(XtScreenOfObject(new));
	w->texted.cursor_x = w->texted.cursor_y = 0;
	w->texted.cursor_valid = False;
	w->texted.refresh = 0;
    }


static Time NoteTime(event)
XEvent *event;
    {
	if (event != NULL)
	    {
		switch (event->type)
		    {
		    case ButtonPress:
		    case ButtonRelease:
			return event->xbutton.time;
			break;
		    case KeyPress:
		    case KeyRelease:
			return event->xkey.time;
			break;
		    case MotionNotify:
			return event->xmotion.time;
			break;
		    case EnterNotify:
		    case LeaveNotify:
			return event->xcrossing.time;
		    }
	    }
	return CurrentTime; /* A kludge, should probably give warning --msa */
    }	    

static void NotePosition(w, event, p)
XeTextEdWidget w;
XEvent *event;
XPoint *p;
    {
	switch (event->type)
	    {
	    case ButtonPress:
	    case ButtonRelease:
		p->x = event->xbutton.x;
		p->y = event->xbutton.y;
		break;
	    case KeyPress:
	    case KeyRelease:
		    {
			XRectangle cursor;
			GetCursorBounds(w, &cursor);
			p->x = cursor.x + cursor.width / 2;;
			p->y = cursor.y + cursor.height / 2;;
		    }
		break;
	    case MotionNotify:
		p->x = event->xmotion.x;
		p->y = event->xmotion.y;
		break;
	    case EnterNotify:
	    case LeaveNotify:
		p->x = event->xcrossing.x;
		p->y = event->xcrossing.y;
		break;
	    default:
		p->x = 0;
		p->y = 0;
		break;
	    }
    }

/*
** XeLocation2Coordinates
**	Return (x,y) coordinates matching the given (Snip, Offset)
**	This function uses some ad hoc rules in deciding the actual
**	position. The reasons for them are somewhat obscure, but they
**	have been arrived by trial and error and do give out least
**	surprises to the user...
**
**	If Snip == NULL, default to the first Snip in content, or
**	origin, if empty content.
**
**	* THIS WORKS ONLY IF THE CONTENT HAS BEEN LAID OUT *
*/
#if NeedFunctionPrototypes
static void XeLocation2Coordinates(XeTextEdWidget, Snip *, int, XPoint *);
#endif
static void XeLocation2Coordinates(t, s, o, xy)
XeTextEdWidget t;
Snip *s;
int o;
XPoint *xy;
    {
	if (s)
	    {
		/*
		** If the offset is beyond the current Snip, this actually
		** means the beginning of the next Snip.
		*/
		if (o > s->length)
		    {
			if (s->next)
			    {
				s = s->next;
				o = 0;
			    }
			else
				o = s->length;
		    }
		xy->y = s->y;
		/*
		** If this snip is a space and point would be in front of it,
		** then use the end of previous snip instead (but, still use
		** the y from the current Snip!).
		*/
		if (o > 0 || !s->space)
		    {
			xy->x = s->x;
			if (o < s->length)
				xy->x += XeTextSnipWidth
					((XeTextWidget)t, s, s->data, o);
			else
				xy->x += s->xWidth;
		    }
		else if (&t->text.first == s->back)
			xy->x = t->text.x;
		else
		    {
			s = PreviousSnip(s);
			if (s->endseq)
				xy->x = t->text.x;
			else
				xy->x = s->x + s->xWidth;
		    }
	    }
	else if (t->text.first)
	    {
		xy->x = t->text.first->x;
		xy->y = t->text.first->y;
	    }
	else
	    {
		xy->x = t->text.x;
		xy->y = t->text.y;
	    }
    }

/*
** XeOffset2Locations
**	Convert virtual offsets into XeTextPositions and matching
**	(x,y) coordinates.
**
**	NOTE:	The offsets will be rearranged to ascending order,
**		if they are not.
**
**	NOTE:	In the returned (snip, offset), it is always true
**		that "offset < VirtualLength(snip)", except when the
**		converted offset was beyond the end of the file, this
**		is the only case when "offset == VirtualLength(snip)".
*/
#if NeedFunctionPrototypes
static void XeOffset2Locations
	(XeTextEdWidget, long *, int, XeTextPosition *, XPoint *);
#endif
static void XeOffset2Locations(t, v, n, p, xy)
XeTextEdWidget t;	/* Text Widget */
long *v;		/* Virtual offsets (points to convert) */
int n;			/* Number of points to convert */
XeTextPosition *p;	/* Return matching (Snip, offset) here */
XPoint *xy;		/* Return matching (x,y) here */
    {
	int done = 0;			/* # of converted virtual offsets */
	long voffset = 0;		/* Cumulative Virtual offset */
	Snip *last_editable = NULL;	/* Last Editable Snip in content */
	long vlength = 0;		/* Virtual Lenght of the last snip */
	Snip *s;
	int x, y, i, j;

	/*
	** First, ensure that the offsets are given in ascending
	** order. Rearrange them if this is not the case. Use brute
	** bubble sort. Never call this with more than 1-4 offsets!
	*/
	for (i = 0; i < n-1; ++i)
		for (j = i + 1; j < n; ++j)
			if (v[i] > v[j])
			    {
				register long temp = v[j];

				v[j] = v[i];
				v[i] = temp;
			    }
	/*
	** Find the matching (Snip,Offset) and (x,y) information for
	** the each virtual offset.
	*/
	y = t->text.y;
	x = t->text.x;
	for (s = t->text.first; done < n && s; s = s->next)
	    {
		if (IsEditableContent(s->mode.bits))
		    {
			last_editable = s;
			voffset += (vlength = VirtualLength(s));
			while (done < n && v[done] < voffset)
			    {
				int l = v[done] - voffset + vlength;

				p[done].snip = s;
				p[done].offset = l;
				XeLocation2Coordinates(t, s, l, &xy[done]);
				done += 1;
			    }
		    }
		/* The following test for endseq only attempts to get the
		   position of the space right at the beginning of line.
		   But, don't do this at end of content.. (would place
		   content cursor in funny way..)
		   --msa */
		if (s->endseq && s->next)
			x = t->text.x;
		else
			x = s->x + s->xWidth;
		y = s->y;
	    }
	/*
	** The remaining un-done virtual offsets point past end of
	** content. Adjust them to point after the last editable
	** Snip.
	*/
	while (done < n)
	    {
		v[done] = voffset;
		xy[done].x = x;
		xy[done].y = y;
		p[done].snip = last_editable;
		p[done].offset = vlength;
		done += 1;
	    }
    }

/*
** XeLocation2Offset
**	Convert location identified by (Snip,offset) tuple into
**	virtual offset.
*/
#if NeedFunctionPrototypes
static long XeLocation2Offset(XeTextEdWidget, XeTextPosition *);
#endif
static long XeLocation2Offset(t, dot)
XeTextEdWidget t;
XeTextPosition *dot;
    {
	Snip *s;
	long voffset;

	for (voffset = 0, s = t->text.first; s; s = s->next)
		if (dot->snip == s)
		    {
			voffset += dot->offset;
			break;
		    }
		else if (IsEditableContent(s->mode.bits))
			voffset += VirtualLength(s);
	if (s == NULL)
		XeWidgetWarningMsg
			((Widget)t, "bugTextEdExistingSnip",
			 "Non-Existing or bad Snip pointer (BUG)",
			 (String *)NULL, (Cardinal)0);
	return voffset;
    }

/*
** XeMoveLocation
**	Move the dot forward or backward the specified amount of virtual
**	positions. (amount < 0 ==> backward, amount > 0 ==> forward). dot
**	must initially point to editable content.
**
**	Returns the amount that didn't get processed (0 means success).
*/
#if NeedFunctionPrototypes
static long XeMoveLocation(XeTextEdWidget, XeTextPosition *, long);
#endif
static long XeMoveLocation(t, dot, amount)
XeTextEdWidget t;
XeTextPosition *dot;
long amount;
    {
	Snip *s = dot->snip;
	Snip *last_editable = NULL;
	long vlength = 0;

	if (s == NULL && (s = t->text.first) == NULL)
		return amount;
	if (amount > 0)
	    {
		amount += dot->offset;
		do
		    {
			if (IsEditableContent(s->mode.bits))
			    {
				last_editable = s;
				amount -= (vlength = VirtualLength(s));
				if (amount < 0)
				    {
					dot->offset = amount + vlength;
					dot->snip = s;
					return 0;
				    }
			    }
			
		    }
		while ((s = s->next) != NULL);
		dot->snip = last_editable;
		dot->offset = vlength;
	    }
	else if (amount < 0)
	    {
		amount -= (VirtualLength(s) - dot->offset);
		for (;; s = PreviousSnip(s))
		    {
			if (IsEditableContent(s->mode.bits))
			    {
				last_editable = s;
				amount += (vlength = VirtualLength(s));
				if (amount >= 0)
				    {
					dot->offset = amount;
					dot->snip = s;
					return 0;
				    }
			    }
			if (s->back == &t->text.first)
				break;
		    }
		dot->snip = last_editable;
		dot->offset = 0;
	    }
	return amount;
    }

/*
** FindPosition
**	Return virtual length of the content starting from the specified
**	Snip (s) up to the position defined by goal.
**
**	NOTE 1)	The code assumes that the starting point is at the
**		beginning of the snip (offset == 0) and that the Snip
**		is the first in line.
**
**	Returns the virtual length.
*/
#if NeedFunctionPrototypes
static long FindPosition(XeTextEdWidget, XPoint *, Snip *);
#endif
static long FindPosition(w, goal, s)
XeTextEdWidget w;
XPoint *goal;
Snip *s;
    {
	int x, y;
	Snip *r = s;
	long position = 0;
	int prev_end_position = 0;
	int has_end_position = 0;
	int vlength;

	if (s && !IsEditableContent(s->mode.bits))
		s = NULL;
	for ( ; r; r = r->next, position += vlength)
	    {
		if (IsEditableContent(r->mode.bits))
		    {
			/*
			** The following code assumes that VirtualLength
			** is defined as 'length' + HasEndPosition!!
			*/
			prev_end_position = has_end_position;
			has_end_position = HasEndPosition(r);
			vlength = has_end_position + r->length;
			s = r;
		    }
		else
			vlength = 0;
		x = r->x;
		y = r->y;
		if (y >= goal->y)
		    {
			if (x >= goal->x)
			    {
				/*
				** Before current Snip. To get pointer
				** positioned properly, backup position
				** by 1 if previous snip had virtual end
				** position (this kludge gives reasonable
				** results now, but may break horribly
				** with weird combinations of Snips. --msa
				*/
				position -= prev_end_position;
				break;	/* Before current snip */
			    }
			else if (r == s && x + r->xWidth >= goal->x)
			    {
				/* Within the current Snip */
				int n = 0;

				while (++n <= r->length)
					if (x + XeTextSnipWidth
					    ((XeTextWidget)w, r, r->data, n)
					    > goal->x)
						break;
				position += n - 1;
				break;
			    }
			else if (HasEndLine(r))
			    {
				/* The is at the end of line */
				position += vlength - 1;
				break; /* At end of line */
			    }
		    }
	    }
	return position;
    }

/*
** RectanglesOverlap (copied from Athena Text.c)
**	Returns TRUE if two rectangles overlap.
*/
static Boolean RectanglesOverlap(rect1, rect2)
XRectangle *rect1, *rect2;
    {
	return ((rect1->x < rect2->x + (short) rect2->width) &&
		(rect2->x < rect1->x + (short) rect1->width) &&
		(rect1->y < rect2->y + (short) rect2->height) &&
		(rect2->y < rect1->y + (short) rect1->height));
    }

/*
** RectangleWithin
**	Returns TRUE if rect2 is completely within rect2.
*/
static Boolean RectangleWithin(rect1, rect2)
XRectangle *rect1, *rect2;
    {
	return ((rect1->x <= rect2->x) &&
		(rect1->x + rect1->width >= rect2->x + rect2->width) &&
		(rect1->y <= rect2->y) &&
		(rect1->y + rect1->height >= rect2->y + rect2->height));
    }

/*
** CheckBounds
**	Check if insert cursor was moved outside parent window
**	(Not really completely thought out, first crack...). If
**	outside, then call notifyCallback with XeCR_NOTIFY_AREA.
*/
#if NeedFunctionPrototypes
static void CheckBounds(XeTextEdWidget, XRectangle *);
#endif
static void CheckBounds(t, r)
XeTextEdWidget t;
XRectangle *r;
    {
	Widget parent;
	int x, y, w, h;
	
	XeNotifyAreaCallbackData data;
	data.reason = XeCR_NOTIFY_AREA;
	data.area = *r;
	data.visible.x = 0;
	data.visible.y = 0;
	data.visible.width = t->core.width;
	data.visible.height = t->core.height;
	parent = XtParent(t);
	/*
	** Compute intersection rectangle between parent window and
	** widget window in widget coordinate space.
	*/
#if 0
	x = -t->core.x;
	y = -t->core.y;
	w = parent->core.width;
	h = parent->core.height;
#else
	    {
		Window root, parent, *children;
		unsigned int nchildren;
		XWindowAttributes attribs;

		if (!XQueryTree(XtDisplay(t), XtWindow(t),
				    &root, &parent, &children, &nchildren))
			return;
		XtFree((char *)children);
		XGetWindowAttributes(XtDisplay(t), XtWindow(t), &attribs);
		x = - attribs.x;
		y = - attribs.y;
		XGetWindowAttributes(XtDisplay(t), parent, &attribs);
		w = attribs.width;
		h = attribs.height;
	    }
#endif
	if (x > data.visible.x)
	    {
		data.visible.width -= x - data.visible.x;
		data.visible.x = x;
	    }
	if (x + w < data.visible.x + data.visible.width)
		data.visible.width = x + w - data.visible.x;
	
	if (y > data.visible.y)
	    {
		data.visible.height -= y - data.visible.y;
		data.visible.y = y;
	    }
	if (y + h < data.visible.y + data.visible.height)
		data.visible.height = y + h - data.visible.y;
	/*
	** Generate XtNnotifyCallback, if the cursor block is outside.
	*/
	if (!RectangleWithin(&data.visible, &data.area))
		XtCallCallbackList((Widget)t, t->basic.notify_callbacks,
				   (XtPointer)&data);
    }


/*
** GetCutBufferNumber (from Xaw)
**	Returns the number of the cut buffer (>= 0)identified by the atom,
**	or -1, if atom is not a cut buffer.
*/
static int GetCutBufferNumber(atom)
register Atom atom;
    {
	if (atom == XA_CUT_BUFFER0)
		return 0;
	else if (atom == XA_CUT_BUFFER1)
		return 1;
	else if (atom == XA_CUT_BUFFER2)
		return 2;
	else if (atom == XA_CUT_BUFFER3)
		return 3;
	else if (atom == XA_CUT_BUFFER4)
		return 4;
	else if (atom == XA_CUT_BUFFER5)
		return 5;
	else if (atom == XA_CUT_BUFFER6)
		return 6;
	else if (atom == XA_CUT_BUFFER7)
		return 7;
	return -1;
    }
static void MarkSelection(t)
XeTextEdWidget t;
    {
	if (t->texted.s.nrect > 0)
		XFillRectangles(XtDisplay(t), XtWindow(t), t->text.mygcXOR,
				t->texted.s.rect, t->texted.s.nrect);
    }

static void Redisplay(w, e, r)
Widget w;
XExposeEvent *e;
Region r;
    {
	XeTextEdWidget t = (XeTextEdWidget)w;
	Region my_r = r;

	if (!XtIsRealized(w))
		return;
	if (my_r == NULL)
	    {
		XRectangle clip;
		/*
		** No Region, make one from exposed area. This should be
		** exceptional, widgets should have the compress exposures
		** mode set and always have a region!
		*/
		if (e == NULL)
			return; /* No event either--someone goofed! */
		clip.x = e->x;
		clip.y = e->y;
		clip.width = e->width;
		clip.height = e->height;
		my_r = XCreateRegion();
		XUnionRectWithRegion(&clip, my_r, my_r); 
	    }
	/*
	** TextEd calls Redisplay Internally, shortcutting the server
	** roundtrip that would be cause by XClearArea. This means that
	** the expose method cannot trust the exposed area to be cleared
	** initially. Call Text expose method without XEvent, which makes
	** it explicitly clear the region.
	*/
	(*superclass->core_class.expose)(w, (XEvent *)NULL, my_r);
	/* Warning: This assumes that clipmask is left active from the
	   Text expose method, thus if cursor position was not exposed
	   it will not be affected either. --msa */
	MarkSelection(t);
	if (t->texted.display_caret)
		InsertCursor(t);
	if (t->basic.expose_callbacks &&
	    t->core.widget_class == xeTextEdWidgetClass)
	    {
		XeExposeCallbackData data;
		data.reason = XeCR_EXPOSE;
		data.event = e;
		data.region = r;
		XtCallCallbackList
			(w, t->basic.expose_callbacks, (XtPointer)&data);
	    }
	if (r != my_r)
		XDestroyRegion(my_r);
    }

static void InitRefreshRegion(w)
XeTextEdWidget w;
    {
	if (!w->texted.refresh)
		w->texted.refresh = XCreateRegion();
    }

static void DoRefreshRegion(w)
XeTextEdWidget w;
    {
	Region r = w->texted.refresh;

	/*
	** If refreshable area exists and is non-empty, then generate
	** smallest possible expose event (1 pixel) by XClearArea to
	** guarantee at least one expose event (which will deal with
	** refresh area). (1 pixel clear area should not cause much
	** noticeable flicker).
	*/
	if (r && !XEmptyRegion(r))
	    {
		w->texted.refresh = 0;
		Redisplay((Widget)w, (XExposeEvent *)NULL, r);
		XDestroyRegion(r);
	    }
    }

/*
** UpdateCursorPosition
**	Change the cursor into a new position and/or refresh cursor.
**
**	if goal, then reset horizontal goal position.
**
**	if expose, then request expose events
**
**	if bounds, then request that cursor is in visible window
*/
#if NeedFunctionPrototypes
static void UpdateCursorPosition(XeTextEdWidget, long, int, int, int);
#endif
static void UpdateCursorPosition(t, position, goal, expose, bounds)
XeTextEdWidget t;
long position;
int goal, expose, bounds;
    {
	XPoint xy;
	XRectangle rect;

	expose = expose && XtIsRealized((Widget)t);
	if (expose)
	    {
		GetCursorBounds(t, &rect); /* to clear out old cursor mark */
		XUnionRectWithRegion
			(&rect, t->texted.refresh, t->texted.refresh);
	    }
	if (t->texted.cursor_valid)
	    {
		t->texted.cursor_position = position -
			XeMoveLocation(t, &t->texted.cursor_location,
				       position - t->texted.cursor_position);
		XeLocation2Coordinates
			(t, t->texted.cursor_location.snip,
			 t->texted.cursor_location.offset, &xy);
	    }
	else
	    {
		XeOffset2Locations(t, &t->texted.cursor_position, 1,
				   &t->texted.cursor_location, &xy);
		t->texted.cursor_valid = True;
	    }
	t->texted.cursor_x = xy.x;
	t->texted.cursor_y = xy.y + insertCursor_height + 1;
	if (goal)
		t->texted.cursor_goal = t->texted.cursor_x;
	GetCursorBounds(t, &rect);
	if (expose)
		XUnionRectWithRegion
			(&rect, t->texted.refresh, t->texted.refresh);
	if (t->texted.cursor_location.snip)
	    {
		rect.y -= t->texted.cursor_location.snip->ascent;
		rect.height += t->texted.cursor_location.snip->ascent +
			t->texted.cursor_location.snip->descent;
	    }
	if (rect.x < 0)
	    {
		/*
		** Because the caret extends to the left of the current
		** point, it would generate this notify on every beginning
		** line. Clip area to positive side of x.
		*/
		if ((int)rect.width + rect.x < 0)
			rect.width = 0;
		else
			rect.width += rect.x;
		rect.x = 0;
	    }
	if (bounds && XtIsRealized((Widget)t))
		CheckBounds(t, &rect);
    }


/*
** ChangeSelection
**	Recomputed the area that must be highlighted from the
**	virtual offsets of the selection.
*/
#if NeedFunctionPrototypes
static void ChangeSelection(XeTextEdWidget, int);
#endif
static void ChangeSelection(t, bounds)
XeTextEdWidget t;
int bounds;	/* Call Check bounds, if non-zero */
    {
	XRectangle rect[3];
	int nrect, i, ascent1, descent1, ascent2, descent2, ascent3;
	int in_one_line;
	XeTextPosition dot[2];
	XPoint xy[2];
	long range[2];
	Snip *s;
	
	nrect = 0;
	if (t->texted.s.range[0] != t->texted.s.range[1])
	    {
		range[0] = t->texted.s.range[0];
		range[1] = t->texted.s.range[1];
		XeOffset2Locations(t, range, 2, dot, xy);
		in_one_line = False;
		for (s = dot[0].snip, ascent1=descent1=0; s; s = s->next)
		    {
			if (s->ascent > ascent1)
				ascent1 = s->ascent;
			if (s->descent > descent1)
				descent1 = s->descent;
			if (s == dot[1].snip)
			    {
				in_one_line = True;
				break;
			    }
			if (HasEndLine(s))
				break;
		    }
		for (s = dot[1].snip, ascent2=descent2=0;s;s=PreviousSnip(s))
		    {
			if (s->ascent > ascent2)
				ascent2 = s->ascent;
			if (s->descent > descent2)
				descent2 = s->descent;
			if (s == dot[0].snip || s->back == &t->text.first ||
			    HasEndLine(s))
				break;
		    }
		for (s = dot[1].snip, ascent3 = 0; s; s = s->next)
		    {
			if (s->ascent > ascent3)
				ascent3 = s->ascent;
			if (HasEndLine(s))
				break;
		    }
		rect[0].x = xy[0].x;
		rect[0].y = xy[0].y - ascent1;
		rect[0].height = ascent1 + descent1;
		if (in_one_line)
		    {
			nrect = 1;
			if (xy[0].x < xy[1].x)
				rect[0].width = xy[1].x - xy[0].x;
			else if (xy[0].x > xy[1].x)
				rect[0].width = xy[0].x - xy[1].x;
			else
				nrect = 0;
		    }
		else
		    {
			if (t->text.w <= (int)rect[0].x)
				rect[0].width = 1;
			else
				rect[0].width = t->text.w - rect[0].x;
			rect[1].x = 0;
			rect[1].y = xy[1].y - ascent3;
			if (xy[1].x > 0)
				rect[1].width = xy[1].x;
			else
				rect[1].width = 1;
			rect[1].height = ascent3 + descent2;
			if (rect[1].height == 0)
				rect[1].height = 1;
			rect[2].x = 0;
			rect[2].y = rect[0].y + rect[0].height;
			rect[2].width = t->text.w > 0 ? t->text.w : 1;
			i = rect[1].y - rect[0].y - rect[0].height;
			if (i > 0)
			    {
				rect[2].height = i;
				nrect = 3;
			    }
			else
			    {
				i = rect[1].y - rect[0].y;
				rect[0].height = i > 0 ? i : 1;
				nrect = 2;
			    }
		    }
	    }
	if (XtIsRealized((Widget)t))
	    {
		Region old = XCreateRegion();
		Region new = XCreateRegion();
		Region xor = XCreateRegion();
		
		for (i = 0; i < t->texted.s.nrect; ++i)
			XUnionRectWithRegion(&t->texted.s.rect[i], old, old);
		for (i = 0; i < nrect; ++i)
		    {
			XUnionRectWithRegion(&rect[i], new, new);
			t->texted.s.rect[i] = rect[i];
		    }
		t->texted.s.nrect = nrect;
		XXorRegion(new, old, xor);
		XUnionRegion(xor, t->texted.refresh, t->texted.refresh);
		XDestroyRegion(old);
		XDestroyRegion(new);
		XDestroyRegion(xor);
		if (bounds && nrect > 0)
		    {
			XRectangle area;
			i = range[1] == t->texted.s.range[1] ? 1 : 0;
			area.x = xy[i].x;
			area.y = xy[i].y;
			area.width = area.height = 1;
			if ((s = dot[i].snip) != NULL)
			    {
				area.y -= s->ascent;
				area.height += s->ascent + s->descent;
			    }
			CheckBounds(t, &area);
		    }
	    }
	t->texted.s.nrect = nrect;
    }

static Boolean SetValues(current, request, new, args, num_args)
Widget current, request, new;
ArgList args;
Cardinal *num_args;
    {
	XeTextEdWidget r = (XeTextEdWidget)request;
	XeTextEdWidget t = (XeTextEdWidget)new;

	if (!r->basic.content_loaded)
	    {
		t->texted.s.range[0] = t->texted.s.range[1] = 0;
		t->texted.cursor_position = 0;
		t->texted.cursor_valid = False;
	    }
	InitRefreshRegion(t);
	UpdateCursorPosition(t, t->texted.cursor_position, False, True, False);
	ChangeSelection(t, False);
	DoRefreshRegion(t);
	return False;
    }

static void Destroy(w)
Widget w;
    {
	XeTextEdWidget t = (XeTextEdWidget)w;

	if (t->texted.refresh)
		XDestroyRegion(t->texted.refresh);
	XFreePixmap(XtDisplay(w), t->texted.cursor);
	XFree((char *)t->texted.s.selections);
    }

/*
**
*/
typedef struct TextSelectionStruct
    {
	XeTextEdWidget w;/* Associated Widget */
	int size;	/* Current Size of data buffer */
	int length;	/* Bytes in the data buffer */
	char *data;	/* Allocated data buffer */
    } TextSelectionStruct;

#if NeedFunctionPrototypes
static int FeedSelect(char *, int, XeTextTag, XtPointer);
#endif
static int FeedSelect(data, length, tag, client_data)
char *data;
int length;
XeTextTag tag;
XtPointer client_data;
    {
	TextSelectionStruct *select = (TextSelectionStruct *)client_data;

	if (length <= 0)
		return 0;
	if (select->size - select->length < length)
	    {
		select->size += ((length + 511) / 512) * 512;
		select->data = XtRealloc(select->data, select->size);
	    }
	memcpy(&select->data[select->length], data, length);
	select->length += length;
	return length;
    }

static void ExtractSelectedContent(t, s, format)
XeTextEdWidget t;
TextSelectionStruct *s;	/* Will be initialized and filled */
XeTextExport format;	/* Defines the format of selection */
    {
	XeTextPosition dot[2];
	XPoint xy[2];
	long range[2];

	s->w = t;
	s->length = 0;
	s->size = 0;
	s->data = 0;
	range[0] = t->texted.s.range[0]; /* Copy to prevent reordering in .. */
	range[1] = t->texted.s.range[1]; /* ..the widgets data. */
	XeOffset2Locations(t, range, 2, dot, xy);
	if (range[0] < range[1])
		XeTextExtractContent((XeTextWidget)t, &dot[0], &dot[1], format,
				     FeedSelect, (XtPointer)s);
    }

static Boolean ConvertSelection
	(w, selection, target, type, value, length, format)
Widget w;
Atom *selection, *target, *type;
XtPointer *value;
unsigned long *length;
int *format;
    {
	Display* d = XtDisplay(w);
	XeTextEdWidget t = (XeTextEdWidget)w;
	XSelectionRequestEvent *req =
		XtGetSelectionRequest(w, *selection, (XtRequestId)NULL);

	if (*target == XA_TARGETS(d))
	    {
		Atom* targetP, * std_targets;
		unsigned long std_length;

		XmuConvertStandardSelection
			(w, req->time, selection, 
			 target, type, (caddr_t *)&std_targets,
			 &std_length, format);
		
		*value = XtMalloc((unsigned)(sizeof(Atom)*(std_length + 3)));
		targetP = *(Atom **)value;
		*length = std_length + 2;
		*targetP++ = XA_STRING;
		*targetP++ = XA_TEXT(d);
		if (t->text.export_format >= XeTextExport_ODIF)
		    {
			*targetP++ = XA_COMPOUND_TEXT(d);
			*length += 1;
		    }
		memcpy((char*)targetP, (char*)std_targets,
		       sizeof(Atom)*std_length);
		XtFree((char*)std_targets);
		*type = XA_ATOM;
		*format = 32;
		return True;
	    }
	if (*target == XA_STRING || *target == XA_TEXT(d) ||
	    *target == XA_COMPOUND_TEXT(d))
	    {
		TextSelectionStruct select;
		XeTextExport ef = t->text.export_format;

		if (*target != XA_STRING && ef >= XeTextExport_ODIF)
			*type = XA_COMPOUND_TEXT(d);
		else
		    {
			*type = XA_STRING;
			ef = XeTextExportFormatted(ef) ?
				XeTextExport_STRING_F : XeTextExport_STRING;
		    }
		ExtractSelectedContent(t, &select, ef);
		if (select.data == NULL)
			return False;
		*value = (XtPointer)select.data;
		*length = select.length;
		*format = 8;
		return True;
	    }
	if (XmuConvertStandardSelection(w, req->time, selection, target, type,
					(caddr_t *)value, length, format))
		return True;
	return False;
    }

static void LoseSelection(w, selection)
Widget w;
Atom *selection;
    {
	XeTextEdWidget t = (XeTextEdWidget)w;
	Atom *p, *r;
	int i;

	InitRefreshRegion(t);
	p = r = t->texted.s.selections;
	for (i = 0; i < t->texted.s.atom_count; i++, r++)
		if (*r != *selection && GetCutBufferNumber(*r) < 0)
			*p++ = *r;
	if ((t->texted.s.atom_count = p - t->texted.s.selections) == 0)
	    {
		t->texted.s.range[1] = t->texted.s.range[0];
		ChangeSelection(t, False);
	    }
	DoRefreshRegion(t);
    }

/*
** CreateCutBuffers
**	(copied from Xaw)
*/
static void CreateCutBuffers(d)
Display *d;
    {
	static struct _DisplayRec
	    {
		struct _DisplayRec *next;
		Display *dpy;
	    } *dpy_list = NULL;
	struct _DisplayRec *dpy_ptr;

	for (dpy_ptr = dpy_list; dpy_ptr != NULL; dpy_ptr = dpy_ptr->next)
		if (dpy_ptr->dpy == d)
			return;

	dpy_ptr = XtNew(struct _DisplayRec);
	dpy_ptr->next = dpy_list;
	dpy_ptr->dpy = d;
	dpy_list = dpy_ptr;

#define Create(buffer) \
	XChangeProperty(d, RootWindow(d, 0), buffer, XA_STRING, 8, \
			PropModeAppend, NULL, 0 );

	Create( XA_CUT_BUFFER0 );
	Create( XA_CUT_BUFFER1 );
	Create( XA_CUT_BUFFER2 );
	Create( XA_CUT_BUFFER3 );
	Create( XA_CUT_BUFFER4 );
	Create( XA_CUT_BUFFER5 );
	Create( XA_CUT_BUFFER6 );
	Create( XA_CUT_BUFFER7 );
#undef Create
    }


#define MAX_CUT_LEN(dpy)  (XMaxRequestSize(dpy) - 64)

#if NeedFunctionPrototypes
static void PostSelections(XeTextEdWidget, Atom *, Cardinal);
#endif

static void PostSelections(t, selections, count)
XeTextEdWidget t;
Atom *selections;
Cardinal count;
    {
	int buffer;
  
	while (count)
	    {
		Atom selection = selections[--count];

		if ((buffer = GetCutBufferNumber(selection)) >= 0)
		    {
			/* This is a cut buffer. */

			unsigned int amount;
			unsigned int max_len = MAX_CUT_LEN(XtDisplay(t));
			TextSelectionStruct s;
			unsigned char *ptr;
			XeTextExport format =
				XeTextExportFormatted(t->text.export_format) ?
					XeTextExport_STRING_F :
						XeTextExport_STRING;
			ExtractSelectedContent(t, &s, format);
			if (buffer == 0)
			    {
				CreateCutBuffers(XtDisplay(t));
				XRotateBuffers(XtDisplay(t), 1);
			    }
			amount = s.length < max_len ? s.length : max_len;
			ptr = (unsigned char *)s.data;
			XChangeProperty
				(XtDisplay(t), RootWindow(XtDisplay(t), 0),
				 selection, XA_STRING, 8, PropModeReplace,
				 ptr, amount);
			while (s.length > max_len)
			    {
				s.length -= max_len;
				ptr += max_len;
				amount = s.length<max_len ? s.length : max_len;
				XChangeProperty
					(XtDisplay(t),
					 RootWindow(XtDisplay(t), 0), 
					 selection, XA_STRING, 8,
					 PropModeAppend, 
					 ptr, amount);
			    }
			XtFree ((char *)s.data);
		    }
		else  if (!XtOwnSelection((Widget)t, selection, t->texted.time,
					  ConvertSelection, LoseSelection,
					  (XtSelectionDoneProc)NULL))
			XeWidgetWarningMsg
				((Widget)t, "noteTextEdSelectFail",
				 "Could not become selection owner",
				 (String *)NULL, (Cardinal)0);
	    }
    }

/*
** TextSelectionList
**	Converts (params, num_params) to a list of atoms & caches the
**	list in the TextWidget instance. (modified from Xaw)
*/
static Atom *TextSelectionList(t, list, nelems)
XeTextEdWidget t;
String *list;
Cardinal nelems;
    {
	Atom *sel = t->texted.s.selections;
	Display *dpy = XtDisplay((Widget)t);
	int n;

	if (nelems > t->texted.s.array_size)
	    {
		sel = (Atom *)XtRealloc((char *)sel, sizeof(Atom) * nelems);
		t->texted.s.array_size = nelems;
		t->texted.s.selections = sel;
	    }
	for (n = nelems; --n >= 0; sel++, list++)
		*sel = XInternAtom(dpy, *list, False);
	t->texted.s.atom_count = nelems;
	return t->texted.s.selections;
    }

/*
** TextSetSelection
**	Sets the current selection. (modified from Xaw)
*/
#if NeedFunctionPrototypes
static void TextSetSelection(XeTextEdWidget, String *, Cardinal);
#endif
static void TextSetSelection(t, list, nelems)
XeTextEdWidget t;
String *list;
Cardinal nelems;
    {
	String defaultSel = "PRIMARY";

	if (nelems == 1 && !strcmp (list[0], "none"))
		return;
	if (nelems == 0)
	    {
		list = &defaultSel;
		nelems = 1;
	    }
	PostSelections(t, TextSelectionList(t,list,nelems), nelems);
    }

#if NeedFunctionPrototypes
static void StartMoving(XeTextEdWidget, XEvent *);
#endif
static void StartMoving(w, e)
XeTextEdWidget w;
XEvent *e;
    {
	if (e)
		w->texted.time = NoteTime(e);
	InitRefreshRegion(w);
	if (w->text.inserting)
	    {
		_XeTextEndContent(w->text.inserting, w->texted.refresh);
		w->text.inserting = NULL;
	    }
    }

#if NeedFunctionPrototypes
static void EndMoving(XeTextEdWidget);
#endif
static void EndMoving(w)
XeTextEdWidget w;
    {
	w->texted.mult = 1;
	DoRefreshRegion(w);
    }

typedef enum
    {
	XeMoveToPoint,
	XeMovePositions,
	XeMoveLines,
	XeMoveWords,
	XeMoveLineBegin,
	XeMoveLineEnd,
	XeMoveFileBegin,
	XeMoveFileEnd
    } XeMoveType;

/*
** Scan
**	Find a new position within content starting from the current
**	cursor position.
*/
#if NeedFunctionPrototypes
static long Scan(XeTextEdWidget, XeMoveType, int);
#endif
static long Scan(w, move, amount)
XeTextEdWidget w;/* TextEd widget being operated */
XeMoveType move;/* Moving type */
int amount;	/* Moving amount and direction */
    {
	Snip *s;
	XeTextPosition dot;
	XPoint xy;
	int n, prev_space;
	long position = w->texted.cursor_position;

	if (move == XeMovePositions)
		return position + amount;
	if (w->texted.cursor_valid)
	    {
		dot = w->texted.cursor_location;
		XeLocation2Coordinates(w, dot.snip, dot.offset, &xy);
	    }
	else
		XeOffset2Locations(w, &position, 1, &dot, &xy);
	if ((s = dot.snip) == NULL)
		return 0;	/* No real content */
	switch (move)
	    {
	    case XeMoveLineEnd:
		/*
		** Move current position to the end of line. End of line
		** can either be real or just result of the line breaking
		** algorithm. (This causes some kludgy gyrations using
		** 'prev_space' counter.. --msa)
		*/
		for (position -= dot.offset, prev_space = 0; s ; s = s->next)
		    {
			if (IsEditableContent(s->mode.bits))
			    {
				n = VirtualLength(s);
				if (s->space)
					prev_space += n;
				else
				    {
					position += prev_space + n;
					prev_space = 0;
				    }
			    }
			if (HasEndLine(s))
			    {
				if (IsEditableContent(s->mode.bits))
					position -= 1;
				else
					position += prev_space - 1;
				break;
			    }
		    }
		break;
	    case XeMoveLineBegin:
		position -= dot.offset;
		while (s->back != &w->text.first)
		    {
			s = PreviousSnip(s);
			if (HasEndLine(s))
				break;
			if (IsEditableContent(s->mode.bits))
				position -= VirtualLength(s);
		    }
		break;
	    case XeMoveFileBegin:
		position = 0;
		break;
	    case XeMoveFileEnd:
		position = LONG_MAX;
		break;
	    case XeMoveWords:
#define IsWhiteSpace(s,o) ((s)->space || ((s)->endseq && (o) == (s)->length))
		if (amount >= 0)
		    {
			int offset = dot.offset;

			do
			    {
				/*
				** Skip white space
				*/
				while (s)
				    {
					if (!IsEditableContent(s->mode.bits))
					    {
						s = s->next;
						continue;
					    }
					if (!IsWhiteSpace(s,offset))
						break;
					if (offset == s->length)
					    {
						position += HasEndPosition(s);
						offset = 0;
						s = s->next;
					    }
					else
					    {
						position += s->length - offset;
						offset = s->length;
					    }
				    }
				while (s)
				    {
					if (!IsEditableContent(s->mode.bits))
					    {
						s = s->next;
						continue;
					    }
					if (IsWhiteSpace(s, offset))
						break;
					if (offset == s->length)
					    {
						position += HasEndPosition(s);
						offset = 0;
						s = s->next;
					    }
					else
					    {
						position += s->length - offset;
						offset = s->length;
					    }
				    }
			    } while (--amount > 0);
		    }
		else if (amount < 0)
		    {
			int offset = dot.offset;

			do
			    {
				/*
				** Skip white space
				*/
				while (s->back != &w->text.first)
				    {
					if (!IsEditableContent(s->mode.bits))
					    {
						s = PreviousSnip(s);
						continue;
					    }
					if (offset == 0)
					    {
						s = PreviousSnip(s);
						position -= HasEndPosition(s);
						offset = s->length;
					    }
					else
					    {
						position -= offset;
						offset = 0;
					    }
					if (!IsWhiteSpace(s,offset))
						break;
				    }
				while (s->back != &w->text.first)
				    {
					if (!IsEditableContent(s->mode.bits))
					    {
						s = PreviousSnip(s);
						continue;
					    }
					if (offset == 0)
					    {
						s = PreviousSnip(s);
						position -= HasEndPosition(s);
						offset = s->length;
					    }
					else
					    {
						position -= offset;
						offset = 0;
					    }
					if (IsWhiteSpace(s, offset))
						break;
				    }
			    } while (++amount < 0);
		    }
		break;
	    case XeMoveLines:
		/*
		** Pass over as many Snip_EndLine's as indicated by
		** amount.
		*/
		position -= dot.offset;
		if (amount > 0)
			do
			    {
				if (IsEditableContent(s->mode.bits))
					position += VirtualLength(s);
				if (HasEndLine(s))
					amount -= 1;
				s = s->next;
			    }
			while (s && amount > 0);
		else if (amount < 0)
			while (s->back != &w->text.first)
			    {
				s = PreviousSnip(s);
				if (HasEndLine(s))
				    {
					amount += 1;
					if (amount > 0)
					    {
						s = s->next;
						break;
					    }
				    }
				if (IsEditableContent(s->mode.bits))
					position -= VirtualLength(s);
			    }
		xy.x = w->texted.cursor_goal;
		xy.y = 0;
		position += FindPosition(w, &xy, s);
		break;
	    default:
		break;
	    }
	return position;
    }

static void Move(w, event, move, amount)
XeTextEdWidget w;
XEvent *event;
XeMoveType move;
int amount;
    {
	long position;

	StartMoving(w, event);
	amount *= w->texted.mult;
	if (move ==  XeMoveToPoint)
	    {
		XPoint p;

		NotePosition(w, event, &p);
		position = FindPosition(w, &p, w->text.first);
	    }
	else
		position = Scan(w, move, amount);
	if (position < 0)
	    {
		position = 0;
		XBell(XtDisplay(w), 0);
	    }
	UpdateCursorPosition(w, position, move != XeMoveLines, True, True);
	EndMoving(w);
    }

static void MoveForwardChar(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	Move((XeTextEdWidget)w, e, XeMovePositions, 1);
    }

static void MoveBackwardChar(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	Move((XeTextEdWidget)w, e, XeMovePositions, -1);
    }

static void MoveForwardWord(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	Move((XeTextEdWidget)w, e, XeMoveWords, 1);
    }

static void MoveBackwardWord(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	Move((XeTextEdWidget)w, e, XeMoveWords, -1);
    }

static void MoveToLineEnd(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	Move((XeTextEdWidget) w, e, XeMoveLineEnd, 1);
    }

static void MoveToLineStart(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	Move((XeTextEdWidget) w, e, XeMoveLineBegin, 1);
    }

static void MoveNextLine(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	Move((XeTextEdWidget) w, e, XeMoveLines, 1);
    }

static void MovePreviousLine(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	Move((XeTextEdWidget) w, e, XeMoveLines, -1);
    }

static void MoveBeginningOfFile(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	Move((XeTextEdWidget) w, e, XeMoveFileBegin, 1);
    }

static void MoveEndOfFile(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	Move((XeTextEdWidget) w, e, XeMoveFileEnd, 1);
    }

/*
** StartEditing
**	Prepare for editing the content.
**
**	Assumption: XEvent (e) is always NULL for programmatic modifications
**	and non-NULL for direct keyboard/mouse actions. If the allowEdit
**	resource has been set to False, keyboard/mouse modifying actions
**	are disabled.
**
**	Returns 0, if editing not started.
**	Returns 1, if editing started
*/
#if NeedFunctionPrototypes
static int StartEditing(XeTextEdWidget, int, XEvent *);
#endif
static int StartEditing(w, mode, e)
XeTextEdWidget w;
int mode;	/* Initial state mode */
XEvent *e;
    {
	XeTextModifyCallbackData data;
	data.reason = XeCR_MODIFY_TEXT;
	data.cancel = False;
	XtCallCallbackList
		((Widget)w, w->texted.modify_callbacks, (XtPointer)&data);
	if (data.cancel)
		return 0;	/* Callback cancelled start edit! */
	InitRefreshRegion(w);
	if (e)
		if (!w->texted.allow_edit)
		    {
			XBell(XtDisplay(w), 0);
			return 0;
		    }
		else
			w->texted.time = NoteTime(e);
	if (!w->text.inserting)
	    {
		if (!w->texted.cursor_valid || !w->texted.cursor_location.snip)
		    {
			XPoint xy;

			XeOffset2Locations
				(w,&w->texted.cursor_position, 1,
				 &w->texted.cursor_location, &xy);
			w->texted.cursor_valid = True;
		    }
		w->text.inserting = _XeTextInsertContent
			((XeTextWidget)w,
			 w->texted.cursor_location.snip,
			 w->texted.cursor_location.offset,
			 mode);
	    }
	return 1;
    }


/*
** EndEditing
**	Is called after any operation that changes the content.
**	The deleted parameter
*/
#if NeedFunctionPrototypes
static void EndEditing(XeTextEdWidget, int);
#endif
static void EndEditing(w, deleted)
XeTextEdWidget w;
int deleted;
    {
	long ref, position;
	int i;
	
	w->texted.mult = 1;
	ref = w->texted.cursor_position;
	if (w->text.inserting)
	    {
		w->texted.cursor_location.snip =
			_XeTextInsertPoint
				(w->text.inserting, w->texted.refresh);
		_XeTextLayout((XeTextWidget)w,
			      w->texted.cursor_location.snip,
			      w->texted.refresh);
		if (w->texted.cursor_location.snip)
		    {
			w->texted.cursor_location.offset =
				VirtualLength(w->texted.cursor_location.snip);
			w->texted.cursor_position = XeLocation2Offset
				(w, &w->texted.cursor_location);
		    }
		else
		    {
			w->texted.cursor_position = 0;
			w->texted.cursor_location.offset = 0;
		    }
		w->texted.cursor_valid = True;
	    }
	position = w->texted.cursor_position;
	/*
	** Do a feeble attempt to update selection range
	*/
	if (deleted < 0)
	    {
		deleted = -deleted;
		ref = position;
	    }
	for (i = 0; i < XtNumber(w->texted.s.range); ++i)
		if (ref <= w->texted.s.range[i])
			if (deleted == 0)
				w->texted.s.range[i] += 
					w->texted.cursor_position - ref;
			else if (ref + deleted > w->texted.s.range[i])
				w->texted.s.range[i] = ref;
			else
				w->texted.s.range[i] -= deleted;

	ChangeSelection(w, False); /* Should fine tune? --msa */
	/*
	** Request resize of if the basic resize resource is set. However,
	** do this only if the current size exceeds the window size.
	*/
	if (w->basic.resize &&
	    (w->core.width < w->text.w || w->core.height < w->text.h))
	    {
		Dimension rw = w->text.w;
		Dimension rh = w->text.h;
		int do_almost = 1;

		while (XtMakeResizeRequest((Widget)w, rw, rh, &rw, &rh)
		       == XtGeometryAlmost && do_almost)
			do_almost = 0;
	    }
	UpdateCursorPosition(w, position, True, True, True);
	DoRefreshRegion(w);
    }

/************************************************************
 *
 * Delete Routines.
 *
 ************************************************************/
#if NeedFunctionPrototypes
static void DeleteOrKill(XeTextEdWidget, XEvent *, XeMoveType, int, int);
#endif

#define TEXT_KILL True		/* Used in function calls just for clarity */
#define TEXT_DELETE False

static void DeleteOrKillRange(w, amount, kill)
XeTextEdWidget w;
int amount, kill;
    {
	Snip *s;
	XeTextPosition dot[2];
	TextSelectionStruct sel;

	s = _XeTextDeleteContent(w->text.inserting, amount);
	if (kill)
	    {
		sel.w = w;
		sel.length = 0;
		sel.size = 0;
		sel.data = 0;
		dot[0].snip = s;
		dot[0].offset = 0;
		dot[1].snip = NULL;
		dot[1].offset = 0;
		XeTextExtractContent((XeTextWidget)w, &dot[0], &dot[1],
				     w->text.export_format, FeedSelect,
				     (XtPointer)&sel);
		if (sel.length > 0)
			XStoreBuffer(XtDisplay(w), sel.data, sel.length, 1);
		XtFree(sel.data);
	    }	
	while (s)
		_XeDeleteSnip(&s);
    }

static void DeleteOrKill(w, event, move, amount, kill)
XeTextEdWidget w;
XEvent *event;
XeMoveType move;
int amount, kill;
    {
	long end;

	if (!StartEditing(w, XeTextInsertMode_CURRENT, event))
		return;
	amount *= w->texted.mult;
	end = Scan(w, move, amount);
	if (move == XeMoveLineEnd && end == w->texted.cursor_position)
		end += 1;	/* Special handling for KILL to END LINE */
	DeleteOrKillRange(w, end - w->texted.cursor_position, kill);
	EndEditing(w, end - w->texted.cursor_position);
    }

static void DeleteForwardChar(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	DeleteOrKill((XeTextEdWidget)w, e, XeMovePositions, 1, TEXT_DELETE);
    }

static void DeleteBackwardChar(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	DeleteOrKill((XeTextEdWidget)w, e, XeMovePositions, -1, TEXT_DELETE);
    }

static void DeleteForwardWord(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	DeleteOrKill((XeTextEdWidget)w, e, XeMoveWords, 1, TEXT_DELETE);
    }

static void DeleteBackwardWord(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	DeleteOrKill((XeTextEdWidget)w, e, XeMoveWords, -1, TEXT_DELETE);
    }

static void KillForwardWord(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	DeleteOrKill((XeTextEdWidget) w, e, XeMoveWords, 1, TEXT_KILL);
    }

static void KillBackwardWord(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	DeleteOrKill((XeTextEdWidget)w, e, XeMoveWords, -1, TEXT_KILL);
    }

static void KillToEndOfLine(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	DeleteOrKill((XeTextEdWidget)w, e, XeMoveLineEnd, 1, TEXT_KILL);
    }

static void KillToEndOfParagraph(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
    }

#if NeedFunctionPrototypes
static void TextZapSelection(XeTextEdWidget, XEvent *, int);
#endif

static void TextZapSelection(w, event, kill)
XeTextEdWidget w;
XEvent *event;
int kill;
    {
	int amount;

	if (w->texted.s.range[0] == w->texted.s.range[1])
		return;
	StartMoving(w, event);
	w->texted.cursor_position = w->texted.s.range[0];
	w->texted.cursor_valid = False;
	EndMoving(w);
	if (!StartEditing(w, XeTextInsertMode_CURRENT, event))
		return;
	amount = w->texted.s.range[1] - w->texted.cursor_position;
	DeleteOrKillRange(w, amount, kill);
	w->texted.s.range[0] = w->texted.s.range[1] = 0;
	ChangeSelection(w, False);
	EndEditing(w, amount);
    }

static void KillCurrentSelection(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	TextZapSelection((XeTextEdWidget)w, e, TEXT_KILL);
    }

static void DeleteCurrentSelection(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	TextZapSelection((XeTextEdWidget)w, e, TEXT_DELETE);
    }

/************************************************************
 *
 * Insertion Routines.
 *
 ************************************************************/

static void InsertNewLineAndBackup(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	XeTextEdWidget t = (XeTextEdWidget)w;
	int count;
	long save;

	if (!StartEditing(t, XeTextInsertMode_CURRENT, e))
		return;
	save = t->texted.cursor_position;
	for (count = 0 ; count < t->texted.mult ; count++)
		_XeTextFeedContent(t->text.inserting, "\r\n", 2);
	EndEditing(t, 0);
	/*
	** Must simulate cursor movement action (basicly, must
	** close the insert state).
	*/
	StartMoving(t, e);
	UpdateCursorPosition(t, save, True, True, True);
	EndMoving(t);
    }

static void InsertNewLine(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	XeTextEdWidget t = (XeTextEdWidget)w;
	int count;

	if (!StartEditing(t, XeTextInsertMode_CURRENT, e))
		return;
	for (count = 0 ; count < t->texted.mult ; count++)
		_XeTextFeedContent(t->text.inserting, "\r\n", 2);
	EndEditing(t, 0);
    }

/************************************************************
 *
 * Selection Routines.
 *
 *************************************************************/

typedef enum
    {
	XeSelectionStart,
	XeSelectionEnd,
	XeSelectionExtendStart,
	XeSelectionMove
    } XeSelectionType;

static void SelectWord(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	XeTextEdWidget t = (XeTextEdWidget) w;

	StartMoving(t, e);
	EndMoving(t);
    }

static void SelectAll(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	XeTextEdWidget t = (XeTextEdWidget) w;
	
	StartMoving(t, e);
	t->texted.s.range[0] = 0;
	t->texted.s.range[1] = LONG_MAX;
	ChangeSelection(t, False);
	EndMoving(t);
    }

static void ModifySelection(t, event, mode, par, num_par)
XeTextEdWidget t;
XEvent *event;
int mode;
String *par;
Cardinal *num_par;
    {
	XPoint p;
	long position;
	long a, b;

	StartMoving(t, event);
	NotePosition(t, event, &p);
	position = FindPosition(t, &p, t->text.first);
	switch (mode)
	    {
	    case XeSelectionStart:
		XeTextUnsetSelection((Widget)t);
		t->texted.s.range[0] = t->texted.s.range[1] = position;
		break;
	    case XeSelectionExtendStart:
		a = position - t->texted.s.range[0];
		b = position - t->texted.s.range[1];
		if (abs(b) > abs(a))
			t->texted.s.range[0] = t->texted.s.range[1];
		t->texted.s.range[1] = position;
		break;
	    case XeSelectionMove:
		t->texted.s.range[1] = position;
		break;
	    case XeSelectionEnd:
		t->texted.s.range[1] = position;
		if (t->texted.s.range[0] == position)
			UpdateCursorPosition(t, position, True, True, True);
		else
			TextSetSelection(t, par, *num_par);
		break;
	    default:
		break;
	    }
	ChangeSelection(t, True);
	EndMoving(t);
    }

static void SelectStart(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	ModifySelection((XeTextEdWidget)w, e, XeSelectionStart, p, n);
    }

static void SelectAdjust(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	ModifySelection((XeTextEdWidget)w, e, XeSelectionMove, p, n);
    }

static void SelectEnd(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	ModifySelection((XeTextEdWidget)w, e, XeSelectionEnd, p, n);
    }

static void ExtendStart(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	ModifySelection((XeTextEdWidget)w, e, XeSelectionExtendStart, p, n);
    }

static void ExtendAdjust(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	ModifySelection((XeTextEdWidget)w, e, XeSelectionMove, p, n);
    }

static void ExtendEnd(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	ModifySelection((XeTextEdWidget)w, e, XeSelectionEnd, p, n);
    }

/************************************************************
 *
 * Misc. Routines.
 *
 ************************************************************/

static void RedrawDisplay(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	XeTextLayout(w);
    }

static void TextFocusIn (w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
    }

static void TextFocusOut(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
    }

static XComposeStatus compose_status = {NULL, 0};

static void InsertChar(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	XeTextEdWidget ctx = (XeTextEdWidget) w;
	char strbuf[BUFSIZ];
	int count;
	KeySym keysym;
	int length;

	if ((length = XLookupString (&e->xkey, strbuf, BUFSIZ,
				     &keysym, &compose_status)) == 0)
		return;
	if (!StartEditing(ctx, XeTextInsertMode_CURRENT, e))
		return;
	for (count = 0 ; count < ctx->texted.mult ; count++)
		_XeTextFeedContent(ctx->text.inserting, strbuf, length);
	EndEditing(ctx, 0);
    }

static void InsertString(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	XeTextEdWidget ctx = (XeTextEdWidget) w;
	int i;

	if (!StartEditing(ctx, XeTextInsertMode_CURRENT, e))
		return;
	for (i = *n; i; i--, p++)
	    {
		unsigned char hexval;
		if ((*p)[0] == '0' &&
		    (*p)[1] == 'x' && (*p)[2] != '\0')
		    {
			char c, *s;
			hexval = 0;
			for (s = *p+2; (c = *s); s++)
			    {
				hexval *= 16;
				if (c >= '0' && c <= '9')
					hexval += c - '0';
				else if (c >= 'a' && c <= 'f')
					hexval += c - 'a' + 10;
				else if (c >= 'A' && c <= 'F')
					hexval += c - 'A' + 10;
				else break;
			    }
			if (c == '\0')
				_XeTextFeedContent(ctx->text.inserting,
						   (char *)&hexval, 1L);
			else
				_XeTextFeedContent(ctx->text.inserting,
						   *p, strlen(*p));
		    }
		else
			_XeTextFeedContent(ctx->text.inserting,*p,strlen(*p));
	    }
	EndEditing(ctx, 0);
    }

static void DisplayCaret(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	XeTextEdWidget ctx = (XeTextEdWidget)w;
	Boolean display_caret = True;
	
	if (e->type == EnterNotify || e->type == LeaveNotify)
	    {
		/* for Crossing events, the default case is to check the focus
		** field and only change the caret when focus==True.  A second
		** argument of "always" will cause the focus field to be
		** ignored.
		*/
		Boolean check_focus = True;
		if (*n == 2 && strcmp(p[1], "always") == 0)
			check_focus = False;
		if (check_focus && !e->xcrossing.focus)
			return;
	    }
	
	if (*n > 0)
	    {
		/* default arg is "True" */
		XrmValue from, to;
		from.size = strlen(from.addr = p[0]);
		XtConvert(w, XtRString, &from, XtRBoolean, &to);
		if (to.addr != NULL)
			display_caret = *(Boolean*)to.addr;
		if (ctx->texted.display_caret == display_caret)
			return;
	    }
	StartMoving(ctx, e);
	if (ctx->texted.display_caret != display_caret)
	    {
		/* State has really changed */

		ctx->texted.display_caret = display_caret;
		UpdateCursorPosition(ctx, ctx->texted.cursor_position,
				     False, True, False);
	    }
	EndMoving(ctx);
    }

/*
** Multiply (from MIT Athena TextAction.c)
**	Multiplies the current action by the number passed.
**
**	The parameter list may contain either a number or the string 'Reset'.
**
**	A number will multiply the current multiplication factor by that
**	number. Many of the text widget actions will will perform n actions,
**	 where n is the multiplication factor.
**
**	The string reset will reset the mutiplication factor to 1.
*/
static void Multiply(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	XeTextEdWidget ctx = (XeTextEdWidget) w;
	int mult;
	
	if (*n != 1)
	    {
		XeWidgetWarningMsg
			(w, "bugTextEdActionError",
			 "The multiply action takes exactly one argument.",
			 (String *)NULL, 0);
		XBell(XtDisplay(w), 0);
		return;
	    }
	
	if ( (p[0][0] == 'r') || (p[0][0] == 'R') ) 
	    {
		XBell(XtDisplay(w), 0);
		ctx->texted.mult = 1;
		return;
	    }
	
	if ( (mult = atoi(p[0])) == 0 )
	    {
		XeWidgetWarningMsg
			(w, "bugTextEdActionError",
"The multiply action's argument must be a number greater than zero, or 'Reset'.",
			 (String *)NULL, 0);
		XBell(XtDisplay(w), 0);
		return;
	    }
	
	ctx->texted.mult *= mult;
    }

/*	Function Name: TransposeCharacters
 *	Description: Swaps the character to the left of the mark with
 *                   the character to the right of the mark.
 *	Arguments: w - the text widget.
 *                 event - the event that cause this action.
 *                 params, num_params *** NOT USED ***.
 *	Returns: none.
 */

static void TransposeCharacters(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	XeTextEdWidget ctx = (XeTextEdWidget) w;

	StartMoving(ctx, e);
	EndMoving(ctx);
    }

static void MoveForwardParagraph(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
    }

static void MoveBackwardParagraph(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
    }

/*
** *************************************************************
** First cract at..
** Selection/paste processing modified from Athena Xaw3d sources..
** *************************************************************
*/
struct SelectionList
    {
	String *params;
	Cardinal count;
	Time time;
    };

static void GetSelection();

/*
** (XtSelectionCallbackProc)
*/
static void SelectionReceived(w,client_data,selection,type,value,length,format)
Widget w;
XtPointer client_data;
Atom *selection, *type;
XtPointer value;
unsigned long *length;
int *format;
    {
	XeTextEdWidget t = (XeTextEdWidget)w;
  
	if (*type == 0 /*XT_CONVERT_FAIL*/ || *length == 0)
	    {
		struct SelectionList* list =
			(struct SelectionList *)client_data;
		if (list != NULL)
			GetSelection(w, list->time, list->params, list->count);
	    }
	else if (StartEditing(t, XeTextInsertMode_CURRENT, (XEvent *)NULL))
	    {
		_XeTextFeedContent(t->text.inserting, (char *)value,
				  (int)((*length * *format)/8));
		EndEditing(t, 0);
	    }
	XtFree(client_data);
	XFree(value);	/* the selection value should be freed with XFree */
    }

static void GetSelection(w, time, params, num_params)
Widget w;
Time time;
String *params;			/* selections in precedence order */
Cardinal num_params;
    {
	Atom selection;
	int buffer =  0;
	
	selection = XInternAtom(XtDisplay(w), *params, False);
	switch (selection)
	    {
	    case XA_CUT_BUFFER7: ++buffer;
	    case XA_CUT_BUFFER6: ++buffer;
	    case XA_CUT_BUFFER5: ++buffer;
	    case XA_CUT_BUFFER4: ++buffer;
	    case XA_CUT_BUFFER3: ++buffer;
	    case XA_CUT_BUFFER2: ++buffer;
	    case XA_CUT_BUFFER1: ++buffer;
	    case XA_CUT_BUFFER0:
		    {
			int nbytes;
			unsigned long length;
			Atom type = XA_STRING;
			int fmt8 = 8;
			char *line = XFetchBuffer(XtDisplay(w),&nbytes,buffer);

			if ((length = nbytes))
				SelectionReceived(w, (XtPointer)NULL,
						  &selection, &type,
						  (XtPointer)line, &length,
						  &fmt8);
			else if (num_params > 1)
				GetSelection(w, time, params+1, num_params-1);
		    }
		break;
	    default:
		    {
			Display *d = XtDisplay(w);
			struct SelectionList* list;
			if (--num_params)
			    {
				list = XtNew(struct SelectionList);
				list->params = params + 1;
				list->count = num_params;
				list->time = time;
			    }
			else
				list = NULL;
			XtGetSelectionValue(w, selection, XA_TEXT(d),
					    SelectionReceived,
					    (XtPointer)list, time);
		    }
		break;
	    }
    }

static void InsertSelection(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	if (((XeTextEdWidget)w)->texted.allow_edit)
		GetSelection(w, NoteTime(e), p, *n);
	else
		XBell(XtDisplay(w), 0);
    }

/*
** NoOp (from MIT Athena Xaw)
**
**	This action performs no action, and allows the user or
**	application programmer to unbind a translation.
**
** Note: If the parameter list contains the string "RingBell" then
**       this action will ring the bell.
*/
static void NoOp(w, e, p, n)
Widget w; XEvent *e; String *p; Cardinal *n;
    {
	if (*n != 1)
		return;
	
	switch(p[0][0])
	    {
	    case 'R':
	    case 'r':
		XBell(XtDisplay(w), 0);
	    default:			/* Fall Through */
		break;
	    }
    }

/*
** User callable convenience function support
** ==========================================
*/

/*
** IsTextEdWidget
**	Returns True, if the widget is of class xeTextEd and
**	False otherwise (also, a warning message is generated).
*/
static int IsTextEdWidget(w)
Widget w;
    {
	String params[2];

	if (XtIsSubclass(w, xeTextEdWidgetClass))
		return True;
	params[0] = w->core.name;
	params[1] = w->core.widget_class->core_class.class_name;
	XeWidgetWarningMsg
		(w, "bugTextEdNotTextEd",
		 "Widget %s is class %s, and not xeTextEdWidget",
		 (String *)params, (Cardinal)2);
	return False;
    }

/*
** FeedFile
**	Routine to write exported data to file.
*/
#if NeedFunctionPrototypes
static int FeedFile(char *, int, XeTextTag, XtPointer);
#endif
static int FeedFile(data, length, tag, client_data)
char *data;
int length;
XeTextTag tag;
XtPointer client_data;
    {
	return length > 0 ? fwrite(data, 1, length, (FILE *)client_data) : 0;
    }

/*
** XeTextSaveAsFile
**	Save the current content of the Text widget into a file. The format
**	of the exported file is determined from by the 'exportFormat'
**	resource (text.export_format). Returns True, if save succeeds
**	and False otherwise.
*/
Boolean XeTextSaveAsFile(w, name)
Widget w;
String name;
    {
	XeTextEdWidget t = (XeTextEdWidget)w;
	XeTextPosition dot[2];
	FILE *fp;

	if (!IsTextEdWidget(w))
		return False;
	if (name == NULL)
	    {
		XeWidgetWarningMsg
			((Widget)t, "noteTextEdNoFilename",
			 "No filename given",
			 (String *)NULL, (Cardinal)0);
		return False;
	    }
	if ((fp = fopen(name, "wb")) == NULL)
	    {
		XeWidgetWarningMsg
			((Widget)t, "noteTextEdCannotWrite",
			 "Cannot write to file %s",
			 (String *)&name, (Cardinal)1);
		return False;
	    }
	dot[0].snip = t->text.first;
	dot[0].offset = 0;
	dot[1].snip = NULL;
	dot[1].offset = 0;
	XeTextExtractContent((XeTextWidget)t, &dot[0], &dot[1],
			     t->text.export_format, FeedFile, (XtPointer)fp);
	fclose(fp);
	return True;
    }

/*
** XeTextGetString
**	Save the current content of the Text widget into a String. The format
**	of the exported data is determined from by the 'exportFormat'
**	resource (text.export_format). Returns a pointer to the allocated
**	String. The actual length of the string is returned into long
**	integer 'length' (if non-NULL).
**
**	Application must release the returned string with XtFree.
**
**	For convenience, the actual memory allocation is 'length+1' and this
**	extra byte is set to NUL. In simple applications, the returned value
**	can often be treated as NUL terminated string and the length return
**	ignored.
*/
String XeTextGetString(w, length)
Widget w;
long *length;
    {
	XeTextEdWidget t = (XeTextEdWidget)w;
	XeTextPosition dot[2];
	TextSelectionStruct s;

	if (!IsTextEdWidget(w))
		return NULL;
	s.w = t;
	s.length = 0;
	s.size = 0;
	s.data = 0;
	dot[0].snip = t->text.first;
	dot[0].offset = 0;
	dot[1].snip = NULL;
	dot[1].offset = 0;
	XeTextExtractContent((XeTextWidget)t, &dot[0], &dot[1],
			     t->text.export_format, FeedSelect, (XtPointer)&s);
	if (length)
		*length = s.length;
	if (s.length)
	    {
		s.data = XtRealloc(s.data, s.length + 1);
		*((char *)s.data + s.length) = 0;
	    }
	return (String)s.data;
    }

/*
** XeTextGetSubstring
**	Same as XeTextGetString, except only characters between the given
**	character positions are returned.
*/
String XeTextGetSubstring(w, length, start, end)
Widget w;
long *length, start, end;
    {
	XeTextEdWidget t = (XeTextEdWidget)w;
	XeTextPosition dot[2];
	XPoint xy[2];
	long range[2];
	TextSelectionStruct s;

	if (!IsTextEdWidget(w))
		return NULL;
	s.w = t;
	s.length = 0;
	s.size = 0;
	s.data = 0;
	range[0] = start;
	range[1] = end;
	XeOffset2Locations(t, range, 2, dot, xy);
	XeTextExtractContent((XeTextWidget)t, &dot[0], &dot[1],
			     t->text.export_format, FeedSelect, (XtPointer)&s);
	if (length)
		*length = s.length;
	if (s.length)
	    {
		s.data = XtRealloc(s.data, s.length + 1);
		*((char *)s.data + s.length) = 0;
	    }
	return (String)s.data;
    }

/*
** XeTextExtract
**	Extract a range of characters from the content using the supplied
**	callback function.
*/
void XeTextExtract(w, start, end, feed, client_data)
Widget w;
long start, end;
XeTextExtractFeed feed;
XtPointer client_data;
    {
	XeTextEdWidget t = (XeTextEdWidget)w;
	XeTextPosition dot[2];
	XPoint xy[2];
	long range[2];

	if (!IsTextEdWidget(w))
		return;
	range[0] = start;
	range[1] = end;
	XeOffset2Locations(t, range, 2, dot, xy);
	XeTextExtractContent((XeTextWidget)t, &dot[0], &dot[1],
			     t->text.export_format, feed, client_data);
    }

/*
** XeTextReplaceTagged
**	Replace content of the text widget between positions (start, end)
**	with new the content. This can be used inserting text by having
**	start == end. The new content will have the specified tag value.
**
**	Insert position will be at the end of the inserted text.
**
**	This function will unconditionally open a new insert context (it
**	will terminate any old one first) and leave it open.
**
*/
void XeTextReplaceTagged(w, start, end, value, length, tag)
Widget w;
long start, end;
char *value;
long length;
XeTextTag tag;
    {
	XeTextEdWidget t = (XeTextEdWidget)w;
	int amount;

	if (!IsTextEdWidget(w))
		return;
	/*
	** Simulate a move to the replace position
	*/
	StartMoving(t, (XEvent *)NULL);
	if (start > end)
	    {
		long tmp = start;
		start = end;
		end = tmp;
	    }
	t->texted.cursor_position = start;
	t->texted.cursor_valid = False;
	EndMoving(t);
	/*
	** Do the actual replace/insert
	*/
	if (!StartEditing(t, XeTextInsertMode_RESET, (XEvent *)NULL))
		return;
	amount = end - t->texted.cursor_position;
	if (amount > 0)
	    {
		Snip *s;

		s = _XeTextDeleteContent(t->text.inserting,
					 end - t->texted.cursor_position);
		while (s)
			_XeDeleteSnip(&s);
	    }
	else
		amount = 0;
	(void)_XeTextSetTag(t->text.inserting, tag);
	if (length > 0 && value)
		_XeTextFeedContent(t->text.inserting, value, length);
	EndEditing(t, amount);
    }

/*
** XeTextInsert
**	inserts a string into current insertion point (an insertion state is
**	opened with defaults, if none is previously open. This function can
**	be used to add more input to the insertion point opened earlier, it
**	does not cause the input state being reset from the initial state
**	like XeTextReplace[Tagged] does.
*/
void XeTextInsert(w, value, length)
Widget w;
char *value;
long length;
    {
	XeTextEdWidget t = (XeTextEdWidget)w;
	if (!IsTextEdWidget(w))
		return;
	if (!StartEditing(t, XeTextInsertMode_CURRENT, (XEvent *)NULL))
		return;
	if (length > 0 && value)
		_XeTextFeedContent(t->text.inserting, value, length);
	EndEditing(t, 0);
    }

/*
** XeTextReplace
**	Replace content of the text widget between positions (start, end)
**	with new the content. This can be used inserting text by having
**	start == end.
**
**	Insert position will be at the end of the inserted text.
**
*/
void XeTextReplace(w, start, end, value, length)
Widget w;
long start, end;
char *value;
long length;
    {
	XeTextReplaceTagged(w, start, end, value, length, (XeTextTag)0);
    }

/*
** XeTextGetInsertionPoint
**	Return the current insert position.
*/
long XeTextGetInsertionPoint(w)
Widget w;
    {
	if (!IsTextEdWidget(w))
		return 0;
	return ((XeTextEdWidget)w)->texted.cursor_position;
    }

/*
** XeTextGetInsertionTag
**	Return the current value of tag at insertion point.
*/
XeTextTag XeTextGetInsertionTag(w)
Widget w;
    {
	XeTextEdWidget t = (XeTextEdWidget)w;

	if (!IsTextEdWidget(w))
		return 0;
	if (t->text.inserting)
		return _XeTextGetTag(t->text.inserting);
	else if (t->texted.cursor_location.snip)
		return t->texted.cursor_location.snip->mode.tag;
	return 0;
    }

/*
** XeTextGetSelectionPosition
**	Return currently selected text range. If the returned start and
**	end are equal, no selection is currently active. The returned
**	'start' will alway be less or equal to the returned 'end'.
*/
void XeTextGetSelectionPosition(w, start, end)
Widget w;
long *start, *end;
    {
	XeTextEdWidget t = (XeTextEdWidget)w;

	if (!IsTextEdWidget(w))
		*start = *end = 0;
	else if (t->texted.s.range[0] > t->texted.s.range[1])
	    {
		*start = t->texted.s.range[1];
		*end = t->texted.s.range[0];
	    }
	else
	    {
		*start = t->texted.s.range[0];
		*end = t->texted.s.range[1];
	    }
    }	

/*
** XeTextSetSelection
**	Set/Change current selection to (start,end).
*/
void XeTextSetSelection(w, start, end)
Widget w;
long start, end;
    {
	XeTextEdWidget t = (XeTextEdWidget)w;

	if (!IsTextEdWidget(w))
		return;
	t->texted.time = XtLastTimestampProcessed(XtDisplay(w));
	t->texted.s.range[0] = start;
	t->texted.s.range[1] = end;
	InitRefreshRegion(t);
	TextSetSelection(t, (String *)NULL, 0);
	ChangeSelection(t, False);
	DoRefreshRegion(t);
    }

/*
** XeTextUnsetSelection
**	Clear current selection.
*/
void XeTextUnsetSelection(w)
Widget w;
    {
	XeTextEdWidget t = (XeTextEdWidget)w;

	if (!IsTextEdWidget(w))
		return;
	InitRefreshRegion(t);
	while (t->texted.s.atom_count > 0)
	    {
		Atom sel = t->texted.s.selections[0];
		if (GetCutBufferNumber(sel) < 0)
			XtDisownSelection(w, sel, t->texted.time);
		LoseSelection(w, &sel);
	    }
	DoRefreshRegion(t);
    }

