/*
** RasterImport module for importing TIFF images into Xew internal
** raster format.
**
** Copyright 1992, 1993, 1994 by Markku Savela and
**	Technical Research Centre of Finland
*/
#if ANSI_INCLUDES
#	include <stdlib.h>
#endif
#include <stdio.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/IntrinsicP.h>
#include <X11/Xmd.h>
#include <X11/StringDefs.h>
#include <X11/Xew/RasterP.h>
#include "ImageTools.h"
#include "RasterTIFF.h"

#ifndef NO_TIFF
#	include "tiffio.h"
#endif

static void RasterImportWarning(w, msg)
XeRasterWidget w;
char *msg;
    {
	XeWidgetWarningMsg
		((Widget)w, "rasterImportWarning", msg, (String *)NULL, 0);
    }


/*
** AdjustLine
**	Widen the (bitspersample < 8) scanlines up to 8 bits per sample.
**	Assume this is not time critical at the moment and do it hard way.
**
**	-- Also, for now.. support only values 2 and 4.
*/
static void AdjustLine(line, bits, width)
unsigned char *line;
int bits, width;
    {
	unsigned char x, *in, *out;

	switch (bits)
	    {
	    default:
		break;
	    case 2:
		width += ((4-width) & 3); /* Make width disible by 4 */
		for (in = &line[width/4], out = &line[width]; out > line;)
		    {
			x = *--in;
			*--out = x & 0x3;
			*--out = (x>>2) & 0x3;
			*--out = (x>>4) & 0x3;
			*--out = (x>>6) & 0x3;
		    }
	    case 4:
		width += (width & 1); /* Make width even */
		for (in = &line[width/2], out = &line[width]; out > line;)
		    {
			x = *--in;
			*--out = x & 0xf;
			*--out = (x>>4) & 0xf;
		    }
		break;
	    case 16:
		for (in = out = line; --width >= 0; in +=2)
			*out++ = (*(INT16 *)in + 128) >> 8;
		break;
	    }
    }

XeRawImage *XeImport_TIFF(w)
XeRasterWidget w;
    {
#ifdef NO_TIFF
	RasterImportWarning(w, "TIFF Support not compiled.");
	return NULL;
#else
	TIFF *in;
	unsigned long width, height;
	unsigned short *red, *green, *blue;
	unsigned short samplesperpixel, bitspersample, planarconfig;
	unsigned short photometric, orientation;
	float resX, resY;
	unsigned short resU;
	register unsigned long row;
	unsigned char *data;
	int i, linesize, planes;
	XeRawImage *raw = NULL;

	if ((in = TIFFOpen(w->basic.content_file, "r")) == NULL)
	    {
		XeWidgetWarningMsg
			((Widget)w,
			 "rasterImportWarning",
			 "Could not open TIFF %s",
			 &w->basic.content_file,
			 1);
		goto import_failed;
	    }
	TIFFGetFieldDefaulted(in, TIFFTAG_SAMPLESPERPIXEL, &samplesperpixel);
	TIFFGetFieldDefaulted(in, TIFFTAG_BITSPERSAMPLE, &bitspersample);
	raw = _XeCreateRawImage(samplesperpixel);
	raw->samples_per_pixel = samplesperpixel;
	raw->bits_per_sample = bitspersample;
	raw->bits_per_component = 8;
	/*
	** Load Colormap if defined, regardless whether photometric
	** indicates it is needed or not.
	*/
	if (TIFFGetField(in, TIFFTAG_COLORMAP, &red, &green, &blue))
	    {
		int mapsize = (1<<bitspersample);
		unsigned char *r, *g, *b;
		int x;

		raw->alloc_map = True;
		raw->color_map = (unsigned char *)XtMalloc(3 * mapsize);
		/*
		** Convert 16-bit colormap to 8-bit colormap
		*/
		r = raw->color_map;
		g = r + mapsize;
		b = g + mapsize;
		for (x = mapsize; --x >= 0; )
		    {
			r[x] = (red[x] * 255) / 65535;
			g[x] = (green[x] * 255) / 65535;
			b[x] = (blue[x] * 255) / 65535;
		    }
	    }
	if (TIFFGetField(in, TIFFTAG_PHOTOMETRIC, &photometric) == 0)
	    {
		/*
		** Missing PHOTOMETRIC.. faulty TIFF. Make some
		** guess so that at least something will be shown.
		*/
		RasterImportWarning(w, "PHOTOMETRIC tag missing from TIFF");
		if (samplesperpixel > 1)
			photometric = PHOTOMETRIC_RGB;
		else if (raw->color_map != NULL)
			photometric = PHOTOMETRIC_PALETTE;
		else
			photometric = PHOTOMETRIC_MINISBLACK;
			
	    }
	TIFFGetFieldDefaulted(in, TIFFTAG_ORIENTATION, &orientation);
	if (orientation != ORIENTATION_TOPLEFT)
		XeWidgetWarningMsg
			((Widget)w,
			 "rasterImportWarning",
			 "Warning orientation=%d",	/* uhh.. --msa */
			 (String *)&orientation,	/* ..... --msa */
			 1);
	TIFFGetField(in, TIFFTAG_IMAGEWIDTH, &width);
	raw->width = width;
	TIFFGetField(in, TIFFTAG_IMAGELENGTH, &height);
	raw->height = height;
	if (TIFFGetField(in, TIFFTAG_XRESOLUTION, &resX))
		raw->x_resolution = resX;
	else
		raw->x_resolution = 1.0;
	if (TIFFGetField(in, TIFFTAG_YRESOLUTION, &resY))
		raw->y_resolution = resY;
	else
		raw->y_resolution = 1.0;
	TIFFGetFieldDefaulted(in, TIFFTAG_RESOLUTIONUNIT, &resU);
	raw->resolution_unit =
		resU == RESUNIT_NONE ? XeImageUnit_NONE :
		resU == RESUNIT_INCH ? XeImageUnit_INCH :
		resU == RESUNIT_CENTIMETER ? XeImageUnit_CM : XeImageUnit_NONE;
	raw->bytes_per_line = linesize = TIFFScanlineSize(in);
	if (linesize < raw->width && bitspersample != 1)
		raw->bytes_per_line = raw->width;
	raw->alloc_data = True;
	TIFFGetField(in, TIFFTAG_PLANARCONFIG, &planarconfig);
	if (raw->num_channels > 1 && planarconfig == PLANARCONFIG_SEPARATE)
	    {
		raw->data = data = (unsigned char *)
			XtMalloc(raw->num_channels*raw->bytes_per_line*height);
		for (i = 0; i < raw->num_channels; ++i)
		    {
			raw->channel[i].addr =
				raw->data + i * raw->bytes_per_line;
			raw->channel[i].w = raw->width;
			raw->channel[i].h = raw->height;
			raw->channel[i].inc = 1;
			raw->channel[i].line =
				raw->num_channels * raw->bytes_per_line;
		    }
		planes = raw->num_channels;
	    }
	else
	    {
		raw->data = data = (unsigned char *)
			XtMalloc(raw->bytes_per_line * height);
		for (i = 0; i < raw->num_channels; ++i)
		    {
			raw->channel[i].addr = raw->data + i;
			raw->channel[i].w = raw->width;
			raw->channel[i].h = raw->height;
			raw->channel[i].inc = raw->num_channels;
			raw->channel[i].line = raw->bytes_per_line;
		    }
		planes = 1;
	    }
	if (data != NULL)
	    {
		for (row = 0; row < height; row++)
			for (i = 0; i < planes; ++i, data+=raw->bytes_per_line)
				if (TIFFReadScanline(in, data, row, i) < 0)
					goto bad_file;
				else if (bitspersample!=8 && bitspersample!=1)
					AdjustLine(data, bitspersample, width);
	    bad_file:
		if (photometric == PHOTOMETRIC_MINISWHITE)
		    {
			/*
			** If 0 = White, we have to invert the image
			*/
			row = linesize * height;
			data = raw->data;
			for ( ; row > 0; row -= 1, data += 1)
				*data = ~*data;
		    }
	    }
	switch (photometric)
	    {
	    case PHOTOMETRIC_MINISWHITE:
	    case PHOTOMETRIC_MINISBLACK:
		if (bitspersample == 1)
			raw->class = XeImageClass_BILEVEL;
		else
			raw->class = XeImageClass_GRAYSCALE;
		break;
	    case PHOTOMETRIC_YCBCR:
		raw->color_space = XeImageSpace_YCbCr;
		raw->class = XeImageClass_FULLCOLOR;
		break;
	    case PHOTOMETRIC_RGB:
		raw->color_space = XeImageSpace_RGB;
		raw->class = XeImageClass_FULLCOLOR;
		break;
	    case PHOTOMETRIC_PALETTE:
		raw->color_space = XeImageSpace_RGB;
		raw->class = XeImageClass_PALETTE;
		break;
	    default:
		raw->class = XeImageClass_UNKNOWN;
		break;
	    }
	TIFFClose(in);
	return raw;
	/*
	** Import failed, cleanup
	*/
    import_failed:
	_XeDestroyRawImage(raw);
	return NULL;
#endif
    }
