/*
** RasterPBM.c
**	Import module for importing PBM, PGM or PPM format images into Xew.
**
** Copyright 1994 by Markku Savela and
**	Technical Research Centre of Finland
*/
#if SYSV_INCLUDES
#	include <memory.h>	/* memset() et al. */
#endif
#if ANSI_INCLUDES
#	include <stdlib.h>
#endif
#include <stdio.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xew/RasterP.h>
#include "ImageTools.h"
#include "RasterPBM.h"

static void RasterImportWarning(w, msg)
XeRasterWidget w;
char *msg;
    {
	XeWidgetWarningMsg
		((Widget)w, "rasterImportWarning", msg, (String *)NULL, 0);
    }


static int getint(p, e)
unsigned char **p, *e;
    {
	int c, n;
	unsigned char *s = *p;
	/*
	** Just skip all non-numeric data until one of '0'..'9'
	** or end of content is reached.
	*/
	while (1)
	    {
		if (s == e)
		    {
			c = '0';  /* Fake '0' forever at end */
			break;
		    }
		else if ((c = *s++) == '#')
			while (s < e && (c = *s++) != '\n');
		else if (c >= '0' && c <= '9') 
			break;
	    }
	n = 0;
	do
		n = n * 10 + c - '0';
	while (s < e && (c = *s++) >= '0' && c <= '9');
	*p = s;
	return n;
    }

static int getbit(p, e)
unsigned char **p, *e;
    {
	int c;
	unsigned char *s = *p;
	/*
	** Just skip all non-numeric data until one of '0'..'9'
	** or end of content is reached.
	*/
	while (1)
	    {
		if (s == e)
		    {
			c = '0';  /* Fake '0' forever at end */
			break;
		    }
		else if ((c = *s++) == '#')
			while (s < e && (c = *s++) != '\n');
		else if (c >= '0' && c <= '9') 
			break;
	    }
	*p = s;
	return c != '0';
    }

static void loadpbm(pic, p, e, maxv, raw)
XeRawImage *pic;
unsigned char *p, *e;
int maxv, raw;
    {
	unsigned char *row, *ptr, *old;
	int i, j, size;

	pic->samples_per_pixel = 1;
	pic->bits_per_sample = 1;
	pic->bits_per_component = 1;
	pic->bytes_per_line = (pic->width + 7) / 8;
	size = pic->height * pic->bytes_per_line;

	if (e - p < size)
	    {
		/*
		** Cannot use the initial allocation, allocate a new image
		*/
		old = pic->data;
		pic->data = (unsigned char *)XtMalloc(size);
	    }
	else
		old = NULL;
	pic->channel[0].addr = row = pic->data;
	if (!raw)
	    {
		int pix, m;

		for (i = 0; i < pic->height; i++, row += pic->bytes_per_line)
		    {
			for (pix=j=0, ptr=row, m=0x80; j < pic->width; j++)
			    {
				if (!getbit(&p, e))
					pix |= m;
				if ((m >>= 1) == 0)
				    {
					*ptr++ = pix;
					m = 0x80;
					pix = 0;
				    }
			    }
			if (m)
				*ptr = pix;
		    }
	    }
	else
	    {
		pic->channel[0].addr = p;
		for (i = 0; i < size; i++, p++)
			*p = ~ *p;
	    }
	pic->channel[0].line = pic->bytes_per_line;
	pic->channel[0].inc = 1;
	pic->channel[0].w = pic->width;
	pic->channel[0].h = pic->height;
	XtFree((char *)old);
    }


static void loadpgm(pic, p, e, maxv, raw)
XeRawImage *pic;
unsigned char *p, *e;
int maxv, raw;
    {
	int i, shift, size;
	unsigned char scale[256], *ptr;
	
	pic->samples_per_pixel = 1;
	pic->bits_per_sample = 8;
	pic->bits_per_component = 8;
	pic->bytes_per_line = pic->width;

	/* if maxv>255, keep dropping bits until it's reasonable */
	shift = 0;
	while (maxv > 255)
	    {
		maxv = maxv >> 1;
		shift++;
	    }
	for (i = 0; i <= maxv; i++)
		scale[i] = (i * 255) / maxv;
	size = pic->width * pic->bytes_per_line;
	if (e - p < size)
	    {
		/* Cope with short files */
		i = p - pic->data;
		pic->data = (unsigned char *)
			XtRealloc((char *)pic->data, i + size);
		e = pic->data + ((e - p) + i);
		p = pic->data + i;
	    }
	if (raw)
	    {
		pic->channel[0].addr = p;
		if (maxv < 255)
			for (i = 0; i < size; i++, p++)
				*p = scale[*p];
	    }
	else
	    {
		ptr = pic->data;
		for (i = 0; i < size; i++)
			*ptr++ = scale[(getint(&p, e) >> shift) & 255];
		/* Discard extra allocated space */
		pic->data = (unsigned char *)XtRealloc((char *)pic->data,size);
		pic->channel[0].addr = pic->data;
	    }
	pic->channel[0].line = pic->bytes_per_line;
	pic->channel[0].inc = 1;
	pic->channel[0].w = pic->width;
	pic->channel[0].h = pic->height;
    }

static void loadppm(pic, p, e, maxv, raw)
XeRawImage *pic;
unsigned char *p, *e;
int maxv, raw;
    {
	unsigned char scale[256], *ptr, *base;
	int i, shift, size;
	
	pic->samples_per_pixel = 3;
	pic->bits_per_sample = 8;
	pic->bits_per_component = 8;
	pic->bytes_per_line = 3 * pic->width;
	size = pic->height * pic->bytes_per_line;

	/* if maxv>255, keep dropping bits until it's reasonable */
	shift = 0;
	while (maxv > 255)
	    {
		maxv = maxv>>1;
		shift++;
	    }
	for (i = 0; i <= maxv; i++)
		scale[i] = (i * 255) / maxv;
	if (e - p < size)
	    {
		/* Cope with short files */
		i = p - pic->data;
		pic->data = (unsigned char *)
			XtRealloc((char *)pic->data, i + size);
		e = pic->data + ((e - p) + i);
		p = pic->data + i;
	    }
	if (raw)
	    {
		base = p;
		if (maxv < 255)
			for (i = 0; i < size; i++, p++)
				*p = scale[*p];
	    }
	else
	    {
		ptr = pic->data;
		for (i = 0; i < size; i++)
			*ptr++ = scale[(getint(&p, e) >> shift) & 255];
		/* Discard extra allocated space */
		base = pic->data = (unsigned char *)XtRealloc
			((char *)pic->data, size);
	    }
	for (i = 0; i < 3; i++)
	    {
		pic->channel[i].addr = base + i;
		pic->channel[i].w = pic->width;
		pic->channel[i].h = pic->height;
		pic->channel[i].inc = 3;
		pic->channel[i].line = pic->bytes_per_line;
	    }
    }

static unsigned char *LoadFromStream(w, f, l)
XeRasterWidget w;
FILE *f;
int *l;
    {
	unsigned char *data;
	/*
	** find the size of the file. This really won't work for pipes
	** and network connections, should be fixed --msa
	*/
	fseek(f, 0L, 2);
	*l = w->basic.content_length = ftell(f);
	fseek(f, 0L, 0);
	data = (unsigned char *)XtMalloc(*l);
	if (fread(data, *l, 1, f) == 1)
		return data;
	XtFree((void *)data);
	RasterImportWarning(w, "PBM data read failed");
	return NULL;
    }

/*
** XeImport_PBM
*/
XeRawImage *XeImport_PBM(w)
XeRasterWidget w;
    {
	unsigned char *data = NULL, *end, *p;
	int length, width, height, maxv;
	XeRawImage *raw;

	/*
        ** Mapping from PBM format ID to Xew internal image class
	*/
	typedef struct FormatMapping
	    {
		XeImageClass class;
		int raw;	/* Non-zero, if RAW PBM (non-ASCII) format */
		int channels;	/* Number of channels */
	    } FormatMapping;
	FormatMapping *fmt;

	static FormatMapping format[] = /* Index by numeric value after 'P' */
	    {
		{ XeImageClass_UNKNOWN,1,1 },	/* P0 not used */
		{ XeImageClass_BILEVEL,0,1 },	/* P1 = Ascii bitmap */
		{ XeImageClass_GRAYSCALE,0,1 },	/* P2 = Ascii greymap */
		{ XeImageClass_FULLCOLOR,0,3 },	/* P3 = Ascii pixmap */
		{ XeImageClass_BILEVEL,1,1 },	/* P4 = Raw bitmap */
		{ XeImageClass_GRAYSCALE,1,1 },	/* P5 = Raw greymap */
		{ XeImageClass_FULLCOLOR,1,3 },	/* P6 = Raw pixmap */
	    };

	if (w->basic.content_string != NULL)
	    {
		length = w->basic.content_length;
		data = (unsigned char *)XtMalloc(length);
		memcpy((void *)data, (void *)w->basic.content_string, length);
	    }
	else if (w->basic.content_stream != NULL)
		data = LoadFromStream(w, w->basic.content_stream, &length);
	else if (w->basic.content_file != NULL)
	    {
		FILE *fl = fopen(w->basic.content_file, "rb");
		if (fl == NULL)
		    {
			String params[1];
			params[0] = w->basic.content_file;
			XeWidgetWarningMsg((Widget)w, "cannotOpen",
					   "Cannot open file '%s'",
					   params, 1);
			goto import_failed;
		    }
		data = LoadFromStream(w, fl, &length);
		fclose(fl);
	    }
	else
		goto import_failed;
	p = data;
	end = data + length;
	if (p[0] != 'P' || p[1] < '1' || p[1] > '6')
	    {
		RasterImportWarning(w, "Not a known PBM format");
		goto import_failed;
	    }
	p += 2;
	fmt = &format[data[1] - '0'];
	/*
	** read in header information
	*/
	width = getint(&p, end);
	height = getint(&p, end);
	/*
	** if we're not reading a bitmap, read the 'max value'
	*/
	if (fmt->class != XeImageClass_BILEVEL)
		maxv = getint(&p, end);
	else
		maxv = 1;
	if (width < 1 || height < 1 || maxv < 1)
	    {
		RasterImportWarning(w, "Bad PBM header information");
		goto import_failed;
	    }
	raw = _XeCreateRawImage(fmt->channels);
	raw->class = fmt->class;
	raw->width = width;
	raw->height = height;
	raw->alloc_data = True;
	raw->data = data;
	switch (fmt->class)
	    {
	    default:
	    case XeImageClass_BILEVEL:
		loadpbm(raw, p, end, maxv, fmt->raw);
		break;
	    case XeImageClass_GRAYSCALE:
		loadpgm(raw, p, end, maxv, fmt->raw);
		break;
	    case XeImageClass_FULLCOLOR:
		loadppm(raw, p, end, maxv, fmt->raw);
		break;
	    }
	return raw;
	/*
	** Failed in load, cleanup
	*/
    import_failed:
	XtFree((void *)data);
	return NULL;
    }  




