/*
** ImageTools.c
**
** Copyright 1992, 1993, 1994 by Markku Savela and
**	Technical Research Centre of Finland
**
** Some of the ideas and code snippets in this module have been taken
** from the Berkeley MPEG decoder.
*/
#include <stdio.h>
#if SYSV_INCLUDES
#	include <memory.h>
#endif
#if ANSI_INCLUDES
#	include <stddef.h>
#	include <stdlib.h>
#endif
#include <assert.h>
#include <math.h>

#include <X11/Xlib.h>
#include <X11/IntrinsicP.h>
#include <X11/Xmd.h>
#include <X11/StringDefs.h>
#include <X11/Xew/VideoP.h>
#include "BasicColor.h"
#include "ImageTools.h"
#include "Raster24to8.h"

#ifdef SH_MEM
#	include <sys/ipc.h>
#	include <sys/shm.h>
#	include <X11/extensions/XShm.h>
#endif

#define MONO(rd,gn,bl) (((rd)*11 + (gn)*16 + (bl)*5) >> 5)  /*.33R+ .5G+ .17B*/

/*
** Functions InitSpline, EvalSpline and GenerateFSGamma have been taken
** from John Bradley's xv package. Currently these are only used in
** computing the Floyd-Steinberg gamma curve. Should probably just compute
** the curve offline and compile it statically in... --msa
** The following copyright applies these three functions:
*/
/*
 * Copyright 1989, 1990, 1991, 1992 by John Bradley and
 *                       The University of Pennsylvania
 *
 * Permission to use, copy, and distribute for non-commercial purposes,
 * is hereby granted without fee, providing that the above copyright
 * notice appear in all copies and that both the copyright notice and this
 * permission notice appear in supporting documentation. 
 *
 * The software may be modified for your own purposes, but modified versions
 * may not be distributed.
 *
 * This software is provided "as is" without any expressed or implied warranty.
 *
 * The author may be contacted via:
 *    US Mail:   John Bradley
 *               GRASP Lab, Room 301C
 *               3401 Walnut St.  
 *               Philadelphia, PA  19104
 *
 *    Phone:     (215) 898-8813
 *    EMail:     bradley@cis.upenn.edu       
 */
static void InitSpline(x, y, n, y2)
int *x, *y;
int n;
double *y2;
    {
	/* given arrays of data points x[0..n-1] and y[0..n-1], computes the
	   values of the second derivative at each of the data points
	   y2[0..n-1] for use in the splint function */
	
	int i,k;
	double p,qn,sig,un,u[10];
	
	y2[0] = u[0] = 0.0;
	
	for (i=1; i<n-1; i++) 
	    {
		sig = ((double) x[i]-x[i-1]) / ((double) x[i+1] - x[i-1]);
		p = sig * y2[i-1] + 2.0;
		y2[i] = (sig-1.0) / p;
		u[i] = (((double) y[i+1]-y[i]) / (x[i+1]-x[i])) - 
			(((double) y[i]-y[i-1]) / (x[i]-x[i-1]));
		u[i] = (6.0 * u[i]/(x[i+1]-x[i-1]) - sig*u[i-1]) / p;
	    }
	qn = un = 0.0;
	
	y2[n-1] = (un-qn*u[n-2]) / (qn*y2[n-2]+1.0);
	for (k=n-2; k>=0; k--)
		y2[k] = y2[k]*y2[k+1]+u[k];
    }

static double EvalSpline(xa, ya, y2a, n, x)
int xa[], ya[];
double y2a[];
int n;
double x;
    {
	int klo,khi,k;
	double h,b,a;
	
	klo = 0;
	khi = n-1;
	while (khi-klo > 1)
	    {
		k = (khi+klo) >> 1;
		if (xa[k] > x) khi = k;
		else klo = k;
	    }
	h = xa[khi] - xa[klo];
	a = (xa[khi]-x)/h;
	b = (x-xa[klo])/h;
	return (a*ya[klo] + b*ya[khi] +
		((a*a*a-a)*y2a[klo] + (b*b*b-b)*y2a[khi]) * (h*h) / 6.0);
    }

static unsigned char fs_gamcr[256];

static void GenerateFSGamma()
    {
	/* this function generates the Floyd-Steinberg gamma curve (fs_gamcr)

	   This function generates a 4 point spline curve to be used as a 
	   non-linear grey 'colormap'.  Two of the points are nailed down at
	   0,0 and 255,255, and can't be changed.  You specify the other two.
	   If you specify points on the line (0,0 - 255,255), you'll get the
	   normal linear reponse curve.  If you specify points of 50,0 and
	   200,255, you'll get grey values of 0-50 to map to black (0), and
	   grey values of 200-255 to map to white (255) (roughly).  Values
	   between 50 and 200 will cover the output range 0-255.  The reponse
	   curve will be slightly 's' shaped. */
	
	int i,j;
	static int x[4] = {0,32,224,255};
	static int y[4] = {0, 0,255,255};
	double yf[4];
	
	InitSpline(x, y, 4, yf);
	
	for (i = 0; i < 256; i++)
	    {
		j = (int) EvalSpline(x, y, yf, 4, (double) i);
		if (j < 0)
			j = 0;
		else if (j >255)
			j = 255;
		fs_gamcr[i] = j;
	    }
    }
/*
** ************************
** End Of Extract from 'xv'
** ************************
*/
/*
** CLAMP_WITH_ARRAY
**	If in your processor
**		x = map[y]
**	is faster than
**		x = y < 0 ? 0 : y > 255 ? 255: y;
**	then define CLAMP_WITH_ARRAY.
*/
#define CLAMP_WITH_ARRAY 1

#if CLAMP_WITH_ARRAY
static unsigned char fs_clamp[256+512];

#	define CLAMP(e) e = fs_clamp[256+(e)]
#else
#	define CLAMP(e)			\
	do				\
	    {				\
		if ((e) & ~0xFF)	\
			if (e < 0)	\
				e = 0;	\
			else		\
				e = 255;\
	    }				\
	while (0)
#endif


/*
** InitFSDither
**	Precompute Floyd-Sternberg error constants for each
**	pixel value in range 0..255.
**
**		----   x    7/16
**		3/16  5/16  1/16
*/
static struct
    {
	int e1;	/* 7/16 */
	int e2;	/* 1/16 */
	int e3;	/* 5/16 */
	int e4;	/* 3/16 (or actually, remainder) */
    }
fs_error[512];



static void InitFSDither()
    {
	int i;

	GenerateFSGamma();
	for (i = -256; i < 0 ; i++)
	    {
		fs_error[256+i].e1 = -((8 -7*i) / 16);
		fs_error[256+i].e2 = -((8 -i  ) / 16);
		fs_error[256+i].e3 = -((8 -5*i) / 16);
		fs_error[256+i].e4 = i - fs_error[256+i].e1 -
			fs_error[256+i].e2 - fs_error[256+i].e3;
	    }
	for (i = 0; i < 256 ; i++)
	    {
		fs_error[256+i].e1 = (7*i+8) / 16;
		fs_error[256+i].e2 = (i  +8) / 16;
		fs_error[256+i].e3 = (5*i+8) / 16;
		fs_error[256+i].e4 = i - fs_error[256+i].e1 -
			fs_error[256+i].e2 - fs_error[256+i].e3;
	    }
#if CLAMP_WITH_ARRAY
	for (i = 0; i < 256; i++)
		fs_clamp[256+i] = i;
	for (i = 512; i < (sizeof(fs_clamp) / sizeof(fs_clamp[0])); i++)
		fs_clamp[i] = 255;
#endif
    }


/*
**	================================================================
**	The following macro definitions have been defined only to
**	minimize typing errors when tuning up the functions. Due to
**	optimizations the code is manually expanded in some places and
**	same sequences of steps keep repeating.
**
**	These definitions do not make the code more readable, quite
**	the opposite is true.
**	================================================================
*/

/*
** COLLECT_ROW
**	accumulates information from the scanline that will
**	be skipped due to scaling down process.
**
**	This macro assumes the following variables defined:
**
**	srcA	start address of the pixel row
**	srcinc	increment for getting next source pixel in row
**	srcW	width of the src row in pixels
**	dstW	width of the dst row in pixels
**	pW	an array[dstW] to accumulate the information
**
**	Note:	when multiplying pixels, only the *first* pW in such
**		case has valid value.
*/
#define COLLECT_ROW					\
do							\
    {							\
	unsigned char *src_ptr = srcA;			\
	int Psum = 0;					\
	dCol = 0;					\
	if (dstW == srcW)				\
	    {						\
		for (; dCol < dstW;)			\
		    {					\
			pW[dCol++] += in[*src_ptr];	\
			src_ptr += srcinc;		\
		    }					\
	    }						\
	else for (; dCol < dstW;)			\
	    {						\
		do					\
		    {					\
			pW[dCol] += in[*src_ptr];	\
			src_ptr += srcinc;		\
		    }					\
		while ((Psum += dstW) < srcW);		\
		do					\
			dCol++;				\
		while ((Psum -= srcW) >= srcW);		\
	    }						\
    }							\
while (0)

/*
** INIT_FS4_ERROR
**	setup lst_err and nxt_err pointers.
*/
#define INIT_FS4_ERROR(errbuf)				\
do							\
    {							\
	nxt_err = (errbuf);				\
	lst_err = (nxt_err) + 1;			\
    }							\
while (0)

/*
** PROPAGATE_FS4_ERROR
**	compute Floyd-Sternberg error terms forward. This macro
**	assumes the following variables exist:
**
**	e	error value to propagate (new forward value left in)
**	nxt_err	an array to store the error terms of the next line
**	dCol	the current column number (incremented)
*/
#define PROPAGATE_FS4_ERROR				\
do							\
    {							\
	nxt_err[dCol++] = e2 + fs_error[256+e].e4;	\
	e2 = e1 + fs_error[256+e].e3;			\
	e1 = fs_error[256+e].e2;			\
	e  = fs_error[256+e].e1;			\
    }							\
while (0)

/*
** END_FS4_ERROR
**	perform the tasks necessary at the end of row for FS dither.
**
**	e2	error term (see above PROPAGATE_FS4_ERROR)
**	nxt_err	an array to store the error terms of the next line
**	lst_err	an array to store the error terms of the previouis line
*/
#define END_FS4_ERROR nxt_err[dstW] = e2

/*
** GetWorkSpace
**	Allocate temporary workspace for the image operations
**
**	The current implementation keeps the work space permanently
**	allocated and resizes dynamically to the maximun need. This
**	may not work if threads are used (that would require allocation
**	to be widget specific at least).
*/
static int *GetWorkSpace(size)
int size;
    {
	static int *wrkbuf;	/* Working space buffers */
	static int wrksiz;	/* Current size of the working space */

	if (wrksiz < size)
	    {
		if (wrkbuf == NULL)
			wrkbuf = (int *)XtMalloc(size);
		else
			wrkbuf = (int *)XtRealloc((char *)wrkbuf, size);
		wrksiz = size;
	    }
	return wrkbuf;
    }

/*
** ScaleDitherChannel1to1
**	Scale 1 bit image into 1 bit image and apply 
**	Floyd-Steinberg dithering while doing it...
**
**		----   x    7/16
**		3/16  5/16  1/16
**
**	********************************************
**	* This filter ignores the dst->inc field!! *
**	* This filter ignores the src->inc field!! *
**	********************************************
*/
static void ScaleDitherChannel1to1(widget, in, val, out, dst, src)
XeBasicWidget widget;
unsigned char *in, *val, *out;
XeImageChannel *dst, *src;
    {
	int dstW = dst->w;
	int dstH = dst->h;
	int srcW = src->w;
	int srcH = src->h;
	unsigned char *srcA = src->addr;
	unsigned char *dstA = dst->addr;
	int dCol, dRow, Qsum, hCount = 0, wCount;
	int *nxt_err, *lst_err, *pW;
	int *errbuf;

	assert(dstW > 0);
	assert(srcW > 0);

	errbuf = GetWorkSpace(((dstW+2) + dstW) * sizeof(int));
	INIT_FS4_ERROR(errbuf);
	pW = lst_err + dstW + 1;
	memset((char *)lst_err, 0, (&pW[dstW] - lst_err) * sizeof(int));
	for (Qsum = dRow = 0; dRow < dstH; srcA += src->line)
	    {
		hCount += 1;
		if ((Qsum += dstH) < srcH)
		    {
			unsigned char *src_ptr = srcA;
			int Psum  = 0, s = 0x80;
			for (dCol = 0; dCol < dstW; )
			    {
				do
				    {
					pW[dCol] += in[(s & *src_ptr) != 0];
					if ((s >>= 1) == 0)
					    {
						s = 0x80;
						src_ptr++;
					    }
				    }
				while ((Psum += dstW) < srcW);
				do
					dCol++;
				while ((Psum -= srcW) >= srcW);
			    }
			continue;
		    }
		do
		    {
			unsigned char *src_ptr = srcA;
			unsigned char *dst_ptr = dstA;
			int m = 0x80, s = 0x80;
			int e = 0, e1 = 0, e2 = 0;
			unsigned char pix = 0;
			int Psum;

			for (dCol = Psum = 0; dCol < dstW;)
			    {
				int v = pW[dCol];

				wCount = 0;
				do
				    {
					v += in[(s & *src_ptr) != 0];
					wCount += hCount;
					if ((s >>= 1) == 0)
					    {
						src_ptr++;
						s = 0x80;
					    }
				    }
				while ((Psum += dstW) < srcW);
				if (wCount > 1)
					v = (v + wCount/2)/wCount;
				do
				    {
					e += v + lst_err[dCol];
					CLAMP(e);
					if (e > 127)
					    {
						pix |= m;
						e -= 255;
					    }
					PROPAGATE_FS4_ERROR;
					if ((m >>= 1) == 0)
					    {
						m = 0x80;
						*dst_ptr++ = pix;
						pix = 0;
					    }
					if (dCol == dstW)
						goto end_row;
				    }
				while ((Psum -= srcW) >= srcW);
			    }
		    end_row:
			memset((void *)pW, 0, dstW * sizeof(int));
			if (m != 0x80)
				*dst_ptr = pix;
			END_FS4_ERROR;
			dstA += dst->line;
			dRow += 1;
		    }
		while ((Qsum -= srcH) >= srcH && dRow < dstH);
		hCount = 0;
	    }
    }
/*
** ScaleDitherChannel8to1
**	Scale and dither Gray scale image into 1 bit image using
**	Floyd-Steinberg dithering:
**
**		----   x    7/16
**		3/16  5/16  1/16
**
**	********************************************
**	* This filter ignores the dst->inc field!! *
**	********************************************
**
*/
static void ScaleDitherChannel8to1(widget, in, val, out, dst, src)
XeBasicWidget widget;
unsigned char *in, *val, *out;
XeImageChannel *dst, *src;
    {
	int dstW = dst->w;
	int dstH = dst->h;
	int srcW = src->w;
	int srcH = src->h;
	int srcinc = src->inc;
	unsigned char *srcA = src->addr;
	unsigned char *dstA = dst->addr;
	int dCol, dRow, Qsum, hCount = 0, wCount;
	int *nxt_err, *lst_err, *pW;
	int *errbuf;

	assert(dstW > 0);
	assert(srcW > 0);
	errbuf = GetWorkSpace(((dstW+2) + dstW) * sizeof(int));
	INIT_FS4_ERROR(errbuf);
	pW = lst_err + dstW + 1;
	memset((char *)lst_err, 0, (&pW[dstW] - lst_err) * sizeof(int));
	if (srcW == dstW && srcH == dstH)
	    {
		for (dRow = 0; dRow < dstH; srcA += src->line)
		    {
			unsigned char *src_ptr = srcA;
			unsigned char *dst_ptr = dstA;
			int m = 0x80, e = 0, e1 = 0, e2 = 0;
			unsigned char pix = 0;

			for (dCol = 0; dCol < dstW; )
			    {
				e += in[*src_ptr] + lst_err[dCol];
				CLAMP(e);
				src_ptr += srcinc;
				if (e > 127)
				    {
					pix |= m;
					e -= 255;
				    }
				PROPAGATE_FS4_ERROR;
				if ((m >>= 1) == 0)
				    {
					m = 0x80;
					*dst_ptr++ = pix;
					pix = 0;
				    }
			    }
			if (m != 0x80)
				*dst_ptr = pix;
			END_FS4_ERROR;
			dstA += dst->line;
			dRow += 1;
		    }
	    }
	else for (Qsum = dRow = 0; dRow < dstH; srcA += src->line)
	    {
		hCount += 1;
		if ((Qsum += dstH) < srcH)
		    {
			COLLECT_ROW;
			continue;
		    }
		do
		    {
			unsigned char *src_ptr = srcA;
			unsigned char *dst_ptr = dstA;
			int m = 0x80;
			int e = 0, e1 = 0, e2 = 0;
			unsigned char pix = 0;

			if (dstW == srcW && hCount == 1)
			    {
				for (dCol = 0; dCol < dstW; )
				    {
					e += in[*src_ptr]+lst_err[dCol];
					CLAMP(e);
					src_ptr += srcinc;
					if (e > 127)
					    {
						pix |= m;
						e -= 255;
					    }
					PROPAGATE_FS4_ERROR;
					if ((m >>= 1) == 0)
					    {
						m = 0x80;
						*dst_ptr++ = pix;
						pix = 0;
					    }
				    }
			    }
			else
			    {
				int Psum;

				for (dCol = Psum = 0;;)
				    {
					int v = pW[dCol];

					wCount = 0;
					do
					    {
						v += in[*src_ptr];
						src_ptr += srcinc;
						wCount += hCount;
					    }
					while ((Psum += dstW) < srcW);
					if (wCount > 1)
						v = (v+wCount/2)/wCount;
					do
					    {
						e += v + lst_err[dCol];
						CLAMP(e);
						if (e > 127)
						    {
							pix |= m;
							e -= 255;
						    }
						PROPAGATE_FS4_ERROR;
						if ((m >>= 1) == 0)
						    {
							m = 0x80;
							*dst_ptr++ = pix;
							pix = 0;
						    }
						if (dCol == dstW)
							goto end_row;
					    }
					while ((Psum -= srcW) >= srcW);
				    }
			    end_row:
				memset((void *)pW, 0, dstW * sizeof(int));
			    }
			if (m != 0x80)
				*dst_ptr = pix;
			END_FS4_ERROR;
			dstA += dst->line;
			dRow += 1;
		    }
		while ((Qsum -= srcH) >= srcH && dRow < dstH);
		hCount = 0;
	    }
    }

static void ScaleDitherChannel8to8(widget, in, val, out, dst, src)
XeBasicWidget widget;
unsigned char *in, *val, *out;
XeImageChannel *dst, *src;
    {
	int dstW = dst->w;
	int dstH = dst->h;
	int dstinc = dst->inc;
	int srcW = src->w;
	int srcH = src->h;
	int srcinc = src->inc;
	unsigned char *srcA = src->addr;
	unsigned char *dstA = dst->addr;
	int dCol, dRow, Qsum, hCount;
	int *nxt_err, *lst_err, *pW;
	int *errbuf;

	assert(dstW > 0);
	assert(srcW > 0);
	errbuf = GetWorkSpace(((dstW+2) + dstW) * sizeof(int));
	INIT_FS4_ERROR(errbuf);
	pW = lst_err + dstW + 1;
	memset((char *)lst_err, 0, (&pW[dstW] - lst_err) * sizeof(int));
	if (srcW == dstW && srcH == dstH)
	    {
		for (dRow = 0; dRow < dstH; srcA += src->line)
		    {
			unsigned char *src_ptr = srcA;
			unsigned char *dst_ptr = dstA;
			int e = 0, e1 = 0, e2 = 0;
			
			for (dCol = 0; dCol < dstW; )
			    {
				e += in[*src_ptr] + lst_err[dCol];
				src_ptr += srcinc;
				CLAMP(e);
				*dst_ptr += out[e];
				dst_ptr += dstinc;
				e -= val[e];
				PROPAGATE_FS4_ERROR;
			    }
			END_FS4_ERROR;
			dstA += dst->line;
			dRow++;
		    }
	    }
	else for (hCount = Qsum = dRow = 0; dRow < dstH; srcA += src->line)
	    {
		hCount += 1;
		if ((Qsum += dstH) < srcH)
		    {
			COLLECT_ROW;
			continue;
		    }
		do
		    {
			unsigned char *src_ptr = srcA;
			unsigned char *dst_ptr = dstA;
			int e = 0, e1 = 0, e2 = 0;

			dCol = 0;
			if (dstW == srcW && hCount == 1)
			    {
				for (; dCol < dstW; )
				    {
					e += in[*src_ptr] + lst_err[dCol];
					src_ptr += srcinc;
					CLAMP(e);
					*dst_ptr += out[e];
					dst_ptr += dstinc;
					e -= val[e];
					PROPAGATE_FS4_ERROR;
				    }
			    }
			else
			    {
				int Psum = 0;
				for (;;)
				    {
					int v = pW[dCol], wCount = 0;

					do
					    {
						v += in[*src_ptr];
						src_ptr += srcinc;
						wCount += hCount;
					    }
					while ((Psum += dstW) < srcW);
					if (wCount != 1)
						v = (v + wCount / 2) / wCount;
					do
					    {
						e += v + lst_err[dCol];
						CLAMP(e);
						*dst_ptr += out[e];
						dst_ptr += dstinc;
						e -= val[e];
						PROPAGATE_FS4_ERROR;
						if (dCol == dstW)
							goto end_row;
					    }
					while ((Psum -= srcW) >= srcW);
				    }
			    end_row:
				memset((void *)pW, 0, dstW * sizeof(int));
			    }
			END_FS4_ERROR;
			dstA += dst->line;
			dRow++;
		    }
		while ((Qsum -= srcH) >= srcH && dRow < dstH);
		hCount = 0;
	    }
    }


static void ScaleMapChannel8to8(widget, in, val, out, dst, src)
XeBasicWidget widget;
unsigned char *in, *val, *out;
XeImageChannel *dst, *src;
    {
	int dstW = dst->w;
	int dstH = dst->h;
	int dstinc = dst->inc;
	int srcW = src->w;
	int srcH = src->h;
	int srcinc = src->inc;
	unsigned char *srcA = src->addr;
	unsigned char *dstA = dst->addr;
	int dCol, dRow, Qsum, hCount;
	int *pW;

	assert(dstW > 0);
	assert(srcW > 0);
	pW = GetWorkSpace(dstW * sizeof(int));
	memset((char *)pW, 0, dstW * sizeof(int));
	if (srcW == dstW && srcH == dstH)
	    {
		for (dRow = 0; dRow < dstH; srcA += src->line)
		    {
			unsigned char *src_ptr = srcA;
			unsigned char *dst_ptr = dstA;

			for (dCol = 0; dCol < dstW; dCol++)
			    {
				*dst_ptr += out[in[*src_ptr]];
				dst_ptr += dstinc;
				src_ptr += srcinc;
			    }
			dstA += dst->line;
			dRow += 1;
		    }
	    }
	else for (hCount = Qsum = dRow = 0; dRow < dstH; srcA += src->line)
	    {
		hCount += 1;
		if ((Qsum += dstH) < srcH)
		    {
			COLLECT_ROW;
			continue;
		    }
		do
		    {
			unsigned char *src_ptr = srcA;
			unsigned char *dst_ptr = dstA;
			int wCount;

			dCol = 0;
			if (dstW == srcW && hCount == 1)
			    {
				for ( ; dCol < dstW; dCol++)
				    {
					*dst_ptr += out[in[*src_ptr]];
					dst_ptr += dstinc;
					src_ptr += srcinc;
				    }
			    }
			else
			    {
				int Psum = 0;
				for (;;)
				    {
					int v = pW[dCol];

					wCount = 0;
					do
					    {
						v += in[*src_ptr];
						src_ptr += srcinc;
						wCount += hCount;
					    }
					while ((Psum += dstW) < srcW);
					if (wCount != 1)
						v = (v + wCount / 2) / wCount;
					v = out[v];
					do
					    {
						*dst_ptr += v;
						dst_ptr += dstinc;
						if (++dCol == dstW)
							goto end_row;
					    }
					while ((Psum -= srcW) >= srcW);
				    }
			    end_row:
				memset((void *)pW, 0, dstW * sizeof(int));
			    }
			dstA += dst->line;
			dRow += 1;
		    }
		while ((Qsum -= srcH) >= srcH && dRow < dstH);
		hCount = 0;
	    }
    }

static void ScaleMapChannel8to16(widget, in, val, out8, dst, src)
XeBasicWidget widget;
unsigned char *in, *val, *out8;
XeImageChannel *dst, *src;
    {
	INT16 *out = (INT16 *)out8;
	int dstW = dst->w;
	int dstH = dst->h;
	int dstinc = dst->inc;
	int srcW = src->w;
	int srcH = src->h;
	int srcinc = src->inc;
	unsigned char *srcA = src->addr;
	unsigned char *dstA = dst->addr;
	int dCol, dRow, Qsum, hCount;
	int *pW;

	assert(dstW > 0);
	assert(srcW > 0);
	pW = GetWorkSpace(dstW * sizeof(int));
	memset((char *)pW, 0, dstW * sizeof(int));
	if (srcW == dstW && srcH == dstH)
	    {
		for (dRow = 0; dRow < dstH; srcA += src->line)
		    {
			unsigned char *src_ptr = srcA;
			unsigned char *dst_ptr = dstA;

			for (dCol = 0; dCol < dstW; dCol++)
			    {
				*((INT16 *)dst_ptr) += out[in[*src_ptr]];
				dst_ptr += dstinc;
				src_ptr += srcinc;
			    }
			dstA += dst->line;
			dRow += 1;
		    }
	    }
	else for (hCount = Qsum = dRow = 0; dRow < dstH; srcA += src->line)
	    {
		hCount += 1;
		if ((Qsum += dstH) < srcH)
		    {
			COLLECT_ROW;
			continue;
		    }
		do
		    {
			unsigned char *src_ptr = srcA;
			unsigned char *dst_ptr = dstA;
			int wCount;

			dCol = 0;
			if (dstW == srcW && hCount == 1)
			    {
				for ( ; dCol < dstW; dCol++)
				    {
					*((INT16 *)dst_ptr)+=out[in[*src_ptr]];
					dst_ptr += dstinc;
					src_ptr += srcinc;
				    }
			    }
			else
			    {
				int Psum = 0;
				for (;;)
				    {
					int v = pW[dCol];

					wCount = 0;
					do
					    {
						v += in[*src_ptr];
						src_ptr += srcinc;
						wCount += hCount;
					    }
					while ((Psum += dstW) < srcW);
					if (wCount != 1)
						v = (v + wCount / 2) / wCount;
					v = out[v];
					do
					    {
						*((INT16 *)dst_ptr) += v;
						dst_ptr += dstinc;
						if (++dCol == dstW)
							goto end_row;
					    }
					while ((Psum -= srcW) >= srcW);
				    }
			    end_row:
				memset((void *)pW, 0, dstW * sizeof(int));
			    }
			dstA += dst->line;
			dRow += 1;
		    }
		while ((Qsum -= srcH) >= srcH && dRow < dstH);
		hCount = 0;
	    }
    }

static void ScaleDitherChannel8to16(widget, in, val, out8, dst, src)
XeBasicWidget widget;
unsigned char *in, *val, *out8;
XeImageChannel *dst, *src;
    {
	INT16 *out = (INT16 *)out8;
	int dstW = dst->w;
	int dstH = dst->h;
	int dstinc = dst->inc;
	int srcW = src->w;
	int srcH = src->h;
	int srcinc = src->inc;
	unsigned char *srcA = src->addr;
	unsigned char *dstA = dst->addr;
	int dCol, dRow, Qsum, hCount;
	int *nxt_err, *lst_err, *pW;
	int *errbuf;

	assert(dstW > 0);
	assert(srcW > 0);
	errbuf = GetWorkSpace(((dstW+2) + dstW) * sizeof(int));
	INIT_FS4_ERROR(errbuf);
	pW = lst_err + dstW + 1;
	memset((char *)lst_err, 0, (&pW[dstW] - lst_err) * sizeof(int));
	if (srcW == dstW && srcH == dstH)
	    {
		for (dRow = 0; dRow < dstH; srcA += src->line)
		    {
			unsigned char *src_ptr = srcA;
			unsigned char *dst_ptr = dstA;
			int e = 0, e1 = 0, e2 = 0;
			
			for (dCol = 0; dCol < dstW; )
			    {
				e += in[*src_ptr] + lst_err[dCol];
				src_ptr += srcinc;
				CLAMP(e);
				*((INT16 *)dst_ptr) += out[e];
				dst_ptr += dstinc;
				e -= val[e];
				PROPAGATE_FS4_ERROR;
			    }
			END_FS4_ERROR;
			dstA += dst->line;
			dRow++;
		    }
	    }
	else for (hCount = Qsum = dRow = 0; dRow < dstH; srcA += src->line)
	    {
		hCount += 1;
		if ((Qsum += dstH) < srcH)
		    {
			COLLECT_ROW;
			continue;
		    }
		do
		    {
			unsigned char *src_ptr = srcA;
			unsigned char *dst_ptr = dstA;
			int e = 0, e1 = 0, e2 = 0;

			dCol = 0;
			if (dstW == srcW && hCount == 1)
			    {
				for (; dCol < dstW; )
				    {
					e += in[*src_ptr] + lst_err[dCol];
					src_ptr += srcinc;
					CLAMP(e);
					*((INT16 *)dst_ptr) += out[e];
					dst_ptr += dstinc;
					e -= val[e];
					PROPAGATE_FS4_ERROR;
				    }
			    }
			else
			    {
				int Psum = 0;
				for (;;)
				    {
					int v = pW[dCol], wCount = 0;

					do
					    {
						v += in[*src_ptr];
						src_ptr += srcinc;
						wCount += hCount;
					    }
					while ((Psum += dstW) < srcW);
					if (wCount != 1)
						v = (v + wCount / 2) / wCount;
					do
					    {
						e += v + lst_err[dCol];
						CLAMP(e);
						*((INT16 *)dst_ptr) += out[e];
						dst_ptr += dstinc;
						e -= val[e];
						PROPAGATE_FS4_ERROR;
						if (dCol == dstW)
							goto end_row;
					    }
					while ((Psum -= srcW) >= srcW);
				    }
			    end_row:
				memset((void *)pW, 0, dstW * sizeof(int));
			    }
			END_FS4_ERROR;
			dstA += dst->line;
			dRow++;
		    }
		while ((Qsum -= srcH) >= srcH && dRow < dstH);
		hCount = 0;
	    }
    }


static void ScaleMapChannel8to32(widget, in, val, out8, dst, src)
XeBasicWidget widget;
unsigned char *in, *val, *out8;
XeImageChannel *dst, *src;
    {
	INT32 *out = (INT32 *)out8;
	int dstW = dst->w;
	int dstH = dst->h;
	int dstinc = dst->inc;
	int srcW = src->w;
	int srcH = src->h;
	int srcinc = src->inc;
	unsigned char *srcA = src->addr;
	unsigned char *dstA = dst->addr;
	int dCol, dRow, Qsum, hCount;
	int *pW;

	assert(dstW > 0);
	assert(srcW > 0);
	pW = GetWorkSpace(dstW * sizeof(int));
	memset((char *)pW, 0, dstW * sizeof(int));
	if (srcW == dstW && srcH == dstH)
	    {
		for (dRow = 0; dRow < dstH; srcA += src->line)
		    {
			unsigned char *src_ptr = srcA;
			unsigned char *dst_ptr = dstA;

			for (dCol = 0; dCol < dstW; dCol++)
			    {
				*((INT32 *)dst_ptr) += out[in[*src_ptr]];
				dst_ptr += dstinc;
				src_ptr += srcinc;
			    }
			dstA += dst->line;
			dRow += 1;
		    }
	    }
	else for (hCount = Qsum = dRow = 0; dRow < dstH; srcA += src->line)
	    {
		hCount += 1;
		if ((Qsum += dstH) < srcH)
		    {
			COLLECT_ROW;
			continue;
		    }
		do
		    {
			unsigned char *src_ptr = srcA;
			unsigned char *dst_ptr = dstA;
			int wCount;

			dCol = 0;
			if (dstW == srcW && hCount == 1)
			    {
				for ( ; dCol < dstW; dCol++)
				    {
					*((INT32 *)dst_ptr)+=out[in[*src_ptr]];
					dst_ptr += dstinc;
					src_ptr += srcinc;
				    }
			    }
			else
			    {
				int Psum = 0;
				for (;;)
				    {
					int v = pW[dCol];

					wCount = 0;
					do
					    {
						v += in[*src_ptr];
						src_ptr += srcinc;
						wCount += hCount;
					    }
					while ((Psum += dstW) < srcW);
					if (wCount != 1)
						v = (v + wCount / 2) / wCount;
					v = out[v];
					do
					    {
						*((INT32 *)dst_ptr) += v;
						dst_ptr += dstinc;
						if (++dCol == dstW)
							goto end_row;
					    }
					while ((Psum -= srcW) >= srcW);
				    }
			    end_row:
				memset((void *)pW, 0, dstW * sizeof(int));
			    }
			dstA += dst->line;
			dRow += 1;
		    }
		while ((Qsum -= srcH) >= srcH && dRow < dstH);
		hCount = 0;
	    }
    }

/*
** ScaleDropPixels8to8
**	Copy and scale from 8 bit to 8 bit by simple pixel dropping.
*/
static void ScaleDropPixels8to8(widget, in, val, out, dst, src)
XeBasicWidget widget;
unsigned char *in, *val, *out;
XeImageChannel *dst, *src;
    {
	int dstW = dst->w;
	int dstH = dst->h;
	int dstinc = dst->inc;
	int srcW = src->w;
	int srcH = src->h;
	int srcinc = src->inc;
	unsigned char *srcA = src->addr;
	unsigned char *dstA = dst->addr;
	int dCol, dRow, Qsum;

	assert(dstW > 0);
	assert(srcW > 0);
	if (srcW == dstW && srcH == dstH)
	    {
		for (dRow = 0; dRow < dstH; srcA += src->line)
		    {
			unsigned char *src_ptr = srcA;
			unsigned char *dst_ptr = dstA;

			for (dCol = 0; dCol < dstW; dCol++)
			    {
				*dst_ptr += out[in[*src_ptr]];
				dst_ptr += dstinc;
				src_ptr += srcinc;
			    }
			dstA += dst->line;
			dRow += 1;
		    }
	    }
	else for (Qsum = dRow = 0; dRow < dstH; srcA += src->line)
	    {
		if ((Qsum += dstH) < srcH)
			continue;
		do
		    {
			unsigned char *src_ptr = srcA;
			unsigned char *dst_ptr = dstA;

			dCol = 0;
			if (dstW == srcW)
			    {
				for ( ; dCol < dstW; dCol++)
				    {
					*dst_ptr += out[in[*src_ptr]];
					dst_ptr += dstinc;
					src_ptr += srcinc;
				    }
			    }
			else
			    {
				int Psum = 0;
				for (;;)
				    {
					int v = out[in[*src_ptr]];

					do
						src_ptr += srcinc;
					while ((Psum += dstW) < srcW);
					do
					    {
						*dst_ptr += v;
						dst_ptr += dstinc;
						if (++dCol == dstW)
							goto end_row;
					    }
					while ((Psum -= srcW) >= srcW);
				    }
			    end_row:
				;
			    }
			dstA += dst->line;
			dRow += 1;
		    }
		while ((Qsum -= srcH) >= srcH && dRow < dstH);
	    }
    }

/*
** ZeroChannel
**	Fill 'dst' channel with Zeroes.
**
**	THIS WORKS ONLY FOR "UNTOUCHED" CHANNEL DESCRIPTORS
**	(NO MIRROR/ROTATION JUGGLING DONE!!!)
*/
static void ZeroChannel(widget, in, val, out, dst, src)
XeBasicWidget widget;
unsigned char *in, *val, *out;
XeImageChannel *dst, *src;
    {
	memset(dst->addr, 0, dst->h * dst->line);
    }

/*
** AllocateColors8
**	Allocate colors for the colormappedd image specified by the
**	'src' channel.
**
**	The r, g and b tables must be at least 256 bytes long!
*/
static void AllocateColors8(widget, r, g, b, dst, src)
XeBasicWidget widget;
unsigned char *r, *g, *b;
XeImageChannel *dst, *src;
    {
	(void)_XeAllocColors(widget, 256, r, g, b,
			     src->addr, src->w, src->h, src->line);
    }

/*
** ColorMapping24to8H
**	Run a 24 bit src image through a color quantification and
**	build 8 bit image with same dimensions into dst.
**
**	The r, g and b tables must be at least 256 bytes long!
*/
static void ColorMapping24to8H(widget, r, g, b, dst, src)
XeBasicWidget widget;
unsigned char *r, *g, *b;
XeImageChannel *dst, *src;
    {
	int ncolors = widget->basic.max_colors;
	
	if (ncolors > 256)
		ncolors = 256;
	ncolors = Conv24to8(src, dst,		/* 24 -> 8 bit image */
			    ncolors,		/* Max colors to use */
			    r, g, b,		/* Colormaps to be filled */
			    XeColorQuantize_HECKBERT);
	(void)_XeAllocColors(widget, ncolors, r, g, b,
			     dst->addr, dst->w, dst->h, dst->line);
    }

/*
** ColorMapping24to8P
**	Run a 24 bit src image through a color quantification and
**	build 8 bit image with same dimensions into dst.
**
**	The r, g and b tables must be at least 256 bytes long!
*/
static void ColorMapping24to8P(widget, r, g, b, dst, src)
XeBasicWidget widget;
unsigned char *r, *g, *b;
XeImageChannel *dst, *src;
    {
	int ncolors = widget->basic.max_colors;
	
	if (ncolors > 256)
		ncolors = 256;
	ncolors = Conv24to8(src, dst,		/* 24 -> 8 bit image */
			    ncolors,		/* Max colors to use */
			    r, g, b,		/* Colormaps to be filled */
			    XeColorQuantize_PPMQUANT);
	(void)_XeAllocColors(widget, ncolors, r, g, b,
			     dst->addr, dst->w, dst->h, dst->line);
    }

/*
** CollapsRows
**	Collect information from 3 source channels into single
**	3*dstW int array.
*/
static void CollapsRows(Qsum, pW, srcA, src, dstW, dstH)
int *Qsum, *pW;
unsigned char **srcA;
XeImageChannel *src;
int dstW, dstH;
    {
	int i;
	/*
	** Collapsing rows from src
	*/
	for (i = 0; i < 3; i++)
	    {
		int hCount, Psum;
		int *w, *wEnd, W, inc;
		unsigned char *s;

		if (Qsum[i] >= src[i].h)
			continue;
		w = pW + i;
		wEnd = w + 3 * dstW;
		W = src[i].w;
		inc = src[i].inc;
		s = srcA[i];
		srcA[i] += src[i].line;
		hCount = (src[i].h - Qsum[i] + dstH - 1) / dstH;
		if (hCount == 1 && dstW == W)
		    {
			Qsum[i] += dstH;
			for (; w < wEnd; s += inc, w += 3)
				*w = *s;
			continue;
		    }
		/* Load the first line fast */
		for (Psum = 0; w < wEnd ;)
		    {
			int v = 0;
			do
				v += *s, s += inc;
			while((Psum += dstW) < W);
			*w = v;
			do
				w += 3;
			while((Psum -= W) >= W);
		    }
		/* Add the other lines (if any) */
		while ((Qsum[i] += dstH) < src[i].h)
		    {
			s = srcA[i];
			srcA[i] += src[i].line;
			for (Psum = 0, w = pW + i; w < wEnd; )
			    {
				do
					*w += *s, s += inc;
				while((Psum += dstW) < W);
				do
					w += 3;
				while((Psum -= W) >= W);
			    }
		    }
		/* Compute the average */
		for (Psum = 0, w = pW + i;;)
		    {
			int v = *w, wCount = 0;
			do
				wCount += hCount;
			while ((Psum += dstW) < W);
			if (wCount != 1)
				v = (v + wCount / 2) / wCount;
			do
			    {
				*w = v;
				if ((w += 3) == wEnd)
					goto done_it;
			    }
			while ((Psum -= W) >= W);
		    }
	    done_it:
		;
	    }
    }

static void ScaleConvertRGBto8(widget, in, val, out, dst, src)
XeBasicWidget widget;
unsigned char *in, *val, *out;
XeImageChannel *dst, *src;
    {
	int dstW = dst->w;
	int dstH = dst->h;
	int dstinc = dst->inc;
	int srcH[3];
	unsigned char *srcA[3];
	unsigned char *dstA = dst->addr;
	int dRow, Qsum[3], *pW, *wEnd, i;

	pW = GetWorkSpace(dstW * 3 * sizeof(int));
	wEnd = &pW[3*dstW];
	for (i = 0; i < 3; ++i)
	    {
		srcH[i] = src[i].h;
		srcA[i] = src[i].addr;
		Qsum[i] = 0;
	    }
	for (dRow = 0; dRow < dstH; )
	    {
		CollapsRows(Qsum, pW, srcA, src, dstW, dstH);
		/*
		** Copy collected data to the destination
		*/
		do
		    {
			unsigned char *dst_ptr = dstA;
			int *w;
			
			for (w = pW; w < wEnd; w += 3)
			    {
				int v[3];
				v[0] = w[0];
				v[1] = w[1];
				v[2] = w[2];
				
				*dst_ptr = out[MONO(v[0], v[1], v[2])];
				dst_ptr += dstinc;
			    }
			dstA += dst->line;
			dRow += 1;
			Qsum[0] -= srcH[0];
			Qsum[1] -= srcH[1];
			Qsum[2] -= srcH[2];
		    }
		while (Qsum[0] >= srcH[0] &&
		       Qsum[1] >= srcH[1] &&
		       Qsum[2] >= srcH[2] &&
		       dRow < dstH);
	    }
    }

static void ScaleCopyRGBto24(widget, in, val, out, dst, src)
XeBasicWidget widget;
unsigned char *in, *val, *out;
XeImageChannel *dst, *src;
    {
	int dstW = dst->w;
	int dstH = dst->h;
	int dstinc = dst->inc;
	int srcH[3];
	unsigned char *srcA[3];
	unsigned char *dstA = dst->addr;
	int dRow, Qsum[3], *pW, *wEnd, i;

	pW = GetWorkSpace(dstW * 3 * sizeof(int));
	wEnd = &pW[3*dstW];
	for (i = 0; i < 3; ++i)
	    {
		srcH[i] = src[i].h;
		srcA[i] = src[i].addr;
		Qsum[i] = 0;
	    }
	for (dRow = 0; dRow < dstH; )
	    {
		CollapsRows(Qsum, pW, srcA, src, dstW, dstH);
		/*
		** Copy collected data to the destination
		*/
		do
		    {
			unsigned char *dst_ptr = dstA;
			int *w;

			for (w = pW; w < wEnd; w += 3)
			    {
				dst_ptr[0] = w[0];
				dst_ptr[1] = w[1];
				dst_ptr[2] = w[2];
				dst_ptr += dstinc;
			    }
			dstA += dst->line;
			dRow += 1;
			Qsum[0] -= srcH[0];
			Qsum[1] -= srcH[1];
			Qsum[2] -= srcH[2];
		    }
		while (Qsum[0] >= srcH[0] &&
		       Qsum[1] >= srcH[1] &&
		       Qsum[2] >= srcH[2] &&
		       dRow < dstH);
	    }
    }

/*
** ycc_rgb_init
**	This is adapted from the IJG jpeg library (jdcolor.c)
*/
#define RIGHT_SHIFT(x,shft)	((x) >> (shft))
#define SCALEBITS	16	/* speedier right-shift on some machines */
#define ONE_HALF	((INT32) 1 << (SCALEBITS-1))
#define FIX(x)		((INT32) ((x) * (1L<<SCALEBITS) + 0.5))

static int * Cr_r_tab;		/* => table for Cr to R conversion */
static int * Cb_b_tab;		/* => table for Cb to B conversion */
static INT32 * Cr_g_tab;	/* => table for Cr to G conversion */
static INT32 * Cb_g_tab;	/* => table for Cb to G conversion */

static void ycc_rgb_init()
    {
	INT32 i, x2;

	Cr_r_tab = (int *)XtMalloc(XeSample_RANGE * sizeof(int));
	Cb_b_tab = (int *)XtMalloc(XeSample_RANGE * sizeof(int));
	Cr_g_tab = (INT32 *)XtMalloc(XeSample_RANGE * sizeof(INT32));
	Cb_g_tab = (INT32 *)XtMalloc(XeSample_RANGE * sizeof(INT32));
	
	for (i = 0; i < XeSample_RANGE; i++)
	    {
		/*
		** i is the actual input pixel value, in the range
		** 0..MAXJSAMPLE
		** The Cb or Cr value we are thinking of is x = i-MAXJSAMPLE/2
		*/
		x2 = 2*i - XeSample_MAX;	/* twice x */
		/* Cr=>R value is nearest int to 1.40200 * x */
		Cr_r_tab[i] = (int)
			RIGHT_SHIFT(FIX(1.40200/2) * x2 + ONE_HALF, SCALEBITS);
		/* Cb=>B value is nearest int to 1.77200 * x */
		Cb_b_tab[i] = (int)
			RIGHT_SHIFT(FIX(1.77200/2) * x2 + ONE_HALF, SCALEBITS);
		/* Cr=>G value is scaled-up -0.71414 * x */
		Cr_g_tab[i] = (- FIX(0.71414/2)) * x2;
		/* Cb=>G value is scaled-up -0.34414 * x */
		/* We also add in ONE_HALF so that need not do it in
		   inner loop */
		Cb_g_tab[i] = (- FIX(0.34414/2)) * x2 + ONE_HALF;
	    }
    }

/*
** ScaleConvertYCbCrTo24
**	Convert from YCbCr to RGB space. The destination is assumed
**	to have 24 bytes per pixel, and in format RGBRGBRGB...
*/
static void ScaleConvertYCbCrTo24(widget, in, val, out, dst, src)
XeBasicWidget widget;
unsigned char *in, *val, *out;
XeImageChannel *dst, *src;
    {
	int dstW = dst->w;
	int dstH = dst->h;
	int dstinc = dst->inc;
	int srcH[3];
	unsigned char *srcA[3];
	unsigned char *dstA = dst->addr;
	int dRow, Qsum[3], *pW, *wEnd, i;

	static int init;

	if (!init)
	    {
		init = 1;
		ycc_rgb_init();
	    }
	pW = GetWorkSpace(dstW * 3 * sizeof(int));
	wEnd = &pW[3*dstW];

	for (i = 0; i < 3; ++i)
	    {
		srcH[i] = src[i].h;
		srcA[i] = src[i].addr;
		Qsum[i] = 0;
	    }
	for (dRow = 0; dRow < dstH; )
	    {
		CollapsRows(Qsum, pW, srcA, src, dstW, dstH);
		/*
		** Copy collected data to the destination
		*/
		do
		    {
			unsigned char *dst_ptr = dstA;
			int *w;

			for (w = pW; w < wEnd; w += 3)
			    {
				register int r, g, b;
				/* w[0] == Y  */
				/* w[1] == Cb */
				/* w[2] == Cr */

				r = g = b = w[0];
				r += Cr_r_tab[w[2]];
				g += (int)RIGHT_SHIFT
					(Cb_g_tab[w[1]] + Cr_g_tab[w[2]],
					 SCALEBITS);
				b += Cb_b_tab[w[1]];
				CLAMP(r);
				CLAMP(g);
				CLAMP(b);
				dst_ptr[0] = r;
				dst_ptr[1] = g;
				dst_ptr[2] = b;
				dst_ptr += dstinc;
			    }
			dstA += dst->line;
			dRow += 1;
			Qsum[0] -= srcH[0];
			Qsum[1] -= srcH[1];
			Qsum[2] -= srcH[2];
		    }
		while (Qsum[0] >= srcH[0] &&
		       Qsum[1] >= srcH[1] &&
		       Qsum[2] >= srcH[2] &&
		       dRow < dstH);
	    }
    }

/*
** ScaleDitherXYZto8
**	A complex tranformation of 3 input channels to single
**	output channel in one phase.
*/

#undef INIT_FS4_ERROR
#define INIT_FS4_ERROR(d, errbuf)			\
do							\
    {							\
	nxt_err = (errbuf);				\
    }							\
while (0)
#undef PROPAGATE_FS4_ERROR
#define PROPAGATE_FS4_ERROR(i)				\
do							\
    {							\
	*err_ptr++ = e2[i] + fs_error[256+e[i]].e4;	\
	e2[i] = e1[i] + fs_error[256+e[i]].e3;		\
	e1[i] = fs_error[256+e[i]].e2;			\
	e[i]  = fs_error[256+e[i]].e1;			\
    }							\
while (0)
#undef END_FS4_ERROR
#define END_FS4_ERROR(i) *err_ptr = e2[i]

static void ScaleDitherXYZto8(widget, in3, val3, out, dst, src)
XeBasicWidget widget;
unsigned char *in3, *val3, *out;
XeImageChannel *dst, *src;
    {
	int dstW = dst->w;
	int dstH = dst->h;
	int dstinc = dst->inc;
	int srcH[3];
	unsigned char *srcA[3];
	unsigned char *dstA = dst->addr;
	unsigned char *in[3], *val[3];
	int dRow, Qsum[3], *pW, *wEnd, *nxt_err, i;

	pW = GetWorkSpace((dstW + dstW + 1) * 3 * sizeof(int));
	wEnd = &pW[3*dstW];
	INIT_FS4_ERROR(3, pW + 3*dstW);
	memset((void *)nxt_err, 0, (dstW + 1) * 3 * sizeof(int));
	for (i = 0; i < 3; ++i)
	    {
		srcH[i] = src[i].h;
		srcA[i] = src[i].addr;
		Qsum[i] = 0;
		in[i] = in3;
		val[i] = val3;
		in3 += 256;
		val3 += 256;
	    }
	for (dRow = 0; dRow < dstH; )
	    {
		CollapsRows(Qsum, pW, srcA, src, dstW, dstH);
		/*
		** Copy collected data to the destination
		*/
		do
		    {
			unsigned char *dst_ptr = dstA;
			int *err_ptr = nxt_err;
			int *w;
			int e[3], e1[3], e2[3], i;

			e[0] = e1[0] = e2[0] = 0;
			e[1] = e1[1] = e2[1] = 0;
			e[2] = e1[2] = e2[2] = 0;
			
			for (w = pW; w < wEnd; w += 3)
			    {
				e[0] += w[0] + err_ptr[3];
				CLAMP(e[0]);
				e[1] += w[1] + err_ptr[4];
				CLAMP(e[1]);
				e[2] += w[2] + err_ptr[5];
				CLAMP(e[2]);
				i = out[in[0][e[0]]+in[1][e[1]]+in[2][e[2]]];
				*dst_ptr = i;
				dst_ptr += dstinc;
				e[0] -= val[0][i];
				e[1] -= val[1][i];
				e[2] -= val[2][i];
				PROPAGATE_FS4_ERROR(0);
				PROPAGATE_FS4_ERROR(1);
				PROPAGATE_FS4_ERROR(2);
			    }
			END_FS4_ERROR(0);
			END_FS4_ERROR(1);
			END_FS4_ERROR(2);
			dstA += dst->line;
			dRow += 1;
			Qsum[0] -= srcH[0];
			Qsum[1] -= srcH[1];
			Qsum[2] -= srcH[2];
		    }
		while (Qsum[0] >= srcH[0] &&
		       Qsum[1] >= srcH[1] &&
		       Qsum[2] >= srcH[2] &&
		       dRow < dstH);
	    }
    }

/*
** ScaleMapXYZto8
**	A complex tranformation of 3 input channels to single
**	output channel in one phase.
*/
static void ScaleMapXYZto8(widget, in3, val3, out, dst, src)
XeBasicWidget widget;
unsigned char *in3, *val3, *out;
XeImageChannel *dst, *src;
    {
	int dstW = dst->w;
	int dstH = dst->h;
	int dstinc = dst->inc;
	int srcH[3];
	unsigned char *srcA[3];
	unsigned char *dstA = dst->addr;
	unsigned char *in[3], *val[3];
	int dRow, Qsum[3], *pW, *wEnd, i;

	pW = GetWorkSpace(dstW * 3 * sizeof(int));
	wEnd = &pW[3*dstW];

	for (i = 0; i < 3; ++i)
	    {
		srcH[i] = src[i].h;
		srcA[i] = src[i].addr;
		Qsum[i] = 0;
		in[i] = in3;
		val[i] = val3;
		in3 += 256;
		val3 += 256;
	    }
	for (dRow = 0; dRow < dstH; )
	    {
		CollapsRows(Qsum, pW, srcA, src, dstW, dstH);
		/*
		** Copy collected data to the destination
		*/
		do
		    {
			unsigned char *dst_ptr = dstA;
			int *w;

			for (w = pW; w < wEnd; w += 3)
			    {
				i = in[0][w[0]] + in[1][w[1]] +  in[2][w[2]];
				*dst_ptr = out[i];
				dst_ptr += dstinc;
			    }
			dstA += dst->line;
			dRow += 1;
			Qsum[0] -= srcH[0];
			Qsum[1] -= srcH[1];
			Qsum[2] -= srcH[2];
		    }
		while (Qsum[0] >= srcH[0] &&
		       Qsum[1] >= srcH[1] &&
		       Qsum[2] >= srcH[2] &&
		       dRow < dstH);
	    }
    }


static void ScaleDitherYCCto8(widget, in3, val3, out, dst, src)
XeBasicWidget widget;
unsigned char *in3, *val3, *out;
XeImageChannel *dst, *src;
    {
	int dstW = dst->w;
	int dstH = dst->h;
	int dstinc = dst->inc;
	int srcH[3];
	unsigned char *srcA[3];
	unsigned char *dstA = dst->addr;
	unsigned char *in[3], *val[3];
	int dRow, Qsum[3], *pW, *wEnd, *nxt_err, i;

	static int init;

	if (!init)
	    {
		init = 1;
		ycc_rgb_init();
	    }
	pW = GetWorkSpace((dstW + dstW + 1) * 3 * sizeof(int));
	wEnd = &pW[3*dstW];
	INIT_FS4_ERROR(3, wEnd);
	memset((void *)nxt_err, 0, (dstW + 1) * 3 * sizeof(int));
	for (i = 0; i < 3; ++i)
	    {
		srcH[i] = src[i].h;
		srcA[i] = src[i].addr;
		Qsum[i] = 0;
		in[i] = in3;
		val[i] = val3;
		in3 += 256;
		val3 += 256;
	    }
	for (dRow = 0; dRow < dstH; )
	    {
		CollapsRows(Qsum, pW, srcA, src, dstW, dstH);
		/*
		** Copy collected data to the destination
		*/
		do
		    {
			unsigned char *dst_ptr = dstA;
			int *err_ptr = nxt_err;
			int *w;
			int e[3], e1[3], e2[3], i;

			e[0] = e1[0] = e2[0] = 0;
			e[1] = e1[1] = e2[1] = 0;
			e[2] = e1[2] = e2[2] = 0;
			
			for (w = pW; w < wEnd; w += 3)
			    {
				/* w[0] == Y  */
				/* w[1] == Cb */
				/* w[2] == Cr */

				e[0] += w[0] + err_ptr[3];
				e[1] += w[0] + err_ptr[4];
				e[2] += w[0] + err_ptr[5];
				e[0] += Cr_r_tab[w[2]];
				e[1] += (int)RIGHT_SHIFT
					(Cb_g_tab[w[1]] + Cr_g_tab[w[2]],
					 SCALEBITS);
				e[2] += Cb_b_tab[w[1]];
				CLAMP(e[0]);
				CLAMP(e[1]);
				CLAMP(e[2]);
				i = out[in[0][e[0]]+in[1][e[1]]+in[2][e[2]]];
				*dst_ptr = i;
				dst_ptr += dstinc;
				e[0] -= val[0][i];
				e[1] -= val[1][i];
				e[2] -= val[2][i];
				PROPAGATE_FS4_ERROR(0);
				PROPAGATE_FS4_ERROR(1);
				PROPAGATE_FS4_ERROR(2);
			    }
			END_FS4_ERROR(0);
			END_FS4_ERROR(1);
			END_FS4_ERROR(2);
			dstA += dst->line;
			dRow += 1;
			Qsum[0] -= srcH[0];
			Qsum[1] -= srcH[1];
			Qsum[2] -= srcH[2];
		    }
		while (Qsum[0] >= srcH[0] &&
		       Qsum[1] >= srcH[1] &&
		       Qsum[2] >= srcH[2] &&
		       dRow < dstH);
	    }
    }

/*
** ScaleMapYCCto8
**	A complex tranformation of 3 input channels to single
**	output channel in one phase.
*/
static void ScaleMapYCCto8(widget, in3, val3, out, dst, src)
XeBasicWidget widget;
unsigned char *in3, *val3, *out;
XeImageChannel *dst, *src;
    {
	int dstW = dst->w;
	int dstH = dst->h;
	int dstinc = dst->inc;
	int srcH[3];
	unsigned char *srcA[3];
	unsigned char *dstA = dst->addr;
	unsigned char *in[3], *val[3];
	int dRow, Qsum[3], *pW, *wEnd, i;

	static int init;

	if (!init)
	    {
		init = 1;
		ycc_rgb_init();
	    }
	pW = GetWorkSpace(dstW * 3 * sizeof(int));
	wEnd = &pW[3*dstW];
	for (i = 0; i < 3; ++i)
	    {
		srcH[i] = src[i].h;
		srcA[i] = src[i].addr;
		Qsum[i] = 0;
		in[i] = in3;
		val[i] = val3;
		in3 += 256;
		val3 += 256;
	    }
	for (dRow = 0; dRow < dstH; )
	    {
		CollapsRows(Qsum, pW, srcA, src, dstW, dstH);
		/*
		** Copy collected data to the destination
		*/
		do
		    {
			unsigned char *dst_ptr = dstA;
			int *w;
			
			for (w = pW; w < wEnd; w += 3)
			    {
				int r, g, b;
				int Y = w[0];
				int Cb = w[1];
				int Cr = w[2];

				r = Y + Cr_r_tab[Cr];
				g = Y + (int)RIGHT_SHIFT
					(Cb_g_tab[Cb] + Cr_g_tab[Cr],
					 SCALEBITS);
				b = Y + Cb_b_tab[Cb];
				CLAMP(r);
				CLAMP(g);
				CLAMP(b);
				i = in[0][r] + in[1][g] + in[2][b];
				*dst_ptr = out[i];
				dst_ptr += dstinc;
			    }
			dstA += dst->line;
			dRow += 1;
			Qsum[0] -= srcH[0];
			Qsum[1] -= srcH[1];
			Qsum[2] -= srcH[2];
		    }
		while (Qsum[0] >= srcH[0] &&
		       Qsum[1] >= srcH[1] &&
		       Qsum[2] >= srcH[2] &&
		       dRow < dstH);
	    }
    }


/*
** MakeImageChannel8
**	allocate temporary space for 8bit image channel to the destination
**	using the width and height of the source channel. The channel is
**	initialized to NULL content.
*/
static void MakeImageChannel8(widget, in, val, out, dst, src)
XeBasicWidget widget;
unsigned char *in, *val, *out;
XeImageChannel *dst, *src;
    {
	dst->addr = (unsigned char *)XtCalloc(1, src->w * src->h);
	dst->w = src->w;
	dst->h = src->h;
	dst->line = dst->w;
	dst->inc = 1;
    }

/*
** MakeImageChannel24
**	allocate temporary space for 24bit image channel to the destination
**	using the width and height of the source channel. The channel is
**	initialized to NULL content.
*/
static void MakeImageChannel24(widget, in, val, out, dst, src)
XeBasicWidget widget;
unsigned char *in, *val, *out;
XeImageChannel *dst, *src;
    {
	dst->addr = (unsigned char *)XtCalloc(1, 3 * src->w * src->h);
	dst->w = src->w;
	dst->h = src->h;
	dst->line = 3 * dst->w;
	dst->inc = 3;
    }

/*
** FreeImageChannel
**	release space allocated to a *temporary* channel defined by dst.
**
**	DO NOT CALL THIS FUNCTION FOR "PERMANENT" CHANNELS!
*/
static void FreeImageChannel(widget, in, val, out, dst, src)
XeBasicWidget widget;
unsigned char *in, *val, *out;
XeImageChannel *dst, *src;
    {
	XtFree((char *)dst->addr);
    }

/*
** InitDisplay_COLORCUBE
**	Initialize Pseudocolor visual for TRUECOLOR images. Allocate
**	colorcube and initialize mapping tables.
**
**	Only RGB and YCbCr colorspaces now!
*/
static void InitDisplay_COLORCUBE(w, im, src, dither, quant)
XeBasicWidget w;
XeImage *im;
XeRawImage *src;
int dither, quant;
    {
	/*
	** Build the mapping tables along the following principles:
	**
	** Each input p [0..255] from channel i [0..2] is mapped into
	** smaller value k in range [0..R[i]-1]. The index to the
	** final X11 pixel value is computed from the k values as
	** follows:
	**	index =  k0 * R[1] * R[2] + k1 * R[2] + k2
	** The R[i] values are computed such that following holds
	**	R[0] * R[1] * R[2] <= maxColors
	** and
	**	R[0] : R[1] : R[2] = 3:3:2 (for RGB)
	**	R[0] : R[1] : R[2] = 2:1:1 (for YCbCr)
	** The actual number of colors to be used will be
	**	ncolors = R[0] * R[1] * R[2] <= maxColors
	**
	** For each channel the mapping tables are:
	**
	** in[channel][sample]
	**	contains precomputed terms for the index value
	**	"k0 * R[1] * R[2]", "k1 * R[2]" or "k2".
	**
	** val[channel][index] (ColorMaps)
	**	is the actual channel value that has been used in
	**	determining the color component. The initial value
	**	is computed from 
	**		(k * 256) / R + ADJ(k,R)
	**	where ADJ(k,R) is correction term so that with k=0
	**	the value will be 0, with k=R-1, the value will be
	**	255.
	** out[index]
	**	is the actual 8bit pixel value for XImage.
	*/
	static XeImageOp dither_8[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ScaleDitherXYZto8,	0,	3*256,	6*256,	0,1,	},
		{NULL,			0,	0,	0,	0,0,	},
	    };
	static XeImageOp map_8[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ScaleMapXYZto8,	0,	3*256,	6*256,	0,1,	},
		{NULL,			0,	0,	0,	0,0,	},
	    };
	static XeImageOp dither_ycc_8[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ScaleDitherYCCto8,	0,	3*256,	6*256,	0,1,	},
		{NULL,			0,	0,	0,	0,0,	},
	    };
	static XeImageOp map_ycc_8[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ScaleMapYCCto8,	0,	3*256,	6*256,	0,1,	},
		{NULL,			0,	0,	0,	0,0,	},
	    };
	char buf[200];
	int ncolors, got;
	int i, j, f, k, a;
	unsigned char *in, *val, *out, *Y[3], *r, *g, *b;
	int C, R[3];
	int N = w->basic.max_colors > 256 ? 256 : w->basic.max_colors;
	int c[3];
#if 0
#define ADJ(i,R) (XeSample_RANGE/(2*(R))) /* Used by Berkeley MPEG decoder */
#else
#define ADJ(i,R) \
	((R)>1?(((i)*XeSample_RANGE+(R)*((R)-(i)-1)-1)/((R)*((R)-1))):0)
#endif
	c[0] = 3;
	c[1] = 3;
	c[2] = 2;
	im->map = (unsigned char *)XtMalloc(256 * 7);
	for (i = 0, C = R[0] = R[1] = R[2] = 1; i < 3; C *= c[i++])
		for (j = 0; j < 3; j++)
			R[j] *= c[j];
	for (i = 0, ncolors = 1; i < 3; ncolors *= R[i++])
		/* This doesn't quite work.. some rounding problem! */
		R[i] = pow((double)(R[i]*N)/(double)C,1.0/3.0)+0.1;
	in = im->map;
	val = in + 3*256;
	out = in + 6*256;
	/*
	** Setup color mapping to RGB
	*/
	Y[0] = r = val;
	Y[1] = g = val + 256;
	Y[2] = b = val + 2*256;
	for (f = 1, j = 3; --j >= 0; f *= R[j])
		for (i = 0; i < ncolors; i++)
		    {	
			k = ( i / f) % R[j];
			a = ADJ(k,R[j]);
			Y[j][i] = (k * XeSample_RANGE) / R[j] + a;
		    }
	for (i = 0; i < ncolors; i++)
		out[i] = i;
	got = _XeAllocColors(w,ncolors, r,g,b, out,ncolors,1,ncolors);
	if (got < ncolors)
	    {
		sprintf(buf,"Warning, got %d exact colors, out of %d requested.",
			got, ncolors);
		XeWidgetWarningMsg((Widget)w,"noteImageToolsColorFail",
				   buf, (String *)NULL, 0);
	    }
	for (i = 0; i < XeSample_RANGE; i++)
		for (f = 1, j = 3; --j >= 0; f *= R[j])
		    {
			k = (R[j] * i) / XeSample_RANGE;
			a = ADJ(k, R[j]);
			in[256*j+i] = k * f;
		    }
	if (src->color_space == XeImageSpace_YCbCr)
		im->image_op = (dither == XeDither_FS4)?dither_ycc_8:map_ycc_8;
	else
		im->image_op = (dither == XeDither_FS4)?dither_8:map_8;
    }

/*
** InitDisplay_COLORQUANT
**	Initialize Pseudocolor visual for TRUECOLOR images. Use a
**	slow color 24 to 8 conversion phase.
**
**	Only RGB and YCbCr colorspaces now!
*/
static void InitDisplay_COLORQUANT(w, im, src, dither, quant)
XeBasicWidget w;
XeImage *im;
XeRawImage *src;
int dither, quant;
    {
	/*
	** Build the mapping tables along the following principles:
	**
	*/
	static XeImageOp ycbcr_8H[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{MakeImageChannel24,	0,	0,	0,	-1,0,},
		{ScaleConvertYCbCrTo24,	0,	0,	0,	-1,1,},
		{ColorMapping24to8H,	1*256,	2*256,	3*256,	0,-1,},
		{FreeImageChannel,	0,	0,	0,	-1,0,},
		{NULL,			0,	0,	0,	0,0,},
	    };

	static XeImageOp rgb_8H[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{MakeImageChannel24,	0,	0,	0,	-1,0,},
		{ScaleCopyRGBto24,	0,	0,	0,	-1,1,},
		{ColorMapping24to8H,	1*256,	2*256,	3*256,	0,-1,},
		{FreeImageChannel,	0,	0,	0,	-1,0,},
		{NULL,			0,	0,	0,	0,0,},
	    };

	static XeImageOp ycbcr_8P[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{MakeImageChannel24,	0,	0,	0,	-1,0,},
		{ScaleConvertYCbCrTo24,	0,	0,	0,	-1,1,},
		{ColorMapping24to8P,	1*256,	2*256,	3*256,	0,-1,},
		{FreeImageChannel,	0,	0,	0,	-1,0,},
		{NULL,			0,	0,	0,	0,0,},
	    };

	static XeImageOp rgb_8P[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{MakeImageChannel24,	0,	0,	0,	-1,0,},
		{ScaleCopyRGBto24,	0,	0,	0,	-1,1,},
		{ColorMapping24to8P,	1*256,	2*256,	3*256,	0,-1,},
		{FreeImageChannel,	0,	0,	0,	-1,0,},
		{NULL,			0,	0,	0,	0,0,},
	    };

	int i;
	unsigned char *map;

	im->map = map = (unsigned char *)XtMalloc(4 * 256);
	for (i = 0; i < 256; i++)
		map[i] = i;	/* Indentity mapping for now.. */

	if (quant == XeColorQuantize_HECKBERT)
		if (src->color_space == XeImageSpace_YCbCr)
			im->image_op = ycbcr_8H;
		else
			im->image_op = rgb_8H;
	else
		if (src->color_space == XeImageSpace_YCbCr)
			im->image_op = ycbcr_8P;
		else
			im->image_op = rgb_8P;

    }

/*
** InitDisplay_PSEUDOCOLOR_GRAY
**	Initialize Pseudocolor display for any image class to be shown
**	in grayscale. Allocate a range of gray shades and initialize
**	mapping tables.
*/
static void InitDisplay_PSEUDOCOLOR_GRAY(w, im, raw, dither, quant)
XeBasicWidget w;
XeImage *im;
XeRawImage *raw;
int dither, quant;
    {
	static XeImageOp dither_1[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ZeroChannel,		0,	0,	0,	0,0,},
		{ScaleDitherChannel1to1,0,	0,	0,	0,1,},
		{NULL,			0,	0,	0,	0,0,},
	    };
	static XeImageOp dither_8[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ZeroChannel,		0,	0,	0,	0,0,},
		{ScaleDitherChannel8to8,0,	1*256,	2*256,	0,1,},
		{NULL,			0,	0,	0,	0,0,},
	    };
	static XeImageOp map_8[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ZeroChannel,		0,	0,	0,	0,0,},
		{ScaleMapChannel8to8,	0,	1*256,	2*256,	0,1,},
		{NULL,			0,	0,	0,	0,0,},
	    };
	static XeImageOp dither_24[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ZeroChannel,		0,	0,	0,	0,0,},
		{MakeImageChannel8,	0,	0,	0,	-1,0,},
		{ScaleConvertRGBto8,	0,	0,	0,	-1,1,},
		{ScaleDitherChannel8to8,0,	1*256,	2*256,	0,-1,},
		{FreeImageChannel,	0,	0,	0,	-1,0,},
		{NULL,			0,	0,	0,	0,0,},
	    };
	static XeImageOp map_24[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ZeroChannel,		0,	0,	0,	0,0,},
		{MakeImageChannel8,	0,	0,	0,	-1,0,},
		{ScaleConvertRGBto8,	0,	0,	0,	-1,1,},
		{ScaleMapChannel8to8,	0,	1*256,	2*256,	0,-1,},
		{FreeImageChannel,	0,	0,	0,	-1,0,},
		{NULL,			0,	0,	0,	0,0,},
	    };
	int ncolors = w->basic.max_colors > 256 ? 256 : w->basic.max_colors;
	int i;
	unsigned char r[256], g[256], b[256], *map, *ir, *ig, *ib;
	int mapsize;

	im->map = map = (unsigned char *)XtMalloc(3 * 256);
	/*
	** Initialize mapping tables for evenly spaced ncolors shades
	** of gray.
	*/
	for (i = 0; i < 256; i++)
	    {
		int p = ((ncolors - 1) * i) / 255;

		map[i] = i;
		map[i+256] = (255 * p) / (ncolors-1);
		map[i+2*256] = p;
	    }
	for (i = 0; i < ncolors; i++)
		r[i] = g[i] = b[i] = ((i * 255) / (ncolors-1));
	switch (raw->class)
	    {
	    default:
	    case XeImageClass_BILEVEL:
		map[0] = 255;	/* 0 = White (background color) */
		map[1] = 0;	/* 1 = Black (foreground color) */
		im->image_op = dither_1;
		break;
	    case XeImageClass_FULLCOLOR:
		if (raw->color_space != XeImageSpace_YCbCr)
		    {
			im->image_op = (dither == XeDither_FS4) ?
				dither_24 : map_24;
			break;
		    }
		/* --- On YCbCr, fall trhough to GRAYSCALE -- */
	    case XeImageClass_GRAYSCALE:
		im->image_op = (dither == XeDither_FS4) ? dither_8 : map_8;
		break;
	    case XeImageClass_PALETTE:
		mapsize = 1 << raw->bits_per_sample;
		ir = raw->color_map;
		ig = ir + mapsize;
		ib = ig + mapsize;
		if (raw->color_space == XeImageSpace_YCbCr)
			for (i = 0; i < mapsize; i++)
				map[i] = ir[i]; /* Assume this is luminance */
		else
			for (i = 0; i < mapsize; i++)
				map[i] = MONO(ir[i], ig[i], ib[i]);
		im->image_op = (dither == XeDither_FS4) ? dither_8 : map_8;
		break;
	    }
	_XeAllocColors((XeBasicWidget)w,ncolors,r,g,b,map+512,256,1,256);
    }

/*
**	Initialize Pseudocolor display for PALETTE images to be shown
**	in COLOR.
*/
static void InitDisplay_PALETTE_COLOR(w, im, raw, dither, quant)
XeBasicWidget w;
XeImage *im;
XeRawImage *raw;
int dither, quant;
    {
	static XeImageOp map_8[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ZeroChannel,		0,	0,	0,	0,0,},
		{ScaleDropPixels8to8,	0,	1*256,	2*256,	0,1,},
		{AllocateColors8,	3*256,	4*256,	5*256,	0,0,},
		{NULL,			0,	0,	0,	0,0,},
	    };
	int i;
	unsigned char *map, *r, *g, *b;
	int mapsize = 1 << raw->bits_per_sample;

	im->map = map = (unsigned char *)XtMalloc(3 * 256 + 3 * 256);
	r = raw->color_map;
	g = r + mapsize;
	b = g + mapsize;
	memset((void *)(map + 3 * 256),0, 3*256);
	memcpy((void *)(map + 3 * 256),r,mapsize);
	memcpy((void *)(map + 4 * 256),g,mapsize);
	memcpy((void *)(map + 5 * 256),b,mapsize);
	for (i = 0; i < 256; i++)
		map[i] = map[i+256] = map[i+2*256] = i;
	im->image_op = map_8;
    }

/*
** InitDisplay_TRUECOLOR_COLOR
**	Initialize TrueColor or DirectColor display for FULLCOLOR
**	images to be shown in color.
*/
static int highbit(ul)
unsigned long ul;
    {
	/* returns position of highest set bit in 'ul' as an integer (0-31),
		   or -1 if none */
	
		int i;
	for (i=31; ((ul&0x80000000) == 0) && i>=0;  i--, ul<<=1);
	return i;
    }

static void InitDisplay_TRUECOLOR_COLOR(w, im, raw, dither, quant)
XeBasicWidget w;
XeImage *im;
XeRawImage *raw;
int dither, quant;
    {
	/*
	** Build the mapping tables along the following principles:
	**
	** in	maps the image sample value to the corresponding
	**	R, G or B value in range 0-255. For FULLCOLOR images
	**	this is usually nul-mapping, for PALETTE images this
	**	table will be the R, G or B colormap.
	** out	maps the R, G or B value into component of the pixel
	**	value that will be added to the final output pixel.
	** val	is the R, G or B value that actually matches the
	**	corresponding out value (used in finding out the error
	**	when dithering).
	*/
	static XeImageOp true_dither_8[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ZeroChannel,		0,	0,	0,	0,0,},
		{ScaleDitherChannel8to8,0,	1*256,	2*256,	0,1,},
		{ScaleDitherChannel8to8,3*256,	4*256,	5*256,	0,2,},
		{ScaleDitherChannel8to8,6*256,	7*256,	8*256,	0,3,},
		{NULL,			0,	0,	0,	0,0,},
	    };
	static XeImageOp true_map_8[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ZeroChannel,		0,	0,	0,	0,0,},
		{ScaleMapChannel8to8,	0,	1*256,	2*256,	0,1,},
		{ScaleMapChannel8to8,	3*256,	4*256,	5*256,	0,2,},
		{ScaleMapChannel8to8,	6*256,	7*256,	8*256,	0,3,},
		{NULL,			0,	0,	0,	0,0,},
	    };
	static XeImageOp palette_dither_8[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ZeroChannel,		0,	0,	0,	0,0,},
		{ScaleDitherChannel8to8,0,	1*256,	2*256,	0,1,},
		{ScaleDitherChannel8to8,3*256,	4*256,	5*256,	0,1,},
		{ScaleDitherChannel8to8,6*256,	7*256,	8*256,	0,1,},
		{NULL,			0,	0,	0,	0,0,},
	    };
	static XeImageOp palette_map_8[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ZeroChannel,		0,	0,	0,	0,0,},
		{ScaleMapChannel8to8,	0,	1*256,	2*256,	0,1,},
		{ScaleMapChannel8to8,	3*256,	4*256,	5*256,	0,1,},
		{ScaleMapChannel8to8,	6*256,	7*256,	8*256,	0,1,},
		{NULL,			0,	0,	0,	0,0,},
	    };
	static XeImageOp true_dither_16[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ZeroChannel,		0,	0,	0,	0,0,},
		{ScaleDitherChannel8to16,0,	1*256,	2*256,	0,1,},
		{ScaleDitherChannel8to16,4*256,	5*256,	6*256,	0,2,},
		{ScaleDitherChannel8to16,8*256,	9*256,	10*256,	0,3,},
		{NULL,			0,	0,	0,	0,0,},
	    };
	static XeImageOp true_map_16[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ZeroChannel,		0,	0,	0,	0,0,},
		{ScaleMapChannel8to16,	0,	1*256,	2*256,	0,1,},
		{ScaleMapChannel8to16,	4*256,	5*256,	6*256,	0,2,},
		{ScaleMapChannel8to16,	8*256,	9*256,	10*256,	0,3,},
		{NULL,			0,	0,	0,	0,0,},
	    };
	static XeImageOp palette_dither_16[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ZeroChannel,		0,	0,	0,	0,0,},
		{ScaleDitherChannel8to16,0,	1*256,	2*256,	0,1,},
		{ScaleDitherChannel8to16,4*256,	5*256,	6*256,	0,1,},
		{ScaleDitherChannel8to16,8*256,	9*256,	10*256,	0,1,},
		{NULL,			0,	0,	0,	0,0,},
	    };
	static XeImageOp palette_map_16[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ZeroChannel,		0,	0,	0,	0,0,},
		{ScaleMapChannel8to16,	0,	1*256,	2*256,	0,1,},
		{ScaleMapChannel8to16,	4*256,	5*256,	6*256,	0,1,},
		{ScaleMapChannel8to16,	8*256,	9*256,	10*256,	0,1,},
		{NULL,			0,	0,	0,	0,0,},
	    };
	static XeImageOp true_map_32[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ZeroChannel,		0,	0,	0,	0,0,},
		{ScaleMapChannel8to32,	0,	1*256,	2*256,	0,1,},
		{ScaleMapChannel8to32,	6*256,	7*256,	8*256,	0,2,},
		{ScaleMapChannel8to32,	12*256,	13*256,	14*256,	0,3,},
		{NULL,			0,	0,	0,	0,0,},
	    };
	static XeImageOp palette_map_32[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ZeroChannel,		0,	0,	0,	0,0,},
		{ScaleMapChannel8to32,	0,	1*256,	2*256,	0,1,},
		{ScaleMapChannel8to32,	6*256,	7*256,	8*256,	0,1,},
		{ScaleMapChannel8to32,	12*256,	13*256,	14*256,	0,1,},
		{NULL,			0,	0,	0,	0,0,},
	    };

	static XeImageOp
		*palette_map[4] =
		    {palette_map_8,palette_map_16,0,palette_map_32},
		*palette_dither[4] =
		    {palette_dither_8,palette_dither_16,0,palette_map_32},
		*true_map[4] = {true_map_8,true_map_16,0,true_map_32},
		*true_dither[4] = {true_dither_8,true_dither_16,0,true_map_32};
	
	int i, rs, gs, bs, out_width;
	unsigned long rm = im->image->red_mask;
	unsigned long gm = im->image->green_mask;
	unsigned long bm = im->image->blue_mask;
	unsigned char *r_in, *g_in, *b_in;
	unsigned char *r_val, *g_val, *b_val;
	unsigned char *r_out, *g_out, *b_out;

	switch (im->image->bits_per_pixel)
	    {
	    default:
		out_width = 1;
		break;
	    case 32:
		out_width = 4;
		break;
	    case 16:
		out_width = 2;
		break;
	    }
	/*
	** Compute the amount of shifting that is required to position
	** the XeSample value into the final output pixel (to align the
	** most significant bits).
	*/
	rs = (XeSample_BITS-1) - highbit(rm);
	gs = (XeSample_BITS-1) - highbit(gm);
	bs = (XeSample_BITS-1) - highbit(bm);
	/*
	** Align the masks to samples MSB
	*/
	rm = rs > 0 ? (rm << rs) : (rm >> -rs);
	gm = gs > 0 ? (gm << gs) : (gm >> -gs);
	bm = bs > 0 ? (bm << bs) : (bm >> -bs);
	im->map = (unsigned char *)XtMalloc(6 * 256 + 3 * out_width * 256);
	r_in  = im->map;
	r_val = r_in  + 256;
	r_out = r_val + 256;
	g_in  = r_out + out_width * 256;
	g_val = g_in  + 256;
	g_out = g_val + 256;
	b_in  = g_out + out_width * 256;
	b_val = b_in  + 256;
	b_out = b_val + 256;
	/*
	** Fill the mapping tables for each channel.
	*/
	for (i = 0; i < XeSample_RANGE; i++)
	    {
		r_val[i] = i & rm;
		g_val[i] = i & gm;
		b_val[i] = i & bm;
	    }
	switch (out_width)
	    {
	    default:
		for (i = 0; i < XeSample_RANGE; i++)
		    {
			r_out[i] = rs>0 ? (r_val[i] >> rs) : (r_val[i] << -rs);
			g_out[i] = gs>0 ? (g_val[i] >> gs) : (g_val[i] << -gs);
			b_out[i] = bs>0 ? (b_val[i] >> bs) : (b_val[i] << -bs);
		    }
		break;
	    case 2:
		for (i = 0; i < XeSample_RANGE; i++)
		    {
			((INT16*)r_out)[i]=rs>0?(r_val[i]>>rs):(r_val[i]<<-rs);
			((INT16*)g_out)[i]=gs>0?(g_val[i]>>gs):(g_val[i]<<-gs);
			((INT16*)b_out)[i]=bs>0?(b_val[i]>>bs):(b_val[i]<<-bs);
		    }
		break;
	    case 4:
		for (i = 0; i < XeSample_RANGE; i++)
		    {
			((INT32*)r_out)[i]=rs>0?(r_val[i]>>rs):(r_val[i]<<-rs);
			((INT32*)g_out)[i]=gs>0?(g_val[i]>>gs):(g_val[i]<<-gs);
			((INT32*)b_out)[i]=bs>0?(b_val[i]>>bs):(b_val[i]<<-bs);
		    }
		break;
	    }
	if (raw->class == XeImageClass_PALETTE)
	    {
		unsigned char *r, *g, *b;
		int mapsize = 1 << raw->bits_per_sample;

		r = raw->color_map;
		g = r + mapsize;
		b = g + mapsize;
		for (i = 0; i < mapsize; i++)
		    {
			r_in[i] = r[i];
			g_in[i] = g[i];
			b_in[i] = b[i];
		    }
		im->image_op = (dither == XeDither_FS4) ?
			palette_dither[out_width-1] : palette_map[out_width-1];
	    }
	else
	    {
		for (i = 0; i < XeSample_RANGE; i++)
			r_in[i] = g_in[i] = b_in[i] = i;
		im->image_op = (dither == XeDither_FS4) ?
			true_dither[out_width-1] : true_map[out_width-1];
	    }
    }

/*
** InitDisplay_ANY_MONO
**	Initialize for displaying the image as bilevel dithered image.
**	Any visual, any image class.
*/
static void InitDisplay_ANY_MONO(w, im, raw, dither, quant)
XeBasicWidget w;
XeImage *im;
XeRawImage *raw;
int dither, quant;
    {
	static XeImageOp dither_1[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ZeroChannel,		0,	0,	0,	0,0,},
		{ScaleDitherChannel1to1,0,	0,	0,	0,1,},
		{NULL,			0,	0,	0,	0,0,},
	    };
	static XeImageOp dither_8[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ZeroChannel,		0,	0,	0,	0,0,},
		{ScaleDitherChannel8to1,0,	0,	0,	0,1,},
		{NULL,			0,	0,	0,	0,0,},
	    };
	static XeImageOp dither_24[] = 
	    {
		/*   Op			in	val	out	dst,src */
		{ZeroChannel,		0,	0,	0,	0,0,},
		{MakeImageChannel8,	0,	0,	0,	-1,0,},
		{ScaleConvertRGBto8,	0,	0,	256,	-1,1,},
		{ScaleDitherChannel8to1,0,	0,	0,	0,-1,},
		{FreeImageChannel,	0,	0,	0,	-1,0,},
		{NULL,			0,	0,	0,	0,0,},
	    };

	int i;
	unsigned char *map, *r, *g, *b;
	int mapsize;

	switch (raw->class)
	    {
	    default:
	    case XeImageClass_BILEVEL:
		im->map = (unsigned char *)XtMalloc(2);
		im->map[0] = 255;	/* 0 = White (background color) */
		im->map[1] = 0;		/* 1 = Black (foreground color) */
		im->image_op = dither_1;
		return;		/* *RETURN USED HERE!* */
	    case XeImageClass_GRAYSCALE:
		im->map = map = (unsigned char *)XtMalloc(256);
		im->image_op = dither_8;
		break;
	    case XeImageClass_PALETTE:
		mapsize = 1 << raw->bits_per_sample;
		im->map = map = (unsigned char *)XtMalloc(256);
		r = raw->color_map;
		g = r + mapsize;
		b = g + mapsize;
		if (raw->color_space == XeImageSpace_YCbCr)
			for (i = 0; i < mapsize; i++)
				map[i] = fs_gamcr[255-r[i]];
		else
			for (i = 0; i < mapsize; i++)
				map[i] = fs_gamcr[255-MONO(r[i], g[i], b[i])];
		im->image_op = dither_8;
		return;		/* *RETURN USED HERE!* */
	    case XeImageClass_FULLCOLOR:
		if (raw->color_space == XeImageSpace_YCbCr)
		    {
			im->map = map = (unsigned char *)XtMalloc(256);
			im->image_op = dither_8;
		    }
		else
		    {
			im->map = map = (unsigned char *)XtMalloc(2*256);
			for (i = 0; i < 256; i++)
				map[256 + i] = i;
			im->image_op = dither_24;
		    }
		break;
	    }
	for (i = 0; i < 256; i++)
		map[i] = fs_gamcr[255-i];
    }

static void InitDisplay_GRAYSCALE(w, im, raw, dither, quant)
XeBasicWidget w;
XeImage *im;
XeRawImage *raw;
int dither, quant;
    {
	switch (w->basic.visual->class)
	    {
	    case TrueColor:
	    case DirectColor:
		/* Should do something different?? */

	    case StaticColor:
	    case GrayScale:
	    case StaticGray:
	    case PseudoColor:
	    default:
		InitDisplay_PSEUDOCOLOR_GRAY(w, im, raw, dither, quant);
		break;
	    }
    }

static void InitImage_GRAYSCALE(w, im, raw, mode, dither, quant)
XeBasicWidget w;
XeImage *im;
XeRawImage *raw;
XeColorMode mode;
int dither, quant;
    {
	if (mode == XeColorMode_MONO)
		InitDisplay_ANY_MONO(w, im, raw, dither, quant);
	else
		InitDisplay_GRAYSCALE(w, im, raw, dither, quant);
    }

static void InitImage_PALETTE(w, im, raw, mode, dither, quant)
XeBasicWidget w;
XeImage *im;
XeRawImage *raw;
XeColorMode mode;
int dither, quant;
    {
	switch (mode)
	    {
	    case XeColorMode_MONO:
	    default:
		InitDisplay_ANY_MONO(w, im, raw, dither, quant);
		break;
	    case XeColorMode_GRAY:
		InitDisplay_GRAYSCALE(w, im, raw, dither, quant);
		break;
	    case XeColorMode_COLOR:
		switch (w->basic.visual->class)
		    {
		    case TrueColor:
		    case DirectColor:
			InitDisplay_TRUECOLOR_COLOR(w, im, raw, dither, quant);
			break;
		    case StaticColor:
		    case GrayScale:
		    case StaticGray:
		    case PseudoColor:
		    default:
			InitDisplay_PALETTE_COLOR(w, im, raw, dither, quant);
			break;
		    }
		break;
	    }
    }

static void InitImage_FULLCOLOR(w, im, raw, mode, dither, quant)
XeBasicWidget w;
XeImage *im;
XeRawImage *raw;
XeColorMode mode;
int dither, quant;
    {
	switch (mode)
	    {
	    default:
	    case XeColorMode_MONO:
		InitDisplay_ANY_MONO(w, im, raw, dither, quant);
		break;
	    case XeColorMode_GRAY:
		InitDisplay_GRAYSCALE(w, im, raw, dither, quant);
		break;
	    case XeColorMode_COLOR:
		switch (w->basic.visual->class)
		    {
		    case TrueColor:
		    case DirectColor:
			InitDisplay_TRUECOLOR_COLOR(w, im, raw, dither, quant);
			break;
		    default:
		    case StaticColor:
		    case PseudoColor:
			if (quant == XeColorQuantize_FAST ||
			    quant == XeColorQuantize_JPEG)
				InitDisplay_COLORCUBE(w,im,raw,dither,quant);
			else
				InitDisplay_COLORQUANT(w,im,raw,dither,quant);
			break;
		    }
		break;
	    }
    }
/*
** _XeCreateImage
**	Create Xew Image structure for displaying the width*height
**	source image on the X Window.
*/
XeImage *_XeCreateImage(w, raw, dither, quant)
XeBasicWidget w;
XeRawImage *raw;
int dither, quant;
    {
	static int init;
	XImage *image;
	XeImage *im;
	Dimension dw, dh;
	int format, depth, bitmap_pad;
	XeColorMode mode;

	if (!init)
	    {
		InitFSDither();
		init = 1;
	    }
	im = (XeImage *)XtCalloc(1, sizeof(XeImage));
	if (w->basic.visual == NULL)
		w->basic.visual = XeGetVisual((Widget)w);
	XeBasicScaling((XeBasicWidget)w, raw->width, raw->height, &dw, &dh);
	/*
	** Request resize?
	*/
	if (w->basic.resize && (w->core.width != dw || w->core.height != dh))
	    {
		Dimension rw = dw;
		Dimension rh = dh;
		int do_almost = 1;
		
		while (XtMakeResizeRequest((Widget)w, rw, rh, &rw, &rh)
		       == XtGeometryAlmost && do_almost)
			do_almost = 0;
	    }
	/*
	** Constrain the requested color_mode by display capabilities and
	** choose XImage parameters according to the result. (If NONE, try
	** for COLOR by default).
	*/
	if ((mode = w->basic.color_mode) == XeColorMode_NONE)
		mode = XeColorMode_COLOR;
	switch (w->basic.visual->class)
	    {
	    case GrayScale:
	    case StaticGray:
		if (mode == XeColorMode_COLOR)
			mode = XeColorMode_GRAY;
		break;
	    case TrueColor:
	    case StaticColor:
	    case PseudoColor:
	    case DirectColor:
		break;
	    default:
		mode = XeColorMode_MONO;
	    }
	if (w->core.depth == 1 || w->basic.max_colors < 2 ||
	    raw->class == XeImageClass_BILEVEL || mode == XeColorMode_MONO)
	    {
		depth = 1;
		bitmap_pad = 32;
		format = XYBitmap;
		mode = XeColorMode_MONO;
	    }
	else
	    {
		depth = w->core.depth;
		bitmap_pad = depth > 8 ? 32 : 8;
		format = ZPixmap;
	    }
	do
	    {
		
#ifdef SH_MEM
		im->shared = False;
		if (w->basic.use_shm == 2)
		    {
			int server = XShmPixmapFormat(XtDisplay(w));
			if (server != format)
				w->basic.use_shm = 1;
		    }
		if (w->basic.use_shm)
		    {
			im->shm.shmid = -1;
			im->shm.shmaddr = (char *)-1;
			im->image = image = XShmCreateImage
				(XtDisplay(w),
				 w->basic.visual,	/* visual */
				 depth,			/* depth */
				 format,		/* format */
				 (char *)NULL,		/* Bitmap */
				 &im->shm,		/* Shared info */
				 (unsigned int)dw,	/* Bitmap width */
				 (unsigned int)dh);	/* Bitmap height*/
			if (image == NULL)
				goto try_non_shared;
			im->shm.shmid =
				shmget(IPC_PRIVATE,
				       image->bytes_per_line * image->height,
				       IPC_CREAT | 0777);
			if (im->shm.shmid < 0)
				goto try_non_shared;
			im->shm.shmaddr = image->data =
				(char *)shmat(im->shm.shmid,0,0);
			if (im->shm.shmaddr == (char *)-1)
				goto try_non_shared;
			im->shm.readOnly = False;
			if (image->data == NULL)
				goto try_non_shared;
			if (!XShmAttach(XtDisplay(w), &im->shm))
				goto try_non_shared;
			XSync(XtDisplay(w), 0);
			/*
			** To prevent unwanted accumulation of shared segments
			** due abnormal program exits, delete the segment
			** already here. It should stay around until detached
			** from all processess.
			*/
			shmctl(im->shm.shmid, IPC_RMID, 0);
			if (w->basic.use_shm == 2)
				im->pixmap = XShmCreatePixmap
					(XtDisplay(w),
					 XtWindow(w),
					 (char *)im->shm.shmaddr,
					 &im->shm,
					 (unsigned int)dw,
					 (unsigned int)dh,
					 depth);
			im->shared = True;
			break;
		    try_non_shared:
			/*
			** Clean up resources that got allocated
			*/
			if (image)
				XDestroyImage(image);
			if (im->shm.shmaddr != (char *)-1)
				shmdt(im->shm.shmaddr);
			if (im->shm.shmid >= 0)
				shmctl(im->shm.shmid, IPC_RMID, 0);
			im->image = NULL;
			im->shared = False;
			im->shm.shmid = -1;
			im->shm.shmaddr = (char *)-1;
			XeWidgetWarningMsg
				((Widget)w, "noteImageToolsNoShm",
				 "Shared memory not usable",
				 (String *)NULL, (Cardinal)0);
		    }
#endif
		im->image = image = XCreateImage
			(XtDisplay(w),
			 w->basic.visual,	/* visual */
			 depth,			/* depth */
			 format,		/* format */
			 0,			/* Offset */
			 (char *)NULL,		/* Bitmap */
			 (unsigned int)dw,	/* Bitmap width */
			 (unsigned int)dh,	/* Bitmap height*/
			 bitmap_pad,		/* bitmap_pad */
			 0);			/* Bytes per line */
		if (!image ||
		    !(image->data =
		      (char *)XtMalloc(dh*image->bytes_per_line)))
		    {
			XFree((char *)image);
			XtFree((char *)im);
			return NULL;
		    }
	    }
	while (0);
	/*
	** XImage successfully created, setup pixel processing for it
	*/
	if (format == XYBitmap)
	    {
		image->bitmap_bit_order = MSBFirst;
		image->byte_order = MSBFirst;
		image->bitmap_unit = 8;
	    }
	switch (raw->class)
	    {
	    default:
	    case XeImageClass_BILEVEL:
		InitDisplay_ANY_MONO(w, im, raw, dither, quant);
		break;
	    case XeImageClass_GRAYSCALE:
		InitImage_GRAYSCALE(w, im, raw, mode, dither, quant);
		break;
	    case XeImageClass_PALETTE:
		InitImage_PALETTE(w, im, raw, mode, dither, quant);
		break;
	    case XeImageClass_FULLCOLOR:
		InitImage_FULLCOLOR(w, im, raw, mode, dither, quant);
		break;
	    }
	return im;
    }

/*
** _XeBuildImage
**	Build the actuall XImage from the raw input image using the
**	control structures.
*/
void _XeBuildImage(w, im, n, data)
XeBasicWidget w;	/* Widget */
XeImage *im;		/* Xew Image Control Structure */
int n;			/* Number of entries in data */
XeImageChannel *data;	/* Raw Image data channels */
    {
	XeImageOp *op = im->image_op;
	unsigned char *map = im->map;
	XeImageChannel temp[3], *src, *dst;

	for ( ;op->op != NULL; op++)
	    {
		if (op->dst >= n || op->src >= n)
			continue;	/* Ref to non-existent data, skip op */
		dst = (op->dst < 0) ? &temp[-op->dst - 1] : &data[op->dst];
		src = (op->src < 0) ? &temp[-op->src - 1] : &data[op->src];
		(*op->op)(w, &map[op->in_offset], &map[op->val_offset],
			  &map[op->out_offset], dst, src);
	    }
    }


void _XePutImage(w, gc, im, src_x, src_y, dst_x, dst_y, width, height)
XeBasicWidget w;
GC gc;
XeImage *im;
int src_x, src_y;
int dst_x, dst_y;
unsigned int width;
unsigned int height;
    {
	if (im == NULL || im->image == NULL)
		return;
	/*
	** Adjust width,height because XShmPutImage seems to be
	** somewhat allergic to outside image references...
	*/
	if (width > im->image->width - src_x)
		width = im->image->width - src_x;
	if (height > im->image->height - src_y)
		height = im->image->height - src_y;
#ifdef SH_MEM
	if (im->shared)
	    {
		(void)XShmPutImage(XtDisplay(w), XtWindow(w),
				   gc, im->image,
				   src_x, src_y, dst_x, dst_y,
				   width, height, False);
		return;
	    }
#endif
	XPutImage(XtDisplay(w), XtWindow(w), gc, im->image,
		  src_x, src_y, dst_x, dst_y, width, height);
    }

void _XeDestroyImage(w, im)
XeBasicWidget w;
XeImage *im;
    {
	if (im == NULL)
		return;
	/*
	** Release old image, if allocated
	*/
#ifdef SH_MEM
	if (im->shared)
	    {
		XShmDetach(XtDisplay(w), &im->shm);
		shmdt(im->shm.shmaddr);
		/* Note: shared memory id is already deleted. no need
		** to call "shmctl(im->shm.shmid, IPC_RMID, 0)"
		*/
		if (im->pixmap)
			XFreePixmap(XtDisplay(w), im->pixmap);
	    }
#endif
	if (im->image)
		XDestroyImage(im->image);
	/*
	** Release old colors if any allocated
	*/
	_XeFreeColors((XeBasicWidget)w);
	XtFree((char *)im->map);
	XtFree((char *)im);
    }

/*
** _XeCreateRawImage
**	Create empty raw image structure which has room for the
**	specified number of channels.
*/
XeRawImage *_XeCreateRawImage(channels)
int channels;
    {
	static XeRawImage init;	/* Static initial empty Image structure */
	int alloc_channels = channels;
	XeRawImage *raw;

	if (alloc_channels < XtNumber(init.channel))
		alloc_channels = XtNumber(init.channel);
	raw = (XeRawImage *)XtMalloc(XtOffsetOf(XeRawImage, channel[0]) +
				     alloc_channels * sizeof(XeImageChannel));
	*raw = init;
	/*
	** Initialize each channel structure to NULLs and Zeroes.
	*/
	while (--alloc_channels > 0)
		raw->channel[alloc_channels] = init.channel[0];
	raw->num_channels = channels;
	return raw;
    }

/*
** _XeDestroyRawImage
*/
void _XeDestroyRawImage(raw)
XeRawImage *raw;
    {
	if (raw == NULL)
		return;
	if (raw->alloc_data)
		XtFree((char *)raw->data);
	if (raw->alloc_map)
		XtFree((char *)raw->color_map);
	XtFree((char *)raw);
    }

