/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGPERIODEDIT_H
#define SKGPERIODEDIT_H
/** @file
 * A period editor.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */

#include "ui_skgperiodedit.h"
#include "skgbasegui_export.h"
/**
 * This file is a period editor.
 */
class SKGBASEGUI_EXPORT SKGPeriodEdit : public QWidget
{
    Q_OBJECT

    /**
     * This enumerate defines the period mode
     */
    Q_ENUMS(PeriodMode)

    /**
     * This enumerate defines the period interval
     */
    Q_ENUMS(PeriodInterval)

    /**
     * Where clause
     */
    Q_PROPERTY(QString whereClause READ getWhereClause NOTIFY changed)

    /**
     * Text
     */
    Q_PROPERTY(QString text READ text NOTIFY changed)

public:
    /**
     * This enumerate defines the period mode
     */
    enum PeriodMode {ALL,        /**< All date*/
                     CURRENT,    /**< Current month, year, ...*/
                     PREVIOUS,   /**< Previous month, year, ...*/
                     LAST,       /**< Last month, year, ...*/
                     CUSTOM,     /**< Custom dates*/
                     TIMELINE    /**< Timeline.*/
                    };

    /**
     * This enumerate defines the period mode
     */
    enum PeriodInterval {DAY = 0, /**< Day*/
                         WEEK = 1,    /**< Week (7 days)*/
                         MONTH = 2,   /**< Month*/
                         QUARTER = 4, /**< Quarter (3 months)*/
                         SEMESTER = 5, /**< Semester (6 months)*/
                         YEAR = 3     /**< Year*/
                        };

    /**
     * Default Constructor
     * @param iParent the parent
     * @param iModeEnabled if true then the widgets are enabled/disabled instead of shown/hidden
     */
    explicit SKGPeriodEdit(QWidget* iParent, bool iModeEnabled = false);

    /**
     * Default Destructor
     */
    virtual ~SKGPeriodEdit();

    /**
     * Get the current state
     * MUST BE OVERWRITTEN
     * @return a string containing all information needed to set the same state.
     * Could be an XML stream
     */
    virtual QString getState();

    /**
     * Set the current state
     * MUST BE OVERWRITTEN
     * @param iState must be interpreted to set the state of the widget
     */
    virtual void setState(const QString& iState);

    /**
     * Get the text
     * @return the text
     */
    virtual QString text() const;

    /**
     * Get the mode
     * @return the mode
     */
    virtual SKGPeriodEdit::PeriodMode mode() const;

    /**
     * Get the where clauses
     * @param iForecast enable forecast where clause
     * @param oWhereClausForPreviousData the where clause for previous data. Can be nullptr.
     * @param oWhereClausForForecastData the where clause for forecast data. Can be nullptr.
     * @return the where clause
     */
    virtual QString getWhereClause(bool iForecast = true, QString* oWhereClausForPreviousData = nullptr, QString*  oWhereClausForForecastData = nullptr) const;

    /**
     * Get begin and end dates
     * @param iPeriod the period
     * @param iInterval the interval
     * @param iValue the number of iInterval
     * @param oBeginDate the begin date of the period
     * @param oEndDate the end date of the period
     * @param iDate the input date
     */
    // cppcheck-suppress passedByValue
    static void getDates(PeriodMode iPeriod, PeriodInterval iInterval, int iValue, QDate& oBeginDate, QDate& oEndDate, const QDate iDate = QDate::currentDate());

    /**
     * Get begin and end dates
     * @param oBeginDate the begin date of the period
     * @param oEndDate the end date of the period
     */
    // cppcheck-suppress passedByValue
    void getDates(QDate& oBeginDate, QDate& oEndDate);

private Q_SLOTS:
    void refresh();

Q_SIGNALS:
    /**
     * Emitted when the period change
     */
    void changed();

private:
    Ui::skgperiodedit_base ui;

    bool m_modeEnable;
    int m_count;
};

#endif  // SKGPERIODEDIT_H
