/***************************************************************************
 * SPDX-FileCopyrightText: 2024 S. MANKOWSKI stephane@mankowski.fr
 * SPDX-FileCopyrightText: 2024 G. DE BURE support@mankowski.fr
 * SPDX-License-Identifier: GPL-3.0-or-later
 ***************************************************************************/
/** @file
 * This file is a test script.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgtestmacro.h"
#include "skgbankincludes.h"
#include "skgimportexportmanager.h"

/**
 * The main function of the unit test
 * @param argc the number of arguments
 * @param argv the list of arguments
 */
int main(int argc, char** argv)
{
    Q_UNUSED(argc)
    Q_UNUSED(argv)

    // Init test
    SKGINITTEST(true)

    QDate d(1970, 1, 1);

    {
        // Test import QIF 1
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGBankObject bank(&document1);
        SKGAccountObject account;
        SKGUnitObject unit_euro(&document1);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_INIT"), err)
            SKGImportExportManager impmissing(&document1, QUrl::fromLocalFile(QLatin1String("missingfile.qif")));
            SKGTESTERROR(QLatin1String("imp1.importFile"), impmissing.importFile(), false)
            impmissing.getParameterDefaultValue(QLatin1String("mapping_date"));

            // Creation bank
            SKGTESTERROR(QLatin1String("BANK:setName"), bank.setName(QLatin1String("CREDIT COOP")), true)
            SKGTESTERROR(QLatin1String("BANK:setNumber"), bank.setNumber(QLatin1String("0003")), true)
            SKGTESTERROR(QLatin1String("BANK:save"), bank.save(), true)

            // Creation account
            SKGTESTERROR(QLatin1String("BANK:addAccount"), bank.addAccount(account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setName"), account.setName(QLatin1String("Courant steph")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setNumber"), account.setNumber(QLatin1String("12345P")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setAgencyNumber"), account.setAgencyNumber(QLatin1String("98765")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setAgencyNumber"), account.setAgencyAddress(QLatin1String("10 rue Dupon, 31000 TOULOUSE")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:save"), account.save(), true)

            // Creation unit
            SKGTESTERROR(QLatin1String("UNIT:setName"), unit_euro.setName(QLatin1String("euro")), true)
            SKGTESTERROR(QLatin1String("UNIT:save"), unit_euro.save(), true)

            // Creation unitvalue
            SKGUnitValueObject unit_euro_val1;
            SKGTESTERROR(QLatin1String("UNIT:addUnitValue"), unit_euro.addUnitValue(unit_euro_val1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setQuantity"), unit_euro_val1.setQuantity(1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setDate"), unit_euro_val1.setDate(d), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:save"), unit_euro_val1.save(), true)
        }
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_BP_QIF"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/ref.qif"));
            SKGTESTERROR(QLatin1String("QIF.setDefaultAccount"), imp1.setDefaultAccount(&account), true)
            SKGTESTERROR(QLatin1String("QIF.setDefaultUnit"), imp1.setDefaultUnit(&unit_euro), true)
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        SKGTESTERROR(QLatin1String("ACCOUNT:load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-935"))

        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_BP_QIF"), err)
            SKGImportExportManager imp1(&document1);
            SKGTESTERROR(QLatin1String("QIF.cleanBankImport"), imp1.cleanBankImport(), true)
            int out = 0;
            SKGTESTERROR(QLatin1String("QIF.findAndGroupTransfers"), imp1.findAndGroupTransfers(out), true)

            SKGImportExportManager exp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportqif1/ref.qif"));
            SKGTESTERROR(QLatin1String("QIF.exportFile"), exp1.exportFile(), true)
        }

        SKGTESTERROR(QLatin1String("ACCOUNT:load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-935"))
    }

    {
        // Test import QIF 2
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGBankObject bank(&document1);
        SKGAccountObject account;
        SKGUnitObject unit_euro(&document1);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_INIT"), err)

            // Creation bank
            SKGTESTERROR(QLatin1String("BANK:setName"), bank.setName(QLatin1String("CREDIT COOP")), true)
            SKGTESTERROR(QLatin1String("BANK:save"), bank.save(), true)

            // Creation account
            SKGTESTERROR(QLatin1String("BANK:addAccount"), bank.addAccount(account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setName"), account.setName(QLatin1String("Courant Guillaume")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setNumber"), account.setNumber(QLatin1String("98765A")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:save"), account.save(), true)

            // Creation unit
            SKGTESTERROR(QLatin1String("UNIT:setName"), unit_euro.setName(QLatin1String("euro")), true)
            SKGTESTERROR(QLatin1String("UNIT:save"), unit_euro.save(), true)

            // Creation unitvalue
            SKGUnitValueObject unit_euro_val1;
            SKGTESTERROR(QLatin1String("UNIT:addUnitValue"), unit_euro.addUnitValue(unit_euro_val1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setQuantity"), unit_euro_val1.setQuantity(1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setDate"), unit_euro_val1.setDate(d), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:save"), unit_euro_val1.save(), true)
        }
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_BP_QIF"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/E0269787.qif"));
            SKGTESTERROR(QLatin1String("QIF.setDefaultAccount"), imp1.setDefaultAccount(&account), true)
            SKGTESTERROR(QLatin1String("QIF.setDefaultUnit"), imp1.setDefaultUnit(&unit_euro), true)
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
            SKGTESTERROR(QLatin1String("QIF.cleanBankImport"), imp1.cleanBankImport(), true)
            int out = 0;
            SKGTESTERROR(QLatin1String("QIF.findAndGroupTransfers"), imp1.findAndGroupTransfers(out), true)


            SKGImportExportManager exp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportqif1/E0269787.qif"));
            SKGTESTERROR(QLatin1String("QIF.exportFile"), exp1.exportFile(), true)
        }

        SKGTESTERROR(QLatin1String("ACCOUNT:load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-1684.58"))

        // Check import with account retrieved from file
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        unit_euro = SKGUnitObject(&document1);
        {
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_INIT"), err)
            // Creation unit
            SKGTESTERROR(QLatin1String("UNIT:setName"), unit_euro.setName(QLatin1String("euro")), true)
            SKGTESTERROR(QLatin1String("UNIT:save"), unit_euro.save(), true)

            // Creation unitvalue
            SKGUnitValueObject unit_euro_val1;
            SKGTESTERROR(QLatin1String("UNIT:addUnitValue"), unit_euro.addUnitValue(unit_euro_val1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setQuantity"), unit_euro_val1.setQuantity(1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setDate"), unit_euro_val1.setDate(d), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:save"), unit_euro_val1.save(), true)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportqif1/E0269787.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
            SKGTESTERROR(QLatin1String("QIF.cleanBankImport"), imp1.cleanBankImport(), true)
            int out = 0;
            SKGTESTERROR(QLatin1String("QIF.findAndGroupTransfers"), imp1.findAndGroupTransfers(out), true)

            SKGImportExportManager exp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportqif1/E0269787_bis.qif"));
            SKGTESTERROR(QLatin1String("QIF.exportFile"), exp1.exportFile(), true)
        }

        // Check multi import
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true) {
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportqif1/ref.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)     // Double import
            SKGTESTERROR(QLatin1String("QIF.cleanBankImport"), imp1.cleanBankImport(), true)
            int out = 0;
            SKGTESTERROR(QLatin1String("QIF.findAndGroupTransfers"), imp1.findAndGroupTransfers(out), true)
        }

        // Check import with account retrieved from files
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true) {
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportqif1/ref.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
            SKGImportExportManager imp2(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportqif1/E0269787.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp2.importFile(), true)
            SKGTESTERROR(QLatin1String("QIF.cleanBankImport"), imp1.cleanBankImport(), true)
            int out = 0;
            SKGTESTERROR(QLatin1String("QIF.findAndGroupTransfers"), imp1.findAndGroupTransfers(out), true)

            SKGImportExportManager exp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportqif1/E0269787_ref.qif"));
            SKGTESTERROR(QLatin1String("QIF.exportFile"), exp1.exportFile(), true)
        }

        // Check import qif multi accounts
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true) {
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportqif1/E0269787_ref.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
            SKGTESTERROR(QLatin1String("QIF.cleanBankImport"), imp1.cleanBankImport(), true)
            int out = 0;
            SKGTESTERROR(QLatin1String("QIF.findAndGroupTransfers"), imp1.findAndGroupTransfers(out), true)
            QStringList oResult;
            SKGTESTERROR(QLatin1String("ACCOUNT:getDistinctValues"), document1.getDistinctValues(QLatin1String("account"), QLatin1String("t_name"), oResult), true)
            SKGTEST(QLatin1String("ACCOUNT:oResult.size"), oResult.size(), 3)
        }
    }

    {
        // Test import QIF 2
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGBankObject bank(&document1);
        SKGAccountObject account;
        SKGUnitObject unit_euro(&document1);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_INIT"), err)

            // Creation bank
            SKGTESTERROR(QLatin1String("BANK:setName"), bank.setName(QLatin1String("ING")), true)
            SKGTESTERROR(QLatin1String("BANK:save"), bank.save(), true)

            // Creation account
            SKGTESTERROR(QLatin1String("BANK:addAccount"), bank.addAccount(account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setName"), account.setName(QLatin1String("ING")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setNumber"), account.setNumber(QLatin1String("ING")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:save"), account.save(), true)

            // Creation unit
            SKGTESTERROR(QLatin1String("UNIT:setName"), unit_euro.setName(QLatin1String("euro")), true)
            SKGTESTERROR(QLatin1String("UNIT:save"), unit_euro.save(), true)

            // Creation unitvalue
            SKGUnitValueObject unit_euro_val1;
            SKGTESTERROR(QLatin1String("UNIT:addUnitValue"), unit_euro.addUnitValue(unit_euro_val1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setQuantity"), unit_euro_val1.setQuantity(1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setDate"), unit_euro_val1.setDate(d), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:save"), unit_euro_val1.save(), true)
        }
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_ING_QIF"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/ing.qif"));
            SKGTESTERROR(QLatin1String("QIF.setDefaultAccount"), imp1.setDefaultAccount(&account), true)
            SKGTESTERROR(QLatin1String("QIF.setDefaultUnit"), imp1.setDefaultUnit(&unit_euro), true)
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
            SKGTESTERROR(QLatin1String("QIF.cleanBankImport"), imp1.cleanBankImport(), true)
            int out = 0;
            SKGTESTERROR(QLatin1String("QIF.findAndGroupTransfers"), imp1.findAndGroupTransfers(out), true)


            SKGImportExportManager exp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportqif1/ing.qif"));
            SKGTESTERROR(QLatin1String("QIF.exportFile"), exp1.exportFile(), true)
        }
    }

    {
        // Test import QIF MOTO
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_ING_QIF"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/moto.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("ACTIF Moto")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("26198.77"))
        }
    }

    {
        // Test import QIF MOTO with euro and franc
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_INIT"), err)

            SKGUnitObject unit;
            SKGTESTERROR(QLatin1String("SKGUnitObject::createCurrencyUnit"), SKGUnitObject::createCurrencyUnit(&document1, QLatin1String("EUR"), unit), true)
            SKGTESTERROR(QLatin1String("SKGUnitObject::createCurrencyUnit"), SKGUnitObject::createCurrencyUnit(&document1, QLatin1String("FRF"), unit), true)
            SKGTESTERROR(QLatin1String("FRANC.addOrModifyUnitValue"), document1.addOrModifyUnitValue(QLatin1String("French Franc (FRF)"), QDate(1963, 1, 1), 1.0 / 6.55957), true)
        }
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_ING_QIF"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/moto.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("ACTIF Moto")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(static_cast<int>(account.getCurrentAmount())), QLatin1String("5009"))
        }
    }

    {
        // Test import QIF KMM
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_KMM_QIF"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/kmm-without-category.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("CCP")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-52.36"))
        }
    }

    {
        // Test import QIF KMM with category
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_KMM_QIF"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/kmm-with-category.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("CCP")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-52.36"))
        }
    }

    {
        // Test import QIF REMI in double to check merge
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_REMI_QIF"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/remi_2.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_REMI_QIF"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/remi_2.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("remi 2")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-1208.63"))
        }
    }

    {
        // Test import QIF REMI in double to check merge
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_DOUBLE"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/double.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("double")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-243"))
        }
    }

    {
        // Test import bug GNUCash 350286
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/350286.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("350286")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("1057.37"))
        }
    }

    {
        // Test import bug GNUCash 393596
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/393596.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("393596")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("12.34"))
        }
    }

    {
        // Test import bug GNUCash 503166
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/503166.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("My Investments")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("20"))
        }
    }

    {
        // Test import bug GNUCash 392707
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/392707.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("392707")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("1219.06"))
        }
    }

    {
        // Test import bug GNUCash 373584
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/373584.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("My Investments")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("1000000"))
        }
    }

    {
        // Test import bug 199818
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/199818.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("199818")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-345.64"))
        }
    }

    {
        // Test import bug 201316
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/201316.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        {
            SKGStringListList listTmp;
            SKGTESTERROR(QLatin1String("QIF.executeSelectSqliteOrder"), document1.executeSelectSqliteOrder(
                             QLatin1String("SELECT TOTAL(f_CURRENTAMOUNT),t_status FROM v_operation_display GROUP BY t_status ORDER BY t_status"), listTmp), true);
            SKGTEST(QLatin1String("QIF:listTmp.count"), listTmp.count(), 4)
            if (listTmp.count() == 4) {
                SKGTEST(QLatin1String("QIF:listTmp.at(1).at(0)"), listTmp.at(1).at(0), QLatin1String("-10"))
                SKGTEST(QLatin1String("QIF:listTmp.at(2).at(0)"), listTmp.at(2).at(0), QLatin1String("-100"))
                SKGTEST(QLatin1String("QIF:listTmp.at(3).at(0)"), listTmp.at(3).at(0), QLatin1String("-1000"))
            }
        }
    }

    {
        // Test import bug 201451
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/201451.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("liability")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-123.45"))
        }
    }

    {
        // Test import bug 214809
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/BNP_CC_virement.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("BNP CC")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("696.64"))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("BNP CEL")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("500"))
        }
    }

    {
        // Test import bug 214851
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/Fortuneo PEA (Caisse).qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)

            SKGImportExportManager imp2(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/Fortuneo PEA.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp2.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("BNP CC")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-3604"))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("Fortuneo PEA")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("86.29"))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("Fortuneo PEA (Caisse)")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("10319.5"))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("Fortuneo Titres (Caisse)")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-6700"))
        }
    }

    {
        // Test import bug 215620
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/Cortal PEA.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)

            SKGImportExportManager imp2(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/EADS.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp2.importFile(), true)
        }
    }

    {
        // Test import bug 216520
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/216520.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("BNP CC")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("11196.64"))
        }
    }

    {
        // Test import investment
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_INIT"), err)

            SKGUnitObject unit;
            SKGTESTERROR(QLatin1String("EUR.setName"), SKGUnitObject::createCurrencyUnit(&document1, QLatin1String("Euro (EUR)"), unit), true)
        }
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/exp_inv.qif"));
            imp1.setCodec(QLatin1String("UTF-8"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)

            SKGImportExportManager exp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportqif1/exp_inv.qif"));
            exp1.setCodec(QLatin1String("UTF-8"));
            SKGTESTERROR(QLatin1String("QIF.exportFile"), exp1.exportFile(), true)
        }
    }

    {
        // Test transfer qif
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/t2.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("COURANT")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("4767.97"))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("CODEVI")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-4767.97"))
        }
    }

    {
        // 233930
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/233930.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }
    }

    {
        // Quicken
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/quicken.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }
    }

    {
        // 267996
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/267996.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }
        {
            SKGImportExportManager exp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportqif1/267996.csv"));
            SKGTESTERROR(QLatin1String("QIF.exportFile"), exp1.exportFile(), true)
        }
    }

    {
        // 271708
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            {
                SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/271708/10Compte A.qif"));
                SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
            }
            {
                SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/271708/20Compte B.qif"));
                SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
            }
            {
                SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/271708/30Compte C.qif"));
                SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
            }
            {
                SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif1/271708/40Compte D.qif"));
                SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
            }
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("10Compte A")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("0"))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("20Compte B")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("10"))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("30Compte C")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("20"))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("40Compte D")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("40"))
        }
    }

    // End test
    SKGENDTEST()
}  // NOLINT(readability/fn_size)
