/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
* This file implements classes SKGServices.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgservices.h"

#include <klocalizedstring.h>
#include <KIO/FileCopyJob>
#include <kiconengine.h>
#include <kiconloader.h>
#include <KDELibs4Support/klocale.h>
#include <sys/time.h>

#include <QSaveFile>
#include <QSqlDatabase>
#include <QSqlQuery>
#include <QSqlRecord>
#include <QSqlError>
#include <QSqlDriver>
#include <QVariant>
#include <qregexp.h>
#include <qfile.h>
#include <qtemporaryfile.h>
#include <QtCrypto>
#include <qmath.h>

#include <QDomDocument>
#include <QScriptEngine>

#include "skgtraces.h"
#include "skgdocument.h"

#define SQLCIPHERHEARDER "SQLCipher format"

int SKGServices::SKGSqlTraces = (SKGServices::getEnvVariable("SKGTRACESQL").isEmpty() ? -1 : SKGServices::stringToInt(SKGServices::getEnvVariable("SKGTRACESQL")));

SKGError SKGServices::m_lastCallbackError;

QString SKGServices::searchCriteriasToWhereClause(const QList< SKGServices::SKGSearchCriteria >& iSearchCriterias, const QStringList& iAttributes, const SKGDocument* iDocument)
{
    QString whereclause;
    int nbCriterias = iSearchCriterias.count();
    int nbAttributes = iAttributes.count();
    for (int i = 0; i < nbCriterias; ++i) {
        SKGSearchCriteria criteria = iSearchCriterias[i];
        QString subWhereClause;

        int nbWords = criteria.words.count();
        for (int w = 0; w < nbWords; ++w) {
            QString subWhereClause2;

            QString word = criteria.words[w].toLower();
            QString att;
            QString op(':');

            // Check if the word follows the format attribut:value
            int pos = word.indexOf(":");
            int pos2 = word.indexOf("<=");
            int pos3 = word.indexOf(">=");
            int pos4 = word.indexOf("=");
            int pos5 = word.indexOf("<");
            int pos6 = word.indexOf(">");
            int pos7 = word.indexOf("#");
            int opLength = 1;
            if (pos2 != -1 && (pos2 < pos || pos == -1)) {
                pos = pos2;
                opLength = 2;
            }
            if (pos3 != -1 && (pos3 < pos || pos == -1)) {
                pos = pos3;
                opLength = 2;
            }
            if (pos4 != -1 && (pos4 < pos || pos == -1)) {
                pos = pos4;
            }
            if (pos5 != -1 && (pos5 < pos || pos == -1)) {
                pos = pos5;
            }
            if (pos6 != -1 && (pos6 < pos || pos == -1)) {
                pos = pos6;
            }
            if (pos7 != -1 && (pos7 < pos || pos == -1)) {
                pos = pos7;
            }

            if (pos != -1) {
                att = word.left(pos);
                op = word.mid(pos, opLength);
                word = word.right(word.count() - pos - op.count());
            }

            for (int j = 0; j < nbAttributes; ++j) {
                QString attDatabase = iAttributes[j];
                QString attForComparison = (iDocument ? iDocument->getDisplay(attDatabase) : attDatabase).toLower();
                if (att.isEmpty() || attForComparison.startsWith(att)) {
                    if (!subWhereClause2.isEmpty()) {
                        subWhereClause2 = subWhereClause2 % " OR ";
                    }

                    if (attDatabase.startsWith(QLatin1String("p_"))) {
                        // Case property
                        QString propName = attDatabase.right(attDatabase.length() - 2);
                        if (op == ":") {
                            subWhereClause2 = subWhereClause2 % "i_PROPPNAME='" % SKGServices::stringToSqlString(propName) % "' AND (lower(i_PROPVALUE) LIKE '%" % word % "%')";
                        } else if (op == "#") {
                            subWhereClause2 = subWhereClause2 % "i_PROPPNAME='" % SKGServices::stringToSqlString(propName) % "' AND REGEXP('" % word % "',i_PROPVALUE)";
                        } else {
                            attDatabase = "i_PROPPNAME='" % SKGServices::stringToSqlString(propName) % "' AND i_PROPVALUE";
                            if (attDatabase.startsWith(QLatin1String("f_")) || attDatabase.startsWith(QLatin1String("i_"))) {
                                subWhereClause2 = subWhereClause2 % attDatabase % op % word;
                            } else {
                                subWhereClause2 = subWhereClause2 % "lower(" % attDatabase % ")" % op % "'" % word % "'";
                            }
                        }
                    } else {
                        // Case normal attribute
                        if (op == ":") {
                            subWhereClause2 = subWhereClause2 % "lower(" % attDatabase % ") LIKE '%" % word % "%'";
                        } else if (op == "#") {
                            subWhereClause2 = subWhereClause2 % "REGEXP('" % word % "'," % attDatabase % ")";
                        } else {
                            if (attDatabase.startsWith(QLatin1String("f_")) || attDatabase.startsWith(QLatin1String("i_"))) {
                                subWhereClause2 = subWhereClause2 % attDatabase % op % word;
                            } else {
                                subWhereClause2 = subWhereClause2 % "lower(" % attDatabase % ")" % op % "'" % word % "'";
                            }
                        }
                    }
                }
            }

            if (!subWhereClause2.isEmpty()) {
                if (!subWhereClause.isEmpty()) {
                    subWhereClause = subWhereClause % " AND ";
                }
                subWhereClause = subWhereClause % "(" % subWhereClause2 % ")";
            } else {
                subWhereClause = "1=0";
            }
        }

        if (!subWhereClause.isEmpty()) {
            if (criteria.mode == '+') {
                if (!whereclause.isEmpty()) {
                    whereclause = whereclause % " OR ";
                }
                whereclause = whereclause % "(" % subWhereClause % ")";
            } else if (criteria.mode == '-') {
                if (!whereclause.isEmpty()) {
                    whereclause = whereclause % " AND NOT";
                } else {
                    whereclause = "NOT";
                }
                whereclause = whereclause % "(" % subWhereClause % ")";
            }
        }
    }
    return whereclause;
}

QList< SKGServices::SKGSearchCriteria > SKGServices::stringToSearchCriterias(const QString& iString)
{
    QList< SKGServices::SKGSearchCriteria > output;

    QStringList words = SKGServices::splitCSVLine(iString, ' ', true);

    int nbwords = words.count();

    SKGServices::SKGSearchCriteria criteria;
    criteria.mode = '+';
    bool atLeastOnePlus = false;
    for (int i = 0; i < nbwords; ++i) {
        QString word = words.at(i);
        bool isWordStartingByPlus = word.startsWith(QLatin1String("+"));
        bool isWordStartingByLess = word.startsWith(QLatin1String("-"));
        if (isWordStartingByPlus || isWordStartingByLess) {
            QChar nextChar;
            if (word.count() > 1) {
                nextChar = word[1];
            }
            if (nextChar < '0' || nextChar > '9') {
                word = word.right(word.length() - 1);
                if (Q_LIKELY(i != 0)) {
                    if (criteria.mode == '-') {
                        output.push_back(criteria);
                    } else {
                        output.push_front(criteria);
                        atLeastOnePlus = true;
                    }
                }
                criteria.words.clear();
                criteria.mode = (isWordStartingByPlus ? '+' : '-');
            }
        }
        criteria.words.push_back(word);
    }
    if (criteria.mode == '-') {
        output.push_back(criteria);
    } else {
        output.push_front(criteria);
        atLeastOnePlus = true;
    }

    if (!atLeastOnePlus) {
        // Add a '+' always true
        SKGServices::SKGSearchCriteria criteria2;
        criteria2.mode = '+';
        criteria2.words.push_back("");
        output.push_front(criteria2);
    }

    return output;
}

QString SKGServices::getEnvVariable(const QString& iAttribute)
{
    return QString::fromUtf8(qgetenv(iAttribute.toUtf8().constData()));
}

QString SKGServices::intToString(qlonglong iNumber)
{
    QString output;
    output.setNum(iNumber);
    return output;
}

qlonglong SKGServices::stringToInt(const QString& iNumber)
{
    if (Q_UNLIKELY(iNumber.isEmpty())) {
        return 0;
    }

    bool ok;
    qlonglong output = iNumber.toLongLong(&ok);
    if (Q_LIKELY(!ok)) {
        SKGTRACE << "WARNING: SKGServices::stringToInt(" << iNumber << ") failed" << endl;
    }

    return output;
}

QString SKGServices::stringToSqlString(const QString& iString)
{
    QString output = iString;
    output.replace('\'', "''");
    return output;
}

QString SKGServices::stringToHtml(const QString& iString)
{
    QString output = iString;
    output.replace('&', "&amp;");  // Must be done first
    output.replace('<', "&lt;");
    output.replace('>', "&gt;");
    output.replace('"', "&quot;");

    return output;
}

QString SKGServices::htmlToString(const QString& iString)
{
    QString output = iString;
    output.replace("&lt;", "<");
    output.replace("&gt;", ">");
    output.replace("&quot;", "\"");
    output.replace("&amp;", "&");

    return output;
}

QString SKGServices::stringsToCsv(const QStringList& iList, const QChar& iSeparator)
{
    QString output;
    int nb = iList.count();
    for (int i = 0; i < nb; ++i) {
        output.append(SKGServices::stringToCsv(iList.at(i)));
        if (Q_LIKELY(i < nb - 1)) {
            output.append(iSeparator);
        }
    }

    return output;
}

QString SKGServices::stringToCsv(const QString& iNumber)
{
    QString output = iNumber;
    output.replace('"', "#SKGDOUBLECOTE#");
    output.replace("#SKGDOUBLECOTE#", "\"\"");
    output = '"' % output % '"';
    return output;
}

double SKGServices::stringToDouble(const QString& iNumber)
{
    if (Q_UNLIKELY(iNumber.isEmpty() || iNumber == "nan")) {
        return 0;
    } else if (Q_UNLIKELY(iNumber == "inf")) {
        return 1e300;
    } else if (Q_UNLIKELY(iNumber == "-inf")) {
        return -1e300;
    }
    QString number = iNumber;
    number.remove(QRegExp("[^0-9-+/eE,.]"));
    if (number.contains("/")) {
        // Use script engine
        QScriptEngine myEngine;
        QScriptValue result = myEngine.evaluate(number);
        if (result.isNumber()) {
            return result.toNumber();
        }
    }

    bool ok;
    double output = number.toDouble(&ok);
    if (Q_LIKELY(!ok)) {
        QString tmp = number;
        tmp.replace(',', '.');
        if (tmp.count('.') > 1) {
            tmp.remove(tmp.indexOf('.'), 1);
        }
        output = tmp.toDouble(&ok);
        if (Q_LIKELY(!ok)) {
            QString tmp2 = number;
            tmp2.replace('.', ',');
            if (tmp2.count(',') > 1) {
                tmp2.remove(tmp2.indexOf(','), 1);
            }
            output = tmp2.toDouble(&ok);
            if (!ok) {
                QString tmp3 = number;
                tmp3.remove(',');
                output = tmp3.toDouble(&ok);
            }
        }
    }
    if (Q_LIKELY(!ok)) {
        SKGTRACE << "WARNING: SKGServices::stringToDouble(" << iNumber << ") failed" << endl;
    }
    return output;
}

QString SKGServices::doubleToString(double iNumber)
{
    QString output;
    output.setNum(iNumber, 'g', 10);
    return output;
}

QString SKGServices::dateToPeriod(const QDate& iDate, const QString& iPeriod)
{
    QString period;
    if (iPeriod == "D") {
        // Day
        period = iDate.toString("yyyy-MM-dd");
    } else if (iPeriod == "W") {
        // Week
        period = iDate.toString("yyyy-W") %  SKGServices::intToString(iDate.weekNumber());
    } else if (iPeriod == "M") {
        // Month
        period = iDate.toString("yyyy-MM");
    } else if (iPeriod == "Q") {
        // Quarter
        period = iDate.toString("yyyy-Q") % (iDate.month() <= 3 ? '1' : (iDate.month() <= 6 ? '2' : (iDate.month() <= 9 ? '3' : '4')));
    } else if (iPeriod == "S") {
        // Semester
        period = iDate.toString("yyyy-S") % (iDate.month() <= 6 ? '1' : '2');;
    } else if (iPeriod == "Y") {
        // Year
        period = iDate.toString("yyyy");
    }
    return period;
}

QString SKGServices::timeToString(const QDateTime& iDateTime)
{
    QDateTime d = iDateTime;
    if (Q_UNLIKELY(!d.isValid())) {
        d = QDateTime::currentDateTime();
    }
    return d.toString("yyyy-MM-dd HH:mm:ss");
}

QString SKGServices::dateToSqlString(const QDate& iDate)
{
    return dateToSqlString(QDateTime(iDate));
}

QString SKGServices::dateToSqlString(const QDateTime& iDateTime)
{
    QDateTime d = iDateTime;
    if (Q_UNLIKELY(!d.isValid())) {
        d = QDateTime::currentDateTime();
    }
    return d.toString("yyyy-MM-dd");
}

int SKGServices::nbWorkingDays(const QDate& iFrom, const QDate& iTo)
{
    int nb = 0;
    QDate min = (iFrom < iTo ? iFrom : iTo);
    QDate max = (iFrom < iTo ? iTo : iFrom);

    while (min != max) {
        if (min.dayOfWeek() <= 5) {
            ++nb;
        }
        min = min.addDays(1);
    }
    if (nb == 0) {
        nb = 1;
    }
    return nb;
}

QDateTime SKGServices::stringToTime(const QString& iDateString)
{
    QDateTime output = QDateTime::fromString(iDateString, "yyyy-MM-dd HH:mm:ss");
    if (Q_UNLIKELY(!output.isValid())) {
        output = QDateTime::fromString(iDateString, "yyyy-MM-dd");
    }

    return output;
}

QDate SKGServices::partialStringToDate(const QString& iDateString, bool iFixupBackward)
{
    QDate result;
    QStringList items = iDateString.split('/');
    int size = items.count();
    bool ok = false;

    if (size == 1) {
        int dayCount = items.at(0).toInt(&ok);

        result = QDate::currentDate();
        result = result.addDays(dayCount - result.day());

        if (iFixupBackward) {
            if (result > QDate::currentDate()) {
                result = result.addMonths(-1);
            }
        } else {
            if (result < QDate::currentDate()) {
                result = result.addMonths(1);
            }
        }
    } else if (size == 2) {
        int dayCount = items.at(0).toInt(&ok);
        int monthCount = items.at(1).toInt(&ok);

        result = QDate::currentDate();
        result = result.addDays(dayCount - result.day());
        result = result.addMonths(monthCount - result.month());

        if (iFixupBackward) {
            if (result > QDate::currentDate()) {
                result = result.addYears(-1);
            }
        } else {
            if (result < QDate::currentDate()) {
                result = result.addYears(1);
            }
        }
    } else if (size == 3) {
        int dayCount = items.at(0).toInt(&ok);
        int monthCount = items.at(1).toInt(&ok);
        int yearCount = items.at(2).toInt(&ok);
        int lengthYear = items.at(2).count();

        result = QDate::currentDate();
        result = result.addDays(dayCount - result.day());
        result = result.addMonths(monthCount - result.month());

        if (lengthYear < 4) {
            int y = int(result.year() / qPow(10, lengthYear)) * qPow(10, lengthYear) + yearCount;
            if (y > result.year() && iFixupBackward) {
                y = y - qPow(10, lengthYear);
            } else if (y < result.year() && !iFixupBackward) {
                y = y + qPow(10, lengthYear);
            }
            result = result.addYears(y - result.year());
        } else {
            result = result.addYears(yearCount - result.year());
        }
    }

    if (!ok) {
        result = QDate();
    }
    return result;
}

QStringList SKGServices::splitCSVLine(const QString& iString, const QChar& iSeparator, bool iCoteDefineBlock, QChar* oRealSeparator)
{
    QStringList items;
    QString item;
    bool isInBlock = false;
    QChar realSeparator = iSeparator;

    QChar cote = ' ';  // Not yet defined
    int nb = iString.length();
    for (int pos = 0; pos < nb; ++pos) {
        QChar c = iString.at(pos);
        if (isInBlock) {
            if (c == cote) {
                if (pos < nb - 1 && iString.at(pos + 1) == cote) {
                    ++pos;
                } else {
                    items.push_back(item);
                    item = "";
                    isInBlock = false;
                    // 320112 vvvv
                    // Reset the block character to autorize mix
                    cote = ' ';
                    // 320112 ^^^^

                    ++pos;  // To ignore next separator
                    if (pos < nb) {
                        realSeparator = iString.at(pos);    // To get the real separator
                    }
                }
            }

            if (isInBlock) {
                item += c;
            }
        } else  if ((c == '\"' || c == '\'') && item.count() == 0 && iCoteDefineBlock) {
            if (cote == ' ') {
                cote = c;    // Set the real cote char
            }
            isInBlock = true;
        } else  if (QString(c) == realSeparator) {
            items.push_back(item);
            item = "";
            isInBlock = false;
            // 320112 vvvv
            // Reset the block character to autorize mix
            cote = ' ';
            // 320112 ^^^^
        } else {
            item += c;
        }
    }

    if (!item.isEmpty() || (nb > 0 && iString.at(nb - 1) == realSeparator)) {
        items.push_back(item);
    }

    if (oRealSeparator) {
        *oRealSeparator = realSeparator;
    }

    if (isInBlock) {
        items.clear();
    }

    return items;
}

QString SKGServices::getDateFormat(const QStringList& iDates)
{
    SKGTRACEINFUNC(2);
    bool f_YYYY_MM_DD = true;
    bool f_YYYYMMDD = true;
    bool f_DDMMYYYY = true;
    bool f_MMDDYYYY = true;
    bool f_MM_DD_YY = true;
    bool f_DD_MM_YY = true;
    bool f_MM_DD_YYYY = true;
    bool f_DD_MM_YYYY = true;
    bool f_DDMMMYYYY = true;
    bool f_DD_MMM_YY = true;
    bool f_DD_MMM_YYYY = true;

    // Build regexp
    QRegExp rx("(.+)-(.+)-(.+)");

    // Check all dates
    int nb = iDates.count();
    for (int i = 0; i < nb; ++i) {
        QString val = iDates.at(i).trimmed();
        if (val.count() > 10) {
            auto l = SKGServices::splitCSVLine(val, ' ');
            val = l[0];
        }
        if (!val.isEmpty()) {
            val = val.replace(' ', '0');
            val = val.replace('\\', '-');
            val = val.replace('/', '-');
            val = val.replace('.', '-');
            val = val.replace("'20", "-20");
            val = val.replace("' ", "-200");
            val = val.replace('\'', "-20");
            val = val.replace("-90", "-1990");
            val = val.replace("-91", "-1991");
            val = val.replace("-92", "-1992");
            val = val.replace("-93", "-1993");
            val = val.replace("-94", "-1994");
            val = val.replace("-95", "-1995");
            val = val.replace("-96", "-1996");
            val = val.replace("-97", "-1997");
            val = val.replace("-98", "-1998");
            val = val.replace("-99", "-1999");
            if (rx.indexIn(val) == -1) {
                f_YYYY_MM_DD = false;
                f_MM_DD_YY = false;
                f_DD_MM_YY = false;
                f_MM_DD_YYYY = false;
                f_DD_MM_YYYY = false;
                f_DD_MMM_YY = false;
                f_DD_MMM_YYYY = false;

                if (val.length() == 8) {
                    int left2 = SKGServices::stringToInt(val.left(2));
                    if (left2 > 12) {
                        f_MMDDYYYY = false;
                    }
                    if (left2 > 31) {
                        f_DDMMYYYY = false;
                    }

                    int mid2 = SKGServices::stringToInt(val.mid(2, 2));
                    if (mid2 > 12) {
                        f_DDMMYYYY = false;
                    }
                    if (mid2 > 31) {
                        f_MMDDYYYY = false;
                    }

                    int mid4 = SKGServices::stringToInt(val.mid(4, 2));
                    if (mid4 > 12) {
                        f_YYYYMMDD = false;
                    }

                    int right2 = SKGServices::stringToInt(val.right(2));
                    if (right2 > 31) {
                        f_YYYYMMDD = false;
                    }

                    f_DDMMMYYYY = false;
                } else if (val.length() == 9) {
                    f_MMDDYYYY = false;
                    f_DDMMYYYY = false;
                    f_YYYYMMDD = false;
                } else {
                    f_MMDDYYYY = false;
                    f_DDMMYYYY = false;
                    f_YYYYMMDD = false;
                    f_DDMMMYYYY = false;
                }
            } else {
                f_YYYYMMDD = false;
                f_DDMMYYYY = false;
                f_MMDDYYYY = false;
                f_DDMMMYYYY = false;

                QString v1 = rx.cap(1);
                QString v2 = rx.cap(2);
                QString v3 = rx.cap(3);

                if (SKGServices::stringToInt(v1) > 12) {
                    f_MM_DD_YY = false;
                    f_MM_DD_YYYY = false;
                }

                if (SKGServices::stringToInt(v2) > 12) {
                    f_DD_MM_YY = false;
                    f_DD_MM_YYYY = false;
                }

                if (v2.length() > 2) {
                    f_MM_DD_YY = false;
                    f_MM_DD_YYYY = false;
                    f_DD_MM_YY = false;
                    f_DD_MM_YYYY = false;
                    f_YYYY_MM_DD = false;
                }

                if (v2.length() != 3) {
                    f_DD_MMM_YYYY = false;
                    f_DD_MMM_YY = false;
                }

                if (SKGServices::stringToInt(v1) > 31 || SKGServices::stringToInt(v2) > 31) {
                    f_MM_DD_YY = false;
                    f_MM_DD_YYYY = false;
                    f_DD_MM_YY = false;
                    f_DD_MM_YYYY = false;
                }

                if (SKGServices::stringToInt(v3) > 31) {
                    f_YYYY_MM_DD = false;
                }

                if (v1.length() == 4) {
                    f_MM_DD_YY = false;
                    f_DD_MM_YY = false;
                    f_MM_DD_YYYY = false;
                    f_DD_MM_YYYY = false;
                } else {
                    // To be more permissive and support mix of date: f_YYYY_MM_DD = false;
                }

                if (v3.length() == 4) {
                    f_YYYY_MM_DD = false;
                    f_MM_DD_YY = false;
                    f_DD_MM_YY = false;
                } else {
                    // To be more permissive and support mix of date: f_MM_DD_YYYY = false;
                    // To be more permissive and support mix of date: f_DD_MM_YYYY = false;
                }
            }
        }
    }

    if (f_YYYYMMDD) {
        return "YYYYMMDD";
    } else if (f_MMDDYYYY) {
        return "MMDDYYYY";
    } else if (f_DDMMYYYY) {
        return "DDMMYYYY";
    } else if (f_DD_MM_YY && f_MM_DD_YY) {
        QString sFormat = QLocale().dateFormat(QLocale::ShortFormat);
        if (sFormat.startsWith(QLatin1String("%m")) || sFormat.startsWith(QLatin1String("%n"))) {
            return "MM-DD-YY";
        }
        return "DD-MM-YY";
    } else if (f_MM_DD_YY) {
        return "MM-DD-YY";
    } else if (f_DD_MM_YY) {
        return "DD-MM-YY";
    } else if (f_DD_MM_YYYY && f_MM_DD_YYYY) {
        QString sFormat = QLocale().dateFormat(QLocale::ShortFormat);
        if (sFormat.startsWith(QLatin1String("%m")) || sFormat.startsWith(QLatin1String("%n"))) {
            return "MM-DD-YYYY";
        }
        return "DD-MM-YYYY";
    } else if (f_MM_DD_YYYY) {
        return "MM-DD-YYYY";
    } else if (f_DD_MM_YYYY) {
        return "DD-MM-YYYY";
    } else if (f_YYYY_MM_DD) {
        return "YYYY-MM-DD";
    } else if (f_DDMMMYYYY) {
        return "DDMMMYYYY";
    } else if (f_DD_MMM_YY) {
        return "DD-MMM-YY";
    } else if (f_DD_MMM_YYYY) {
        return "DD-MMM-YYYY";
    }

    return "";
}

QString SKGServices::toPercentageString(double iAmount, int iNbDecimal)
{
    return toCurrencyString(iAmount, QString(), iNbDecimal) % " %";
}

QString SKGServices::toCurrencyString(double iAmount, const QString& iSymbol, int iNbDecimal)
{
    if (iSymbol == "%") {
        return toPercentageString(iAmount, iNbDecimal);
    }
    return KLocale::global()->formatMoney(iAmount, iSymbol.isEmpty() ? " " : iSymbol, iNbDecimal).trimmed();
}

QString SKGServices::dateToSqlString(const QString& iDate, const QString& iFormat)
{
    QString input = iDate;
    if (input.count() > 10) {
        auto l = SKGServices::splitCSVLine(input, ' ');
        input = l[0];
    }

    QString format = "yyyy-MM-dd";
    QString YYYY = "0000";
    QString MM = "00";
    QString DD = "00";
    if (iFormat == "YYYYMMDD") {
        YYYY = input.mid(0, 4);
        MM = input.mid(4, 2);
        DD = input.mid(6, 2);
    } else if (iFormat == "DDMMYYYY" || iFormat == "DDMMYY") {
        YYYY = input.mid(4, 4);
        MM = input.mid(2, 2);
        DD = input.mid(0, 2);
    } else if (iFormat == "DDMMMYYYY" || iFormat == "DDMMMYY") {
        YYYY = input.mid(5, 4);
        MM = input.mid(2, 3);
        DD = input.mid(0, 2);
        format = "yyyy-MMM-dd";
    } else if (iFormat == "MMDDYYYY" || iFormat == "MMDDYY") {
        YYYY = input.mid(4, 4);
        MM = input.mid(0, 2);
        DD = input.mid(2, 2);

    } else {
        QString val = input;
        val = val.replace(' ', '0');
        val = val.replace('\\', '-');
        val = val.replace('/', '-');
        val = val.replace('.', '-');
        val = val.replace("'20", "-20");
        val = val.replace("' ", "-200");
        val = val.replace('\'', "-20");
        val = val.replace("-90", "-1990");
        val = val.replace("-91", "-1991");
        val = val.replace("-92", "-1992");
        val = val.replace("-93", "-1993");
        val = val.replace("-94", "-1994");
        val = val.replace("-95", "-1995");
        val = val.replace("-96", "-1996");
        val = val.replace("-97", "-1997");
        val = val.replace("-98", "-1998");
        val = val.replace("-99", "-1999");
        QRegExp rx("(.+)-(.+)-(.+)");
        if (rx.indexIn(val) != -1) {
            QString v1 = rx.cap(1);
            QString v2 = rx.cap(2);
            QString v3 = rx.cap(3);
            if (iFormat == "YYYY-MM-DD") {
                YYYY = v1;
                MM = v2;
                DD = v3;
            } else if (iFormat == "MM/DD/YY" || iFormat == "MM-DD-YY" || iFormat == "MM/DD/YYYY" || iFormat == "MM-DD-YYYY") {
                MM = v1;
                DD = v2;
                YYYY = v3;
            } else if (iFormat == "DD/MM/YY" || iFormat == "DD-MM-YY" || iFormat == "DD/MM/YYYY" || iFormat == "DD-MM-YYYY") {
                DD = v1;
                MM = v2;
                YYYY = v3;
            } else if (iFormat == "DD/MMM/YY" || iFormat == "DD-MMM-YY" || iFormat == "DD/MMM/YYYY" || iFormat == "DD-MMM-YYYY") {
                DD = v1;
                MM = v2;
                YYYY = v3;
                format = "yyyy-MMM-dd";
            }
        }
    }

    if (MM.length() == 1) {
        MM = '0' % MM;
    }
    if (DD.length() == 1) {
        DD = '0' % DD;
    }
    if (YYYY.length() == 1) {
        YYYY = '0' % YYYY;
    }
    if (YYYY.length() == 2) {
        if (stringToInt(YYYY) > 70) {
            YYYY = "19" % YYYY;
        } else {
            YYYY = "20" % YYYY;
        }
    }

    QString date = YYYY % '-' % MM % '-' % DD;
    date.replace(' ', '0');
    return dateToSqlString(QDateTime::fromString(date, format));
}

QString SKGServices::getPeriodWhereClause(const QString& iPeriod, const QString& iDateAttribute)
{
    QString output = "1=0";
    if (iPeriod == "ALL") {
        output = "1=1";
    } else if (iPeriod.length() == 4) {
        // 2014
        output = "STRFTIME('%Y'," + SKGServices::stringToSqlString(iDateAttribute) + ")='" + SKGServices::stringToSqlString(iPeriod) + '\'';
    } else if (iPeriod.length() == 7 && iPeriod[4] == '-') {
        if (iPeriod[5] == 'S') {
            // 2014-S1
            output = "STRFTIME('%Y'," + SKGServices::stringToSqlString(iDateAttribute) + ")||'-S'||(CASE WHEN STRFTIME('%m'," + SKGServices::stringToSqlString(iDateAttribute) + ")<='06' THEN '1' ELSE '2' END)='" + SKGServices::stringToSqlString(iPeriod) + '\'';
        } else if (iPeriod[5] == 'Q') {
            // 2014-Q1
            output = "STRFTIME('%Y'," + SKGServices::stringToSqlString(iDateAttribute) + ")||'-Q'||(CASE WHEN STRFTIME('%m'," + SKGServices::stringToSqlString(iDateAttribute) + ")<='03' THEN '1' WHEN STRFTIME('%m'," + SKGServices::stringToSqlString(iDateAttribute) + ")<='06' THEN '2' WHEN STRFTIME('%m'," + SKGServices::stringToSqlString(iDateAttribute) + ")<='09' THEN '3' ELSE '4' END)='" + SKGServices::stringToSqlString(iPeriod) + '\'';
        } else {
            // 2014-07
            output = "STRFTIME('%Y-%m'," + SKGServices::stringToSqlString(iDateAttribute) + ")='" + SKGServices::stringToSqlString(iPeriod) + '\'';
        }
    }
    return output;
}

QDate SKGServices::periodToDate(const QString& iPeriod)
{
    QDate output;

    if (iPeriod == "ALL") {
        output = QDate(0, 1, 1);
    } else if (iPeriod.length() == 4) {
        // 2014
        output = QDate::fromString(iPeriod, "yyyy").addYears(1).addDays(-1);
    } else if (iPeriod.length() == 7) {
        if (iPeriod[5] == 'S') {
            // 2014-S1
            output = QDate::fromString(iPeriod, "yyyy-SM");
            output = output.addMonths(output.month() * 6 - output.month());  // convert semester in month
            output = output.addMonths(1).addDays(-1);
        } else if (iPeriod[5] == 'Q') {
            // 2014-Q1
            output = QDate::fromString(iPeriod, "yyyy-QM");
            output = output.addMonths(output.month() * 3 - output.month());  // convert quarter in month
            output = output.addMonths(1).addDays(-1);
        } else {
            // 2014-07
            output = QDate::fromString(iPeriod, "yyyy-MM").addMonths(1).addDays(-1);
        }
    }
    return output;
}

QString SKGServices::getNeighboringPeriod(const QString& iPeriod, int iDelta)
{
    QString output = "1=0";
    if (iPeriod.length() == 4) {
        // 2014
        QDate date = QDate::fromString(iPeriod, "yyyy").addYears(iDelta);
        output = date.toString("yyyy");
    } else if (iPeriod.length() == 7) {
        if (iPeriod[5] == 'S') {
            // 2014-S1
            QDate date2 = QDate::fromString(iPeriod, "yyyy-SM");
            date2 = date2.addMonths(date2.month() * 6 - date2.month());  // convert semester in month
            date2 = date2.addMonths(6 * iDelta);
            output = date2.toString("yyyy-S") % (date2.month() <= 6 ? '1' : '2');
        } else if (iPeriod[5] == 'Q') {
            // 2014-Q1
            QDate date2 = QDate::fromString(iPeriod, "yyyy-QM");
            date2 = date2.addMonths(date2.month() * 3 - date2.month());  // convert quarter in month
            date2 = date2.addMonths(3 * iDelta);
            output = date2.toString("yyyy-Q") % (date2.month() <= 3 ? '1' : (date2.month() <= 6 ? '2' : (date2.month() <= 9 ? '3' : '4')));
        } else {
            // 2014-07
            QDate date2 = QDate::fromString(iPeriod, "yyyy-MM").addMonths(iDelta);
            output = date2.toString("yyyy-MM");
        }
    }
    return output;
}

QStringList SKGServices::tableToDump(const SKGStringListList& iTable, SKGServices::DumpMode iMode)
{
    SKGTRACEINFUNC(10);
    // initialisation
    QStringList oResult;

    // Compute max size of each column
    int* maxSizes = NULL;
    if (iMode == DUMP_TEXT) {
        int nb = iTable.count();
        for (int i = 0; i < nb; ++i) {
            QStringList line = iTable.at(i);

            if (maxSizes == NULL) {
                int size = line.size();
                maxSizes = new int[size];
                if (maxSizes) {
                    for (int j = 0; j < size; ++j) {
                        maxSizes[j] = 0;
                    }
                }
            }

            int nb2 = line.size();
            for (int j = 0; j < nb2; ++j) {
                QString s = line.at(j);
                if (s.length() > maxSizes[j]) {
                    maxSizes[j] = s.length();
                }
            }
        }
    }

    // dump
    int nb = iTable.count();
    for (int i = 0; i < nb; ++i) {
        QString lineFormated;
        if (iMode == DUMP_TEXT) {
            lineFormated = "| ";
        }

        QStringList line = iTable.at(i);
        int nb2 = line.size();
        for (int j = 0; j < nb2; ++j) {
            QString s = line.at(j);
            s.remove('\n');

            if (iMode == DUMP_CSV) {
                if (j > 0) {
                    lineFormated += ';';
                }
                lineFormated += stringToCsv(s);
            } else {
                s = s.leftJustified(maxSizes[j], ' ');
                lineFormated += s % " | ";
            }
        }
        oResult.push_back(lineFormated);
    }

    // delete
    if (maxSizes) {
        delete [] maxSizes;
        maxSizes = NULL;
    }

    return oResult;
}

QString SKGServices::getRealTable(const QString& iTable)
{
    QString output = iTable;
    if (output.length() > 2 && output.startsWith(QLatin1String("v_"))) {
        output = output.mid(2, output.length() - 2);

        int pos = output.indexOf("_");
        if (pos != -1) {
            output = output.left(pos);
        }
    }

    return output;
}

SKGError SKGServices::downloadToStream(const QUrl& iSourceUrl, QByteArray& oStream)
{
    SKGError err;
    SKGTRACEINFUNCRC(10, err);
    QString tmpFile;
    err = download(iSourceUrl, tmpFile);
    IFOK(err) {
        // Open file
        QFile file(tmpFile);
        if (Q_UNLIKELY(!file.open(QIODevice::ReadOnly))) {
            err.setReturnCode(ERR_FAIL).setMessage(i18nc("An information message", "Open file '%1' failed", tmpFile));
        } else {
            oStream = file.readAll();

            // close file
            file.close();
        }
        QFile(tmpFile).remove();
    }
    return err;
}

SKGError SKGServices::download(const QUrl& iSourceUrl, QString& oTemporaryFile)
{
    SKGError err;
    SKGTRACEINFUNCRC(10, err);
    QTemporaryFile tmpFile;
    tmpFile.setAutoRemove(false);
    if (tmpFile.open()) {
        err = upload(iSourceUrl, QUrl::fromLocalFile(tmpFile.fileName()));
        IFOK(err) oTemporaryFile = tmpFile.fileName();
    }
    return err;
}

SKGError SKGServices::upload(const QUrl& iSourceUrl, const QUrl& iDescUrl)
{
    SKGError err;
    SKGTRACEINFUNCRC(10, err);
    if (iDescUrl != iSourceUrl) {
        KIO::FileCopyJob* getJob = KIO::file_copy(iSourceUrl, iDescUrl, -1, KIO::Overwrite | KIO::HideProgressInfo);
        if (!getJob->exec()) {
            err.setReturnCode(ERR_ABORT).setMessage(getJob->errorString());
            err.addError(ERR_ABORT, i18nc("Error message", "Impossible to copy '%1' to '%2'", iSourceUrl.toDisplayString(), iDescUrl.toDisplayString()));
        }
    }
    return err;
}

SKGError SKGServices::cryptFile(const QString& iFileSource, const QString& iFileTarget, const QString& iPassword, bool iEncrypt, const QString& iHeaderFile, bool& oModeSQLCipher)
{
    SKGError err;
    SKGTRACEINFUNCRC(10, err);
    SKGTRACEL(10) << "Input parameter [iFileSource]=[" << iFileSource << ']' << endl;
    SKGTRACEL(10) << "Input parameter [iFileTarget]=[" << iFileTarget << ']' << endl;
    SKGTRACEL(10) << "Input parameter [iPassword]  =[" << iPassword << ']' << endl;
    SKGTRACEL(10) << "Input parameter [iHeaderFile]=[" << iHeaderFile << ']' << endl;

    oModeSQLCipher = false;

    QCA::Initializer init;
    SKGTRACEL(10) << "QCA::Initializer done" << endl;
    if (!iPassword.isEmpty() && !QCA::isSupported("aes128-ecb")) {
        // Set error message
        err.setReturnCode(ERR_INSTALL);  // To avoid password request
        err.setMessage(i18nc("An error message about encryption", "AES128 encryption is not supported (%1). Please install qca-ossl.", QCA::supportedFeatures().join(",")));
    } else {
        // Create a random key - you'd probably use one from another
        // source in a real application
        QCA::SymmetricKey key(QByteArray("skrooge"));
        QCA::Cipher* cipher = NULL;

        // Read document
        QByteArray input;
        err = downloadToStream(QUrl::fromUserInput(iFileSource), input);
        IFOK(err) {
            bool isFileEncrypted = (input.startsWith(QByteArray((iHeaderFile % "_ENCRYPT").toLatin1())));
            SKGTRACEL(10) << "isFileEncrypted=[" << static_cast<unsigned int>(isFileEncrypted) << ']' << endl;
            QCA::InitializationVector iv(iPassword.toLatin1());

            // Create a 128 bit AES cipher object using Cipher Block Chaining (CBC) mode
            if ((isFileEncrypted || iEncrypt) && !iPassword.isEmpty()) cipher = new QCA::Cipher("aes128", QCA::Cipher::CBC,
                        // use Default padding, which is equivalent to PKCS7 for CBC
                        QCA::Cipher::DefaultPadding,
                        iEncrypt ? QCA::Encode : QCA::Decode,
                        key, iv);

            // BUG 249955 vvv
            if (!cipher && isFileEncrypted) {
                err = SKGError(ERR_ENCRYPTION, i18nc("Error message about encrypting a file", "Encryption failed"));
            }
            // BUG 249955 ^^^

            // Suppress header
            SKGTRACEL(10) << "input=[" << input.left(50) << "...]" << endl;
            if (!iHeaderFile.isEmpty() && input.startsWith(iHeaderFile.toLatin1())) {
                input = input.right(input.length() - iHeaderFile.length() - 11);
            }
            bool sqliteMode = (input.left(15) == "SQLite format 3");
#ifdef SKGCIPHER
            if (iEncrypt && !sqliteMode) {
                input = input.insert(0, SQLCIPHERHEARDER);
            }
#endif
            SKGTRACEL(10) << "input without header=[" << input.left(50) << "...]" << endl;

            QCA::SecureArray u;
            QByteArray uba;
            if (cipher) {
                if (!err) {
                    // Process encryption or decryption
                    u = cipher->process(input);

                    // We need to check if that update() call worked.
                    if (!cipher->ok()) {
                        err = SKGError(ERR_UNEXPECTED, i18nc("Error message about encrypting a file", "Encryption failed"));
                    } else {
                        uba = u.toByteArray();
                    }
                }
            } else {
                uba = input;
            }

            IFOK(err) {
                // Check if decryption is OK
                SKGTRACEL(10) << "output 1=[" << uba.left(50) << "...]" << endl;
                if (!iEncrypt) {
                    if (!uba.startsWith(QByteArray("SQLite format 3"))) {
                        if (!uba.startsWith(SQLCIPHERHEARDER)) {
                            if (isFileEncrypted) {
                                err = SKGError(ERR_ENCRYPTION, i18nc("Error message", "Wrong password"));
                            } else {
#ifdef SKGCIPHER
                                oModeSQLCipher = true;
#endif
                            }
                        }
#ifdef SKGCIPHER
                        else {
                            uba = uba.right(uba.length() - QString(SQLCIPHERHEARDER).length());
                            oModeSQLCipher = true;
                        }
#endif
                    }
                }
            }

            IFOK(err) {
                // Add headers
                if (iEncrypt && !iHeaderFile.isEmpty()) {
                    QByteArray h = (iHeaderFile % (cipher ? "_ENCRYPT" : "_DECRYPT") % (sqliteMode ? "ED-" : "E2-")).toLatin1();
                    uba = uba.insert(0, h);
                }
            }
            SKGTRACEL(10) << "output 2=[" << uba.left(50) << "...]" << endl;

            // output the results of that stage
            IFOK(err) {
                QSaveFile fileOutput(iFileTarget);
                if (!fileOutput.open(QIODevice::WriteOnly)) {
                    err = SKGError(ERR_WRITEACCESS, i18nc("Error message: writing a file failed", "Write file '%1' failed", iFileTarget));
                } else {
                    // Write document
                    fileOutput.write(uba);

                    // Close the file
                    if (!fileOutput.commit()) {
                        IFOK(err) {
                            err = SKGError(ERR_WRITEACCESS, i18nc("Error message: writing a file failed", "Write file '%1' failed", iFileTarget));
                        }
                    }
                }
            }
        }

        delete cipher;
        cipher = NULL;
    }
    SKGTRACEL(10) << "Output parameter [oModeSQLCipher]=[" << static_cast<unsigned int>(oModeSQLCipher) << ']' << endl;
    return err;
}

SKGError SKGServices::copySqliteDatabaseToXml(QSqlDatabase* iDb, QDomDocument& oDocument)
{
    SKGError err;
    SKGTRACEINFUNCRC(10, err);
    if (iDb) {
        oDocument = QDomDocument("SKGML");
        QDomElement document = oDocument.createElement("document");
        oDocument.appendChild(document);

        // Copy the tables
        QStringList listTables = iDb->tables();
        int nb = listTables.count();
        for (int i = 0; !err && i < nb; ++i) {
            QString tableName = listTables.at(i);
            if (!tableName.startsWith(QLatin1String("sqlite_")) && !tableName.startsWith(QLatin1String("vm_"))) {
                QDomElement table = oDocument.createElement("table");
                document.appendChild(table);
                table.setAttribute("name", tableName);

                SKGStringListList listRows;
                err = SKGServices::executeSelectSqliteOrder(iDb, "SELECT * FROM " % tableName, listRows);
                int nbRows = listRows.count();
                if (nbRows) {
                    QStringList titles = listRows.at(0);
                    for (int j = 1; !err && j < nbRows; ++j) {  // Forget title
                        QStringList values = listRows.at(j);

                        QDomElement row = oDocument.createElement("row");
                        table.appendChild(row);

                        int nbVals = values.count();
                        for (int k = 0; k < nbVals; ++k) {
                            row.setAttribute(titles.at(k), values.at(k));
                        }
                    }
                }
            }
        }
    }
    return err;
}

SKGError SKGServices::copySqliteDatabase(QSqlDatabase* iFileDb, QSqlDatabase* iMemoryDb, bool iFromFileToMemory, const QString& iPassword)
{
    SKGError err;
    SKGTRACEINFUNCRC(10, err);
    if (iFileDb && iMemoryDb) {
        SKGTRACEL(20) << "Input parameter [iFileDb]=[" << iFileDb->databaseName() << ']' << endl;
        SKGTRACEL(20) << "Input parameter [iMemoryDb]=[" << iMemoryDb->databaseName() << ']' << endl;
        SKGTRACEL(10) << "Input parameter [iFromFileToMemory]=[" << (iFromFileToMemory ? "FILE->MEMORY" : "MEMORY->FILE") << ']' << endl;

        QString dbFileName = iFileDb->databaseName();
        // Copy the tables
        SKGStringListList listTables;
        int nb = 0;
        IFOK(err) {
            err = SKGServices::executeSelectSqliteOrder((iFromFileToMemory ? iFileDb : iMemoryDb),
                    "SELECT sql, tbl_name FROM sqlite_master WHERE type='table' AND sql NOT NULL and name NOT LIKE 'sqlite_%'",
                    listTables);

            nb = listTables.count();
            for (int i = 1; !err && i < nb; ++i) {  // Forget header
                QString val = listTables.at(i).at(0);
                err = SKGServices::executeSqliteOrder((iFromFileToMemory ? iMemoryDb : iFileDb), val);
            }
        }
        // Attach db
        IFOK(err) {
            QString add;
            if (!iPassword.isEmpty()) {
                add = " KEY '" % SKGServices::stringToSqlString(iPassword) % "'";
            }
            err = SKGServices::executeSqliteOrder(iMemoryDb, "ATTACH DATABASE '" % dbFileName % "' as source" % add);
        }

        // Copy records
        IFOK(err) {
            err = SKGServices::executeSqliteOrder(iMemoryDb, "BEGIN");
            IFOK(err) {
                for (int i = 1; !err && i < nb; ++i) {  // Forget header
                    QString val = listTables.at(i).at(1);
                    if (iFromFileToMemory)  {
                        err = SKGServices::executeSqliteOrder(iMemoryDb, "insert into main." % val % " select * from source." % val);
                    } else {
                        err = SKGServices::executeSqliteOrder(iMemoryDb, "insert into source." % val % " select * from main." % val);
                    }
                }
            }
            SKGServices::executeSqliteOrder(iMemoryDb, "COMMIT");
        }

        // Detach
        {
            SKGError err2 = SKGServices::executeSqliteOrder(iMemoryDb, "DETACH DATABASE source");
            if (!err && err2) {
                err = err2;
            }
        }

        // Optimization
        IFOK(err) {
            QStringList optimization;
            optimization << "PRAGMA case_sensitive_like=true"
                         << "PRAGMA journal_mode=MEMORY"
                         << "PRAGMA temp_store=MEMORY"
                         << "PRAGMA locking_mode=EXCLUSIVE"
                         << "PRAGMA synchronous = OFF"
                         << "PRAGMA recursive_triggers=true";
            err = SKGServices::executeSqliteOrders(iFromFileToMemory ? iMemoryDb : iFileDb, optimization);
        }

        // Copy the indexes
        IFOK(err) {
            SKGStringListList listSqlOrder;
            err = SKGServices::executeSelectSqliteOrder((iFromFileToMemory ? iFileDb : iMemoryDb),
                    "SELECT sql FROM sqlite_master WHERE type='index' AND sql NOT NULL and name NOT LIKE 'sqlite_%'",
                    listSqlOrder);

            int nb2 = listSqlOrder.count();
            for (int i = 1; !err && i < nb2; ++i) {  // Forget header
                QString val = listSqlOrder.at(i).at(0);
                err = SKGServices::executeSqliteOrder((iFromFileToMemory ? iMemoryDb : iFileDb), val);
            }
        }

        // Copy the views
        IFOK(err) {
            SKGStringListList listSqlOrder;
            err = SKGServices::executeSelectSqliteOrder((iFromFileToMemory ? iFileDb : iMemoryDb),
                    "SELECT sql FROM sqlite_master WHERE type='view' AND sql NOT NULL and name NOT LIKE 'sqlite_%'",
                    listSqlOrder);

            int nb2 = listSqlOrder.count();
            for (int i = 1; !err && i < nb2; ++i) {  // Forget header
                QString val = listSqlOrder.at(i).at(0);
                err = SKGServices::executeSqliteOrder((iFromFileToMemory ? iMemoryDb : iFileDb), val);
            }
        }

        // Copy the triggers, must be done after the views
        IFOK(err) {
            SKGStringListList listSqlOrder;
            err = SKGServices::executeSelectSqliteOrder((iFromFileToMemory ? iFileDb : iMemoryDb),
                    "SELECT sql FROM sqlite_master WHERE type='trigger' AND sql NOT NULL and name NOT LIKE 'sqlite_%'",
                    listSqlOrder);

            int nb2 = listSqlOrder.count();
            for (int i = 1; !err && i < nb2; ++i) {  // Forget header
                QString val = listSqlOrder.at(i).at(0);
                err = SKGServices::executeSqliteOrder((iFromFileToMemory ? iMemoryDb : iFileDb), val);
            }
        }

        // Check if created file exists
        if (!err && !iFromFileToMemory && !QFile(dbFileName).exists()) {
            err.setReturnCode(ERR_FAIL).setMessage(i18nc("An error message: creating a file failed", "Creation file '%1' failed", dbFileName));
        }
    }
    IFKO(err) {
        err.addError(SQLLITEERROR + ERR_FAIL, i18nc("Error message: something failed", "%1 failed", QString("SKGServices::copySqliteDatabase()")));
    }
    return err;
}

SKGError SKGServices::executeSqliteOrders(QSqlDatabase* iDb, const QStringList& iSqlOrders)
{
    SKGError err;
    _SKGTRACEINFUNCRC(10, err);
    int nb = iSqlOrders.count();
    for (int i = 0; !err && i < nb; ++i) {
        err = SKGServices::executeSqliteOrder(iDb, iSqlOrders.at(i));
    }
    return err;
}

SKGError SKGServices::executeSqliteOrder(QSqlDatabase* iDb, const QString& iSqlOrder, const QMap<QString, QVariant>& iBind, int* iLastId)
{
    SKGError err;
    _SKGTRACEINFUNCRC(10, err);
    SKGTRACEL(20) << "Input parameter [iSqlOrder]=[" << iSqlOrder << ']' << endl;

    if (iDb == NULL) {
        err = SKGError(ERR_POINTER, i18nc("Error message", "No database defined"));
    } else {
        QSqlQuery query(QString(), *iDb);
        query.setForwardOnly(true);

        double elapse = 0;
        if (SKGServices::SKGSqlTraces != -1) {
            elapse = SKGServices::getMicroTime();
        }

        // Prepare sql order
        bool prep = query.prepare(iSqlOrder);

        // Bind values
        QMapIterator<QString, QVariant> i(iBind);
        while (i.hasNext()) {
            i.next();
            query.bindValue(i.key(), i.value());
        }

        if (!prep || !query.exec()) {
            QSqlError sqlError = query.lastError();
            if (sqlError.number() != 19 /*SQLITE_CONSTRAINT*/) {
                SKGTRACE << "WARNING: " << iSqlOrder << endl;
                SKGTRACE << "         returns :" << sqlError.text() << endl;
            }

            err = SKGError(SQLLITEERROR + sqlError.number(), iSqlOrder);
            err.addError(SQLLITEERROR + sqlError.number(), sqlError.text());

            if (sqlError.number() == 19 && iSqlOrder.startsWith(QLatin1String("INSERT "))) {
                err.addError(ERR_FAIL, i18nc("Error message", "Creation failed. The object already exists."));
            }
        } else {
            if (iLastId) {
                *iLastId = query.lastInsertId().toInt();
            }
        }
        if (SKGServices::SKGSqlTraces != -1) {
            elapse = SKGServices::getMicroTime() - elapse;
            if (elapse >= SKGServices::SKGSqlTraces) {
                SKGTRACE << "executeSqliteOrder :" << iSqlOrder << " TIME=" << elapse << " ms" << endl;
            }
        }
    }
    return err;
}

SKGError SKGServices::executeSqliteOrder(QSqlDatabase* iDb, const QString& iSqlOrder, int* iLastId)
{
    return executeSqliteOrder(iDb, iSqlOrder, QMap< QString, QVariant >(), iLastId);
}

SKGError SKGServices::dumpSelectSqliteOrder(QSqlDatabase* iDb, const QString& iSqlOrder, QTextStream* oStream, SKGServices::DumpMode iMode)
{
    SKGError err;
    _SKGTRACEINFUNCRC(10, err);
    SKGTRACEL(20) << "Input parameter [iSqlOrder]=[" << iSqlOrder << ']' << endl;

    // initialisation
    QStringList oResult;
    err = SKGServices::dumpSelectSqliteOrder(iDb, iSqlOrder, oResult, iMode);
    IFOK(err) {
        // dump
        int nb = oResult.size();
        for (int i = 0; i < nb; ++i) {
            if (oStream == NULL) {
                SKGTRACESUITE << oResult.at(i) << endl;
            } else {
                *oStream << oResult.at(i) << endl;
            }
        }
    }
    return err;
}

SKGError SKGServices::dumpSelectSqliteOrder(QSqlDatabase* iDb, const QString& iSqlOrder, QString& oResult, SKGServices::DumpMode iMode)
{
    SKGError err;
    _SKGTRACEINFUNCRC(10, err);
    // initialisation
    oResult = "";

    QStringList oResultTmp;
    err = SKGServices::dumpSelectSqliteOrder(iDb, iSqlOrder, oResultTmp, iMode);
    IFOK(err) {
        // dump
        int nb = oResultTmp.size();
        for (int i = 0; i < nb; ++i) {
            oResult += oResultTmp.at(i) % '\n';
        }
    }
    return err;
}

SKGError SKGServices::dumpSelectSqliteOrder(QSqlDatabase* iDb, const QString& iSqlOrder, QStringList& oResult, SKGServices::DumpMode iMode)
{
    SKGError err;
    _SKGTRACEINFUNCRC(10, err);

    // Execution of sql order
    SKGStringListList oResultTmp;
    err = executeSelectSqliteOrder(iDb, iSqlOrder, oResultTmp);
    IFOK(err) oResult = tableToDump(oResultTmp, iMode);
    return err;
}

SKGError SKGServices::executeSingleSelectSqliteOrder(QSqlDatabase* iDb, const QString& iSqlOrder, QString& oResult)
{
    SKGStringListList result;
    SKGError err = executeSelectSqliteOrder(iDb, iSqlOrder, result);
    oResult = result.value(1).value(0);
    return err;
}

SKGError SKGServices::executeSelectSqliteOrder(QSqlDatabase* iDb, const QString& iSqlOrder, SKGStringListList& oResult)
{
    SKGError err;
    _SKGTRACEINFUNCRC(10, err);
    // initialisation
    oResult.clear();

    if (iDb == NULL) {
        err = SKGError(ERR_POINTER, i18nc("Error message", "No database defined"));
    } else {
        QSqlQuery query(QString(), *iDb);
        query.setForwardOnly(true);
        double elapse = 0;
        if (SKGServices::SKGSqlTraces != -1) {
            elapse = SKGServices::getMicroTime();
        }

        if (!query.exec(iSqlOrder)) {
            QSqlError sqlError = query.lastError();
            SKGTRACE << "WARNING: " << iSqlOrder << endl;
            SKGTRACE << "         returns :" << sqlError.text() << endl;
            err = SKGError(SQLLITEERROR + sqlError.number(), iSqlOrder);
            err.addError(SQLLITEERROR + sqlError.number(), sqlError.text());
        } else {
            double elapse1 = 0;
            if (SKGServices::SKGSqlTraces != -1) {
                elapse1 = SKGServices::getMicroTime() - elapse;
            }

            // Addition of column names
            QSqlRecord rec = query.record();
            QStringList line;
            int index = 0;
            while (index != -1) {
                QString val = rec.fieldName(index);
                if (!val.isEmpty()) {
                    line.push_back(val);
                    ++index;
                } else {
                    index = -1;
                }
            }
            oResult.push_back(line);

            // Addition of rows
            while (query.next()) {
                QStringList line2;
                int index2 = 0;
                while (index2 != -1) {
                    QVariant val = query.value(index2);
                    if (val.isValid()) {
                        line2.push_back(val.toString());
                        ++index2;
                    } else {
                        index2 = -1;
                    }
                }
                oResult.push_back(line2);
            }
            if (SKGServices::SKGSqlTraces != -1) {
                double elapse2 = SKGServices::getMicroTime() - elapse;
                if (elapse1 >= SKGServices::SKGSqlTraces) {
                    SKGTRACE << "executeSqliteOrder:" << iSqlOrder << " TIME=" << elapse1 << " ms,  (with fetch):" << elapse2 << " ms" << endl;
                }
            }
        }
    }
    return err;
}

SKGError SKGServices::readPropertyFile(const QString& iFileName, QHash< QString, QString >& oProperties)
{
    SKGError err;
    oProperties.clear();

    // Open file
    QFile file(iFileName);
    if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) {
        err = SKGError(ERR_FAIL, i18nc("An erro message", "Open file '%1' failed", iFileName));
    } else {
        // Read file
        QTextStream stream(&file);
        while (!stream.atEnd() && !err) {
            // Read line
            QString line = stream.readLine().trimmed();
            if (!line.isEmpty() && !line.startsWith(QLatin1String("#"))) {
                int pos = line.indexOf("=");
                if (pos != -1) {
                    oProperties[line.left(pos).trimmed().toLower()] = line.right(line.count() - pos - 1);
                }
            }
        }

        // close file
        file.close();
    }
    return err;
}

double SKGServices::getMicroTime()
{
#ifdef Q_OS_WIN
    return static_cast<double>(GetTickCount());
#else
    struct timeval tv;
    struct timezone tz;

    // get time
    gettimeofday(&tv, &tz);

    // return time
    return (static_cast<double>(1000.0 * tv.tv_sec)) + (static_cast<double>(tv.tv_usec / 1000));
#endif
}

SKGStringListList SKGServices::getBase100Table(const SKGStringListList& iTable)
{
    SKGTRACEINFUNC(10);

    // Build history
    SKGStringListList output;

    output.push_back(iTable.at(0));

    int nblines = iTable.count();
    int nbCols = 0;
    if (nblines) {
        nbCols = iTable.at(0).count();
    }

    // Create table
    for (int i = 1; i < nblines; ++i) {
        QStringList newLine;
        newLine.push_back(iTable.at(i).at(0));

        double valInitial = 0;

        for (int j = 1; j < nbCols; ++j) {
            double val = SKGServices::stringToDouble(iTable.at(i).at(j));
            if (j == 1) {
                valInitial = val;
                val = 100.0;
            } else {
                if (valInitial != 0.0) {
                    val = 100.0 * val / valInitial;
                }
            }
            newLine.push_back(SKGServices::doubleToString(val));
        }
        output.push_back(newLine);
    }

    return output;
}

SKGStringListList SKGServices::getPercentTable(const SKGStringListList& iTable, bool iOfColumns, bool iAbsolute)
{
    SKGTRACEINFUNC(10);

    // Build history
    SKGStringListList output;

    output.push_back(iTable.at(0));

    int nblines = iTable.count();
    int nbCols = 0;
    if (nblines) {
        nbCols = iTable.at(0).count();
    }

    // Compute sums
    QList<double> sums;
    if (iOfColumns) {
        // Compute sum of columns
        for (int j = 1; j < nbCols; ++j) {
            // Compute sum
            double sum = 0;
            for (int i = 1; i < nblines; ++i) {
                double v = SKGServices::stringToDouble(iTable.at(i).at(j));
                sum += (iAbsolute ? qAbs(v) : v);
            }

            sums.push_back(sum);
        }
    } else {
        // Compute sum of lines
        for (int j = 1; j < nblines; ++j) {
            // Compute sum
            double sum = 0;
            for (int i = 1; i < nbCols; ++i) {
                double v = SKGServices::stringToDouble(iTable.at(j).at(i));
                sum += (iAbsolute ? qAbs(v) : v);
            }

            sums.push_back(sum);
        }
    }

    // Create table
    for (int i = 1; i < nblines; ++i) {
        QStringList newLine;
        newLine.push_back(iTable.at(i).at(0));

        for (int j = 1; j < nbCols; ++j) {
            double val = SKGServices::stringToDouble(iTable.at(i).at(j));
            val = (iAbsolute ? qAbs(val) : val);
            double sum = (iOfColumns ? sums.at(j - 1) : sums.at(i - 1));
            newLine.push_back(SKGServices::doubleToString(sum == 0.0 ? 0.0 : 100.0 * val / sum));
        }
        output.push_back(newLine);
    }

    return output;
}

SKGStringListList SKGServices::getHistorizedTable(const SKGStringListList& iTable)
{
    SKGTRACEINFUNC(10);

    // Build history
    SKGStringListList output;

    output.push_back(iTable.at(0));

    int nblines = iTable.count();
    int nbCols = 0;
    if (nblines) {
        nbCols = iTable.at(0).count();
    }
    for (int i = 1; i < nblines; ++i) {
        QStringList newLine;
        newLine.push_back(iTable.at(i).at(0));

        double sum = 0;
        for (int j = 1; j < nbCols; ++j) {
            sum += SKGServices::stringToDouble(iTable.at(i).at(j));
            newLine.push_back(SKGServices::doubleToString(sum));
        }
        output.push_back(newLine);
    }

    return output;
}

QString SKGServices::encodeForUrl(const QString& iString)
{
    return QUrl::toPercentEncoding(iString);
}

QIcon SKGServices::fromTheme(const QString& iName, const QStringList& iOverlays)
{
    QIcon output;
    if (iOverlays.count() > 0) {
        output = QIcon(new KIconEngine(iName,  KIconLoader::global(), iOverlays));
    } else {
        output = QIcon(new KIconEngine(iName,  KIconLoader::global()));
    }
    if (output.isNull() && !iName.isEmpty()) {
        SKGTRACE << "WARNING: Icon [" << iName << "] not found" << endl;
    }
    return output;
}

QString SKGServices::getMajorVersion(const QString& iVersion)
{
    QString output = iVersion;
    int pos = output.indexOf('.');
    if (pos != -1) {
        pos = output.indexOf('.', pos + 1);
        if (pos != -1) {
            output = output.left(pos);
        }
    }
    return output;
}
