<?php
/**
 * The Auth_smb class provides an SMB implementation of the Horde
 * authentication system.
 *
 * This module requires the smbauth extension for PHP:
 *   http://www.tekrat.com/smbauth.php
 *
 * At the time of this writing, the extension, and thus this module, only
 * supported authentication against a domain, and pdc and bdc must be
 * non-null and not equal to each other. In other words, to use this module
 * you must have a domain with at least one PDC and one BDC.
 *
 * Required parameters:
 * ====================
 *   'bdc_hostspec'      The hostname of the BDC SMB server.
 *   'domain'            The domain to authenticate against.
 *   'pdc_hostspec'      The hostname of the PDC SMB server.
 *
 *
 * $Horde: horde/lib/Auth/smb.php,v 1.10 2003/07/10 21:42:57 slusarz Exp $
 *
 * Copyright 1999-2003 Jon Parise <jon@horde.org>
 * Copyright 2002-2003 Marcus I. Ryan <marcus@riboflavin.net>
 *
 * See the enclosed file COPYING for license information (LGPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/lgpl.html.
 *
 * @author  Jon Parise <jon@horde.org>
 * @author  Marcus I. Ryan <marcus@riboflavin.net>
 * @version $Revision: 1.10 $
 * @since   Horde 3.0
 * @package horde.auth
 */
class Auth_smb extends Auth {

    /**
     * An array of capabilities, so that the driver can report which
     * operations it supports and which it doesn't.
     *
     * @var array $capabilities
     */
    var $capabilities = array('add'         => false,
                              'update'      => false,
                              'remove'      => false,
                              'list'        => false,
                              'transparent' => false,
                              'loginscreen' => false);

    /**
     * Constructs a new SMB authentication object.
     *
     * @access public
     *
     * @param optional array $params  A hash containing connection parameters.
     */
    function Auth_smb($params = array())
    {
        if (!Horde::extensionExists('smbauth')) {
            Horde::fatal(PEAR::raiseError(_("Auth_smbauth: Required smbauth extension not found."), __FILE__, __LINE__));
        }

        /* Ensure we've been provided with all of the necessary parameters. */
        $required_params = array('pdc_hostspec', 'bdc_hostspec', 'domain');
        foreach ($required_params as $val) {
            if (!array_key_exists($val, $params)) {
                Horde::fatal(PEAR::raiseError(sprintf(_("Required '%s' not specified in authentication configuration."), $val)), __FILE__, __LINE__);
            }
        }

        $this->_params = $params;
    }

    /**
     * Find out if the given set of login credentials are valid.
     *
     * @access private
     *
     * @param string $userID      The userID to check.
     * @param array $credentials  An array of login credentials.
     *
     * @return boolean  True on success or a PEAR_Error object on failure.
     */
    function _authenticate($userID, $credentials)
    {
        if (!array_key_exists('password', $credentials)) {
            Horde::fatal(PEAR::raiseError(_("No password provided for SMB authentication.")), __FILE__, __LINE__);
        }

        /* Authenticate */
        $rval = smbauth_validate($userID,
                                 $credentials['password'],
                                 $this->_params['pdc_hostspec'],
                                 $this->_params['bdc_hostspec'],
                                 $this->_params['domain']);

        if ($rval === 0) {
            return true;
        } else {
            if ($rval === 1) {
                $this->_setAuthError(_("Failed to connect to SMB server."));
            } else {
                $this->_setAuthError(smbauth_err2str($rval));
            }
            return false;
        }
    }

}
