/*************************************************************************************
 *  Copyright (C) 2013 by Alejandro Fiestas Fiestas <afiestas@kde.org>               *
 *                                                                                   *
 *  This program is free software; you can redistribute it and/or                    *
 *  modify it under the terms of the GNU General Public License                      *
 *  as published by the Free Software Foundation; either version 2                   *
 *  of the License, or (at your option) any later version.                           *
 *                                                                                   *
 *  This program is distributed in the hope that it will be useful,                  *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of                   *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                    *
 *  GNU General Public License for more details.                                     *
 *                                                                                   *
 *  You should have received a copy of the GNU General Public License                *
 *  along with this program; if not, write to the Free Software                      *
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA   *
 *************************************************************************************/

#include "ObexFtpDaemon.h"
#include "createsessionjob.h"
#include "dbus_object_manager.h"
#include "version.h"

#include <QHash>
#include <QDebug>
#include <QDBusConnection>

#include <KAboutData>
#include <KPluginFactory>
#include <KLocalizedString>

#include <bluedevil/bluedevilmanager.h>
#include <bluedevil/bluedeviladapter.h>

using namespace BlueDevil;
K_PLUGIN_FACTORY_WITH_JSON(ObexFtpFactory,
                           "obexftpdaemon.json",
                           registerPlugin<ObexFtpDaemon>();)

struct ObexFtpDaemon::Private
{
    enum Status {
        Online = 0,
        Offline
    } m_status;

    QHash <QString, QString> m_sessionMap;
    QHash <QString, QString> m_reverseSessionMap;
    QHash <QString, CreateSessionJob*> m_wipSessions;
    QDBusServiceWatcher *m_serviceWatcher;
    OrgFreedesktopDBusObjectManagerInterface *m_interface;
};

ObexFtpDaemon::ObexFtpDaemon(QObject *parent, const QList<QVariant>&)
    : KDEDModule(parent)
    , d(new Private)
{
    qDBusRegisterMetaType<DBusManagerStruct>();
    qDBusRegisterMetaType<QVariantMapMap>();

    d->m_status = Private::Offline;

    KAboutData aboutData(QStringLiteral("obexftpdaemon"),
                         i18n("ObexFtp Daemon"),
                         bluedevil_version,
                         i18n("ObexFtp Daemon"),
                         KAboutLicense::GPL,
                         i18n("(c) 2010, UFO Coders"));

    aboutData.addAuthor(i18n("Alejandro Fiestas Olivares"), i18n("Maintainer"),
                        QStringLiteral("afiestas@kde.org"), QStringLiteral("http://www.afiestas.org"));

    connect(Manager::self(), SIGNAL(usableAdapterChanged(Adapter*)),
            SLOT(usableAdapterChanged(Adapter*)));

    d->m_interface = new OrgFreedesktopDBusObjectManagerInterface(QStringLiteral("org.bluez.obex"),
                                                                  QStringLiteral("/"),
                                                                  QDBusConnection::sessionBus(), this);

    connect(d->m_interface, SIGNAL(InterfacesRemoved(QDBusObjectPath,QStringList)),
            SLOT(interfaceRemoved(QDBusObjectPath,QStringList)));

    d->m_serviceWatcher = new QDBusServiceWatcher(QStringLiteral("org.bluez.obex"),
                                                  QDBusConnection::sessionBus(),
                                                  QDBusServiceWatcher::WatchForUnregistration, this);

    connect(d->m_serviceWatcher, SIGNAL(serviceUnregistered(QString)), SLOT(serviceUnregistered(QString)));

    //WARNING this blocks if org.bluez in system bus is dead
    if (Manager::self()->usableAdapter()) {
        onlineMode();
    }
}

ObexFtpDaemon::~ObexFtpDaemon()
{
    if (d->m_status == Private::Online) {
        offlineMode();
    }
    delete d;
}

void ObexFtpDaemon::onlineMode()
{
    qCDebug(OBEXDAEMON);
    if (d->m_status == Private::Online) {
        qCDebug(OBEXDAEMON) << "Already in onlineMode";
        return;
    }

    d->m_status = Private::Online;
}

void ObexFtpDaemon::offlineMode()
{
    qCDebug(OBEXDAEMON) << "Offline mode";
    if (d->m_status == Private::Offline) {
        qCDebug(OBEXDAEMON) << "Already in offlineMode";
        return;
    }

    d->m_sessionMap.clear();
    d->m_reverseSessionMap.clear();

    d->m_status = Private::Offline;
}

void ObexFtpDaemon::usableAdapterChanged(Adapter *adapter)
{
    if (!adapter) {
        offlineMode();
        return;
    }

    onlineMode();
}

QString ObexFtpDaemon::session(QString address, const QDBusMessage& msg)
{
    qCDebug(OBEXDAEMON) << address;
    address.replace(QLatin1Char('-'), QLatin1Char(':'));

    if (d->m_sessionMap.contains(address)) {
        return d->m_sessionMap[address];
    }

    //At this point we always want delayed reply
    msg.setDelayedReply(true);
    if (d->m_wipSessions.contains(address)) {
        d->m_wipSessions[address]->addMessage(msg);
        return QString();
    }

    CreateSessionJob *job = new CreateSessionJob(address, msg);
    connect(job, SIGNAL(finished(KJob*)), SLOT(sessionCreated(KJob*)));
    job->start();

    d->m_wipSessions.insert(address, job);
    return QString();
}

void ObexFtpDaemon::sessionCreated(KJob* job)
{
    CreateSessionJob* cJob = qobject_cast<CreateSessionJob*>(job);
    qCDebug(OBEXDAEMON) << cJob->path();

    d->m_wipSessions.remove(cJob->address());
    d->m_sessionMap.insert(cJob->address(), cJob->path());
    d->m_reverseSessionMap.insert(cJob->path(), cJob->address());

    const QList<QDBusMessage> messages = cJob->messages();
    Q_FOREACH(const QDBusMessage &msg, messages) {
        QDBusMessage reply = msg.createReply(cJob->path());
        QDBusConnection::sessionBus().asyncCall(reply);
    }
}

void ObexFtpDaemon::serviceUnregistered(const QString& service)
{
    if (service != QLatin1String("org.bluez.obex")) {
        return;
    }

    d->m_sessionMap.clear();
    d->m_reverseSessionMap.clear();
}

void ObexFtpDaemon::interfaceRemoved(const QDBusObjectPath &dbusPath, const QStringList& interfaces)
{
    qCDebug(OBEXDAEMON) << dbusPath.path() << interfaces;
    const QString path = dbusPath.path();
    if (!d->m_reverseSessionMap.contains(path)) {
        qCDebug(OBEXDAEMON) << d->m_reverseSessionMap;
        return;
    }

    QString address = d->m_reverseSessionMap.take(path);
    qCDebug(OBEXDAEMON) << address;
    qCDebug(OBEXDAEMON) << d->m_sessionMap.remove(address);
}

Q_LOGGING_CATEGORY(OBEXDAEMON, "ObexDaemon")

#include "ObexFtpDaemon.moc"
