/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// KDE includes
#include <klocale.h>
#include <kdebug.h>
#include <kurldrag.h>

// Qt includes
#include <qpainter.h>
#include <qimage.h>
#include <qpixmap.h>
#include <qframe.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qcursor.h>

// STL includes
#include <algorithm>
  using std::min;
  using std::max;

// WorKflow includes
#include "imageviewer.h"

using namespace WorKflow;

class ImageViewer::Private
{
public:
  Private(ImageViewer* v);

  void adjustControlWidget();

  QPixmap pixmap;
  QPoint pos;
  double zoom;
  bool translating;
  bool zoomEnabled;
  bool dropEnabled;
  bool controlWidgetEnabled;
  bool autoScale;
  bool autoLoad;

  QFrame* controlWidget;
  QLabel* dragLabel;
  QPushButton* zoomInButton;
  QPushButton* zoomOutButton;

private:
  ImageViewer* p;
};

ImageViewer::Private::Private(ImageViewer* v)
  : zoom(1), translating(false), zoomEnabled(true), dropEnabled(true), controlWidgetEnabled(true), autoScale(true), autoLoad(true), p(v)
{
}

void ImageViewer::Private::adjustControlWidget()
{
  controlWidget->adjustSize();
  const int margin = 6;
  controlWidget->setGeometry(p->width() - (p->verticalScrollBar()->isVisible() ? p->verticalScrollBar()->width() : 0)
                                  - controlWidget->width() - margin,
                                margin,
                                controlWidget->width(),
                                controlWidget->height());
}

ImageViewer::ImageViewer(QWidget* parent, const char* name, WFlags f)
  : QScrollView(parent, name, f | Qt::WNoAutoErase)
{
  d = new Private(this);

  // control widget
  d->controlWidget = new QFrame(this);
  d->controlWidget->setFrameShape(QFrame::StyledPanel);
  d->controlWidget->setFrameShadow(QFrame::Raised);
  d->controlWidget->setLineWidth(1);
  QHBoxLayout* hbox = new QHBoxLayout(d->controlWidget, 2, 2);
  d->dragLabel = new QLabel(i18n("Drop image"), d->controlWidget);
  d->zoomInButton = new QPushButton("+", d->controlWidget);
  d->zoomOutButton = new QPushButton("-", d->controlWidget);
  hbox->addWidget(d->dragLabel);
  hbox->addWidget(d->zoomInButton);
  hbox->addWidget(d->zoomOutButton);

  viewport()->setAcceptDrops(true);

  d->controlWidget->hide();

  connect(d->zoomInButton, SIGNAL(clicked()), this, SLOT(zoomIn()));
  connect(d->zoomOutButton, SIGNAL(clicked()), this, SLOT(zoomOut()));
}

ImageViewer::~ImageViewer()
{
  delete d;
}

void ImageViewer::contentsContextMenuEvent(QContextMenuEvent* e)
{
}

void ImageViewer::contentsDragEnterEvent(QDragEnterEvent* e)
{
  e->accept(e->provides("text/uri-list"));
}

void ImageViewer::contentsDropEvent(QDropEvent* e)
{
  if (QUriDrag::canDecode(e)) {
    KURL::List urls;
    if (KURLDrag::decode(e, urls)) {
      for (KURL::List::ConstIterator i = urls.begin(); i != urls.end(); ++i) {
        QString path = (*i).path();
        if (!d->autoLoad || (d->autoLoad && setImage(path))) {
          emit dropped(path);
          e->accept(true);
          return;
        }
      }
    }
  }

  e->accept(false);
}

void ImageViewer::viewportMouseMoveEvent(QMouseEvent* e)
{
  if (d->translating) {
    QPoint delta = d->pos - e->pos();
    scrollBy(delta.x(), delta.y());
    d->pos = e->pos();
    e->accept();
  } else {
    QScrollView::viewportMouseMoveEvent(e);
  }
}

void ImageViewer::viewportMousePressEvent(QMouseEvent* e)
{
  if (e->button() & Qt::LeftButton) {
    d->pos = e->pos();
    d->translating = true;
    setCursor(QCursor(Qt::SizeAllCursor));
    e->accept();
  } else {
    QScrollView::viewportMousePressEvent(e);
  }
}

void ImageViewer::viewportMouseReleaseEvent(QMouseEvent* e)
{
  if (e->button() & Qt::LeftButton) {
    d->translating = false;
    unsetCursor();
    e->accept();
  } else {
    QScrollView::viewportMouseReleaseEvent(e);
  }
}

void ImageViewer::drawContents(QPainter* p, int cx, int cy, int cw, int ch)
{
  p->scale(d->zoom, d->zoom);
  QWMatrix invscale;
  invscale.scale(1.0/d->zoom, 1.0/d->zoom);

  QRect exposedRect = invscale.mapRect(QRect(cx, cy, cw, ch));
  exposedRect.addCoords(-1, -1, 1, 1);

  p->drawPixmap(QPoint(exposedRect.x(), exposedRect.y()), d->pixmap, exposedRect);
}

bool ImageViewer::setImage(const QString& fileName)
{
  QImage img;
  if (img.load(fileName)) {
    setImage(img);
    return true;
  }
  return false;
}

void ImageViewer::setImage(const QImage& image)
{
  if (d->autoScale) {
    QImage img = image.scale(visibleWidth(), visibleHeight());
    d->pixmap.convertFromImage(img);
  } else
    d->pixmap.convertFromImage(image);

  resizeContents(d->pixmap.width() * d->zoom, d->pixmap.height() * d->zoom);
  updateContents();
}

void ImageViewer::setZoom(double newZoom)
{
  d->zoom = newZoom;
  resizeContents(d->pixmap.width() * newZoom, d->pixmap.height() * newZoom);
  viewport()->erase();
  updateContents();

  emit zoomChanged(newZoom);
}

double ImageViewer::zoom() const
{
  return d->zoom;
}

void ImageViewer::setZoomEnabled(bool enabled)
{
  d->zoomEnabled = enabled;
  d->zoomInButton->setShown(enabled);
  d->zoomOutButton->setShown(enabled);
}

void ImageViewer::setDropEnabled(bool enabled)
{
  d->dropEnabled = enabled;
  d->dragLabel->setShown(enabled);
  viewport()->setAcceptDrops(enabled);
}

void ImageViewer::setControlWidgetEnabled(bool enabled)
{
  d->controlWidgetEnabled = enabled;
  if (!enabled)
    d->controlWidget->hide();
}

void ImageViewer::setAutoScale(bool enabled)
{
  d->autoScale = enabled;
}

void ImageViewer::setAutoLoad(bool enabled)
{
  d->autoLoad = enabled;
}

bool ImageViewer::isZoomEnabled() const
{
  return d->zoomEnabled;
}

bool ImageViewer::isDropEnabled() const
{
  return d->dropEnabled;
}

bool ImageViewer::isControlWidgetEnabled() const
{
  return d->controlWidgetEnabled;
}

bool ImageViewer::isAutoScale() const
{
  return d->autoScale;
}

bool ImageViewer::isAutoLoad() const
{
  return d->autoLoad;
}

void ImageViewer::enterEvent(QEvent*)
{
  if (d->controlWidgetEnabled) {
    d->adjustControlWidget();
    d->controlWidget->show();
  }
}

void ImageViewer::leaveEvent(QEvent*)
{
  d->controlWidget->hide();
}

void ImageViewer::resizeEvent(QResizeEvent* e)
{
  d->adjustControlWidget();
  QScrollView::resizeEvent(e);
}

void ImageViewer::zoomIn()
{
  setZoom(min(zoom() * 2.0, 16.0));
}

void ImageViewer::zoomOut()
{
  setZoom(max(zoom() * 0.5, 0.25));
}

#include "imageviewer.moc"
