/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// KDE includes
#include <kapplication.h>
#include <klocale.h>
#include <kdebug.h>
#include <kaction.h>
#include <kstatusbar.h>

// Qt includes
#include <qlayout.h>
#include <qlabel.h>
#include <qcombobox.h>
#include <qtimer.h>

// WorKflow includes
#include "mainwindow.h"
#include "document.h"
#include "librarymanager.h"
#include "commandmanager.h"
#include "commandfactory.h"
#include "commanddescription.h"
#include "command.h"
#include "category.h"
#include "commandwidget.h"

// debugger includes
#include "parameterinspector.h"
#include "resultinspector.h"

using namespace WorKflow;

MainWindow::MainWindow(const QString& category, const QString& command)
  : KMainWindow(0, "MainWindow"), m_command(0), m_commandWidget(0)
{
  m_document = new Document(this);

  QWidget* outer = new QWidget(this);
  QGridLayout* grid = new QGridLayout(outer, 3, 2);
  QLabel* catLabel = new QLabel(i18n("Category"), outer);
  QLabel* cmdLabel = new QLabel(i18n("Command"), outer);
  catLabel->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
  cmdLabel->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
  m_categoryCombo = new QComboBox(outer);
  m_commandCombo = new QComboBox(outer);
  m_mainSpace = new QWidget(outer);
  m_mainSpace->setBackgroundColor(Qt::white);
  m_mainLayout = new QVBoxLayout(m_mainSpace);
  m_mainLayout->addItem(new QSpacerItem(0, 0, QSizePolicy::Fixed, QSizePolicy::Expanding));

  grid->addWidget(catLabel, 0, 0);
  grid->addWidget(m_categoryCombo, 0, 1);
  grid->addWidget(cmdLabel, 1, 0);
  grid->addWidget(m_commandCombo, 1, 1);
  grid->addMultiCellWidget(m_mainSpace, 2, 2, 0, 1);

  setCentralWidget(outer);

  setupActions();
  statusBar()->show();
  setupGUI();

  connect(m_categoryCombo, SIGNAL(activated(int)), this, SLOT(fillCommandCombo(int)));
  connect(m_commandCombo, SIGNAL(activated(int)), this, SLOT(showCommand(int)));

  connect(LibraryManager::self(), SIGNAL(allLibrariesLoaded()), this, SLOT(slotLibrariesLoaded()));
  LibraryManager::self()->loadAllLibraries();

  connect(&m_statusBarTimer, SIGNAL(timeout()), this, SLOT(statusBarTimeout()));

  connect(ParameterInspector::self(), SIGNAL(visibilityChanged()), this, SLOT(updateActionState()));
  connect(ResultInspector::self(), SIGNAL(visibilityChanged()), this, SLOT(updateActionState()));

/*  if (!category.isEmpty()) {
    Category* cat = CommandManager::self()->findCategory(category);
    if (cat) {
      m_categoryCombo->
    }
  }*/
}

MainWindow::~MainWindow()
{
  kdDebug() << k_funcinfo << endl;
}

void MainWindow::setupActions()
{
  KStdAction::quit(kapp, SLOT(quit()), actionCollection());

  m_paramInspector = new KToggleAction(i18n("Parameter Inspector..."), "find", CTRL + Key_I,
                                this, SLOT(showParameterInspector()),
                                actionCollection(), "param_inspector_action");

  m_resultInspector = new KToggleAction(i18n("Result Inspector..."), "find", CTRL + Key_R,
                                this, SLOT(showResultInspector()),
                                actionCollection(), "result_inspector_action");

  m_execute = new KAction(i18n("Execute Command"), "run", CTRL + Key_X,
                                this, SLOT(executeCommand()),
                                actionCollection(), "execute_command_action");

  updateActionState();
}


void MainWindow::slotLibrariesLoaded()
{
  kdDebug() << k_funcinfo << endl;
  m_categories = CommandManager::self()->categories();
  for (CatList::ConstIterator i = m_categories.begin(); i != m_categories.end(); ++i) {
    m_categoryCombo->insertItem(QString("%1 (%2)").arg((*i)->name()).arg((*i)->id()));
  }
}

void MainWindow::fillCommandCombo(int category)
{
  kdDebug() << k_funcinfo << endl;
  m_commandCombo->clear();

  Category* cat = *(m_categories.at(category));

  m_commands = CommandManager::self()->commands(cat->id());

  kdDebug() << cat << endl;
  kdDebug() << cat->name() << endl;

  for (CommandList::ConstIterator i = m_commands.begin(); i != m_commands.end(); ++i) {
    m_commandCombo->insertItem(QString("%1 (%2)").arg((*i)->name()).arg((*i)->id()));
  }
}

void MainWindow::showCommand(int command)
{
  kdDebug() << k_funcinfo << endl;
  if (m_commandWidget) {
    if (m_command->widget())
      m_command->widget()->reparent(0, QPoint());
    delete m_commandWidget;
    delete m_command;
  }

  CommandFactoryBase* factory = (*(m_commands.at(command)))->factory();
  if (factory) {
    m_document->clear();

    delete m_mainLayout;
    m_mainLayout = new QVBoxLayout(m_mainSpace);
    m_mainLayout->addItem(new QSpacerItem(0, 0, QSizePolicy::Fixed, QSizePolicy::Expanding));

    m_command = factory->create(m_document);

    m_commandWidget = new CommandWidget(m_mainSpace, m_command);

    m_mainLayout->add(m_commandWidget);
    m_commandWidget->show();

    m_mainLayout->addItem(new QSpacerItem(0, 0, QSizePolicy::Fixed, QSizePolicy::Expanding));

    connect(m_command, SIGNAL(modified(Command*)), this, SLOT(slotCommandChanged()));
    connect(m_command, SIGNAL(sizeChanged()), this, SLOT(slotCommandResized()));
    ParameterInspector::self()->setCommand(m_command);
    ResultInspector::self()->setCommand(m_command);

    setCaption(m_command->title());

    updateActionState();
  }
}

void MainWindow::slotCommandChanged()
{
  statusBar()->message(i18n("Command changed"));
  m_statusBarTimer.start(1000);
}

void MainWindow::slotCommandResized()
{
  statusBar()->message(i18n("Command resized"));
  m_statusBarTimer.start(1000);
}

void MainWindow::statusBarTimeout()
{
  statusBar()->clear();
}

void MainWindow::showParameterInspector()
{
  ParameterInspector::self()->setShown(m_paramInspector->isChecked());
}

void MainWindow::showResultInspector()
{
  if (m_resultInspector->isChecked())
    ResultInspector::self()->updateInspector();
  ResultInspector::self()->setShown(m_resultInspector->isChecked());
}

void MainWindow::executeCommand()
{
  if (m_command) {
    m_command->updateParameters();
    m_command->execute();
    m_resultInspector->setChecked(true);
    showResultInspector();
  }
}

void MainWindow::updateActionState()
{
  m_execute->setEnabled(m_command != 0);
  m_paramInspector->setChecked(ParameterInspector::self()->isVisible());
  m_resultInspector->setChecked(ResultInspector::self()->isVisible());
}

#include "mainwindow.moc"
