/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

#ifndef LIBRARYMANAGER_H
#define LIBRARYMANAGER_H

// Qt includes
#include <qobject.h>
#include <qvaluelist.h>

// forward declarations
class KPluginInfo;

namespace WorKflow
{
  class Library;
}

namespace WorKflow
{
  /**
   * @short Loads and unloads command libraries.
   *
   * This singleton class's resposiblities are to find, to load and to unload the
   * available command libraries.
   *
   * WorKflow libraries are built upon KDE's plugin system, and in particular the
   * KPluginInfo class. To get a list of KPluginInfo instances representing the
   * available libraries, call availableLibraries(). To get plugin information
   * about a library, call libraryInfo().
   *
   * To load a library, call loadLibrary(), to unload it, call unloadLibrary().
   * The loadedLibraries() method returns a list of Library instances from all
   * loaded libraries. To load all libraries, call loadAllLibraries(). This will
   * emit the allLibrariesLoaded() signal.
   *
   * @author Thomas Kadauke
   */
  class LibraryManager : public QObject
  {
    Q_OBJECT

  public:
    /**
     * Typedef for a list of Library instances.
     */
    typedef QValueList<Library*> LibraryList;

    /**
     * Typedef for a list of meta-info objects for libraries.
     */
    typedef QValueList<KPluginInfo*> PluginList;

    /**
     * Returns the singleton instance of this class.
     * @return The singleton instance.
     */
    static LibraryManager* self();

    /**
     * Returns the list of KPluginInfo objects representing all installed
     * libraries.
     * @return A list of all installed libraries.
     */
    PluginList availableLibraries();

    /**
     * Returns the list of loaded libraries. This list contains instances of
     * Library subclasses.
     * @return The list of loaded libraries.
     */
    LibraryList loadedLibraries();

    /**
     * Returns meta information about the given library.
     * @param lib The Library object for which meta information shall be
     *   returned.
     * @return The KPluginInfo meta information object for the library.
     */
    KPluginInfo* libraryInfo(Library* lib);

    /**
     * Loads all installed libraries and emits the allLibrariesLoaded() signal.
     */
    void loadAllLibraries();

    /**
     * Loads the library with the given ID.
     * @param id The ID of the library to load.
     * @return The Library object exported by the library plugin.
     */
    Library* loadLibrary(const QString& id);

    /**
     * Unloads the library with the given ID.
     * @param id The ID of the library to unload.
     */
    void unloadLibrary(const QString& id);

  signals:
    /**
     * Indicates that a library was loaded successfully.
     * @param lib The loaded library.
     */
    void libraryLoaded(Library* lib);

    /**
     * Indicates that all libraries were loaded.
     */
    void allLibrariesLoaded();

  private:
    LibraryManager();
    ~LibraryManager();

    class Private;
    Private* d;
  };
}

#endif
