/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

#ifndef COMMANDFACTORY_H
#define COMMANDFACTORY_H

// Qt includes
#include <qstring.h>
#include <qstringlist.h>

// forward declarations
class QObject;

namespace WorKflow
{
  class Command;
  class CommandDescription;
  class Document;
}

namespace WorKflow
{
  /**
   * @short Base class for command factories.
   *
   * This class is the abstract base class for command factories. Subclasses of
   * this class should not do anything else besides creating commands of a
   * certain type.
   *
   * You can not use this abstract class directly. Rather use the CommandFactory
   * template class, or implement a custom subclass. Together with the
   * CommandFactory class, this class implements the abstract factory pattern.
   *
   * The constructor of this class needs the ID of the command to create. All
   * needed information about the command with that ID needs to be specified in
   * the library's XML file. Additionally, an instance of a concrete subclass of
   * this class needs to be created in your library's reimplemented
   * Library::init() method. See CommandFactory for details.
   *
   * @author Thomas Kadauke
   */
  class CommandFactoryBase
  {
  public:
    /**
     * Constructor. Takes the command's ID as an arguments.
     * @param id The command's ID.
     */
    CommandFactoryBase(const QString& id);

    /**
     * Destructor.
     */
    virtual ~CommandFactoryBase();

    /**
     * Abstract creation method. You need to implement this method in subclasses.
     * If you don't have special needs when creating your command, simply use the
     * CommandFactory template class, which reimplements this method for the
     * provided template parameter.
     * @param parent The created command's parent document.
     * @return A new Command instance.
     */
    virtual Command* create(Document* parent) = 0;

  protected:
    CommandDescription* m_description;
  };

  /**
   * @short Convenience command factory template
   *
   * This template class is a convenience class that implements a concrete
   * command factory. Add methods are @c inline, adding little overhead to the
   * CommandFactoryBase base class.
   *
   * The constructor takes the commands' meta information as arguments. The
   * create() method returns a new instance of the class given by the type
   * parameter @p Cmd.
   *
   * To make your concrete command factory available to the WorKflow library,
   * you need to instantiate this class with the corrent template parameter in
   * your library's reimplemented Library::init() method.
   *
   * Example:
   *
   * @code
   * void MyLibrary::init()
   * {
   *   // your command's factory
   *   addCommandFactory(new CommandFactory<MyCommand>("org.kde.MyLib.MyCommand"));
   *
   *   // add other commands here ...
   * }
   * @endcode
   *
   * @param Cmd The actual Command subclass to create.
   */
  template<class Cmd>
  class CommandFactory : public CommandFactoryBase
  {
  public:
    /**
     * Constructor. Takes the commands' meta information as arguments.
     * @param id The command factory's unique ID.
     */
    inline CommandFactory(const QString& id)
      : CommandFactoryBase(id) {}

    /**
     * Creates a new instance of the Cmd type parameter, which must be a subclass
     * of the Command class.
     * @param parent The new command's parent document.
     * @return A new Command instance.
     */
    inline virtual Command* create(Document* parent)
    {
      return new Cmd(parent, m_description);
    }
  };
}

#endif
