/***************************************************************************
                          kbudgetdoc.cpp  -  description
                             -------------------
    begin                : Sat Aug 10 16:03:35 CST 2002
    copyright            : (C) 2002 by Richard Garand
    email                : richard@garandnet.net
    $Id: kbudgetdoc.cpp,v 1.27 2002/09/13 02:38:18 richard Exp $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// include files for Qt
#include <qdir.h>
#include <qwidget.h>

// include files for KDE
#include <klocale.h>
#include <kmessagebox.h>
#include <kio/job.h>
#include <kio/netaccess.h>
#include <ktempfile.h>
#include <ksavefile.h>

// application specific includes
#include "kbudgetdoc.h"
#include "kbudget.h"
#include "kbudgetview.h"
#include "kbundoaction.h"

// in case getAccount fails
Account blankAcct;
// in case getTransaction fails
Transaction blankTransaction;

QList<KBudgetView> *KBudgetDoc::pViewList = 0L;

KBudgetDoc::KBudgetDoc(QWidget *parent, const char *name) : QObject(parent, name), nextAccount(0),
  nextTransaction(0), lastSavedPos(0), inUndo(false), ugroup(0)
{
  if(!pViewList)
  {
    pViewList = new QList<KBudgetView>();
  }

  pViewList->setAutoDelete(true);

  undos.setAutoDelete(true);
}

KBudgetDoc::~KBudgetDoc()
{
}

void KBudgetDoc::addView(KBudgetView *view)
{
  pViewList->append(view);
}

void KBudgetDoc::removeView(KBudgetView *view)
{
  pViewList->remove(view);
}
void KBudgetDoc::setURL(const KURL &url)
{
  doc_url=url;
}

void KBudgetDoc::setModified(bool status)
{
  modified = status;
  emit saveableStatus(status);
}

const KURL& KBudgetDoc::URL() const
{
  return doc_url;
}

void KBudgetDoc::slotUpdateAllViews(KBudgetView *sender)
{
  KBudgetView *w;
  if(pViewList)
  {
    for(w=pViewList->first(); w!=0; w=pViewList->next())
    {
      if(w!=sender)
        w->repaint();
    }
  }

}

bool KBudgetDoc::saveModified()
{
  bool completed=true;

  if(modified)
  {
    KBudgetApp *win=(KBudgetApp *) parent();
    int want_save = KMessageBox::warningYesNoCancel(win,
                                         i18n("The current file has been modified.\n"
                                              "Do you want to save it?"),
                                         i18n("Warning"));
    switch(want_save)
    {
      case KMessageBox::Yes:
           if (doc_url.fileName() == i18n("Untitled"))
           {
             win->slotFileSaveAs();
           }
           else
           {
             saveDocument(URL());
       	   };

       	   deleteContents();
           completed=true;
           break;

      case KMessageBox::No:
           modified=false;
           emit saveableStatus(true);
           deleteContents();
           completed=true;
           break;

      case KMessageBox::Cancel:
           completed=false;
           break;

      default:
           completed=false;
           break;
    }
  }

  return completed;
}

void KBudgetDoc::closeDocument()
{
  deleteContents();
}

bool KBudgetDoc::newDocument()
{
  closeDocument();
  emit accountAddDel();
  doc_url.setFileName(i18n("Untitled"));
  modified=false;
  emit saveableStatus(true);

  return true;
}
#include <stdio.h>
bool KBudgetDoc::openDocument(const KURL& url, const char *format /*=0*/)
{
  bool done = false;

  QString tmpfile;
  if ( !url.isLocalFile() )
    KIO::NetAccess::download( url, tmpfile );
  else
    tmpfile = url.path();

  QFile ifile(tmpfile);
  if ( !ifile.open(IO_ReadOnly) )
  	return false;
  if ( !ifile.size() )
    done = true;

  QTextStream inf(&ifile);
  QString type, name;
  inf >> type;
  name = inf.readLine().stripWhiteSpace();
  if ( !done && (type != "KBudget" || (name != "doc 0.1" && name != "doc 0.2")) ) {
    KMessageBox mb;
    QString error = i18n("Error: %1 isn't a valid KBudget file").arg(tmpfile);
    error.arg(tmpfile);
    mb.error(0, error, i18n("Couldn't open file"));
    done = true;
  }
  if ( !done )
	closeDocument();

  int version = name == "doc 0.1" ? 1 : 2;
  int id;
  while ( !done ) {
    inf >> type;
    if ( type == "Transaction" ) {
      inf >> id;
      if ( transactions.find(id) != transactions.end() ) {
        printf("Duplicate transaction %i\n", id);
        inf.readLine();
      } else {
        transactions[id] = Transaction(this, id, inf, version);
      }
      if ( id >= nextTransaction )
        nextTransaction = id + 1;
    } else if ( type == "Account" ) {
      int type;
      inf >> id >> type;
      if ( haveAccount(id) ) {
        printf("Duplicate account %i\n", id);
        inf.readLine();
      } else {
        accounts[type][id] = Account(this, id, (Account::types)type, inf, version);
      }
      if ( id >= nextAccount )
        nextAccount = id + 1;
    }
    done = inf.atEnd();
  }
  ifile.close();

  if ( !url.isLocalFile() )
	KIO::NetAccess::removeTempFile( tmpfile );

  QValueList<int> txs = transactions.keys();
  for ( unsigned int i = 0; i < txs.size(); i++ ) {
    if ( isValidAccount(transactions[txs[i]].from()) )
      getAccount(transactions[txs[i]].from()).addTransaction(txs[i]);
    if ( isValidAccount(transactions[txs[i]].to()) )
      getAccount(transactions[txs[i]].to()).addTransaction(txs[i]);
  }
  for ( int i = 0; i < 3; i++ )
    for ( QMap<int,Account>::const_iterator ac = accounts[i].begin(); ac != accounts[i].end(); ac++ )
      (*ac).dump();

      
  emit accountAddDel();
  
  modified=false;
  doc_url = url;
  undos.clear();
  emit undoStatus(false, false);
  emit saveableStatus(false);
  return true;
}

bool KBudgetDoc::saveDocument(const KURL& url, const char *format /*=0*/)
{
  if ( url.isMalformed() ) {
    KMessageBox::sorry(0, i18n("%1 is a malformed URL.").arg(url.prettyURL()));
    return false;
  }
  
  if ( url.isLocalFile() ) {
    if ( url != doc_url && QFile::exists(url.path()) ) {
      if ( KMessageBox::warningYesNo(0, i18n("That file already exists. Would you like to overwrite the old copy?"))\
        != KMessageBox::Yes)
        return false;
    }
    return writeDocument(url.path());
  }

  KTempFile tf;
  tf.setAutoDelete(true);
  return writeDocument(tf.name());
  // TODO: upload?
}

void KBudgetDoc::deleteContents()
{
  for ( int i = 0; i < 3; i++ )
    accounts[i].clear();
  transactions.clear();
  nextAccount = 0;
  nextTransaction = 0;
}

bool KBudgetDoc::haveAccount(int id)
{
  for ( int i = 0; i < 3; i++ )
	  if ( accounts[i].find(id) != accounts[i].end() )
	    return true;
  return false;
}

Account& KBudgetDoc::getAccount(int id)
{
  for ( int i = 0; i < 3; i++ )
	if ( accounts[i].find(id) != accounts[i].end() )
	  return accounts[i][id];
  return blankAcct;
}

Account KBudgetDoc::getAccount(int id, QDate month)
{
  return getAccount(id).clone(month);
}

QValueList<Account> KBudgetDoc::getAccounts(Account::types type, QDate from, QDate to) const
{
  QValueList<Account> retList;
  QMap<int,Account>::const_iterator account = accounts[type].begin();
  for ( ; account != accounts[type].end(); account++ ) {
//    (*account).dump();
    Account temp = (*account).clone(from, to);
    temp.dump();
    retList.push_back(temp);
  }
  return retList;
}

QValueList<Account> KBudgetDoc::getAllAccounts() const
{
  QValueList<Account> ret = accounts[0].values();
  ret += accounts[1].values();
  ret += accounts[2].values();
  return ret;
}

Transaction& KBudgetDoc::getTransaction(int id)
{
  if ( transactions.find(id) != transactions.end() )
    return transactions[id];
  return blankTransaction;
}

int KBudgetDoc::newAccount(Account::types type, QString name, float initialBalance)
{
  accounts[type].insert(nextAccount, Account(this, nextAccount, type, name, initialBalance));
  nextAccount++;

  if ( !inUndo )
    addUndo(new KBAccountUndo(this, KBAccountUndo::UA_ADD, nextAccount - 1));
    
  emit accountAddDel();
  modified=true;
  emit saveableStatus(true);
  return nextAccount - 1;
}

void KBudgetDoc::deleteAccount(int id)
{
  if ( id < 0 )
    return;

  QMap<int,Account>::iterator acct;
  int i;
  for ( i = 0; i < 3; i++ ) {
    acct = accounts[i].find(id);
    if ( acct != accounts[i].end() )
      break;
  }
  if ( acct == accounts[2].end() )
    return;

  if ( !inUndo )
    addUndo(new KBAccountUndo(this, KBAccountUndo::UA_DEL, id));
  
  accounts[i].remove(acct);
  setModified(true);
  emit accountAddDel();
}

bool KBudgetDoc::isValidAccount(int id)
{
  for ( int i = 0; i < 3; i++ )
    if ( accounts[i].find(id) != accounts[i].end() )
      return true;
  return false;
}

int KBudgetDoc::newTransaction(int from, int to, float value, QDate date, QString description)
{
  transactions.insert(nextTransaction, Transaction(this, nextTransaction, from, to, value, date, description));
  getAccount(from).addTransaction(nextTransaction);
  getAccount(to).addTransaction(nextTransaction);

  if ( !inUndo )
    addUndo(new KBTransactionUndo(this, KBTransactionUndo::UT_ADD, nextTransaction));

  nextTransaction++;  
  emit accountChanged(from);
  emit accountChanged(to);
  modified=true;
  emit saveableStatus(true);
  return nextTransaction - 1;
}

void KBudgetDoc::deleteTransaction(int id)
{
  if ( id < 0 )
    return;
    
  if ( transactions.find(id) != transactions.end() ) {
    if ( !inUndo )
      addUndo(new KBTransactionUndo(this, KBTransactionUndo::UT_DEL, id));
    
    Transaction tx = transactions[id];
    getAccount(tx.from()).removeTransaction(id);
    getAccount(tx.to()).removeTransaction(id);
    transactions.remove(id);

    emit accountChanged(tx.from());
    emit accountChanged(tx.to());
    modified=true;
    emit saveableStatus(true);
  }
}

void KBudgetDoc::undo()
{
  inUndo = true;
  if ( undos.current() ) {
    undos.current()->undo();
    emit undoStatus( undos.prev() ? true : false, true);
  }
  if ( (lastSavedPos < 0 && !undos.current()) || lastSavedPos == undos.at() )
    setModified(false);
  else
    setModified(true);
  inUndo = false;
}

void KBudgetDoc::redo()
{
  inUndo = true;
  if ( undos.current() )
    undos.next()->redo();
  else if ( undos.first() ) {
    undos.first()->redo();
  } else {
    emit undoStatus(false, false);
    return;
  }

  emit undoStatus( true, undos.current() != undos.getLast() ? true : false);
  if ( (lastSavedPos < 0 && !undos.current()) || lastSavedPos == undos.at() )
    setModified(false);
  else
    setModified(true);
  inUndo = false;
}

bool KBudgetDoc::canAddUndo()
{
  return !inUndo;
}

void KBudgetDoc::beginUndoGroup()
{
  if ( !ugroup )
    ugroup = new KBUndoGroup(this);
}

void KBudgetDoc::endUndoGroup()
{
  if ( !ugroup )
    return;

  KBUndoGroup* grp = ugroup;
  ugroup = 0;
  addUndo(grp);
}

void KBudgetDoc::signalAcctChanged(int acct)
{
  modified=true;
  emit saveableStatus(true);
  emit accountChanged(acct);
}

bool KBudgetDoc::writeDocument(QString file)
{
  QFileInfo info(file);
  if ( info.isDir() ) {
    KMessageBox::sorry(0, i18n("You selected a directory!"));
    return false;
  }

  KSaveFile ofile(file, 0600);
  if ( ofile.status() ) {
    KMessageBox::sorry(0, i18n("Couldn't open file for writing"));
    ofile.abort();
    return false;
  }

  QTextStream &outf = *ofile.textStream();
  outf << "KBudget doc 0.2\n";

  QMap<int,Transaction>::iterator t = transactions.begin();
  for ( ; t != transactions.end(); t++ )
    (*t).serialize(outf);

  QMap<int,Account>::iterator acct;
  for ( int i = 0; i < 3; i++ )
    for ( acct = accounts[i].begin(); acct != accounts[i].end(); acct++ )
      (*acct).serialize(outf);

  if ( ofile.status() ) {
    KMessageBox::error(0, i18n("Error while writing file; file was not saved"));
    ofile.abort();
    return false;
  }

  ofile.close();
  if ( ofile.status() ) {
    KMessageBox::sorry(0, i18n("Error after saving file; data may not have been saved"));
    return false;
  }

  lastSavedPos = undos.at();
  modified=false;
  emit saveableStatus(false);
  return true;
}

void KBudgetDoc::insertAccount(int id, Account* copy)
{
  accounts[(int)copy->type()][id] = *copy;
  emit accountAddDel();
}

void KBudgetDoc::insertTransaction(int id, Transaction* copy)
{
  transactions[id] = *copy;
  emit accountChanged(0);
}

void KBudgetDoc::addUndo(KBUndoAction* a)
{
  if ( ugroup ) {
    ugroup->add(a);
    return;
  }

  /** if the current item is not the last, remove everything after the current item */
  int remove = undos.count() - 1 - undos.at();
  if ( remove > 0 )
    for ( int i = 0; i < remove; i++ )
      undos.removeLast();

  /** append the new item */
  undos.append(a);

  /** the current item is the last one, so we can go back but not forward */
  emit undoStatus(true, false);
}
