/***************************************************************************
                          kbudgetview.cpp  -  description
                             -------------------
    begin                : Sat Aug 10 16:03:35 CST 2002
    copyright            : (C) 2002 by Richard Garand
    email                : richard@garandnet.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// system includes
#include <time.h>
#include <string.h>
#include <stdio.h>

// include files for Qt
#include <qprinter.h>
#include <qpainter.h>
#include <klocale.h>

// application specific includes
#include "kbudgetview.h"
#include "kbudgetdoc.h"
#include "kbudget.h"

const char* months[] = { "January", "February", "March", "April", "May", "June", "July",
						 "August", "September", "October", "November", "December" };
const int COLUMNS = 4;

KBudgetView::KBudgetView(QWidget *parent, const char *name) : QVBox(parent, name), iColour(0,128,0),
  eColour(255,0,0), cColour(0,0,0), type(VT_NONE)
{
  setBackgroundMode(PaletteBase);

  periodDisplay = new QHBox(this);
  
  dtab = new QTable(30, COLUMNS, this);
  setStretchFactor(dtab, 1);
  dtab->setShowGrid(false);
  dtab->setReadOnly(true);
  dtab->horizontalHeader()->hide();
  dtab->verticalHeader()->hide();
  dtab->setTopMargin(0);
  dtab->setLeftMargin(0);
  dtab->setSelectionMode(QTable::SingleRow);

  time_t curTime = time(0);
  tm* datetime = localtime(&curTime);

  selectMonth(1900 + datetime->tm_year, datetime->tm_mon);

  connect(getDocument(), SIGNAL(accountAddDel()), this, SLOT(slotAccountAddDel()));
  connect(getDocument(), SIGNAL(accountChanged(int)), SLOT(slotAccountChanged(int)));
}

KBudgetView::~KBudgetView()
{
}

void KBudgetView::selectMonth(int yr, int mn)
{
  if ( !from.setYMD(yr, mn + 1, 1) )
    return;

  if ( type != VT_MONTH ) {
    cleanupPeriodDisplay();

    prevButton = new QPushButton(periodDisplay);
    connect(prevButton, SIGNAL(clicked()), this, SLOT(slotPrevClick()));

    QLabel* spacer1 = new QLabel(periodDisplay);
    periodDisplay->setStretchFactor(spacer1, 1);

    month = new KComboBox(periodDisplay);
    connect(month, SIGNAL(activated(int)), this, SLOT(slotMonthActivate(int)));
    for ( int i = 0; i < 12; i++ )
      month->insertItem(months[i]);

    year = new KIntNumInput(periodDisplay);
    connect(year, SIGNAL(valueChanged(int)), this, SLOT(slotYearChange(int)));
    year->setMinValue(1752);
    year->setMaxValue(8000);

    spacer2 = new QLabel(periodDisplay);
    periodDisplay->setStretchFactor(spacer2, 1);

    nextButton = new QPushButton(periodDisplay);
    connect(nextButton, SIGNAL(clicked()), this, SLOT(slotNextClick()));

    type = VT_MONTH;
  }

  QString filler;

  int pmonth = mn - 1;
  if ( pmonth < 0 ) pmonth = 11;
  prevButton->setText((QString)"< " + (QString)months[pmonth] + filler.fill(' ', 10 - strlen(months[pmonth])));

  month->setCurrentItem(mn);

  year->setValue(yr);

  int nmonth = mn + 1;
  if ( nmonth > 11 ) nmonth = 0;
  nextButton->setText(filler.fill(' ', 10 - strlen(months[nmonth])) + (QString)months[nmonth] + (QString)" >");

  KBudgetDoc* doc = getDocument();
  for ( int i = 0; i < 3; i++ ) {
    accounts[i] = doc->getAccounts((Account::types)i, QDate(yr, mn + 1, 1));
    for ( unsigned int j = 0; j < accounts[i].size(); j++ )
      accounts[i][j].dump();
  }
  createSummary();
}

void KBudgetView::rebuildSummary()
{
	createSummary();
}

KBudgetDoc *KBudgetView::getDocument() const
{
  KBudgetApp *theApp=(KBudgetApp *) parentWidget();

  return theApp->getDocument();
}

void KBudgetView::print(QPrinter *pPrinter)
{
  QPainter printpainter;
  printpainter.begin(pPrinter);

  // TODO: add your printing code here

  printpainter.end();
}

void KBudgetView::slotPrevClick()
{
	if ( type == VT_MONTH ) {
		QDate newdate = from.addMonths(-1);
		selectMonth(newdate.year(), newdate.month() - 1);
	}
}

void KBudgetView::slotMonthActivate(int month)
{
	if ( type == VT_MONTH )
		selectMonth(from.year(), month);
}

void KBudgetView::slotYearChange(int year)
{
	if ( type == VT_MONTH )
		selectMonth(year, from.month() - 1);
}

void KBudgetView::slotNextClick()
{
	if ( type == VT_MONTH ) {
		QDate newdate = from.addMonths(1);
		selectMonth(newdate.year(), newdate.month() - 1);
	}
}

void KBudgetView::slotAccountAddDel()
{
  if ( type == VT_MONTH )
    selectMonth(from.year(), from.month() - 1);
}

void KBudgetView::slotAccountChanged(int it)
{
  slotAccountAddDel();
}

void KBudgetView::cleanupPeriodDisplay()
{
	switch ( type ) {
	case VT_MONTH:
		delete prevButton;
		delete spacer1;
		delete month;
		delete year;
		delete spacer2;
		delete nextButton;
		break;
	case VT_YEAR:
		break;
	case VT_NONE:
		break;
	}
	type = VT_NONE;
}

void KBudgetView::createSummary()
{
  printf( "Creating summary\n" );
  dtab->setNumRows(accounts[0].size()+accounts[1].size()+accounts[2].size()+12);

  int row = 1;
  addSummaryHeader(row, "Income");
  row++;
  addSummaryRow(row, "Account", "Budgeted", "Received", "Remaining");
  row++;
  createSection(0, row);
  for ( int i = 0; i < COLUMNS; i++ )
    dtab->clearCell(row, i);
  row++;

  addSummaryHeader(row, "Expenses");
  row++;
  addSummaryRow(row, "Account", "Budgeted", "Spent", "Remaining");
  row++;
  createSection(1, row);
  for ( int i = 0; i < COLUMNS; i++ )
    dtab->clearCell(row, i);
  row++;

  addSummaryHeader(row, "Accounts");
  row++;
  addSummaryRow(row, "Account", "Start", "Change", "End");
  row++;  
  createSection(2, row);
  printf( "Summary created\n" );
}

void KBudgetView::addSummaryHeader(int row, QString text/*, QColor colour*/)
{
//  QLabel* label = new QLabel(text, vb);
//  label->setPaletteForegroundColor(colour);
//  QFont font = label->font();
//  font.setBold(true);
//  label->setFont(font);
//  label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
//  label->setBackgroundMode(Qt::PaletteBase);
//  dataGrid->addMultiCellWidget(label, row, row, 1, 4);
  dtab->setText(row, 0, text);
  dtab->item(row, 0)->setSpan(1,4);
}

void KBudgetView::addSummaryRow(int row, QString text1, QString text2, QString text3, QString text4/*, QColor colour*/)
{
//  QLabel* label;
  QString* text[4] = {&text1, &text2, &text3, &text4};
  for ( int i = 0; i < COLUMNS; i++ ) {
//    label = new QLabel(*text[i], 0);
//    label->setAlignment(Qt::AlignRight);
//    label->setPaletteForegroundColor(colour);
//    label->setBackgroundMode(Qt::PaletteBase);
//    dataGrid->addWidget(label, row, i + 1);
    dtab->setText(row, i, *text[i]);
//    dtab->setCellWidget(row, i, label);
    if ( i == 0 )
      dtab->item(row, 0)->setSpan(1,1);
  }
}

void KBudgetView::createSection(int number, int &row)
{
  float sumStart = 0.0f, sumChange = 0.0f, sumEnd = 0.0f;
  float start, change, end;
  for ( unsigned int i = 0; i < accounts[number].size(); i++ ) {
    QString name = accounts[number][i].name();
    if ( number != 2 ) {
      start = accounts[number][i].startingBudget();
      change = accounts[number][i].change();
      end = accounts[number][i].budget();
    } else {
      start = accounts[number][i].startingBalance();
      end = accounts[number][i].getBalance();
      change = end - start;
    }
    sumStart += start;
    sumEnd += end;
    sumChange += change;
    addSummaryRow(row, name, createValStr(start), createValStr(change), createValStr(end));
    row++;
  }
  addSummaryRow(row, "Total", createValStr(sumStart), createValStr(sumChange), createValStr(sumEnd));
  row++;
}

QString KBudgetView::createValStr(float value)
{
  KLocale* locale = KGlobal::locale();
  if ( locale )
    return locale->formatMoney(value);
  
  QString ret;
  bool negative = value < 0;
  if ( value < -0.005f )
    value *= -1;
  if ( value < 0.005f )
    value = 0;
  ret.sprintf("$%.2f", value);
  if ( negative )
    ret.prepend('-');
  return ret;
}
