/*
 * KHeiseReg
 *
 * A utility to search for articles within the Heise register.
 *
 * Copyright (C) 2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <qglobal.h>
#include <qnamespace.h>
#include <qsize.h>
#include <qlayout.h>
#include <qwhatsthis.h>
#include <qfile.h>
#include <qpainter.h>
#include <qfont.h>
#include <qfontmetrics.h>
#include <qpaintdevice.h>
#include <qpaintdevicemetrics.h>

#include <kapplication.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kstdaction.h>
#include <kmenubar.h>
#include <ktoolbar.h>
#include <ktoolbarbutton.h>
#include <kstatusbar.h>
#include <kpopupmenu.h>
#include <kurl.h>
#include <kmessagebox.h>
#include <krun.h>
#include <kprinter.h>

#include "global.h"
#include "configure.h"
#include "articleview.h"
#include "appenddlg.h"
#include "kheisereg.h"



KHeiseReg::KHeiseReg(QWidget *parent, const char *name): KMainWindow(parent, name)
{
	config = kapp->config();

	initActions();
	initMenuBar();
	initToolBar();
	initStatusBar();

	view = new KHeiseView(this, "kheiseview");
	Q_CHECK_PTR(view);
	view->setMinimumSize(view->sizeHint());
	view->plugContextMenu(articleView);
	view->plugContextMenu(articlePrint);
	setCentralWidget(view);

	readOptions();

	searchdlg = new SearchDlg(0, "searchdlg");
	Q_CHECK_PTR(searchdlg);

	connect(view, SIGNAL(items(int)), SLOT(slotStatusItems(int)));
	connect(view, SIGNAL(hasSelection(bool)), SLOT(slotHasSelection(bool)));
	connect(searchdlg, SIGNAL(search(const RegMask &)), SLOT(slotSearch(const RegMask &)));
}


KHeiseReg::~KHeiseReg()
{
}


void KHeiseReg::readOptions()
{
	config->setGroup("MainWindow Options");

	QSize geoSize(704, 506);
	resize(config->readSizeEntry("Geometry", &geoSize));
	settingsShowToolbar->setChecked(config->readBoolEntry("Show Toolbar", true));
	slotSettingsShowToolbar();
	settingsShowStatusbar->setChecked(config->readBoolEntry("Show Statusbar", true));
	slotSettingsShowStatusbar();
	toolBar()->setBarPos((KToolBar::BarPosition)config->readNumEntry("ToolbarPos", (int)KToolBar::Top));

	view->readOptions();
}


void KHeiseReg::initActions()
{
	fileOpen = new KAction(i18n("&Append Register File..."), "fileopen", CTRL+Key_A, this, SLOT(slotFileOpen()), actionCollection(), "file_open");
	Q_CHECK_PTR(fileOpen);
	fileOpen->setWhatsThis(i18n("Click this button to open the dialog for appending register files to the data base.<br><br>" \
		"You can also select the command <b>Append Register File</b> from the \"File\" menu."));
	fileQuit = KStdAction::quit(this, SLOT(slotFileQuit()), actionCollection());
	fileQuit->setWhatsThis(i18n("Click this button to quit KHeiseReg.<br><br>" \
		"You can also select the command <b>Quit</b> from the \"File\" menu."));

	articleSearch = KStdAction::find(this, SLOT(slotArticleSearch()), actionCollection());
	articleSearch->setWhatsThis(i18n("Click this button to open the search dialog.<br><br>" \
		"You can also select the command <b>Search</b> from the \"Article\" menu."));
	articleView = new KAction(i18n("&View"), "cdrom_unmount", CTRL+Key_V, this, SLOT(slotArticleView()), actionCollection(), "article_view");
	Q_CHECK_PTR(articleView);
	articleView->setWhatsThis(i18n("Click this button to view the selected article from the c't-ROM.<br><br>" \
		"You can also select the command <b>View</b> from the \"Article\" menu."));
	articleView->setEnabled(false);
	articlePrint = new KAction(i18n("&Print List..."), "fileprint", CTRL+Key_P, this, SLOT(slotArticlePrint()), actionCollection(), "article_print");
	Q_CHECK_PTR(articlePrint);
	articlePrint->setWhatsThis(i18n("Click this button to print the current search result list.<br><br>" \
		"You can also select the command <b>Print List</b> from the \"Article\" menu."));
	articlePrint->setEnabled(false);

	settingsShowToolbar = KStdAction::showToolbar(this, SLOT(slotSettingsShowToolbar()), actionCollection());
	settingsShowStatusbar = KStdAction::showStatusbar(this, SLOT(slotSettingsShowStatusbar()), actionCollection());
	settingsSaveOptions = new KAction(i18n("Save &Options"), 0, this, SLOT(slotSettingsSaveOptions()), actionCollection(), "settings_save_options");
	Q_CHECK_PTR(settingsSaveOptions);
	settingsConfigure = new KAction(i18n("&Configure KHeiseReg..."), "configure", 0, this, SLOT(slotSettingsConfigure()), actionCollection(), "settings_configure");
	Q_CHECK_PTR(settingsConfigure);
	settingsConfigure->setWhatsThis(i18n("Opens the Configure Dialog.<br><br>" \
		"You can also select the command <b>Configure KHeiseReg</b> from the \"Settings\" menu."));
}


void KHeiseReg::initMenuBar()
{
	QPopupMenu *fileMenu, *articleMenu, *settingsMenu;

	fileMenu = new QPopupMenu(0, "filemenu");
	Q_CHECK_PTR(fileMenu);
	fileOpen->plug(fileMenu);
	fileMenu->insertSeparator();
	fileQuit->plug(fileMenu);

	articleMenu = new QPopupMenu(0, "articlemenu");
	Q_CHECK_PTR(articleMenu);
	articleSearch->plug(articleMenu);
	articleView->plug(articleMenu);
	articlePrint->plug(articleMenu);

	settingsMenu = new QPopupMenu(0, "settingsmenu");
	Q_CHECK_PTR(settingsMenu);
	settingsShowToolbar->plug(settingsMenu);
	settingsShowStatusbar->plug(settingsMenu);
	settingsMenu->insertSeparator();
	settingsSaveOptions->plug(settingsMenu);
	settingsMenu->insertSeparator();
  settingsConfigure->plug(settingsMenu);

	menuBar()->insertItem(i18n("&File"), fileMenu);
	menuBar()->insertItem(i18n("&Article"), articleMenu);
	menuBar()->insertItem(i18n("&Settings"), settingsMenu);
	menuBar()->insertSeparator();
	menuBar()->insertItem(i18n("&Help"), helpMenu());
}


void KHeiseReg::initToolBar()
{
	fileOpen->plug(toolBar());
	toolBar()->insertSeparator();
	articleSearch->plug(toolBar());
	articleView->plug(toolBar());
	articlePrint->plug(toolBar());
	toolBar()->insertSeparator();
	settingsConfigure->plug(toolBar());
	toolBar()->insertSeparator();
	toolBar()->insertButton(BarIcon("help"), ID_HELP_CONTENTS, SIGNAL(clicked()), this, SLOT(appHelpActivated()), true, i18n("Help"));
	QWhatsThis::add(toolBar()->getButton(ID_HELP_CONTENTS),
		i18n("Invokes the KDE help system with the KHeiseReg help pages displayed.<br><br>" \
			"You can also select the command <b>Contents</b> from the \"Help\" menu."));
	toolBar()->alignItemRight(fileQuit->itemId(fileQuit->plug(toolBar())));
}


void KHeiseReg::initStatusBar()
{
	statusBar()->insertItem(i18n("Ready."), ID_STATUS_MSG, 1);
	statusBar()->setItemAlignment(ID_STATUS_MSG, AlignLeft | AlignVCenter);
	statusBar()->insertFixedItem(i18n(" %1 Articles ").arg(88888), ID_STATUS_ITEMS);
	statusBar()->changeItem("", ID_STATUS_ITEMS);
}


void KHeiseReg::slotFileOpen()
{
	AppendDlg appenddlg(this);
	appenddlg.exec();
}


void KHeiseReg::slotFileQuit()
{
	close();
}


void KHeiseReg::slotArticleSearch()
{
	if (searchdlg->isVisible())
		searchdlg->raise();
	else
		searchdlg->show();
}


void KHeiseReg::slotArticleView()
{
	QString fname, s;
	KURL url;
	QFile file;
	QDir checkCD;
	int page, count;

	config->setGroup(QString::fromLatin1(ID_PREF_GROUP_GENERAL));
	fname = config->readEntry(ID_PREF_GEN_CDMOUNTPT, QString::fromLatin1(PREF_GEN_CDMOUNTPT)) + "/html";

	checkCD.setPath(fname);
	if (!checkCD.isReadable()) {
		KMessageBox::sorry(this, i18n("The directory \"%1\" is not readable!\nThe mount point may be wrong or the CD is not mounted.").arg(fname));
		return;
	}

	s.sprintf("/%02d/%02d", view->articleYear(), view->articleEdition());
	fname += s;

	checkCD.setPath(fname);
	if (!checkCD.isReadable()) {
		KMessageBox::sorry(this, i18n("The directory \"%1\" is not readable!\nThe wrong c't-ROM might be inserted.").arg(fname));
		return;
	}

	page = view->articlePage();
	for (count = 0; count < 10; count++) {
		if (!page)
			break;
		s.sprintf("/%03d/art.htm", page--);
		file.setName(fname + s);
		if (file.open(IO_ReadOnly))
			break;
	}

	if (!file.isOpen()) {
		KMessageBox::sorry(this, i18n("The article could not be found on CD!\nThe wrong c't-ROM might be inserted."));
		return;
	}
	file.close();

	url.setFileName(fname + s);

	config->setGroup(ID_PREF_GROUP_CTROM);
	if (config->readBoolEntry(ID_PREF_CTROM_ALTBROWSER, PREF_CTROM_ALTBROWSER)) {
		int pos;
		QString com = config->readEntry(ID_PREF_CTROM_BROWSERCOM, PREF_CTROM_BROWSERCOM);

		while ((pos = com.find("%u")) != -1)
			com.replace(pos, 2, url.url());

		proc.clearArguments();

		if ((pos = com.find(' ')) == -1)
			proc << com;
		else {
			proc << com.left(pos);
			proc << com.mid(pos + 1);
			printf("Command: \"%s\"\n", com.left(pos).latin1());
			printf("Arguments: \"%s\"\n", com.mid(pos + 1).latin1());
		}

		if (!proc.start(KProcess::DontCare)) {
			KMessageBox::sorry(this, i18n("The external browser could not be started."));
		}
	} else
		(void)new KRun(url, 0, true);
}


void KHeiseReg::slotArticlePrint()
{
	KListView *list;
	KPrinter printer;

	list = view->articleList();
	if (!list->childCount())
		return;

	if (printer.setup(this)) {
		QPainter painter;
		ArticleViewItem *item;
		int page, ySize, yPos;
		QString m;
		QFont fb("Helvetica", 20, QFont::Bold);
		QFont fn("Times", 18, QFont::Normal);

		if (!painter.begin(&printer))
			return;

		ySize = 1000 * painter.window().height() / painter.window().width();
		painter.setWindow(0, 0, 1000, ySize);
		painter.setFont(fn);
		QFontMetrics fm = painter.fontMetrics();

		page = 0;
		yPos = 0;
		for (item = (ArticleViewItem *)list->firstChild(); (item); item = (ArticleViewItem *)item->itemBelow()) {
			if (yPos >= ySize) {
				printer.newPage();
				yPos = 0;
			}
			if (!yPos) {
				page++;
				yPos = fm.lineSpacing();
				painter.drawText(0, yPos, i18n("KHeiseReg Search Results"));
				m = i18n("Page %1").arg(page);
				painter.drawText(1000 - fm.width(m), yPos, m);
				yPos += 10;

				painter.setFont(fb);
				yPos += painter.fontMetrics().lineSpacing();
				painter.drawText(0, yPos, i18n("Title"));
				painter.drawText(820, yPos, i18n("Edition"));
				m = i18n("Page");
				painter.drawText(1000 - painter.fontMetrics().width(m), yPos, m);
				painter.fillRect(0, yPos + 3, 1000, 2, Qt::black);
				yPos += 6;
				painter.setFont(fn);
				yPos += painter.fontMetrics().lineSpacing();
			}

			if (item->subTitle().isEmpty())
				m = item->title();
			else
				m = item->title() + QString::fromLatin1(" - ") + item->subTitle();
			if (fm.width(m) > 800) {
				int w = 800 - fm.width(QString::fromLatin1("..."));
				while (fm.width(m) > w)
					m.truncate(m.length() - 1);
				m += QString::fromLatin1("...");
			}
			painter.drawText(0, yPos, m);
			if (item->magazineID() == REG_MAGAZINE_CT)
				m = QString::fromLatin1("c't");
			else if (item->magazineID() == REG_MAGAZINE_IX)
				m = QString::fromLatin1("iX");
			else
				m = QString::null;
			painter.drawText(820, yPos, m);
			m.sprintf("%d/%d", item->edition(), item->year());
			painter.drawText(920 - fm.width(m), yPos, m);
			m = QString::number(item->page());
			painter.drawText(1000 - fm.width(m), yPos, m);
			yPos += fm.lineSpacing();
		}

		painter.end();
	}
}


void KHeiseReg::slotSettingsShowToolbar()
{
	if (settingsShowToolbar->isChecked())
		toolBar()->show();
	else
		toolBar()->hide();
}


void KHeiseReg::slotSettingsShowStatusbar()
{
	if (settingsShowStatusbar->isChecked())
		statusBar()->show();
	else
		statusBar()->hide();
}


void KHeiseReg::slotSettingsSaveOptions()
{
	config->setGroup("MainWindow Options");
	
	config->writeEntry("Geometry", size());
	config->writeEntry("Show Toolbar", toolBar()->isVisible());
	config->writeEntry("Show Statusbar", statusBar()->isVisible());
	config->writeEntry("ToolbarPos", (int)toolBar()->barPos());

	view->saveOptions();
}


void KHeiseReg::slotSettingsConfigure()
{
	Configure configure(this, "configure");
	configure.exec();
}


void KHeiseReg::slotHasSelection(bool ct)
{
	articleView->setEnabled(ct);
}


void KHeiseReg::slotStatusMsg(const QString &text)
{
	statusBar()->clear();
	statusBar()->changeItem(text, ID_STATUS_MSG);
}


void KHeiseReg::slotStatusHelpMsg(const QString &text)
{
	statusBar()->message(text, 2000);
}


void KHeiseReg::slotStatusItems(int items)
{
	statusBar()->changeItem(i18n(" %1 Articles ").arg(KGlobal::locale()->formatNumber((double)items, 0)), ID_STATUS_ITEMS);
	slotStatusMsg(i18n("Ready."));
	articlePrint->setEnabled(items > 0);
}


void KHeiseReg::slotSearch(const RegMask &mask)
{
	slotStatusMsg(i18n("Searching..."));
	view->search(mask);
}
