/**
 * uploader.cpp
 *
 * Copyright (C)  2004  Zack Rusin <zack@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 */
#include "uploader.h"
#include "kconfigcreator.h"
#include "applicationtoken.h"

#include <kio/job.h>
#include <kio/scheduler.h>

#include <qcstring.h>

namespace KCfgCreator
{

Uploader::Uploader( QObject* parent )
    : QObject( parent ),
      m_creator( new KConfigCreator( this ) )
{
    connect( m_creator, SIGNAL(done(ApplicationToken*,const QCString&)),
             SLOT(slotConfigCreated(ApplicationToken*,const QCString&)) );
}

void
Uploader::upload( ApplicationToken* token )
{
    m_uploadQueue.append( token );
    m_creator->createConfig( token );
}

void
Uploader::slotConfigCreated( ApplicationToken* token, const QCString& data )
{
    KIO::Job *job = KIO::put( token->url(), -1, false, false, false );
    PutData pd;
    pd.url = token->url();
    pd.data = data;
    pd.offset = 0;
    m_putJobs.insert( job, pd );

    connect( job, SIGNAL(dataReq(KIO::Job*,QByteArray&)),
             SLOT(slotDataReq(KIO::Job*,QByteArray&)) );
    connect( job, SIGNAL(result(KIO::Job*)),
             SLOT(slotResult(KIO::Job*)) );
}

void
Uploader::slotDataReq( KIO::Job* job, QByteArray& data )
{
    // send the data in 64 KB chunks
    const int MAX_CHUNK_SIZE = 64*1024;
    QMap<KIO::Job*, PutData>::Iterator it = m_putJobs.find( job );
    Q_ASSERT( it != m_putJobs.end() );
    int remainingBytes = (*it).data.size() - (*it).offset;
    if( remainingBytes > MAX_CHUNK_SIZE ) {
        // send MAX_CHUNK_SIZE bytes to the receiver (deep copy)
        data.duplicate( (*it).data.data() + (*it).offset, MAX_CHUNK_SIZE );
        (*it).offset += MAX_CHUNK_SIZE;
        //kdDebug( 5006 ) << "Sending " << MAX_CHUNK_SIZE << " bytes ("
        //                << remainingBytes - MAX_CHUNK_SIZE << " bytes remain)\n";
    } else {
        // send the remaining bytes to the receiver (deep copy)
        data.duplicate( (*it).data.data() + (*it).offset, remainingBytes );
        (*it).data = QByteArray();
        (*it).offset = 0;
        //kdDebug( 5006 ) << "Sending " << remainingBytes << " bytes\n";
    }
}

void
Uploader::slotResult( KIO::Job* job )
{
    QMap<KIO::Job*, PutData>::Iterator it = m_putJobs.find( job );

    Q_ASSERT( it != m_putJobs.end() );

    if ( job->error() ) {
        if ( job->error() == KIO::ERR_FILE_ALREADY_EXIST )
            job->showErrorDialog();
    }
    m_putJobs.remove( it );
}

}

#include "uploader.moc"
