/**
 * kcfgparser.cpp
 *
 * Copyright (C)  2004  Zack Rusin <zack@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 */


#include "kcfgparser.h"
#include "entrytoken.h"
#include "grouptoken.h"
#include "applicationtoken.h"
#include "hosttoken.h"
#include "hostdata.h"

#include <kdebug.h>

#include <qdom.h>
#include <qcolor.h>

namespace KCfgCreator
{

KCfgParser::KCfgParser( QObject* parent )
    : ParserInterface( parent )
{
}

void
KCfgParser::parseData( HostData* hdata )
{
    QDomDocument doc( "kcfg" );
    QPtrList<FileData> files = hdata->kcfgFiles();
    QPtrListIterator<FileData> itr( files );
    HostToken* hostToken = new HostToken( hdata->url(), hdata->url().host() );

    for ( ; itr.current(); ++itr ) {
        if ( !doc.setContent( itr.current()->data ) ) {
            kdError() << "Unable to load document." << endl;
            continue;
        }
        QDomElement cfgElement = doc.documentElement();
        if ( cfgElement.isNull() ) {
            kdError() << "No document in cfg file" << endl;
            continue;
        }
        parseFile( itr.current()->url(), cfgElement, hostToken, 0 );
    }
    emit parsed( hostToken );
}

ApplicationToken*
KCfgParser::parseDataMerging( ApplicationToken* app, FileData* fdata )
{
    QDomDocument doc( "kcfg" );

    if ( !doc.setContent( fdata->data ) ) {
        kdError() << "Unable to load document." << endl;
        return app;
    }
    QDomElement cfgElement = doc.documentElement();
    if ( cfgElement.isNull() ) {
        kdError() << "No document in cfg file" << endl;
        return app;
    }
    parseFile( fdata->url(), cfgElement, 0, app );

    return app;
}

ApplicationToken*
KCfgParser::parseAndAddFile( HostToken* parent, FileData* fdata )
{
    if ( fdata->data.isEmpty() )
        return 0;

    QDomDocument doc( "kcfg" );
    if ( !doc.setContent( fdata->data ) ) {
        kdError() << "Unable to load document." << endl;
        return 0;
    }
    QDomElement cfgElement = doc.documentElement();
    if ( cfgElement.isNull() ) {
        kdError() << "No document in cfg file" << endl;
        return 0;
    }
    return parseFile( fdata->url(), cfgElement, parent,
                      parent->application( fdata->url() ) );
}

ApplicationToken*
KCfgParser::parseFile( const KURL& url, const QDomElement& el, HostToken* token,
                       ApplicationToken* app )
{
    QDomNode n;
    GroupToken *curGroup =0;
    QStringList includeList;

    for ( n = el.firstChild(); !n.isNull(); n = n.nextSibling() ) {
        QDomElement e = n.toElement();

        QString tag = e.tagName();
        if ( tag == "kcfgfile" ) {
            if ( token && !app )
                app = token->application( url );
            if ( !app ) {
                app = new ApplicationToken( url, e.attribute( "name" ) );
                if ( token )
                    token->addApp( app );
            }
            app->setIncludeList( includeList );
            QDomElement parameter = e.firstChild().toElement();
            while ( parameter.tagName() == "parameter" ) {
                app->addParameter( parameter.attribute( "name" ) );
                parameter = parameter.nextSibling().toElement();
            }
        } else if ( tag == "include" ) {
            if ( app )
                app->addIncludeFile( e.text() );
            else
                includeList.append( e.text() );
        } else if ( tag == "group" ) {
            QString group = e.attribute( "name" );
            if ( group.isEmpty() ) {
                kdError() << "Group without name" << endl;
                continue;
            }
            if ( !app ) {
                kdWarning()<<"KCFG file at "<< url.prettyURL() <<" is invalid!!"<<endl;
                kdWarning()<<"The group appears before the kcfgfile entry"<<endl;
                return 0;
            }

            curGroup = app->group( group );
            if ( !curGroup )
                curGroup = app->createGroup( group, true );

            QDomNode n2;
            for( n2 = e.firstChild(); !n2.isNull(); n2 = n2.nextSibling() ) {
                QDomElement e2 = n2.toElement();
                parseEntry( curGroup, e2 );
            }
        }
    }
    return app;
}

void
KCfgParser::parseEntry( GroupToken* group, const QDomElement &element )
{
    QString label;
    EntryToken::DefaultValue defaultValue;
    EntryToken::Parameter parameter;
    EntryToken::Value min;
    EntryToken::Value max;
    EntryToken::Choice choice;
    QValueList<EntryToken::Choice> choiceList;
    QString whatsThis;
    QString code;
    QStringList choices;
    QStringList values;
    QString type = element.attribute( "type" );
    QString name = element.attribute( "name" );
    QString key  = element.attribute( "key" );
    bool hidden  = ( element.attribute( "hidde" ) == "true" );

    if ( name.isEmpty() && key.isEmpty() )
        return;//FIXME: report an error?

    QDomNode n;
    for ( n = element.firstChild(); !n.isNull(); n = n.nextSibling() ) {
        QDomElement e = n.toElement();
        QString tag = e.tagName();
        if ( tag == "label" ) label = e.text();
        else if ( tag == "whatsthis" ) whatsThis = e.text();
        else if ( tag == "min" ) {
            min.value = e.text();
            if ( e.attribute( "code" ) == "true" )
                min.isCode = true;
            else
                min.isCode = false;
        } else if ( tag == "max" ) {
             max.value = e.text();
            if ( e.attribute( "code" ) == "true" )
                max.isCode = true;
            else
                max.isCode = false;
        } else if ( tag == "code" ) code = e.text();
        else if ( tag == "values" ) {
            QDomNode n3;
            for ( n3 = e.firstChild(); !n3.isNull(); n3 = n3.nextSibling() ) {
                QDomElement e3 = n3.toElement();
                if ( e3.tagName() == "value" ) {
                    values.append( e3.text() );
                }
            }
        }
        else if ( tag == "parameter" )
        {
            parameter.name = e.attribute( "name" );
            parameter.type = e.attribute( "type" );
            if ( parameter.name.isEmpty() ) {
                kdError() << "Parameter must have a name: " << endl;
            }
            if ( parameter.type.isEmpty() ) {
                kdError() << "Parameter must have a type: " << endl;
            }
            if ( (parameter.type == "Int") || (parameter.type == "UInt") )
            {
                bool ok;
                parameter.max = e.attribute( "max" ).toInt( &ok );
                if ( !ok ) {
                    kdError() << "Integer parameter must have a maximum (e.g. max=\"0\"): "
                              << endl;
                }
            }
            else if ( parameter.type == "Enum" ) {
                QDomNode n2;
                for ( n2 = e.firstChild(); !n2.isNull(); n2 = n2.nextSibling() ) {
                    QDomElement e2 = n2.toElement();
                    if ( e2.tagName() == "values" ) {
                        QDomNode n3;
                        for ( n3 = e2.firstChild(); !n3.isNull(); n3 = n3.nextSibling() ) {
                            QDomElement e3 = n3.toElement();
                            if ( e3.tagName() == "value" ) {
                                parameter.values.append( e3.text() );
                            }
                        }
                        break;
                    }
                }
                if ( parameter.values.isEmpty() )
                {
                    kdError() << "No values specified for parameter '" << endl;
                }
                parameter.max = parameter.values.count() - 1;
            }
            else
            {
                kdError() << "Parameter '" << parameter.name << "' has type " << parameter.type
                          << " but must be of type int, uint or Enum." << endl;
            }
        }
        else if ( tag == "default" ) {
            defaultValue.param = e.attribute( "param" );
            if (e.attribute( "code" ) == "true")
                defaultValue.isCode = true;
            defaultValue.value = e.text();
        } else if ( tag == "choices" ) {
            QDomNode n2;
            for( n2 = e.firstChild(); !n2.isNull(); n2 = n2.nextSibling() ) {
                QDomElement e2 = n2.toElement();
                if ( e2.tagName() == "choice" ) {
                    QDomNode n3;
                    EntryToken::Choice choice;
                    choice.name = e2.attribute( "name" );
                    if ( choice.name.isEmpty() ) {
                        kdError() << "Tag <choice> requires attribute 'name'." << endl;
                    }
                    for( n3 = e2.firstChild(); !n3.isNull(); n3 = n3.nextSibling() ) {
                        QDomElement e3 = n3.toElement();
                        if ( e3.tagName() == "label" )
                            choice.label = e3.text();
                        if ( e3.tagName() == "whatsthis" )
                            choice.whatsThis = e3.text();
                    }
                    choiceList.append( choice );
                }
            }
        }
    }

    if ( key.isEmpty() ) {
        key = name;
    }

    /*if ( name.isEmpty() ) {
        name = key;
        name = name.remove( " " );
    } else {

    }*/

    if ( type.isEmpty() ) type = "String";

    EntryToken* token = group->entry( key );
    if ( !token ) {
        token =  group->createEntry( key,
                                     type,
                                     name,
                                     hidden );
        Q_ASSERT( token );
    } else {
        //FIXME: might need to switch types here
        Q_ASSERT( token->typeName() == type );
        token->setHidden( hidden );
    }

    token->setLabel( label );
    token->setWhatsThis( whatsThis );
    token->setDefaultValue( defaultValue );
    token->setChoices( choiceList );
    token->setParameter( parameter );
    token->setMax( max );
    token->setMin( min );
    token->setCode( code );
    token->setValues( values );
}


}

#include "kcfgparser.moc"
