/**
 * grouptoken.cpp
 *
 * Copyright (C)  2004  Zack Rusin <zack@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 */

#include "grouptoken.h"

#include "entries.h"
#include "entrytoken.h"
#include "undomanager.h"
#include "memento.h"

#include <kdebug.h>

using namespace KCfgCreator;

GroupToken::GroupToken( ApplicationToken* parent,  const QString& name, bool readOnly )
    : Token( name, readOnly ), m_parent( parent )
{
    m_entryList.setAutoDelete( true );
}

GroupToken::~GroupToken()
{
}

ApplicationToken*
GroupToken::parent() const
{
    return m_parent;
}

GroupToken::EntryList
GroupToken::entryList() const
{
    return m_entryList;
}

EntryToken*
GroupToken::entry( const QString& key ) const
{
    QPtrListIterator<EntryToken> itr( m_entryList );

    while( itr.current() ) {
        if ( itr.current()->key() == key )
            return itr.current();
        ++itr;
    }
    return 0;
}

EntryToken*
GroupToken::entryContaining( const QString& str ) const
{
    QPtrListIterator<EntryToken> itr( m_entryList );

    while( itr.current() ) {
        if ( itr.current()->contains( str ) )
            return itr.current();
        ++itr;
    }
    return 0;
}

void
GroupToken::addEntry( const QString& name, const QString& type,
                      const QString& key, bool hidden )
{
    //not a read only group anymore
    setReadOnly( false );

    EntryToken * token = createEntry( name, type, key, hidden );
    UndoManager::Instance()->addMemento(
        new GroupMemento( this, token, GroupMemento::Added ) );

    incrementModificationCount();
}

bool
GroupToken::removeEntry( const QString& name )
{
    QPtrListIterator<EntryToken> itr( m_entryList );

    while( itr.current() ) {
        if ( itr.current()->key() == name ) {
            setReadOnly( false );

            EntryToken *token = m_entryList.take(
                m_entryList.find( itr.current() ) );
            UndoManager::Instance()->addMemento(
                new GroupMemento( this, token, GroupMemento::Removed ) );
            incrementModificationCount();
            //should _always_ succeed;
            return true;
        }
        ++itr;
    }
    return false;
}

EntryToken*
GroupToken::createEntry( const QString& name, const QString& typeName,
                         const QString& key, bool hidden )
{
    EntryToken* token = 0;
    EntryToken::Type type = KCfgCreator::Utils::stringToType( typeName );

    setReadOnly( false );

    switch( type ) {
    case EntryToken::String:
        token = new StringEntry( this, name, key, hidden );
        break;
    case EntryToken::Password:
        token = new StringEntry( this, name, key, hidden );
        break;
    case EntryToken::StringList:
        token = new StringListEntry( this, name, key, hidden );
        break;
    case EntryToken::Font:
        token = new FontEntry( this, name, key, hidden );
        break;
    case EntryToken::Rect:
        token = new RectEntry( this, name, key, hidden );
        break;
    case EntryToken::Size:
        token = new SizeEntry( this, name, key, hidden );
        break;
    case EntryToken::Color:
        token = new ColorEntry( this, name, key, hidden );
        break;
    case EntryToken::Point:
        token = new PointEntry( this, name, key, hidden );
        break;
    case EntryToken::Int:
        token = new IntEntry( this, name, key, hidden );
        break;
    case EntryToken::UInt:
        token = new UIntEntry( this, name, key, hidden );
        break;
    case EntryToken::Bool:
        token = new BoolEntry( this, name, key, hidden );
        break;
    case EntryToken::Double:
        token = new DoubleEntry( this, name, key, hidden );
        break;
    case EntryToken::DateTime:
        token = new DateTimeEntry( this, name, key, hidden );
        break;
    case EntryToken::Int64:
        token = new Int64Entry( this, name, key, hidden );
        break;
    case EntryToken::UInt64:
        token = new UInt64Entry( this, name, key, hidden );
        break;
    case EntryToken::IntList:
        token = new IntListEntry( this, name, key, hidden );
        break;
    case EntryToken::Enum:
        token = new EnumEntry( this, name, key, hidden );
        break;
    case EntryToken::Path:
        token = new PathEntry( this, name, key, hidden );
        break;
    case EntryToken::Invalid:
    default:
        kdDebug()<<"Wrong type with "<< typeName <<endl;
        break;
    }

    if ( token ) m_entryList.append( token );

    incrementModificationCount();

    return token;
}

EntryToken*
GroupToken::changeType( const QString& name, const QString& type )
{
    EntryToken* newToken = 0;

    setReadOnly( false );

    EntryToken* token = entry( name );
    m_entryList.take( m_entryList.find( token ) );
    if ( token ) {
        newToken = createEntry( name, type, token->key() );
        newToken->copyFromToken( token );
    }
    delete token;

    incrementModificationCount();

    return newToken;
}

void
GroupToken::deleteEntry( EntryToken* token )
{
    setReadOnly( false );

    m_entryList.remove( token );

    incrementModificationCount();
}
