/***************************************************************************
                          kncview.cpp  -  description
                             -------------------
    begin                : Tue Aug 29 14:51:43 CEST 2000
    copyright            : (C) 2000 by Henrik Stormer
    email                : stormer@ifi.unizh.ch
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <iostream.h>

// include files for Qt
#include <qprinter.h>
#include <qpainter.h>

// application specific includes
#include <kurl.h>
#include <kmimetype.h>
#include <klocale.h>

#include "helpfuncs.h"
#include "kncdirlister.h"
#include "kncview.h"
#include "knclistviewitem.h"
#include "knc.h"
#include "dirsizedialog.h"

KNCListView::KNCListView(QWidget *parent, const char *name)
: QListView(parent, name), _bShowHiddenFiles(true), _BeginDrag(false), _TryOpen(false),
  _NumSelectedFiles(0), _NumSelectedDirs(0), _SelectedSize(0), _DragOverItem(0)
{
  setFocusPolicy(QWidget::StrongFocus);
  setShowSortIndicator(true);
  viewport()->setAcceptDrops(true);

  _DirLister = new KNCDirLister();
  connect(_DirLister, SIGNAL(newItems(const KFileItemList&)), SLOT (slotNewItems(const KFileItemList&)));
  connect(_DirLister, SIGNAL(deleteItem(KFileItem*)), SLOT(slotDeleteItem(KFileItem*)));
  connect(_DirLister, SIGNAL(refreshItems(const KFileItemList&)), SLOT(slotRefreshItems(const KFileItemList&)));

  connect(this, SIGNAL(doubleClicked(QListViewItem*)), SLOT(slotEnter()));

  _ncName = addColumn(i18n("Name"), 190);
  _ncSize = addColumn(i18n("Size"), 90);
  _ncAttrib = addColumn(i18n("Attribute"), 100);
  _ncOwner = addColumn(i18n("Owner"), 90);
  _ncGroup = addColumn(i18n("Group"), 90);
  _nclMod = addColumn(i18n("Last Modified Time"), 220);
  _nclAccess = addColumn(i18n("Last Access Time"), 220);
  _nclRead = addColumn(i18n("Last Read Time"), 220);

  setColumnAlignment (1, AlignRight);
  setColumnAlignment (2, AlignCenter);

   setMultiSelection (true);
  setAllColumnsShowFocus(true);
  _Dir = "/";

  _DirSizeDialog = new DirSizeDialog(this);
  connect (_DirSizeDialog, SIGNAL(cancel()), SLOT(slotCancelCalculateDirSize()));
}

KNCListView::~KNCListView()
{
}

void KNCListView::slotResult(KIO::Job* Job)
{
  if (!Job->error()) //successful, emit that the directory changed
  {
    if (_TryOpen) clear();
    _NumSelectedFiles = _NumSelectedDirs = _SelectedSize = 0;
    if ((_KeyPressed == false) && (_DirName == NULL)) slotScrollBegin();
    _Dir = _DirLister->url();
    emit directoryChanged(_Dir);
    emit cursorChanged();
    emit selectChanged();
    _DirLister->slotResult(NULL);
  }
  else
  {
    if ((Job->error() == KIO::ERR_IS_FILE) || (Job->error() == KIO::ERR_UNSUPPORTED_PROTOCOL)
        || (Job->error() == KIO::ERR_UNSUPPORTED_ACTION))
    {
      emit executeFile(_Dir);

      _DirLister->slotResult(NULL);
    }
    else _DirLister->slotResult(Job);

    _DirLister->setURL(_OldDir);
    _Dir = _OldDir;
  }
  _TryOpen = false;
  _DirName = (char*)NULL;
}

void KNCListView::openURL(const KURL& cURL, const QString& DirName)
{
  KURL URL = cURL;
  _DirName = DirName;
  _KeyPressed = false;
  _TryOpen = true;
  _Dir = URL;
  _OldDir = _DirLister->url();
  slotResetSubString();

  if (URL.isLocalFile())
  {
    KMimeType::Ptr MimeType = KMimeType::findByURL(URL);
    if (MimeType->name() == QString::fromLatin1("application/x-tar")  ||
        MimeType->name() == QString::fromLatin1("application/x-tgz"))
    {
      URL.setProtocol(QString::fromLatin1("tar"));
      URL.setPath(URL.path() + '/');
    }
  }

  _DirLister->openURL(URL, _bShowHiddenFiles);

  disconnect( _DirLister->job(), SIGNAL(result(KIO::Job*)), 0, 0 );
  connect(_DirLister->job(), SIGNAL(result(KIO::Job*)), SLOT(slotResult(KIO::Job*)));
}


void KNCListView::slotEnter()
{
  if (!currentItem()) return;
  KNCListViewItem* Item = (KNCListViewItem*) currentItem();

  KURL Dir = _Dir;
  if (Dir.cd(Item->getFileItem()->name()))
   openURL(Dir);
}


void KNCListView::slotNewItems(const KFileItemList& FileList)
{
 //the opening succeeds, clear all old items
 if (_TryOpen)
 {
   clear();
   _TryOpen = false;
 }

 KFileItemListIterator it(FileList);
 for ( ; it.current(); ++it )
 {
   KFileItem* Item = it.current();
   QString Size, MTime, ATime, CTime;
   if (Item->isDir())Size = "<DIR>"; else Size = formatLong(Item->size());

   QString Attrib = createAttribString(Item->permissions());

   QDateTime T1, T2, T3;

   T1.setTime_t(Item->time(KIO::UDS_MODIFICATION_TIME));
   T2.setTime_t(Item->time(KIO::UDS_ACCESS_TIME));
   T3.setTime_t(Item->time(KIO::UDS_CREATION_TIME));

   KNCListViewItem* ViewItem = new KNCListViewItem (Item, this, Item->name(), Size, Attrib, Item->user(), Item->group(), T1.toString(), T2.toString(), T3.toString());
   ViewItem->setPixmap(0, Item->pixmap(16));
   if ((_DirName) && (_DirName == Item->name()))
   {
     setCurrentItem(ViewItem);
     makeItemVisible(ViewItem);
   }
 }
}

void KNCListView::slotRefreshItems(const KFileItemList& FileList)
{
 KFileItemListIterator it(FileList);

 for (;it.current();++it)
 {
   KFileItem* Item = it.current();
   slotDeleteItem(Item);
 }
 slotNewItems(FileList);
}

void KNCListView::slotDeleteItem(KFileItem* FileItem)
{
 KNCListViewItem* Item = findItem(FileItem);
 if (Item != NULL) delete Item;
}


void KNCListView::slotRefresh()
{
  _DirLister->updateDirectory(_Dir);
}

KURL::List KNCListView::getSelectedURLs()
{
 KURL::List SelectedList;
 KNCListViewItem* Item = (KNCListViewItem*)firstChild();
 while (Item != NULL)
 {
   if (Item->isSelected())
   {
     SelectedList.append(Item->getFileItem()->url());
   }
   Item = (KNCListViewItem*)Item->itemBelow();
 }

 if ((SelectedList.isEmpty()) && (currentItem()))
   SelectedList.append(((KNCListViewItem*)currentItem())->getFileItem()->url());

 return SelectedList;
}

QList<KFileItem> KNCListView::getSelectedItems()
{
 QList<KFileItem> SelectedList;
 KNCListViewItem* Item = (KNCListViewItem*)firstChild();
 while (Item != NULL)
 {
   if (Item->isSelected())
   {
     SelectedList.append(Item->getFileItem());
   }
   Item = (KNCListViewItem*)Item->itemBelow();
 }

 if ((SelectedList.isEmpty()) && (currentItem()))
   SelectedList.append(((KNCListViewItem*)currentItem())->getFileItem());

 return SelectedList;
}

QList<KNCListViewItem> KNCListView::getSelectedViewItems()
{
 QList<KNCListViewItem> SelectedList;
 KNCListViewItem* Item = (KNCListViewItem*)firstChild();
 while (Item != NULL)
 {
   if (Item->isSelected())
   {
     SelectedList.append(Item);
   }
   Item = (KNCListViewItem*)Item->itemBelow();
 }

 if ((SelectedList.isEmpty()) && (currentItem()))
   SelectedList.append((KNCListViewItem*)currentItem());

 return SelectedList;
}

void KNCListView::saveSettings(KConfig* config, const QString& Group)
{
  config->setGroup(Group);
  config->writeEntry("Show Hidden Files", _bShowHiddenFiles);
  config->writeEntry("Directory", _Dir.url());
  config->writeEntry("Name Length", columnWidth(0));
  config->writeEntry("Size Length", columnWidth(1));
  config->writeEntry("Attrib Length", columnWidth(2));
  config->writeEntry("Owner Length", columnWidth(3));
  config->writeEntry("Group Length", columnWidth(4));
  config->writeEntry("Last Modified Length", columnWidth(5));
  config->writeEntry("Last Access Length", columnWidth(6));
  config->writeEntry("Last Read Length", columnWidth(7));
  config->writeEntry("Sort Column", _SortColumn);
  config->writeEntry("Sort Ascending", _SortAscending);
}

void KNCListView::readSettings(KConfig* config, const QString& Group)
{
  config->setGroup(Group);
  _bShowHiddenFiles = config->readBoolEntry("Show Hidden Files", true);
  KURL URL(config->readEntry("Directory", "/"));
  openURL(URL);
  setColumnWidth(0, config->readNumEntry("Name Length", 190));
  setColumnWidth(1, config->readNumEntry("Size Length", 90));
  setColumnWidth(2, config->readNumEntry("Attrib Length", 100));
  setColumnWidth(3, config->readNumEntry("Owner Length", 90));
  setColumnWidth(4, config->readNumEntry("Group Length", 90));
  setColumnWidth(5, config->readNumEntry("Last Modified Length", 220));
  setColumnWidth(6, config->readNumEntry("Last Access Length", 220));
  setColumnWidth(7, config->readNumEntry("Last Read Length", 220));
  _SortColumn = config->readNumEntry("Sort Column", 0);
  _SortAscending = config->readBoolEntry("Sort Ascending", true);
  setSorting (_SortColumn, _SortAscending);
}
