/*
    KMLOVoice

    A utility to process voice messages received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 2000 Oliver Gantz <Oliver.Gantz@epost.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/soundcard.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>

#include <qglobal.h>
#include <qscrollbar.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qbitmap.h>
#include <qgroupbox.h>

#include <kapp.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kmenubar.h>

#include "playback.h"
#include "exportdlg.h"
#include "maildlg.h"
#include "adpcm.h"
#include "global.h"

#include "bitmaps/volume.xbm"



PlayBackWidget::PlayBackWidget(QWidget *parent, const char *name) : QWidget(parent, name)
{
	QBoxLayout *hbox, *vbox, *shbox, *svbox;
	QGroupBox *gb;
	QLabel *vlabel, *minlabel;
	int i;
	
	timer = 0;
	mixer_fd = -1;
	pcm_stereo = false;
	tracked = false;

	vbox = new QVBoxLayout(this, 8);

	vbox->addStretch();

	hbox = new QHBoxLayout(12);
	vbox->addLayout(hbox);
	
	gb = new QGroupBox(i18n("Level Meter"), this);
	
	svbox = new QVBoxLayout(gb, 6);
	svbox->addSpacing(16);
	
	shbox = new QHBoxLayout(4);
	svbox->addLayout(shbox);

	for (i=0; i < 12; i++) {
		meter_frames[i] = new QFrame(gb);
		meter_frames[i]->setFrameStyle(QFrame::Panel|QFrame::Sunken);
		meter_frames[i]->setMinimumWidth(10);
		meter_frames[i]->setFixedHeight(12);
		shbox->addWidget(meter_frames[i]);
	}

	svbox->activate();

	hbox->addWidget(gb, 3);

	gb = new QGroupBox(i18n("Volume"), this);
	
	shbox = new QVBoxLayout(gb, 6);
	
	svbox = new QVBoxLayout(4);
	shbox->addLayout(svbox);

	svbox->addSpacing(8);

	QBitmap volbm(volume_width, volume_height, volume_bits, true);
	vlabel = new QLabel(gb);
	vlabel->setPixmap(volbm);
	vlabel->setFixedSize(vlabel->sizeHint());
	svbox->addWidget(vlabel, 0, AlignBottom|AlignRight);

	volume = new QSlider(0, 100, 1, 0, QSlider::Horizontal, gb);
	volume->setMinimumWidth(60);
	volume->setFixedHeight(volume->sizeHint().height());
	svbox->addWidget(volume);

	svbox->activate();

	hbox->addWidget(gb, 1);

	vbox->addStretch();

	svbox = new QVBoxLayout(1);
	vbox->addLayout(svbox);
	
	pos = new QSlider(0, 60, 1, 0, QSlider::Horizontal, this);
	pos->setTickmarks(QSlider::Below);
	pos->setFixedHeight(pos->sizeHint().height());

	svbox->addWidget(pos);

	hbox = new QHBoxLayout(8);
	svbox->addLayout(hbox);

	minlabel = new QLabel("0:00", this);
	minlabel->adjustSize();
	hbox->addWidget(minlabel);
	hbox->addStretch();
	midlabel = new QLabel("0:00", this);
	midlabel->adjustSize();
	hbox->addWidget(midlabel);
	hbox->addStretch();
	maxlabel = new QLabel("0:00", this);
	maxlabel->adjustSize();
	hbox->addWidget(maxlabel);

	vbox->addStretch();

	vbox->activate();

	setLevel(0);

	if (initMixer()) {
		connect(volume, SIGNAL(valueChanged(int)), this, SLOT(setVolume(int)));
		connect(volume, SIGNAL(sliderPressed()), this, SLOT(volumePressed()));
		connect(volume, SIGNAL(sliderReleased()), this, SLOT(volumeReleased()));
	}
	else {
		vlabel->setEnabled(false);
		volume->setEnabled(false);
	}
	connect(pos, SIGNAL(sliderPressed()), this, SLOT(posPressed()));
	connect(pos, SIGNAL(sliderReleased()), this, SLOT(posReleased()));
}


PlayBackWidget::~PlayBackWidget()
{
	if (timer)
		delete timer;
	if (mixer_fd != -1)
		::close(mixer_fd);
}


bool PlayBackWidget::initMixer()
{
	int mask;
	KConfig *config = kapp->config();
	
	config->setGroup(ID_PREF_GROUP_PLAY);
	if ((mixer_fd = open(config->readEntry(ID_PREF_PLAY_MIXER, PREF_PLAY_MIXER), O_RDONLY, 0)) == -1)
		return false;
	fcntl(mixer_fd, F_SETFD, FD_CLOEXEC);
	
	if (ioctl(mixer_fd, SOUND_MIXER_READ_DEVMASK, &mask) == -1) {
		::close(mixer_fd);
		mixer_fd = -1;
		return false;
	}
	
	if (!(mask & (1 << SOUND_MIXER_PCM))) {
		::close(mixer_fd);
		mixer_fd = -1;
		return false;
	}
	
	if (ioctl(mixer_fd, SOUND_MIXER_READ_STEREODEVS, &mask) == -1) {
		::close(mixer_fd);
		mixer_fd = -1;
		return false;
	}
	pcm_stereo = (bool)(mask & (1 << SOUND_MIXER_PCM));
	
	timer = new QTimer(this);
	CHECK_PTR(timer);
	connect(timer, SIGNAL(timeout()), this, SLOT(monitorVolume()));
	timer->start(500);

	return true;
}


void PlayBackWidget::volumePressed()
{
	if (timer)
		timer->stop();
}


void PlayBackWidget::volumeReleased()
{
	if (timer)
		timer->start(500);
}


void PlayBackWidget::monitorVolume()
{
	int vol;

	if (ioctl(mixer_fd, SOUND_MIXER_READ_PCM, &vol) != -1) {
		if (pcm_stereo)
			volume->setValue((vol >> 9) + ((vol & 0xff) >> 1));
		else
			volume->setValue(vol);
	}
}


void PlayBackWidget::posPressed()
{
	tracked = true;
}


void PlayBackWidget::posReleased()
{
	tracked = false;
	emit position(pos->value());
}


void PlayBackWidget::setFileSize(int size, int secs)
{
	QString s;

	pos->setRange(0, size);
	pos->setTickInterval(size / 2);
	pos->setEnabled(size > 0);
	s.sprintf("%d:%02d", secs / 120, (secs / 2) % 60);
	midlabel->setText(s);
	midlabel->adjustSize();
	s.sprintf("%d:%02d", secs /  60,  secs      % 60);
	maxlabel->setText(s);
	maxlabel->adjustSize();
}


void PlayBackWidget::setFilePos(int pos_)
{
	if (!tracked)
		pos->setValue(pos_);
}


void PlayBackWidget::setLevel(int level)
{
	static int last_level = 12;
	static const QColor on_colors[12] = { green, green, green, green, green, green, green, green, green, green, red, red };
	static const QColor off_colors[12] = { darkGreen, darkGreen, darkGreen, darkGreen, darkGreen, darkGreen, darkGreen, darkGreen, darkGreen, darkGreen, darkRed, darkRed };
	int i;

	level = level * 12 / 100;

	if (level > last_level) {
		for (i=last_level; i < level; i++)
			meter_frames[i]->setBackgroundColor(on_colors[i]);
	}
	else {
		for (i=level; i < last_level; i++)
			meter_frames[i]->setBackgroundColor(off_colors[i]);
	}

	last_level = level;
}


void PlayBackWidget::setVolume(int val)
{
	if (mixer_fd != -1) {
		if (pcm_stereo)
			val |= val << 8;
		ioctl(mixer_fd, SOUND_MIXER_WRITE_PCM, &val);
	}
}

	




PlayBack::PlayBack(QWidget *parent, const char *name) : KMainWindow(parent, name, WType_TopLevel)
{
	config = kapp->config();
	
	m_type = VOI_UNKNOWN;
	m_speed = 1;
	file_size = 0;
	file_pos = 0;
	playing = false;
	dsp_fd = 0;
	data_size = 0;
	data_buff = 0;
	sn = 0;
	tmp_f = 0;

	initMenuBar();
	initToolBar();
	initStatusBar();
	
	view = new PlayBackWidget(this, "playbackwidget");
	CHECK_PTR(view);
	setCentralWidget(view);

	initKeyAccel();

	connect(view, SIGNAL(position(int)), this, SLOT(slotPosition(int)));

	readOptions();

	enableCommand(ID_PRE_FILE_STOP, false);

	setCaption(i18n("Playback"));

	resize(sizeHint());
}


PlayBack::~PlayBack()
{
	resetFile();
}


void PlayBack::playMessage(const QString& name, uchar type)
{
	int nbits = 0;
	FILE *in_f;

	resetFile();

	slotStatusName("");
	slotStatusSpeed(0);
	slotStatusBits(0);

	if ((type == VOI_ADPCM2_7) || (type == VOI_ADPCM4_7) || (type == VOI_PCM8L_7))
		m_speed = 7200;
	if ((type == VOI_ADPCM2_8) || (type == VOI_ADPCM4_8) || (type == VOI_PCM8L_8))
		m_speed = 8000;
	
	if ((type == VOI_ADPCM2_7) || (type == VOI_ADPCM2_8))
		nbits = 2;
	if ((type == VOI_ADPCM4_7) || (type == VOI_ADPCM4_8))
		nbits = 4;
	if ((type == VOI_PCM8L_7) || (type == VOI_PCM8L_8))
		nbits = 8;

	in_f = fopen(expandPath(name), "r");
	if (in_f == 0) {
		KMessageBox::sorry(this, i18n("Cannot open file\n\"%1\".").arg(name), i18n("File Error"));
		return;
	}
	
	m_name = name;
	m_type = type;

	slotStatusName(m_name);
	slotStatusSpeed(m_speed);
	slotStatusBits(nbits);
	
	if ((tmp_f = tmpfile()) == 0) {
		fclose(in_f);
		KMessageBox::sorry(this, i18n("Cannot create temporary file."), i18n("File Error"));
		return;
	}
	
	if (!adpcm_decode (in_f, tmp_f, nbits)) {
		fclose(tmp_f);
		tmp_f = 0;
		fclose(in_f);
		KMessageBox::sorry(this, i18n("Cannot convert input file."), i18n("File Error"));
		return;
	}
	fclose(in_f);
	file_size = ftell(tmp_f);
	rewind(tmp_f);

	view->setFileSize(file_size, file_size / m_speed);
	
	if (isVisible())
		raise();
	else
		show();

	slotFilePlay();
}


void PlayBack::enableCommand(int id_, bool enable)
{
	menuBar()->setItemEnabled(id_, enable);
	toolBar()->setItemEnabled(id_, enable);
}


void PlayBack::resetFile()
{
	if (playing) {
		slotFileStop();
	}
		
	if (tmp_f) {
		fclose(tmp_f);
		tmp_f = 0;
		file_size = 0;
		file_pos = 0;
		view->setFilePos(0);
		view->setFileSize(0, 0);
	}
}

		
void PlayBack::readOptions()
{
	config->setGroup("PlayBack Options");

	bool toolBarOn = config->readBoolEntry("Show ToolBar", true);
	settingsMenu->setItemChecked(ID_PRE_SETTINGS_TOOLBAR, toolBarOn);
	if (!toolBarOn)
		toolBar()->hide();

	bool statusBarOn = config->readBoolEntry("Show StatusBar", true);
	settingsMenu->setItemChecked(ID_PRE_SETTINGS_STATUSBAR, statusBarOn);
	if (!statusBarOn)
		statusBar()->hide();

	toolBar()->setBarPos((KToolBar::BarPosition)config->readNumEntry("ToolBarPos", (int)KToolBar::Top));
	
	QSize geoSize(sizeHint());
	resize(config->readSizeEntry("Geometry", &geoSize));
}


void PlayBack::saveOptions()
{
	config->setGroup("PlayBack Options");
	
	config->writeEntry("Geometry", size());
	config->writeEntry("Show ToolBar", toolBar()->isVisible());
	config->writeEntry("Show StatusBar", statusBar()->isVisible());
	config->writeEntry("ToolBarPos", (int)toolBar()->barPos());
}


void PlayBack::initMenuBar()
{
	fileMenu = new QPopupMenu(0, "playfilemenu");
	CHECK_PTR(fileMenu);
	fileMenu->insertItem(SmallIcon("1rightarrow"), i18n("&Play"), ID_PRE_FILE_PLAY);
	fileMenu->insertItem(SmallIcon("player_stop"), i18n("&Stop"), ID_PRE_FILE_STOP);
	fileMenu->insertSeparator();
	fileMenu->insertItem(SmallIcon("filesave"), i18n("&Export..."), ID_PRE_FILE_EXPORT);
	fileMenu->insertItem(SmallIcon("mail_send"), i18n("&Mail..."), ID_PRE_FILE_MAIL);
	fileMenu->insertSeparator();
	fileMenu->insertItem(SmallIcon("fileclose"), i18n("&Close"), ID_PRE_FILE_CLOSE);

	settingsMenu = new QPopupMenu(0, "playsettingsmenu");
	CHECK_PTR(settingsMenu);
	settingsMenu->setCheckable(true);
	settingsMenu->insertItem(i18n("Show &Toolbar"), ID_PRE_SETTINGS_TOOLBAR);
	settingsMenu->setItemChecked(ID_PRE_SETTINGS_TOOLBAR, true);
	settingsMenu->insertItem(i18n("Show Status&bar"), ID_PRE_SETTINGS_STATUSBAR);
	settingsMenu->setItemChecked(ID_PRE_SETTINGS_STATUSBAR, true);
	settingsMenu->insertSeparator();
	settingsMenu->insertItem(i18n("Save Options"), ID_PRE_SETTINGS_SAVEOPTIONS);

	menuBar()->insertItem(i18n("&File"), fileMenu);
	menuBar()->insertItem(i18n("&Settings"), settingsMenu);
	
	connect(fileMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
	connect(settingsMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
}


void PlayBack::initToolBar()
{
	toolBar()->insertButton(BarIcon("1rightarrow"), ID_PRE_FILE_PLAY, true, i18n("Play Message"));
	toolBar()->insertButton(BarIcon("player_stop"), ID_PRE_FILE_STOP, true, i18n("Stop Message"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(BarIcon("filesave"), ID_PRE_FILE_EXPORT, true, i18n("Export Message"));
	toolBar()->insertButton(BarIcon("mail_send"), ID_PRE_FILE_MAIL, true, i18n("Mail Message"));
	toolBar()->insertButton(BarIcon("fileclose"), ID_PRE_FILE_CLOSE, true, i18n("Close"));
	toolBar()->alignItemRight(ID_PRE_FILE_CLOSE, true);
	
	connect(toolBar(), SIGNAL(clicked(int)), SLOT(commandCallback(int)));
}


void PlayBack::initStatusBar()
{
	statusBar()->insertItem("", ID_PRE_STATUS_NAME, 1);
	statusBar()->insertFixedItem(" 8000 Bps", ID_PRE_STATUS_SPEED);
	statusBar()->insertFixedItem(" 8 Bits ", ID_PRE_STATUS_BITS);
	statusBar()->changeItem("", ID_PRE_STATUS_SPEED);
	statusBar()->changeItem("", ID_PRE_STATUS_BITS);
}


void PlayBack::initKeyAccel()
{
	keyAccel = new KAccel(this);
	CHECK_PTR(keyAccel);

	keyAccel->insertItem(i18n("Play File"), "PlayFile", Key_Space);
	keyAccel->insertItem(i18n("Stop File"), "StopFile", Key_BackSpace);
	keyAccel->insertItem(i18n("Export File"), "ExportFile", CTRL+Key_E);
	keyAccel->insertItem(i18n("Mail File"), "MailFile", CTRL+Key_M);
	
	keyAccel->insertItem(i18n("Show Toolbar"), "ShowToolbar", CTRL+Key_T);
	keyAccel->insertItem(i18n("Show Statusbar"), "ShowStatusbar", CTRL+Key_S);
	
	keyAccel->readSettings();

	keyAccel->connectItem("PlayFile", this, SLOT(slotFilePlay()));
	keyAccel->connectItem("StopFile", this, SLOT(slotFileStop()));
	keyAccel->connectItem("ExportFile", this, SLOT(slotFileExport()));
	keyAccel->connectItem("MailFile", this, SLOT(slotFileMail()));
	keyAccel->connectItem(KStdAccel::Close, this, SLOT(hide()));
	
	keyAccel->connectItem("ShowToolbar", this, SLOT(slotSettingsShowToolbar()));
	keyAccel->connectItem("ShowStatusbar", this, SLOT(slotSettingsShowStatusbar()));
	
	keyAccel->connectItem(KStdAccel::Help, this, SLOT(appHelpActivated()));

	keyAccel->changeMenuAccel(fileMenu, ID_PRE_FILE_PLAY, "PlayFile");
	keyAccel->changeMenuAccel(fileMenu, ID_PRE_FILE_STOP, "StopFile");
	keyAccel->changeMenuAccel(fileMenu, ID_PRE_FILE_EXPORT, "ExportFile");
	keyAccel->changeMenuAccel(fileMenu, ID_PRE_FILE_MAIL, "MailFile");
	keyAccel->changeMenuAccel(fileMenu, ID_PRE_FILE_CLOSE, KStdAccel::Close);
	
	keyAccel->changeMenuAccel(settingsMenu, ID_PRE_SETTINGS_TOOLBAR, "ShowToolbar");
	keyAccel->changeMenuAccel(settingsMenu, ID_PRE_SETTINGS_STATUSBAR, "ShowStatusbar");
}


void PlayBack::commandCallback(int id_)
{
	switch (id_) {
		case ID_PRE_FILE_PLAY:
			slotFilePlay();
			break;
		case ID_PRE_FILE_STOP:
			slotFileStop();
			break;
		case ID_PRE_FILE_EXPORT:
			slotFileExport();
			break;
		case ID_PRE_FILE_MAIL:
			slotFileMail();
			break;
		case ID_PRE_FILE_CLOSE:
			hide();
			break;
		case ID_PRE_SETTINGS_TOOLBAR:
			slotSettingsShowToolbar();
			break;
		case ID_PRE_SETTINGS_STATUSBAR:
			slotSettingsShowStatusbar();
			break;
		case ID_PRE_SETTINGS_SAVEOPTIONS:
			saveOptions();
			break;
		default:
			break;
	}
}


void PlayBack::slotFilePlay()
{
	int speed, frag, format = AFMT_U8, channels = 1;

	if (playing)
		return;

	config->setGroup(ID_PREF_GROUP_PLAY);
	if ((dsp_fd = open(config->readEntry(ID_PREF_PLAY_DEVICE, PREF_PLAY_DEVICE), O_WRONLY, 0)) == -1) {
		KMessageBox::sorry(this, i18n("Cannot open DSP device."), i18n("File Error"));
		return;
	}
	fcntl(dsp_fd, F_SETFD, FD_CLOEXEC);
	
	data_size = m_speed / 50;		/* Latency ~200ms */
	for (frag = 0; data_size != 1; frag++)
		data_size >>= 1;

	frag |= 0x00020000;
	if (ioctl(dsp_fd, SNDCTL_DSP_SETFRAGMENT, &frag) == -1) {
	
	}
			
	if ((ioctl(dsp_fd, SNDCTL_DSP_SETFMT, &format) == -1) || (format != AFMT_U8)) {
		::close(dsp_fd);
		KMessageBox::sorry(this, i18n("Cannot set DSP to 8 bits unsigned format."), i18n("DSP Error"));
		return;
	}
	
	if ((ioctl(dsp_fd, SNDCTL_DSP_CHANNELS, &channels) == -1) || (channels != 1)) {
		::close(dsp_fd);
		KMessageBox::sorry(this, i18n("Cannot set DSP to 1 channel format."), i18n("DSP Error"));
		return;
	}
	
	speed = m_speed;
	if ((ioctl(dsp_fd, SNDCTL_DSP_SPEED, &speed) == -1) || (speed != m_speed)) {
		::close(dsp_fd);
		KMessageBox::sorry(this, i18n("Cannot set DSP to %1 Bps format.").arg(m_speed), i18n("DSP Error"));
		return;
	}
	
	if (ioctl(dsp_fd, SNDCTL_DSP_GETBLKSIZE, &data_size) == -1) {
	
	}
	
	data_buff = (uchar *)malloc(data_size);

	sn = new QSocketNotifier(dsp_fd, QSocketNotifier::Write);
	connect(sn, SIGNAL(activated(int)), this, SLOT(slotPlayData(int)));

	playing = true;

	enableCommand(ID_PRE_FILE_PLAY, false);
	enableCommand(ID_PRE_FILE_STOP, true);
}


void PlayBack::slotFileStop()
{
	if (!playing)
		return;

	if (sn) {
		delete sn;
		sn = 0;
	}
	if (data_buff) {
		free(data_buff);
		data_buff = 0;
	}
	if (dsp_fd) {
		::close(dsp_fd);
		dsp_fd = 0;
	}
	if (tmp_f)
		rewind(tmp_f);
	file_pos = 0;
	
	view->setFilePos(file_pos);

	playing = false;
		
	view->setLevel(0);

	enableCommand(ID_PRE_FILE_PLAY, true);
	enableCommand(ID_PRE_FILE_STOP, false);
}


void PlayBack::slotPosition(int pos)
{
	if (tmp_f) {
		if (!fseek(tmp_f, (long)pos, SEEK_SET))
			file_pos = pos;
	}
}
		

void PlayBack::slotFileExport()
{
	exportdlg->exportMessage(m_name, m_type);
}


void PlayBack::slotFileMail()
{
	maildlg->mailMessage(m_name, m_type);
}


void PlayBack::slotSettingsShowToolbar()
{
	bool toolBarOn = !settingsMenu->isItemChecked(ID_PRE_SETTINGS_TOOLBAR);

	menuBar()->setItemChecked(ID_PRE_SETTINGS_TOOLBAR, toolBarOn);
	if (toolBarOn)
		toolBar()->show();
	else
		toolBar()->hide();
}


void PlayBack::slotSettingsShowStatusbar()
{
	bool statusBarOn = !settingsMenu->isItemChecked(ID_PRE_SETTINGS_STATUSBAR);

	menuBar()->setItemChecked(ID_PRE_SETTINGS_STATUSBAR, statusBarOn);
	if (statusBarOn)
		statusBar()->show();
	else
		statusBar()->hide();
}


void PlayBack::slotStatusName(const QString& text)
{
	statusBar()->changeItem(text, ID_PRE_STATUS_NAME);
}


void PlayBack::slotStatusSpeed(int speed)
{
	if (speed) {
		QString s = i18n("%1 Bps").arg(speed);
		statusBar()->changeItem(s, ID_PRE_STATUS_SPEED);
	}
	else
		statusBar()->changeItem("", ID_PRE_STATUS_SPEED);
}


void PlayBack::slotStatusBits(int bits)
{
	if (bits) {
		QString s = i18n("%1 Bits").arg(bits);
		statusBar()->changeItem(s, ID_PRE_STATUS_BITS);
	}
	else
		statusBar()->changeItem("", ID_PRE_STATUS_BITS);
}


void PlayBack::slotPlayData(int)
{
	int result, i, max;
	uchar *p;

	result = fread((void *)data_buff, 1, data_size, tmp_f);
	if (result == 0) {
		slotFileStop();
		return;
	}

	max = 0;
	p = data_buff;
	for (i=0; i < result; i++) {
		if (max < abs(*p - 0x80))
			max = abs(*p - 0x80);
		p++;
	}
	view->setLevel(max * 100 / 128);
	write(dsp_fd, data_buff, result);
	file_pos += result;
	view->setFilePos(file_pos);
}


void PlayBack::hide()
{
	resetFile();
	KMainWindow::hide();
}
