/*
    KMLOVoice

    A utility to process voice messages received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 2000 Oliver Gantz <Oliver.Gantz@epost.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include "global.h"
#include "adpcm.h"
#include "filters.h"



MLOFilter::MLOFilter()
{
	init();
}


MLOFilter::MLOFilter(const QString& infile, uchar type)
{
	init();
	
	setFile(infile, type);
}


MLOFilter::~MLOFilter()
{
}


void MLOFilter::setFile(const QString& infile, uchar type)
{
	m_infile = expandPath(infile);
	m_type = type;
}


bool MLOFilter::convertStream(FILE *)
{
	return true;
}


bool MLOFilter::convertFile(const QString& outfile)
{
	FILE *stream;
	bool result;

	if (!(stream = fopen(outfile.latin1(), "w"))) {
		return false;
	}

	result = convertStream(stream);
	
	fclose(stream);

	return result;
}


void MLOFilter::init()
{
	m_infile = QString::null;
	m_type = VOI_UNKNOWN;
}



MLO2WavFilter::MLO2WavFilter() : MLOFilter()
{
}


MLO2WavFilter::MLO2WavFilter(const QString& infile, uchar type) : MLOFilter(infile, type)
{
}


MLO2WavFilter::~MLO2WavFilter()
{
}


bool MLO2WavFilter::convertStream(FILE *stream)
{
	FILE *in_f = 0, *tmp_f;
	int nbits = 0, fsize;
	char buff[1024];
	static struct {
		char riff_id[4];
		Q_UINT32 riff_len;
		char wave_id[4];
		char fmt_id[4];
		Q_UINT32 fmt_len;
		Q_INT16 fmt_format;
		Q_UINT16 fmt_channels;
		Q_UINT32 fmt_samples_per_sec;
		Q_UINT32 fmt_bytes_per_sec;
		Q_UINT16 fmt_block_align;
		Q_UINT16 fmt_bits_per_sample;
		char data_id[4];
		Q_UINT32 data_len;
	} wave_header = {
		{ 'R', 'I', 'F', 'F' },	0, { 'W', 'A', 'V', 'E' }, { 'f', 'm', 't', ' ' }, 16, 1,	1, 0,	0, 1,	8,
		{ 'd', 'a', 't', 'a' },	0
	};

	if ((m_type == VOI_ADPCM2_7) || (m_type == VOI_ADPCM2_8))
		nbits = 2;
	if ((m_type == VOI_ADPCM4_7) || (m_type == VOI_ADPCM4_8))
		nbits = 4;
	if ((m_type == VOI_PCM8L_7) || (m_type == VOI_PCM8L_8))
		nbits = 8;
	
	in_f = fopen(m_infile.latin1(), "r");
	tmp_f = tmpfile();
	if (!adpcm_decode(in_f, tmp_f, nbits)) {
	
	}
	fclose(in_f);

	fsize = ftell(tmp_f);
	rewind(tmp_f);
	
	wave_header.riff_len = (Q_UINT32)(fsize + sizeof(wave_header) - 8);

	if ((m_type == VOI_ADPCM2_7) || (m_type == VOI_ADPCM4_7) || (m_type == VOI_PCM8L_7))
		wave_header.fmt_samples_per_sec = 7200;
	if ((m_type == VOI_ADPCM2_8) || (m_type == VOI_ADPCM4_8) || (m_type == VOI_PCM8L_8))
		wave_header.fmt_samples_per_sec = 8000;

	wave_header.fmt_bytes_per_sec = wave_header.fmt_samples_per_sec;
	wave_header.data_len = (Q_UINT32)fsize;

	fwrite((void *)&wave_header, sizeof(wave_header), 1, stream);
	while ((fsize = fread((void *)buff, 1, sizeof(buff), tmp_f)) > 0)
		fwrite((void *)buff, 1, fsize, stream);

	fclose(tmp_f);

	return true;
}
