/*
    KMLOFax
    
    A utility to process facsimiles received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 1999-2001 Oliver Gantz <Oliver.Gantz@epost.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
    PostScript(R) is a registered trademark of Adobe Systems Incorporated.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <qlayout.h>
#include <qradiobutton.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qtooltip.h>
#include <qsize.h>
#include <qdir.h>

#include <kapp.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <kseparator.h>

#include "printdlg.h"
#include "global.h"
#include "mlofile.h"
#include "filters.h"




PrintDlg::PrintDlg(QWidget *parent, const char *name) : KDialog (parent, name, true)
{
	QBoxLayout *vbox, *hbox, *fvbox, *fhbox;
	QRadioButton *rb;
	QGroupBox *gb;
	KSeparator *separator;
	QPushButton *help, *ok, *cancel;
	QLabel *l1, *l2, *l3;
	QSize size;

	config = kapp->config();

	to_printer = true;
	paper_num = 0;

	vbox = new QVBoxLayout(this , marginHint());

	destination = new QButtonGroup(this);
	destination->setTitle(i18n("Print Destination"));
	fvbox = new QVBoxLayout(destination, 12, 0);
	fvbox->addSpacing(8);
	rb = new QRadioButton(i18n("Print to &printer:"), destination);
	rb->setMinimumSize(rb->sizeHint());
	rb->setChecked(true);
	fvbox->addWidget(rb);
	fhbox = new QHBoxLayout();
	fvbox->addLayout(fhbox);
	fhbox->addSpacing(19);
	printer = new QLineEdit(destination);
	printer->setMinimumWidth(printer->sizeHint().width());
	printer->setFixedHeight(printer->sizeHint().height());
	fhbox->addWidget(printer, 3);

	fvbox->addSpacing(6);

	rb = new QRadioButton(i18n("Print to &file:"), destination);
	rb->setMinimumSize(rb->sizeHint());
	fvbox->addWidget(rb);
	fhbox = new QHBoxLayout();
	fvbox->addLayout(fhbox);
	fhbox->addSpacing(19);
	file = new QLineEdit(destination);
	file->setMinimumWidth(file->sizeHint().width());
	file->setFixedHeight(file->sizeHint().height());
	file->setText("facsimile.ps");
	file->setEnabled(false);
	fhbox->addWidget(file);

	fhbox->addSpacing(6);
	
	browse = new QPushButton(destination);
	browse->setAutoDefault(false);
	browse->setPixmap(SmallIcon("fileopen"));
	browse->setFixedSize(browse->sizeHint()+QSize(4, 4));	
	browse->setEnabled(false);
	fhbox->addWidget(browse);
	
	fvbox->activate();
	vbox->addWidget(destination);

	hbox = new QHBoxLayout(8);
	vbox->addLayout(hbox);

	paper_format = new QButtonGroup(i18n("Paper Format"), this);
	fvbox = new QVBoxLayout(paper_format, 10);
	fvbox->addSpacing(8);
	rb = new QRadioButton(QString("%1 (210297 mm)").arg(i18n("A4")), paper_format);
	rb->setMinimumSize(rb->sizeHint());
	rb->setChecked(true);
	fvbox->addWidget(rb);
	rb = new QRadioButton(QString("%1 (257364 mm)").arg(i18n("B4")), paper_format);
	rb->setMinimumSize(rb->sizeHint());
	fvbox->addWidget(rb);
	rb = new QRadioButton(QString("%1 (182257 mm)").arg(i18n("B5")), paper_format);
	rb->setMinimumSize(rb->sizeHint());
	fvbox->addWidget(rb);
	rb = new QRadioButton(QString("%1 (811 in.)").arg(i18n("Letter")), paper_format);
	rb->setMinimumSize(rb->sizeHint());
	fvbox->addWidget(rb);
	rb = new QRadioButton(QString("%1 (814 in.)").arg(i18n("Legal")), paper_format);
	rb->setMinimumSize(rb->sizeHint());
	fvbox->addWidget(rb);
	rb = new QRadioButton(QString("%1 (710 in.)").arg(i18n("Executive")), paper_format);
	rb->setMinimumSize(rb->sizeHint());
	fvbox->addWidget(rb);
	fvbox->activate();

	hbox->addWidget(paper_format);

	gb = new QGroupBox(i18n("Options"), this);
	fvbox = new QVBoxLayout(gb, 12);
	fvbox->addSpacing(8);
	fhbox = new QHBoxLayout();
	fvbox->addLayout(fhbox);
	l1 = new QLabel(i18n("F&rom page:"), gb);
	fhbox->addWidget(l1);
	from_page = new QSpinBox(1, 999, 1, gb);
	l1->setBuddy(from_page);
	from_page->setValue(1);
	from_page->setMinimumWidth(from_page->sizeHint().width());
	from_page->setFixedHeight(from_page->sizeHint().height());
	fhbox->addWidget(from_page, 1);
	fhbox = new QHBoxLayout();
	fvbox->addLayout(fhbox);
	l2 = new QLabel(i18n("&To page:"), gb);
	fhbox->addWidget(l2);
	to_page = new QSpinBox(1, 999, 1, gb);
	l2->setBuddy(to_page);
	to_page->setValue(999);
	to_page->setMinimumWidth(to_page->sizeHint().width());
	to_page->setFixedHeight(to_page->sizeHint().height());
	fhbox->addWidget(to_page, 1);

	separator = new KSeparator(KSeparator::HLine, gb);
	fvbox->addWidget(separator, 1);
	
	fhbox = new QHBoxLayout();
	fvbox->addLayout(fhbox);
	l3 = new QLabel(i18n("Number of &copies:"), gb);
	fhbox->addWidget(l3);
	copies = new QSpinBox(1, 99, 1, gb);
	l3->setBuddy(copies);
	copies->setValue(1);
	copies->setMinimumWidth(copies->sizeHint().width());
	copies->setFixedHeight(copies->sizeHint().height());
	fhbox->addWidget(copies, 1);

	separator = new KSeparator(KSeparator::HLine, gb);
	fvbox->addWidget(separator, 1);
	
	level2 = new QCheckBox(i18n("&Language Level 2"), gb);
	level2->setMinimumSize(level2->sizeHint());
	fvbox->addWidget(level2);

	interpolate = new QCheckBox(i18n("&Interpolate"), gb);
	interpolate->setMinimumSize(interpolate->sizeHint());
	fvbox->addWidget(interpolate);

	size = l1->sizeHint().expandedTo(l2->sizeHint()).expandedTo(l3->sizeHint());
	l1->setMinimumSize(size);
	l2->setMinimumSize(size);
	l3->setMinimumSize(size);

	fvbox->activate();

	hbox->addWidget(gb);

	separator = new KSeparator(KSeparator::HLine, this);
	vbox->addWidget(separator);
	
	hbox = new QHBoxLayout();
	vbox->addLayout(hbox);

	help = new QPushButton(i18n("&Help"), this);
	ok = new QPushButton(i18n("&OK"), this);
	ok->setDefault(true);
	cancel = new QPushButton(i18n("&Cancel"), this);
	size = help->sizeHint().expandedTo(ok->sizeHint().expandedTo(cancel->sizeHint()));

	help->setFixedSize(size);
	ok->setFixedSize(size);
	cancel->setFixedSize(size);

	hbox->addWidget(help);
	hbox->addStretch();
	hbox->addWidget(ok);
	hbox->addWidget(cancel);

	vbox->activate();

	config->setGroup(ID_PREF_GROUP_PRINT);
	printer->setText(config->readEntry(ID_PREF_PRT_COMMAND, PREF_PRT_COMMAND));
	paper_format->setButton(config->readNumEntry(ID_PREF_PRT_PAPER, PREF_PRT_PAPER));
	level2->setChecked(config->readBoolEntry(ID_PREF_PRT_LEVEL2, PREF_PRT_LEVEL2));
	interpolate->setChecked(config->readBoolEntry(ID_PREF_PRT_INTERPOL, PREF_PRT_INTERPOL));
	interpolate->setEnabled(level2->isChecked());
	
	config->setGroup(ID_PREF_GROUP_GENERAL);
	if (config->readBoolEntry(ID_PREF_GEN_TOOLTIPS, PREF_GEN_TOOLTIPS)) {
		QToolTip::add(printer, i18n("Command to invoke for printing"));
		QToolTip::add(file, i18n("Name of file to print to"));
		QToolTip::add(browse, i18n("Select file to print to"));
		QToolTip::add(paper_format, i18n("Paper format used by the printer"));
		QToolTip::add(from_page, i18n("First page to print"));
		QToolTip::add(to_page, i18n("Last page to print"));
		QToolTip::add(copies, i18n("Number of copies to print"));
		QToolTip::add(level2, i18n("Use PostScript Language Level 2"));
		QToolTip::add(interpolate, i18n("Interpolate for smoother output"));
	}
	
	connect(destination, SIGNAL(clicked(int)), SLOT(destinationSelected(int)));
	connect(browse, SIGNAL(clicked()), SLOT(slotBrowse()));
	connect(paper_format, SIGNAL(clicked(int)), SLOT(formatSelected(int)));
	connect(level2, SIGNAL(toggled(bool)), SLOT(level2Selected(bool)));
	connect(help, SIGNAL(clicked()), SLOT(slotHelp()));
	connect(ok, SIGNAL(clicked()), SLOT(slotOk()));
	connect(cancel, SIGNAL(clicked()), SLOT(reject()));

	setCaption(i18n("Print"));

	resize(sizeHint());
}


PrintDlg::~PrintDlg()
{
}


void PrintDlg::printFax(const QString& name)
{
	MLOFile mlofile;

	mlofile.setName(name);

	if (!mlofile.open()) {
		KMessageBox::sorry(this, i18n("Cannot open facsimile file."), i18n("File Error"));
		return;
	}

	from_page->setRange(1, mlofile.pages());
	from_page->setValue(1);
	to_page->setRange(1, mlofile.pages());
	to_page->setValue(mlofile.pages());
	mlofile.close();

	m_name = name;
}


void PrintDlg::slotBrowse()
{
	QString s = file->text();
	
	if (s.isEmpty())
		s = QDir::currentDirPath();

	s = KFileDialog::getSaveFileName(s, i18n("*.ps|PostScript Files (*.ps)\n*|All Files (*)"), this);
	if (!s.isEmpty())
		file->setText(s);
}


void PrintDlg::slotHelp()
{
	kapp->invokeHelp("PRINT-HELP");
}


void PrintDlg::slotOk()
{
	MLO2PSFilter *filter;
	FILE *ps;

	filter = new MLO2PSFilter(m_name);
	filter->setFormat(paper_num);
	filter->setLevel2(level2->isChecked());
	filter->setInterpolate(interpolate->isChecked());
	filter->setRange(from_page->value(), to_page->value());
	filter->setCopies(copies->value());
	
	config->setGroup(ID_PREF_GROUP_PRINT);
	filter->setMargins(
		config->readNumEntry(ID_PREF_PRT_LMARGIN, PREF_PRT_LMARGIN),
		config->readNumEntry(ID_PREF_PRT_RMARGIN, PREF_PRT_RMARGIN),
		config->readNumEntry(ID_PREF_PRT_TMARGIN, PREF_PRT_TMARGIN),
		config->readNumEntry(ID_PREF_PRT_BMARGIN, PREF_PRT_BMARGIN)
	);


	if (to_printer) {
		got_sigpipe = false;
		if ((ps = popen(printer->text(), "w")) <= 0)
			KMessageBox::sorry(this, i18n("Error executing print command."), i18n("Printer Error"));
		else {
			kapp->setOverrideCursor(waitCursor);
			filter->convertFile(ps);
			pclose(ps);
			kapp->restoreOverrideCursor();
			if (got_sigpipe)
				KMessageBox::sorry(this, i18n("Error executing print command."), i18n("Printer Error"));
		}
	}
	else {
		if ((ps = fopen(file->text(), "w")) == 0)
			KMessageBox::sorry(this, i18n("Cannot create file for output."), i18n("Printer Error"));
		else {
			kapp->setOverrideCursor(waitCursor);
			filter->convertFile(ps);
			fclose(ps);
			kapp->restoreOverrideCursor();
		}
	}		

	delete filter;

	accept();
}


void PrintDlg::destinationSelected(int id_)
{
	to_printer = (id_ == 0);
	printer->setEnabled(to_printer);
	file->setEnabled(!to_printer);
	browse->setEnabled(!to_printer);
}


void PrintDlg::formatSelected(int id_)
{
	paper_num = id_;
}


void PrintDlg::level2Selected(bool on)
{
	interpolate->setEnabled(on);
}
