/*
    KMLOFax

    A utility to process facsimiles received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 1999-2001 Oliver Gantz <Oliver.Gantz@epost.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
    PostScript(R) is a registered trademark of Adobe Systems Incorporated.
*/

#include <stdlib.h>
#include <string.h>

#include <qdatetime.h>

#include <kapp.h>
#include <klocale.h>

#include "global.h"
#include "senderaliases.h"
#include "filters.h"



MLOFilter::MLOFilter()
{
	init();
}


MLOFilter::MLOFilter(const QString& infile)
{
	init();
	
	setFile(infile);
}


MLOFilter::~MLOFilter()
{
}


void MLOFilter::setFile(const QString& infile)
{
	mlofile.setName(infile);
}


bool MLOFilter::convertFile(FILE *stream)
{
	f = stream;

	return true;
}

void MLOFilter::init()
{
	f = 0;
}



MLO2PSFilter::MLO2PSFilter() : MLOFilter()
{
	init();
}


MLO2PSFilter::MLO2PSFilter(const QString& infile) : MLOFilter(infile)
{
	init();
}


MLO2PSFilter::~MLO2PSFilter()
{
}


void MLO2PSFilter::setFormat(int format)
{
	/* A4, B4, B5, Letter, Legal, Executive */
	static int papers_x[] = { 595,  729, 516, 612,  612, 540 };
	static int papers_y[] = { 842, 1032, 729, 792, 1008, 720 };

	paper_x = papers_x[format];
	paper_y = papers_y[format];
}


void MLO2PSFilter::setLevel2(bool l2)
{
	level2 = l2;
}


void MLO2PSFilter::setInterpolate(bool interp)
{
	interpolate = interp;
}


void MLO2PSFilter::setRange(int first, int last)
{
	first_page = first;
	last_page = last;
}


void MLO2PSFilter::setCopies(int copies)
{
	doc_copies = copies;
}


void MLO2PSFilter::setMargins(int left, int right, int top, int bottom)
{
	l_margin = left;
	r_margin = right;
	t_margin = top;
	b_margin = bottom;
}


bool MLO2PSFilter::convertFile(FILE *stream)
{
	page_info_t info;
	int page, ps_pages, sub_page;
	int *size_y_buff;
	int maxlines;
	QString s;

	f = stream;

	/* Size of viewport (in pixel, 1/72 dpi) */
	view_x = paper_x - (int)((double)(l_margin + r_margin) * 2.8346);
	view_y = paper_y - (int)((double)(t_margin + b_margin) * 2.8346);
	
	/* Bottom-left border (in pixel, 1/72 dpi) */
	trans_x = (int)((double)l_margin * 2.8346);
	trans_y = (int)((double)b_margin * 2.8346);
	
	/* Calculate real number of pages */
	if (!mlofile.open()) {
		return false;
	}

	ps_pages = mlofile.pages();
	
	if (first_page == 0) {
		first_page = 1;
		last_page = ps_pages;
	}

	if ((first_page > last_page) || (first_page > ps_pages)) {
		mlofile.close();
		return false;
	}

	if (last_page > ps_pages)
		last_page = ps_pages;

	ps_pages = 0;
	size_y_buff = (int *)malloc((last_page - first_page + 1) * sizeof(int));
	for (page = first_page; page <= last_page; page++) {
		mlofile.readPageInfo(page, &info);

		/* Horizontal size of image (in pixel, 204 dpi) */
		size_x = info.width;
		/* Vertical size of image (in pixel, 98 or 196 lpi) */
		size_y = mlofile.pageUsedHeight(page);
		size_y_buff[page-first_page] = size_y;

		scale_x = size_x * 72 / 204;
		maxlines = (view_y * info.lpi) / 72;

		if (scale_x > view_x)
			maxlines = (maxlines * scale_x) / view_x;

		while (size_y > maxlines) {
			size_y -= maxlines;
			ps_pages++;
		}
		ps_pages++;
	}

	fprintf(f, "%%!PS-Adobe-3.0\n");
	fprintf(f, "%%%%BoundingBox: %d %d %d %d\n", trans_x, trans_y, trans_x + view_x, trans_y + view_y);
	fprintf(f, "%%%%Creator: KMLOFax Version %s\n", VERSION);
	fprintf(f, "%%%%CreationDate: %s\n", (const char *)QDateTime::currentDateTime().toString());
	fprintf(f, "%%%%DocumentData: Clean7Bit\n");
	if (level2)
		fprintf(f, "%%%%LanguageLevel: 2\n");
	fprintf(f, "%%%%Orientation: Portrait\n");
	fprintf(f, "%%%%Pages: %d\n", ps_pages);
	fprintf(f, "%%%%PageOrder: Ascend\n");
	fprintf(f, "%%%%Title: ");
	s = SENDER_ALIAS(mlofile.sender());
	if (s.isEmpty())
		fprintf(f, i18n("Facsimile"));
	else
		fprintf(f, i18n("Facsimile from %1").arg(s));
	fprintf(f, "\n%%%%EndComments\n");
	fprintf(f, "%%%%EndProlog\n");
	fprintf(f, "%%%%BeginSetup\n");
	if (level2)
		fprintf(f, "/DeviceGray setcolorspace\n");
	fprintf(f, "/#copies %d def\n", doc_copies);
	fprintf(f, "%%%%EndSetup\n");

	ps_pages = 0;
	for (page = first_page; page <= last_page; page++) {
		mlofile.readPageInfo(page, &info);

		size_x = info.width;
		size_y = size_y_buff[page-first_page];

		scale_x = (size_x * 72) / 204;
		scale_y = (size_y * 72) / info.lpi;
		maxlines = (view_y * info.lpi) / 72;

		if (scale_x > view_x) {
			scale_y = (scale_y * view_x) / scale_x;
			maxlines = (maxlines * scale_x) / view_x;
			scale_x = view_x;
		}

		mlofile.gotoPage(page);

		sub_page = 0;
		while (size_y > maxlines) {
			if (!convertPage(page, ++sub_page, ++ps_pages, maxlines, view_y)) {
				mlofile.close();
				return false;
			}
			size_y -= maxlines;
			scale_y -= view_y;
		}
		if (!convertPage(page, ++sub_page, ++ps_pages, size_y, scale_y)) {
			mlofile.close();
			return false;
		}
	}

	free(size_y_buff);

	fprintf(f, "%%%%Trailer\n%%%%EOF\n");

	mlofile.close();

	return true;
}


void MLO2PSFilter::init()
{
	setFormat(PAPER_A4);
	level2 = true;
	interpolate = false;
	first_page = 0;
	last_page = 0;
	doc_copies = 1;
	l_margin = 0;
	r_margin = 0;
	t_margin = 0;
	b_margin = 0;

	resetImageData();
}


bool MLO2PSFilter::convertPage(int fax_page, int sub_page, int ps_page, int lines, int strech_y)
{
	static const char n_hex[16] = {
		'f', '7', 'b', '3', 'd', '5', '9', '1',
		'e', '6', 'a', '2', 'c', '4', '8', '0'
	};
	int bpl, i, j;
	uchar img_in_buff[MAX_IMG_BUFF], huf_in_buff[MAX_HUF_BUFF], c;
	char out_buff[66];

	bpl = size_x >> 3;

	if (fprintf(f, "%%%%Page: %d.%d %d\n", fax_page, sub_page, ps_page) <= 0)
		return false;

	fprintf(f, "%%%%BeginPageSetup\n/pagelevel save def\n%%%%EndPageSetup\n");

	fprintf(f, "%d %d translate\n", trans_x, trans_y + view_y - strech_y);
	fprintf(f, "%d %d scale\n", scale_x, strech_y);

	if (level2) {
		fprintf(f, "{\n");
		fprintf(f, "/G3DICT <<\n");
		fprintf(f, " /Columns %d\n", size_x);
		fprintf(f, " /K 0\n");
		fprintf(f, " /Uncompressed false\n");
		fprintf(f, " /EndOfLine true\n");
		fprintf(f, " /EndOfBlock true\n");
		fprintf(f, " /BlackIs1 false\n");
		fprintf(f, ">> def\n");
		fprintf(f, "/Data1 currentfile /ASCII85Decode filter def\n");
		fprintf(f, "/Data2 Data1 G3DICT /CCITTFaxDecode filter def\n");
		fprintf(f, "<<\n");
		fprintf(f, " /ImageType 1\n");
		fprintf(f, " /Width %d\n", size_x);
		fprintf(f, " /Height %d\n", lines);
		fprintf(f, " /BitsPerComponent 1\n");
		fprintf(f, " /Decode [ 0 1 ]\n");
		fprintf(f, " /ImageMatrix [ %d 0 0 %d 0 %d ]\n", size_x, -lines, lines);
		fprintf(f, " /Interpolate %s\n", interpolate ? "true" : "false");
		fprintf(f, " /DataSource Data2\n");
		fprintf(f, ">> image\n");
		fprintf(f, "Data1 flushfile\n");
		fprintf(f, "} exec\n");
	}
	else {
		fprintf(f, "/bpl %d string def\n", bpl);
		fprintf(f, "%d %d 1 [ %d 0 0 %d 0 %d ]\n", size_x, lines, size_x, -lines, lines);
		fprintf(f, "{currentfile bpl readhexstring pop}\nimage\n");	
	}
	
	if (level2) {
		writeBase85(0x00);
		writeBase85(0x01);

		while (lines--) {
			bpl = mlofile.readHuffLine(huf_in_buff);
			for (i=0; i < bpl; i++)
				writeBase85(huf_in_buff[i]);
		}

		writeBase85(0x00); /* Write End-Of-Block */
		writeBase85(0x01);
		writeBase85(0x00);
		writeBase85(0x10);
		writeBase85(0x01);
		writeBase85(0x00);
		writeBase85(0x10);
		writeBase85(0x01);

		flushBase85();
		flushImageData();
		resetImageData();
		
		fprintf(f, "~>\n");
	}
	else {
		while (lines--) {
			mlofile.readImgLine(img_in_buff, true);
			i = 0;
			for (j=0; j < bpl; j++) {
				c = img_in_buff[j];
				out_buff[i++] = n_hex[c & 0x0f];
				out_buff[i++] = n_hex[c >> 4];
				if (i == 64) {
					out_buff[i++] = '\n';
					out_buff[i] = 0;
					fprintf(f, out_buff);
					i = 0;
				}
			}
			if (i) {
				out_buff[i++] = '\n';
				out_buff[i] = 0;
				fprintf(f, out_buff);
			}
		}
	}

	fprintf(f, "pagelevel restore\nshowpage\n");

	return true;
}


void MLO2PSFilter::writeBase85(uchar c)
{
	base85_buff[base85_ind] = c;

	if (base85_ind)
		base85_ind--;
	else
		flushBase85();
}


void MLO2PSFilter::flushBase85()
{
	static ulong power85[5] = { 85*85*85*85, 85*85*85, 85*85, 85, 1 };
	ulong v, c;
	int i;

	if (base85_ind == 3)
		return;

	v = *((ulong *)base85_buff);

	if (v)
		for (i=0; i < 5; i++) {
			c = v / power85[i];
			writeImageData('!' + (char)c);
			v -= c * power85[i];
		}
	else
		writeImageData('z');

	*((ulong *)base85_buff) = 0;
	base85_ind = 3;
}


void MLO2PSFilter::writeImageData(char c)
{
	image_buff[image_ind] = c;

	if (++image_ind == 253)
		flushImageData();
}


void MLO2PSFilter::flushImageData()
{
	if (!image_ind)
		return;

	image_buff[image_ind] = 0;
	
	if ((image_buff[0] == '%') && (image_buff[1] == '%'))
		fprintf(f, "%% %s\n", &image_buff[1]);
	else
		fprintf(f, "%s\n", image_buff);

	image_ind = 0;
}


void MLO2PSFilter::resetImageData()
{
	*((ulong *)base85_buff) = 0;
	base85_ind = 3;
	
	image_ind = 0;
}



MLO2TiffFilter::MLO2TiffFilter() : MLOFilter()
{
	init();
}


MLO2TiffFilter::MLO2TiffFilter(const QString& infile) : MLOFilter()
{
	init();
	
	setFile(infile);
}


MLO2TiffFilter::~MLO2TiffFilter()
{
}


void MLO2TiffFilter::setFile(const QString& infile)
{
	int year, month, day, hour, minute, second;
	
	MLOFilter::setFile(infile);
	
	mloDateSplit(infile, &year, &month, &day, &hour, &minute, &second);
	sprintf(time_str, "%04d:%02d:%02d %02d:%02d:%02d", year, month, day, hour, minute, second);
}


void MLO2TiffFilter::setRange(int first, int last)
{
	first_page = first;
	last_page = last;
}


void MLO2TiffFilter::setCompression(int compress)
{
	static const uint16 comp_tab[] = {
		COMPRESSION_NONE,
		COMPRESSION_CCITTFAX3,
		COMPRESSION_LZW,
		COMPRESSION_PACKBITS,
	};

	compression = comp_tab[compress];
}


bool MLO2TiffFilter::convertFile(const QString& outfile)
{
	int page, pages;
	
	if (!mlofile.open()) {
		return false;
	}

	pages = mlofile.pages();
	
	if (first_page == 0) {
		first_page = 1;
		last_page = pages;
	}

	if ((first_page > last_page) || (first_page > pages)) {
		mlofile.close();
		return false;
	}

	if (last_page > pages)
		last_page = pages;

	
	
	tif = TIFFOpen(outfile, "w");
	if (!tif) {
		mlofile.close();
		return false;
	}

	for (page = first_page; page <= last_page; page++) {
		convertPage(page);
	}

	TIFFClose(tif);

	mlofile.close();

	return true;
}


void MLO2TiffFilter::init()
{
	tif = 0;
	first_page = 0;
	last_page = 0;
	compression = COMPRESSION_NONE;
	time_str[0] = 0;
}


bool MLO2TiffFilter::convertPage(int page)
{
	page_info_t info;
	int h, i, bpl;
	uchar buffer[MAX_IMG_BUFF];

	if (!mlofile.readPageInfo(page, &info))
		return false;
	
	bpl = info.width >> 3;
	h = mlofile.pageHeight(page);

	TIFFSetField(tif, TIFFTAG_SUBFILETYPE, FILETYPE_PAGE);
	TIFFSetField(tif, TIFFTAG_IMAGEWIDTH, (uint32)info.width);
	TIFFSetField(tif, TIFFTAG_IMAGELENGTH, (uint32)h);
	TIFFSetField(tif, TIFFTAG_BITSPERSAMPLE, 1);
	TIFFSetField(tif, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_MINISWHITE);
	TIFFSetField(tif, TIFFTAG_ORIENTATION, ORIENTATION_TOPLEFT);
	TIFFSetField(tif, TIFFTAG_SAMPLESPERPIXEL, 1);
	TIFFSetField(tif, TIFFTAG_ROWSPERSTRIP, 1);
	TIFFSetField(tif, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);
	TIFFSetField(tif, TIFFTAG_FILLORDER, FILLORDER_MSB2LSB);
	TIFFSetField(tif, TIFFTAG_XRESOLUTION, (float)204.);
	TIFFSetField(tif, TIFFTAG_YRESOLUTION, (float)info.lpi);
	TIFFSetField(tif, TIFFTAG_RESOLUTIONUNIT, RESUNIT_INCH);
	TIFFSetField(tif, TIFFTAG_SOFTWARE, "KMLOFax " VERSION);
	TIFFSetField(tif, TIFFTAG_IMAGEDESCRIPTION, (const char *)SENDER_ALIAS(mlofile.sender()));
	TIFFSetField(tif, TIFFTAG_DATETIME, time_str);
	TIFFSetField(tif, TIFFTAG_MAKE, "ELSA");
	TIFFSetField(tif, TIFFTAG_MODEL, "MicroLink Office");
	TIFFSetField(tif, TIFFTAG_COMPRESSION, compression);
	if (compression == COMPRESSION_CCITTFAX3) {
		TIFFSetField(tif, TIFFTAG_GROUP3OPTIONS, 0);
		info.params &= ~(3 << 9);	/* Always reencoded to 1-D MR */
		TIFFSetField(tif, TIFFTAG_FAXRECVPARAMS, (uint32)info.params);
	}

	mlofile.gotoPage(page);
	for (i=0; i < h; i++) {
		mlofile.readImgLine(buffer, false);
		TIFFWriteEncodedStrip(tif, (tstrip_t)i, (tdata_t)buffer, (tsize_t)bpl);
	}

	TIFFWriteDirectory(tif);

	return true;
}
