/***************************************************************************
			  ktransferimpl.h  -  description
                             -------------------
    begin                : Mon Oct 16 2000
    copyright            : (C) 2000 by Sergio Moretti
    email                : sermore@libero.it
    revision             : $Revision: 1.3 $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KTRANSFERIMPL_H
#define KTRANSFERIMPL_H

#include <qdatetime.h>
#include <kurl.h>
#include <qcstring.h>
#include <qfile.h>
#include <qtimer.h>
//#include "ktransfer.h"
#include "kobjectimpl.h"

class KTManagerImpl;
class KTransfer;

/**abstract class to manage download process
  *@author Sergio Moretti
  */

class KTransferImpl : public KObjectImpl {
  
  Q_OBJECT

  friend class KTransfer;

public:
  static const char DOCID[];
  static const char MIMETYPE[][50];
  static const char DOCTYPE[];
  			
  virtual ~KTransferImpl();

  /** document name */
  virtual const char * docId() const { return DOCID; }
  /** dom document type */
  virtual const char * docType() const { return DOCTYPE; }
  /** mime type */
  virtual const char * mimeType(int i) const { return MIMETYPE[i]; }

  // references to other objects

  /** reference to parent manager */
  KTManagerImpl * manager() const;
  /** reference to global instance */
  const KTransferImpl * global() const;

  // read-only tmp state

  int len() const { return _len; }
  int partial() const { return _partial; }
  int retry() const { return _retry; }
  /** get the status of the transfer*/
  Status status() const { return _status; }
  /** set transfer status */
  void setStatus(Status s) { _status = s; }
  /** get the status of the transfer*/
  ResumeStatus rsmStatus() const { return _rsmStatus; }
  /** get the istantaneaous bandwidth */
  int bandwidth() const { return _bandwidth; }
  /** get the medium bandwidth */
  int mediumBandwidth() const { return _mediumBandwidth; }
  /** get start time */
  QTime startTime() const { return _timeStart; }
  /* get status converted in string form */
  QString statusStr() const;
  /* get status converted in string form */
  QString cmdStatusStr() const;
  /** get status converted in string form */
  QString filStatusStr() const;
  /** get status converted in string form */
  QString cnnStatusStr() const;
  /** get the resumable state string: Yes, No, ??? */
  QString rsmStatusStr() const;
  /** get the fatal error log */
  QString fatalErrorLog() const { return _fatalErrorLog; }

  // set/get operation on persistent state
  
  /** get the local file */
  const KURL & local() const { return _local; }
  /** set local file */
  void setLocal(const KURL &l) { _local = l; }
  /** get the url */
  const KURL & remote() const { return _remote; }
  /** set url */
  void setRemote(const KURL &r) { _remote = r; }
  /** get number of resumes */
  int resumed() const { return _resumed; }
  /** set number of resumes */
  void setResumed(int r) { _resumed = r; }
  /** get priority */
  int priority() const { return _priority; }
  /** set priority */
  void setPriority(int p) { _priority = p; }
  /** status of command */
  CommandStatus cmdStatus() const { return _cmdStatus; }
  void setCmdStatus(CommandStatus s) { _cmdStatus = s; }

  // OPERATIONS on transfer

  /** check for resume capabilities */
  void checkResumeCapability();
  /** check if protocol is supported */
  virtual bool isProtocolSupported(const QString &proto) const = 0;
  /** true if transfer is complete */
  bool isComplete() const;
  /** check if process is running */
  virtual bool isRunning() const;
  /** check if the process can be started */
  virtual bool isRunnable(bool automatic) const;
  /** check for transfer resume capability */
  bool isResumable() const { return rsmStatus() == RSM_YES; }
  /** download estimated time */
  QTime estTime();
  /** get the temporary file */
  virtual KURL getTmpFile() const;
  /** get the temporary file */
  virtual KURL getTempFile(const KURL &rmt, const KURL &lcl) const;
  /** start the command process */
  virtual bool start();
  /** kill the command process */
  virtual bool kill(bool setKilled = true);
  /** compare priority, return 1, 0, -1 */
  int cmpPriority(const KTransferImpl *t) const;

  // CONFIG
  
  bool getCheckResume() const;
  void setCheckResume(bool c) { _cfg_checkResume = c; }
  int getMaxResume() const;
  void setMaxResume(int r) { _cfg_maxResume = r; }
  bool getAutoResume() const;
  void setAutoResume(bool m) { _cfg_autoResume = m; }
  int getWaitResume() const;
  void setWaitResume(int w) { _cfg_waitResume = w; }
  int getMaxRetry() const;
  void setMaxRetry(int m) { _cfg_maxRetry = m; }
  int getWaitRetry() const;
  void setWaitRetry(int w) { _cfg_waitRetry = w; }
  int getBandSampleTime() const;
  void setBandSampleTime(int t) { _cfg_bandSampleTime = t; }
  int getReadNotifyTime() const;
  void setReadNotifyTime(int t) { _cfg_readNotifyTime = t; }
  int getPriorityPolicy() const;
  void setPriorityPolicy(int p) { _cfg_priorityPolicy = p; }
  bool getLogProcess() const;
  void setLogProcess(bool l) { _cfg_logProcess = l; }
  
protected:
  KTransferImpl(int type);
  /** load state from dom */
  virtual void loadData();
  /** update dom state entries */
  virtual void updateState();
  /** update dom config entries */
  virtual void updateConf();

  /** set transfer status */
  void setRsmStatus(ResumeStatus s) { _rsmStatus = s; }
  void setLen(int l);
  void setPartial(int p);

  void raiseFatalError();

  /** execute a check for transfer resume capability, blocking until check is
   * done, but have to update event loop */
  virtual bool processCheckResume();

  // abstract operations for derived classes

  /** send a start request to the transfer process */
  virtual bool processStart() = 0;
  /** send a stop request to transfer process */
  virtual bool processKill() = 0;

  // operations that derived class have to call to maintain the state

  /** called when process start */
  virtual void opProcessStart(); 
  /** called when process end */
  virtual void opProcessEnd(int exitStatus);
  /** called when a run is started */
  virtual void opRunStart();
  /** called when a run is finished */
  virtual void opRunEnd();
  /** called when connection is up */
  virtual void opConnect();
  /** called when download start */
  virtual void opDownload();
  /** called when data is read */
  virtual void opDataRead(int datalen);
  /** called when an error occurred */
  virtual void opError(const QString &err);
  /** called when a fatal error occurred (must kill the transfer */
  virtual void opFatalError(const QString &err);
  /** called when process send output */
  virtual void opProcessOutput(const QString &data);

  /** calculate bandwidth */
  void calcBandwidth();

protected slots:
  /** called when timeout expires */
  virtual void slotTimeout();

signals:
  /** forward process output */
  void sigProcessOutput(const QString &data);
  
private: // Private state attributes
  /** url to dowload */
  KURL _remote;
  /** write data to file */
  KURL _local;
  /** status of the transfer */
  Status _status;
  /** resume status */
  ResumeStatus _rsmStatus;
  /** command status */
  CommandStatus _cmdStatus;
  /** num of resuming */
  int _resumed;
  /** time to wait before resuming */
  QTimer _wait;
  /** time of process start */
  QTime _timeStart;
  /** last read time */
  QTime _lastReadTime;
  /** last bytes read */
  int _lastBytesRead;
  /** starting from */
  int _lastBytesStart;
  /** medium bandwidth */
  int _mediumBandwidth;
  /** instantaneaous bandwidth */
  int _bandwidth;
  /** estimate time */
  QTime _timeEst;
  /** last read data notification */
  QTime _lastReadNotify;
  /** priority */
  int _priority;
  /** file descriptor of log file */
  QFile _logFile;
  /** true if process killed by user */
  bool _killed;
  /** number of retries */
  int _retry;
  /** len of file to download */
  int _len;
  /** partial downloaded */
  int _partial;
  /** fatal error flag */
  bool _fatalError;
  /** fatal error message */
  QString _fatalErrorLog;
  
  // config attributes

  /** check for resumable */
  bool _cfg_checkResume;
  /** max num of retries */
  int _cfg_maxResume;
  /** autoresume after a broken connection */
  bool _cfg_autoResume;
  /** seconds to wait before resuming */
  int _cfg_waitResume;
  /** max num of retries */
  int _cfg_maxRetry;
  /** wait before retrying */
  int _cfg_waitRetry;
  /** time interval between band sampling */
  int _cfg_bandSampleTime;
  /** time interval between read notification */
  int _cfg_readNotifyTime;
  /** priority policy */
  int _cfg_priorityPolicy;
  /** enable logging of process output */
  bool _cfg_logProcess;
};

#endif

/* Local Variables: */
/* mode: c++ */
/* End: */
