/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */
#ifndef KBABELVIEW_H
#define KBABELVIEW_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

class MyMultiLineEdit;
class QSplitter;
class KConfig;
class Catalog;
class CatalogItem;
class GotoDialog;
class QPopupMenu;


#include <qwidget.h>
#include <qstrlist.h>
#include <kbabeliface.h>
#include "searchresultbox.h"


/**
 * This is the main view class for KBabel.  Most of the non-menu,
 * non-toolbar, and non-statusbar (e.g., non frame) GUI code should go
 * here.
 * @short Main view
 * @author Matthias Kiefer <matthias.kiefer@gmx.de>
 * @version 0.1
 */
class KBabelView : public QWidget, virtual public KBabelIface
{
    Q_OBJECT
public:
	/**
	 * Default constructor
	 */
    KBabelView(Catalog* catalog,QWidget *parent);

	/**
	 * Destructor
	 */
    virtual ~KBabelView();

    QString currentURL() const;
    bool isLastView() const;
    bool isModified() const;
    /** the edit mode of the entry-editors*/
    bool isOverwriteMode() const;
    bool isReadOnly() const;
    /** the edit mode of the entry-editors*/
    void setOverwriteMode(bool ovr);
    bool isSearching() const;

    void saveView();
    void restoreView();
    void saveSession(KConfig*);
    void restoreSession(KConfig*);

    void readSettings();
    void saveSettings();

    SearchSettings searchSettings() const;
    EditorSettings editorSettings() const{return _settings;}

    virtual void openURL(QCString url);
    void open(QString url=QString::null);
    void openTemplate(QString openURL,QString saveURL);
    bool saveFile(bool checkSyntax=true);
    bool saveFileAs(QString filename=QString::null, bool checkSyntax=true);
    /**
    * Checks, if the file has been modified. If true, it askes the user if he wants
    * to save, discard or cancel. If the users chose save, it saves the file.
    * @return true, if it is allowed to open a new file. false, if the user wants
    * to edit the file again.
    */
    bool checkModified();


    /**
    * Checks syntax of the current catalog. If the catalog is modified it saves it under
    * a temporary filename ( using @ref Catalog::saveTempFile ).
    *
    * @param msgOnlyAtError flag, if a message should be shown, only if a error occured.
    * @param question flag, if only a information about the result should be shown or
    *        a question, whether the user wants to continue or cancel
    *
    * @return true, if no error occured or if an error occured but the user wants to
    *               continue anyway.
    */
    bool checkSyntax(bool msgOnlyAtError, bool question);

    /**
    * this is called from the catalog when updating his views
    * @param index the index of the entry that has changed
    */
    void updateEntry(uint index);
    /**
    * this is called from the catalog when updating his views
    */
    void updateHeader(QString newHeader);

    Catalog* catalog(){return _catalog;}

    void processUriDrop(QStrList& uriList, const QPoint & pos);

public slots:
    void setSettings(SearchSettings settings);
    void setSettings(EditorSettings settings);

    void textCut();
    void textCopy();
    void textPaste();
    void selectAll();
    void gotoFirst();
    void gotoLast();
    void gotoNext();
    void gotoPrev();
    void gotoEntry();
    void gotoNextFuzzyOrUntrans();
    void gotoPrevFuzzyOrUntrans();
    void gotoNextFuzzy();
    void gotoPrevFuzzy();
    void gotoNextUntranslated();
    void gotoPrevUntranslated();
    void gotoNextError();
    void gotoPrevError();

    /** hides or shows the editor for the comments */
    void showComments(bool);
    /** hides or shows the searchresults window */
    void showSearchResults(bool);

    void removeFuzzyStatus();
    /** opens the header editor for the po-file */
    void editHeader();

    bool checkSyntax();

    void stopSearch();
    void startSearch();
    void startCatSearch();
    void startAuxSearch();
    void startComSearch();
    void startSelectionCatSearch();
    void startSelectionAuxSearch();
    void startSelectionComSearch();

protected:
    virtual void dragEnterEvent(QDragEnterEvent *event);
    virtual void dropEvent(QDropEvent *event);
    virtual bool eventFilter(QObject*, QEvent* event);

signals:
    /** emited when a fuzzy catalogentry is shown */
    void signalFuzzyDisplayed(bool);
    /** emited when a untranslated catalogentry is shown */
    void signalUntranslatedDisplayed(bool);
    /** emited when the first catalogentry is shown */
    void signalFirstDisplayed(bool);
    /** emited when the last catalog entry is shown */
    void signalLastDisplayed(bool);
    /**
    * emited when a new entry is shown
    * index: the index of the currently shown entry
    */
    void signalDisplayed(uint index);

    /**
    * emited when new entry is displayed and there is no
    * fuzzy entry afterwards in the catalog
    */
    void signalFuzzyAfterwards(bool);
    /**
    * emited when new entry is displayed and there is no
    * fuzzy entry in front of it in the catalog
    */
    void signalFuzzyInFront(bool);
    /**
    * emited when new entry is displayed and there is no
    * untranslated entry afterwards in the catalog
    */
    void signalUntranslatedAfterwards(bool);
    /**
    * emited when new entry is displayed and there is no
    * fuzzy entry in fornt of it in the catalog
    */
    void signalUntranslatedInFront(bool);

    void signalErrorAfterwards(bool);
    void signalErrorInFront(bool);

    /**
     * Use this signal to change the content of the statusbar
     */
    void signalChangeStatusbar(const QString& text);
    /**
     * Use this signal to change the content of the caption
     */
    void signalChangeCaption(const QString& text);

    void signalResetProgressBar(QString,int);
    void signalProgress(int);
    void signalClearProgressBar();

    void signalSearchActive(bool);
    void signalEnableCatSearch(bool);
    void signalEnableComSearch(bool);
    void signalEnableAuxSearch(bool);

    /**
    * emitted when a search is started, but the searchbox is not visible.
    * It will then be shown and this signal will be emitted.
    */
    void signalSearchResultsShown();

    void signalSearchSettingsChanged(SearchSettings);

private:
   /**
   * inserts the content of the current catalog entry into
   * the fields in the view
   * @param delay flag, if the auto search should be started delayed
   * this is useful when a new file is opened
   */
   void updateEditor(bool delay=false);

   /**
   * checks the status of the displayed entry: last, first, fuzzy,...
   * and emits the appropriate signals
   */
   void emitEntryState();

   void startSearch(bool delay);

   void gotoEntry(int index);

private slots:
   void autoRemoveFuzzyStatus();

   /** connected to the catalog. it is called when a new file is opened*/
   void newFileOpened(bool readOnly);

   /**
   * This function is called by signal textChanged from msgstrEdit and stores
   * the changes in the _catalog
   */
   void takeMsgstrChanges();
   /**
   * This function is called by signal textChanged from commentEdit and stores
   * the changes in the _catalog
   */
   void takeCommentChanges();

   void showError(QString message);

   void catSearchEnabled(bool);
   void comSearchEnabled(bool);
   void auxSearchEnabled(bool);

private:
   MyMultiLineEdit* msgstrEdit;
   MyMultiLineEdit* commentEdit;
   MyMultiLineEdit* msgidLabel;
   SearchResultBox* searchBox;
   GotoDialog* _gotoDialog;
   QPopupMenu* _dropMenu;

   QSplitter* _mainEditSplitter;
   QSplitter* _toolBoxSplitter;
   QSplitter* _viewSplitter;

   QWidget* _commentWidget;
   QWidget* _searchWidget;

   Catalog* _catalog;
   uint _currentIndex;

   EditorSettings _settings;

   bool _autoSearch;
   bool _autoSearchTempDisabled;
   SearchMode _defaultSearchMode;
};

#endif // KBABELVIEW_H
