/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */
#include "catalogitem.h"
#include "resources.h"

#include <qtextstream.h>

CatalogItem::CatalogItem()
{
   clear();
}

CatalogItem::CatalogItem(QString itemStr)
{
   QTextStream stream(&itemStr,IO_ReadOnly);

   read(stream);
}

CatalogItem::~CatalogItem()
{
}



QStringList CatalogItem::msgidAsList() const
{
   QStringList list=QStringList::split("\n",msgid());

   if(msgid().left(1)=="\n")
      list.prepend("");

   if(list.isEmpty())
      list.append("");

   return list;
}


QStringList CatalogItem::msgstrAsList() const
{
   QStringList list=QStringList::split("\n",msgstr());

   if(msgstr().left(1)=="\n")
      list.prepend("");

   if(list.isEmpty())
      list.append("");

   return list;
}

QStringList CatalogItem::commentAsList() const
{
   QStringList list=QStringList::split("\n",comment());

   return list;
}


bool CatalogItem::isFuzzy() const
{
   bool flag=false;
   if(_comment.contains(", fuzzy"))
     flag=true;

   return flag;
}

bool CatalogItem::isCformat() const
{
   bool flag=false;
   if(_comment.contains("\\n?#,.*,? c-format"))
     flag=true;

   return flag;
}

bool CatalogItem::isUntranslated() const
{
   return _msgstr.isEmpty();
}

int CatalogItem::totalLines() const
{
   int lines=0;
   if(!_comment.isEmpty())
   {
      lines = _comment.contains("\n")+1;
   }
   lines+=( _msgid.contains("\n")+1 );
   lines+=( _msgstr.contains("\n")+1 );

   return lines;
}


void CatalogItem::removeFuzzy()
{
   if(isFuzzy())
   {
       QStringList list=QStringList::split("\n",_comment);
       QString temp;

       for(uint i=0;i<list.count();i++)
       {
          list[i].replace(QRegExp(", fuzzy"),"");

          // remove empty comment line
          if(!list[i].contains(QRegExp("^#\\s*$")))
          {
             temp+=("\n"+list[i]);
          }
       }
       // remove first newline
       _comment=temp.remove(0,1);
   }

   return;
}


void CatalogItem::clear()
{
   _comment="";
   _msgid="";
   _msgstr="";
   _valid=true;
}


QString CatalogItem::asString()
{
   QString temp;
   QTextStream stream(&temp,IO_WriteOnly);

   write(stream);
   return temp;
}


void CatalogItem::write(QTextStream& stream) const
{
   if(!comment().isEmpty())
   {
      stream << comment() << "\n";
   }

   QStringList list=msgidAsList();
   QValueList<QString>::ConstIterator lit;

   stream << "msgid ";
   for( lit = list.begin(); lit != list.end(); ++lit )
   {
      stream << "\"" << (*lit) << "\"\n";
   }

   list=msgstrAsList();
   stream << "msgstr ";
   for( lit = list.begin(); lit != list.end(); ++lit )
   {
      stream << "\"" << (*lit) << "\"\n";
   }
}


CatalogItem::IOStatus CatalogItem::read(QTextStream& stream)
{
   enum {Begin,Comment,Msgid,Msgstr} part=Begin;

   clear();

   _valid=false;

   QString line;
   bool cancelLoop=false;
   bool error=false;

   while(!stream.eof() && !cancelLoop)
   {
       int pos=stream.device()->at();

       line=stream.readLine();
       // remove whitespaces from beginning and end of line
       line=line.stripWhiteSpace();

       if(part==Begin)
       {
           // ignore trailing newlines
           if(line.isEmpty())
              continue;

           if(line.contains(QRegExp("^#~")))
           {
              return Obsolete;
           }
           else if(line.contains(QRegExp("^#")))
           {
               part=Comment;
               _comment=line;
           }
           else if(line.contains(QRegExp("^msgid\\s*\".*\"$")))
           {
               part=Msgid;

               // remove quotes at beginning and the end of the lines
               line.replace(QRegExp("^msgid\\s*\""),"");
               line.replace(QRegExp("\"$"),"");

               _msgid=line;
           }
           else
           {
              kDebugInfo(KBABEL,"no comment or msgid found after a comment");

               cancelLoop=true;
               error=true;
           }
       }
       else if(part==Comment)
       {
            if(line.isEmpty())
               continue;
            else if(line.contains(QRegExp("^#~")))
            {
               return Obsolete;
            }
            else if(line.contains(QRegExp("^#")))
            {
               _comment+=("\n"+line);
            }
            else if(line.contains(QRegExp("^msgid\\s*\".*\"$")))
            {
               part=Msgid;

               // remove quotes at beginning and the end of the lines
               line.replace(QRegExp("^msgid\\s*\""),"");
               line.replace(QRegExp("\"$"),"");

               _msgid=line;
            }
            else
            {
               kDebugInfo(KBABEL,"no comment or msgid found after a comment while parsing: %s",
                       _comment.ascii());

               error=true;
               cancelLoop=true;
            }
        }
        else if(part==Msgid)
        {
            if(line.isEmpty())
               continue;
            else if(line.contains(QRegExp("^\".*\\n?\"$")))
            {
               // remove quotes at beginning and the end of the lines
               line.replace(QRegExp("^\""),"");
               line.replace(QRegExp("\"$"),"");

               // add Msgid line to item
               _msgid+=("\n"+line);
            }
            else if(line.contains(QRegExp("^msgstr\\s*\".*\\n?\"$")))
            {
               part=Msgstr;

               // remove quotes at beginning and the end of the lines
               line.replace(QRegExp("^msgstr\\s*\""),"");
               line.replace(QRegExp("\"$"),"");

               _msgstr=line;
            }
            else
            {
               kDebugInfo(KBABEL,"no msgstr or msgid found after a msgid while parsing: %s",
                     _msgid.ascii());

               cancelLoop=true;
               error=true;
            }
        }
        else if(part==Msgstr)
        {
            if(line.isEmpty())
               continue;
            // another line of the msgstr
            else if(line.contains(QRegExp("^\".*\\n?\"$")))
            {
               // remove quotes at beginning and the end of the lines
               line.replace(QRegExp("^\""),"");
               line.replace(QRegExp("\"$"),"");

               _msgstr+=("\n"+line);
            }
            else if(line.contains(QRegExp("^\\s*msgid")) || line.contains(QRegExp("^\\s*#")))
            {
               cancelLoop=true;
               stream.device()->at(pos);// reset position in stream to beginning of this line
               break;
            }
            else
            {
               kDebugInfo(KBABEL,"no msgid or comment found after a msgstr while parsing: %s",
                     _msgstr.ascii());

               cancelLoop=true;
               error=true;
            }
        }
    }

    if(error)
       return ParseError;
    else
    {
      _valid=true;
      return Ok;
    }
}

