/* 
 *  Copyright (C) 1999-2001 Bernd Gehrmann
 *                          bernd@physik.hu-berlin.de
 *
 * This program may be distributed under the terms of the Q Public
 * License as defined by Trolltech AS of Norway and appearing in the
 * file LICENSE.QPL included in the packaging of this file.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */


#include <qtooltip.h>
#include <qpainter.h>
#include <qapp.h>
#include "misc.h"

#include "logtree.h"
#include "logtree.moc"


static const int BORDER = 8;
static const int INSPACE = 3;

static bool static_initialized = false;
static int  static_width;
static int  static_height;


class LogTreeTip : public QToolTip
{
public:
    LogTreeTip(QWidget *parent)
        : QToolTip(parent) {}

protected:
    virtual void maybeTip(const QPoint &pos);
};


class LogTreeItem
{
public:
    QString rev;
    QString author;
    QString comment;
    QString tagcomment;
    QString taglist;
    QString branchpoint;
    bool firstonbranch;
    int row;
    int col;
    bool selected;
};


class LogTreeConnection
{
public:
    LogTreeItem *start;
    LogTreeItem *end;
};


LogTreeView::LogTreeView(QWidget *parent, const char *name)
    : QTableView(parent, name)
{
    if (!static_initialized)
	{
	    static_initialized = true;
	    QFontMetrics fm( fontMetrics() );
	    static_width = fm.width("1234567890") + 2*BORDER + 2*INSPACE;
	    static_height = 2*fm.height() + 2*BORDER + 3*INSPACE;
	}

    items.setAutoDelete(true);
    connections.setAutoDelete(true);
    setNumCols(0);
    setNumRows(0);
    setAutoUpdate (false);

    setTableFlags( Tbl_autoVScrollBar|Tbl_autoHScrollBar|
		   Tbl_smoothVScrolling | Tbl_smoothHScrolling );
    setFrameStyle( QFrame::WinPanel | QFrame::Sunken );
    setBackgroundMode(PaletteBase);
    setFocusPolicy(ClickFocus);
    
    setCellWidth(0);
    setCellHeight(0);

    (void) new LogTreeTip(this);
}



void LogTreeView::addRevision(QString rev, QString author,
                              QString comment, QString taglist, QString tagcomment)
{
    int pos;
    QString branchpoint, branch;

    branch = "";
    branchpoint = rev;

    // find branch
    if ((pos = branchpoint.findRev('.')) != -1)
        {
            branchpoint.truncate(pos);
            if ((pos = branchpoint.findRev('.')) != -1)
                {
                    // e. g. for rev = 1.1.2.3 we have
                    // branch = 1.1.2, rev2 = 1.1
                    branch = branchpoint;
                    branchpoint.truncate(pos);
                }
        }

    if (branch.isEmpty())
        {
            // Most probably we are on the trunk
            setNumRows(numRows()+1);
            setNumCols(1);
            LogTreeItem *item = new LogTreeItem;
            item->rev = rev;
            item->author = author;
            item->comment = comment;
            item->tagcomment = tagcomment;
            item->taglist = taglist;
            item->branchpoint = branchpoint;
            item->firstonbranch = false;
            item->row = numRows()-1;
            item->col = 0;
            item->selected = false;
            items.append(item);
            return;
        }
    
    // look whether we have revisions on this branch
    // shift them up
    int row = -1, col;
    QListIterator<LogTreeItem> it(items);
    for (; it.current(); ++it)
        {
            if (branch == (it.current()->rev).left(branch.length()))
                {
                    it.current()->firstonbranch = false;
                    row = it.current()->row;
                    col = it.current()->col;
                    it.current()->row--;
                    // Are we at the top of the widget?
                    if (row == 0)
                        {
                            QListIterator<LogTreeItem> it2(items);
                            for (; it2.current(); ++it2)
                                it2.current()->row++;
                            setNumRows(numRows()+1);
                            row = 1;
                        }
                }
        }

    if (row == -1)
        {
            // Ok, so we must open a new branch
            // Let's find the branch point
            QListIterator<LogTreeItem> it3(items);
            for (it3.toLast(); it3.current(); --it3)
                {
                    if (branchpoint == it3.current()->rev)
                        {
                            // Move existing branches to the right
                            QListIterator<LogTreeItem> it4(items);
                            for (; it4.current(); ++it4)
                                if (it4.current()->col > it3.current()->col)
                                    {
                                        it4.current()->col++;
                                    }
                            setNumCols(numCols()+1);
                            row = it3.current()->row-1;
                            col = it3.current()->col+1;
                            if (row == -1)
                                {
                                    QListIterator<LogTreeItem> it5(items);
                                    for (; it5.current(); ++it5)
                                        it5.current()->row++;
                                    setNumRows(numRows()+1);
                                    row = 0;
                                }
                            break;
                        }
                }
        }

    LogTreeItem *item = new LogTreeItem;
    item->rev = rev;
    item->author = author;
    item->comment = comment;
    item->tagcomment = tagcomment;
    item->taglist = taglist;
    item->branchpoint = branchpoint;
    item->firstonbranch = true;
    item->row = row;
    item->col = col;
    item->selected = false;
    items.append(item);

#if 0
    cout << "Dump: " << endl;
    cout << "Rows: " << numRows() << "Cols: " << numCols() << endl;
    QListIterator<LogTreeItem> it5(items);
    for (; it5.current(); ++it5)
	{
	    cout << "Rev: "<< it5.current()->rev << endl;
	    cout << "row: "<< it5.current()->row << ", col: " << it5.current()->col << endl;
	    cout << "fob: "<< it5.current()->firstonbranch << endl;
	}
    cout << "End Dump" << endl;
#endif
    
}


void LogTreeView::collectConnections()
{
    QListIterator<LogTreeItem> it(items);
    for (; it.current(); ++it)
	{
            QString rev = it.current()->rev;
            
            QListIterator<LogTreeItem> it2(items);
            for (it2=it,++it2; it2.current(); ++it2)
                if (it2.current()->branchpoint == rev &&
                    it2.current()->firstonbranch)
                    {
                        LogTreeConnection *conn = new LogTreeConnection;
                        conn->start = it.current();
                        conn->end = it2.current();
                        connections.append(conn);
                    }
        }
}


void LogTreeView::setSelectedPair(QString selectionA, QString selectionB)
{
    QListIterator<LogTreeItem> it(items);
    for(; it.current(); ++it)
	{
            bool oldstate = it.current()->selected;
            bool newstate = ( selectionA == it.current()->rev ||
                              selectionB == it.current()->rev );
            if (oldstate != newstate)
                {
                    it.current()->selected = newstate;
                    repaint(false);
                }
    }
}


void LogTreeView::lookupTip(const QPoint &p, QRect *r, QString *s)
{
    int row = findRow( p.y() );
    int col = findCol( p.x() );
    
    *r = QRect(0, 0, -1, -1);
    *s = "";

    QListIterator<LogTreeItem> it(items);
    for(; it.current(); ++it)
        if (it.current()->row == row
            && it.current()->col == col)
            {
                int left; colXPos(col, &left);
                int top; rowYPos(row, &top);
                int width = cellWidth(col);
                int height = cellHeight(row);
                *r = QRect(left, top, width, height);
                *s = it.current()->comment;
                if (!it.current()->tagcomment.isEmpty())
                    {
                        *s += "\n";
                        *s += it.current()->tagcomment;
                    }
                return;
            }
}


QSize LogTreeView::sizeHint() const
{
    return QSize(2 * static_width, 3 * static_height);
}


void LogTreeView::setupPainter(QPainter *p)
{
    p->setBackgroundColor(colorGroup().base());
}


void LogTreeView::paintCell(QPainter *p, int row, int col)
{
    bool followed, branched;
    LogTreeItem *item;

    branched = false;
    followed = false;
    item = 0;
    
    QListIterator<LogTreeItem> it(items);
    for(; it.current(); ++it)
        {
            int itcol = it.current()->col;
            int itrow = it.current()->row;
            if (itrow == row-1 && itcol == col)
                followed = true;
            if (itrow == row && itcol == col)
                item = it.current();
        }
    QListIterator<LogTreeConnection> it2(connections);
    for (; it2.current(); ++it2)
        {
            int itcol1 = it2.current()->start->col;
            int itcol2 = it2.current()->end->col;
            int itrow = it2.current()->start->row;
            if (itrow == row && itcol1 <= col && itcol2 > col)
                branched = true;
        }

    p->fillRect(0, 0, cellWidth(col), cellHeight(row),
                colorGroup().base());
    p->setPen(colorGroup().text());
    if (item)
        paintRevisionCell(p, row, col, item->author, item->taglist, item->rev, followed, branched, item->selected);
    else if (followed || branched)
        paintConnector(p, row, col, followed, branched);
}


void LogTreeView::paintConnector(QPainter *p,
				 int row, int col, bool followed, bool branched)
{
    int midx = colWidths[col] / 2;
    int midy = rowHeights[row] / 2;

    p->drawLine(0, midy, branched ? colWidths[col] : midx, midy);
    if (followed)
        p->drawLine(midx, midy, midx, 0);
}


void LogTreeView::paintRevisionCell(QPainter *p, 
                                    int row, int col,
                                    QString line1, QString line2, QString line3,
                                    bool followed, bool branched, bool selected)
{
    QFontMetrics fm(p->fontMetrics());
    
    int boxwidth, boxheight;
    
    QRect r1 = fm.boundingRect(0, 0, 0, 0, 0, line1);
    QRect r2 = fm.boundingRect(0, 0, 0, 0, 0, line2);
    QRect r3 = fm.boundingRect(0, 0, 0, 0, 0, line3);
    
    boxwidth = QMAX(static_width-2*BORDER, QMAX(r1.width(), r3.width()));
    boxheight = r1.height() + r3.height() + 3 * INSPACE;
    
    if (!line2.isEmpty())
        {
            if (r2.width() > boxwidth)
                boxwidth = r2.width();
            boxheight += r2.height() + INSPACE;
        }
    
    boxwidth += 2 * INSPACE;
    
    int x = (colWidths[col] - boxwidth) / 2;
    int midx = colWidths[col] / 2;
    int y = (rowHeights[row] - boxheight) / 2;
    int midy = rowHeights[row] / 2;
    
    // Connectors
    if (followed)
        p->drawLine(midx, 0, midx, y);
    
    if (branched)
        p->drawLine(midx + boxwidth / 2, midy, colWidths[col], midy);
    
    p->drawLine(midx, y + boxheight, midx, rowHeights[row]);
    
    // The box itself
    if (selected)
        {
            if (style() == WindowsStyle)
                {
                    p->fillRect(x, y, boxwidth, boxheight, QApplication::winStyleHighlightColor());
                    p->setPen(white);
                }
            else
                {
                    p->fillRect(x, y, boxwidth, boxheight, colorGroup().text());
                    p->setPen(colorGroup().base());
                }
        }
    else
        {
            p->drawRoundRect(x, y, boxwidth, boxheight, 10, 10);
        }
    
    x += INSPACE;
    y += INSPACE;
    boxwidth -= 2 * INSPACE;
    
    p->drawText(x, y, boxwidth, boxheight, AlignHCenter, line1);
    y += r1.height() + INSPACE;
    
    if (!line2.isEmpty())
        {
            QFont font(p->font());
            QFont underline(font);
            
            underline.setUnderline(true);
            p->setFont(underline);
            p->drawText(x, y, boxwidth, boxheight, AlignHCenter, line2);
            p->setFont(font);
            y += r2.height() + INSPACE;
        }
    
    p->drawText(x, y, boxwidth, boxheight, AlignHCenter, line3);
}


void LogTreeView::mousePressEvent(QMouseEvent *e)
{
    if ( e->button() == MidButton ||
         e->button() == LeftButton)
	{
	    int row = findRow( e->pos().y() );
	    int col = findCol( e->pos().x() );
            
	    QListIterator<LogTreeItem> it(items);
	    for(; it.current(); ++it)
		if (it.current()->row == row
		    && it.current()->col == col)
		    {
			emit revisionClicked(it.current()->rev,
					     e->button() == MidButton);
			break;
		    }
	}
}


void LogTreeView::recomputeCellSizes ()
{
    int boxwidth, boxheight;
    int v = static_width;
    colWidths.fill(v, numCols());
    v = static_height;
    rowHeights.fill(v, numRows());
    
    QFontMetrics fm(fontMetrics());
    QListIterator<LogTreeItem> it(items);
    
    for (; it.current(); ++it)
        {
            LogTreeItem *item = it.current();
            
            computeBoxSize(fm, item->rev, item->author, item->taglist, &boxwidth, &boxheight);
            
            boxwidth += 2 * BORDER;
            boxheight += 2 * BORDER;
            
            if (boxwidth > colWidths[item->col])
                colWidths[item->col] = boxwidth;
            
            if (boxheight > rowHeights[item->row])
                rowHeights[item->row] = boxheight;
	}
    
    setAutoUpdate(true);
    updateTableSize();
    update();
}

void LogTreeView::computeBoxSize(QFontMetrics &fm, QString line1, QString line2, QString line3,
                                 int *outWidth, int *outHeight)
{
    QRect r = fm.boundingRect(0, 0, 0, 0, AlignCenter, line1);
    *outWidth = r.width();
    *outHeight = r.height() + 3 * INSPACE;
    
    if (!line2.isEmpty())
        {
            r = fm.boundingRect(0, 0, 0, 0, AlignCenter, line2);
            if (r.width() > *outWidth)
                *outWidth = r.width();
            *outHeight += r.height() + INSPACE;
        }
    
    r = fm.boundingRect(0, 0, 0, 0, AlignCenter, line3);
    if (r.width() > *outWidth)
        *outWidth = r.width();
    *outHeight += r.height();
    
    *outWidth += 2 * INSPACE;
}


int LogTreeView::cellWidth(int col)
{
    if (col < 0 || col >= (int)colWidths.size())
        return 0;
    
    return colWidths[col];
}

int LogTreeView::cellHeight(int row)
{
    if (row < 0 || row >= (int)rowHeights.size())
        return 0;
    
    return rowHeights[row];
}

void LogTreeTip::maybeTip(const QPoint &pos)
{
    if (!parentWidget()->inherits("LogTreeView"))
	return;

    LogTreeView *logtree = static_cast<LogTreeView*>(parentWidget());

    QRect r;
    QString s;
    logtree->lookupTip(pos, &r, &s);
    if (r.isValid())
        tip( r, s );
}

// Local Variables:
// c-basic-offset: 4
// End:
