/***************************************************************************
                          setiloc.cpp  -  description
                             -------------------
    begin                : Wed Oct 27 1999
    copyright            : (C) 1999 by Gordon Machel
    email                : gmachel@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include <kapp.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kiconloader.h>

#include <qmessagebox.h>
#include <qbitmap.h>
#include <qdir.h>
#include <qtextstream.h>
#include <qlistview.h>

#include <math.h>

#include "setiloc.h"
#include "analysis.h"
#include "datainfo.h"
#include "userinfo.h"
#include "csvdatabase.h"

extern bool HMS;
extern bool		NEWPEAK_RECORD;
extern bool		NEWPEAK_CURRENTWU;
extern WUScore Record;

struct csvHeaderField	{
											const char* text;
											int type;
											};

csvHeaderField csv_userinfo[] =   {
                                  {"date", 1},
                                  {"id", 0},
                                  {"key", 0},
                                  {"email_addr", 1},
                                  {"user_name", 1},
                                  {"url", 1},
                                  {"country", 1},
                                  {"postal_code", 0},
                                  {"show_name", 1},
                                  {"show_email", 1},
                                  {"venue", 0},
                                  {"register_time", 1},
                                  {"last_wu_time", 1},
                                  {"last_result_time", 1},
                                  {"nwus", 0},
                                  {"nresults", 0},
                                  {"total_cpu", 0},
                                  {"params_index", 0}
                                  };
#define CSV_USERINFO_COLUMNS 18

csvHeaderField csv_datainfo[] =   {
                                  {"task", 1},
                                  {"version", 0},
                                  {"name", 1},
                                  {"data_type", 1},
                                  {"data_class", 0},
                                  {"splitter_version", 1},
                                  {"start_ra", 0},
                                  {"start_dec", 0},
                                  {"end_ra", 0},
                                  {"end_dec", 0},
                                  {"angle_range", 0},
                                  {"time_recorded", 1},
                                  {"subband_center", 0},
                                  {"subband_base", 0},
                                  {"subband_sample_rate", 0},
                                  {"fft_len", 0},
                                  {"ifft_len", 0},
                                  {"subband_number", 0},
                                  {"receiver", 1},
                                  {"nsamples", 0},
                                  {"tape_version", 0}
                                  };
#define CSV_DATAINFO_COLUMNS 21

csvHeaderField csv_state[] =  {
                              {"ncfft", 0},
                              {"cr", 0},
                              {"fl", 0},
                              {"cpu", 0},
                              {"prog", 0},
                              {"bs_power", 0},
                              {"bs_score", 0},
                              {"bs_bin", 0},
                              {"bs_fft_ind", 0},
                              {"bs_chirp_rate", 0},
                              {"bs_fft_len", 0},
                              {"bg_score", 0},
                              {"bg_power", 0},
                              {"bg_chisq", 0},
                              {"bg_bin", 0},
                              {"bg_fft_ind", 0},
                              {"bg_chirp_rate", 0},
                              {"bg_fft_len", 0},
                              {"bp_score", 0},
                              {"bp_power", 0},
                              {"bp_mean", 0},
                              {"bp_period", 0},
                              {"bp_freq_bin", 0},
                              {"bp_time_bin", 0},
                              {"bp_chirp_rate", 0},
                              {"bp_fft_len", 0},
                              {"bt_score", 0},
                              {"bt_power", 0},
                              {"bt_mean", 0},
                              {"bt_period", 0},
                              {"bt_bperiod", 0},
                              {"bt_tpotind0_0", 0},  														
                              {"bt_tpotind0_1", 0},  														
                              {"bt_tpotind1_0", 0},  														
                              {"bt_tpotind1_1", 0},  														
                              {"bt_tpotind2_0", 0},  														
                              {"bt_tpotind2_1", 0},
                              {"bt_freq_bin", 0},
                              {"bt_time_bin", 0},
                              {"bt_chirp_rate", 0},
                              {"bt_scale", 0},
                              {"bt_fft_len", 0}  														
                              };
#define CSV_STATE_COLUMNS 42

/*------------------------------------------------------------------------ */
SetiLoc::SetiLoc(const QString& dir, const QString& loc, QColor col)
        : SetiContainer(dir)
{
sets.directory = dir;
sets.description = loc;
sets.color       = col;
setDirectory(dir);

for(int i=0;i<3;i++)
  {
	graphvsbl[i] = false;
	graphwidget[i] = 0;
	}
sresult_timestamp = "";

// this ensures that all data are read when checking for the first time
mymax.spike.power    = -1.0;
mymax.gaussian.score = -1.0;
mymax.pulse.score    = -1.0;
mymax.triplet.score  = -1.0;
	
for(int i=0; i<3; i++) listitem[i] = 0;
	
statusIcons = new KIconLoader();
status_icon[0] = statusIcons->loadIcon("mini-setistopped", KIcon::User);
status_icon[1] = statusIcons->loadIcon("mini-setirunning", KIcon::User);	
status_icon[2] = statusIcons->loadIcon("mini-setifinished", KIcon::User);
status_icon[3] = statusIcons->loadIcon("mini-setiloading", KIcon::User);
delete statusIcons;
	
connect(this, SIGNAL(progressIncreased()),
              SLOT(slotUpdateAnalysisTimeData()));
connect(this, SIGNAL(newSpike(SpikeScore)),
              SLOT(slotUpdateAnalysisSpikeData(SpikeScore)));	
connect(this, SIGNAL(newGaussian(GaussianScore)),
              SLOT(slotUpdateAnalysisGaussianData(GaussianScore)));	
connect(this, SIGNAL(newPulse(PulseScore)),
              SLOT(slotUpdateAnalysisPulseData(PulseScore)));	
connect(this, SIGNAL(newTriplet(TripletScore)),
              SLOT(slotUpdateAnalysisTripletData(TripletScore)));
connect(this, SIGNAL(stateChanged(int,int)),
              SLOT(slotShowNewState(int,int)));
connect(this, SIGNAL(newWorkUnit(WorkUnitData)),
              SLOT(slotInitMaxScore(WorkUnitData)));	
}

/*------------------------------------------------------------------------ */
SetiLoc::~SetiLoc()
{
}

/*------------------------------------------------------------------------ */
void SetiLoc::setOptions(const LocSettings& ls)
{
sets = ls;
sets.arguments = sets.arguments.simplifyWhiteSpace();
}

/*------------------------------------------------------------------------ */
void SetiLoc::updateListViews()
{
// this ensures that all data are read when checking for the first time
mymax.spike.power    = -1.0;
mymax.gaussian.score = -1.0;
mymax.pulse.score    = -1.0;
mymax.triplet.score  = -1.0;

slotUpdateAnalysisTimeData();
slotUpdateAnalysisSpikeData(spikeScore());
slotUpdateAnalysisGaussianData(gaussianScore());
slotUpdateAnalysisPulseData(pulseScore());
slotUpdateAnalysisTripletData(tripletScore());
// this triggers a refresh of the state icon
setClientState(-1);

slotUpdateWorkUnitData();

slotUpdateUserInfoData();
}

/*------------------------------------------------------------------------ */
QListViewItem* SetiLoc::addItem(int list, QListView* lv)
{
QListViewItem* it;

switch(list)
  {
  case AnalysisList:
	  it = new AnalysisListViewItem(lv);
    listitem[list] = it;
	  slotUpdateAnalysisTimeData();
	  slotUpdateAnalysisSpikeData(spikeScore());
	  slotUpdateAnalysisGaussianData(gaussianScore());
	  slotUpdateAnalysisPulseData(pulseScore());
	  slotUpdateAnalysisTripletData(tripletScore());
	  // this triggers a refresh of the state icon
	  setClientState(-1);
	  break;
  case DataInfoList:
  	it = new DataInfoListViewItem(lv);
    listitem[list] = it;
    slotUpdateWorkUnitData();
  	break;
  case UserInfoList:
  	it = new UserInfoListViewItem(lv);
    listitem[list] = it;
    slotUpdateUserInfoData();
  	break;
  default:
  	it = new QListViewItem(lv);
    listitem[list] = it;
  	break;
	}

return(it);
}

/*------------------------------------------------------------------------ */
void SetiLoc::removeItem(int list)
{
if(listitem[list]) delete listitem[list];
listitem[list] = 0;
}

/*------------------------------------------------------------------------ */
void SetiLoc::removeAllItems()
{
for(int i=0;i<3;i++) removeItem(i);
}

/*------------------------------------------------------------------------ */
void SetiLoc::slotUpdateAnalysisTimeData()
{
QString val;

// exit if listitem doesn't exist
if(listitem[AnalysisList])
  {
  // set location text
  listitem[AnalysisList]->setText(0, description());

  if(exists(SC_StateFile))
    {      	
    double prog(0.0);
    if(clientState() != Finished)
      {
      // set the actual chirp rate
      val.setNum(chirpRate(), 'f', 4);
      listitem[AnalysisList]->setText(2, val);
            		
      // set the progress in %
      prog = progress();
      val.setNum(prog, 'f', 3);
      listitem[AnalysisList]->setText(3, val);
      }

    // set the elapsed CPU time		
    double cputime = cpuTime();
    val = convertTime(cputime, HMS);
    listitem[AnalysisList]->setText(4, val);

    // calculate and set the progress rate
    val.setNum(progressRate(), 'f', 5);
    listitem[AnalysisList]->setText(6, val);

    // MFlops    		
    val.setNum(megaFlopsPerSecond(), 'f', 2);
    listitem[AnalysisList]->setText(7, val);
        		
    if(clientState() != Finished)
      {
      // calculate and set the time required to finish this work unit
      double timeleft = remainingTime();
      val = (timeleft > 0.0) ? convertTime(timeleft, HMS) : QString("");
      listitem[AnalysisList]->setText(5, val);
      }
    }
  else
    for(int i=2;i<=7;i++) listitem[AnalysisList]->setText(i, 0);
  }
}

/*------------------------------------------------------------------------ */
void SetiLoc::slotUpdateAnalysisSpikeData(SpikeScore newspike)
{
// Display a message if a new record is found.
if(NEWPEAK_RECORD)
	{
	if(newspike.power > Record.spike.power)
		{
		QString msgbuf, title;
		msgbuf.sprintf(i18n("You have found a new record peak:\n"
                        "peak = %.3lf at %.4lf Hz/s chirp rate,\n"
                        "found in work unit %s.\n\n"
                        "Previous peak:\n"
                        "peak = %.3lf at %.4lf Hz/s chirp rate,\n"
                        "found in work unit %s."),
                        newspike.power, newspike.chirprate,
                        (const char*)newspike.wu_name,
                        Record.spike.power, Record.spike.chirprate,
                        (const char*)Record.spike.wu_name);
    title = i18n("New Record Spike for %1").arg(description());			
    QMessageBox* mb = new QMessageBox(title, msgbuf, QMessageBox::Information,
												QMessageBox::Ok|QMessageBox::Default, 0, 0, 0, 0, false,
                        WStyle_DialogBorder|WDestructiveClose);
    mb->show();
    }
  }

// Display a message if a new high signal is found. Checking mymax for -1
// ensures that the message is not shown when starting Ksetiwatch.
if(NEWPEAK_CURRENTWU && mymax.spike.power != -1.0)
  {			
  if(newspike.power > mymax.spike.power &&
     newspike.power <= Record.spike.power)
    {
    QString msgbuf, title;
    msgbuf.sprintf(i18n("You have found a new high peak in the current WU:\n"
                        "peak = %.3lf at %.4lf Hz/s chirp rate.\n\n"
                        "Previous peak:\n"
                        "peak = %.3lf at %.4lf Hz/s chirp rate."),
                        newspike.power, newspike.chirprate,
                        mymax.spike.power, mymax.spike.chirprate);
    title = i18n("New High Spike for %1").arg(description());
    QMessageBox* mb = new QMessageBox(title, msgbuf, QMessageBox::Information,
                            QMessageBox::Ok|QMessageBox::Default, 0, 0, 0, 0, false,
                            WStyle_DialogBorder|WDestructiveClose);
    mb->show();
    }
	}

// update highscores
if(newspike.power > mymax.spike.power || mymax.spike.power == -1.0)
  {
	mymax.spike = newspike;
	}
if(newspike.power > Record.spike.power)
  {
	Record.spike = newspike;
	}

if(listitem[AnalysisList])
  {
  // set the highest peak so far
  QString val;
  if(mymax.spike.power > 0.0)
    {
    if(mymax.spike.power > 1.0e6)
      val.setNum(mymax.spike.power, 'e', 8);
    else    				
      val.setNum(mymax.spike.power, 'f', 3);
    }
  else
    val = "";
  listitem[AnalysisList]->setText(8, val);
  }
}

/*------------------------------------------------------------------------ */
void SetiLoc::slotUpdateAnalysisGaussianData(GaussianScore newgaussian)
{
// Display a message if a new record is found.
if(NEWPEAK_RECORD)
	{
	if(newgaussian.score > Record.gaussian.score)
		{
    QString msgbuf, title;
    msgbuf.sprintf(i18n("You have found a new record gaussian:\n"
      "score = %.4lf, power = %.2lf, fit = %.2lf at %.4lf Hz/s chirp rate,\n"
      "found in work unit %s.\n\n"
      "Previous gaussian:\n"
      "score = %.4lf, power = %.2lf, fit = %.2lf at %.4lf Hz/s chirp rate,\n"
      "found in work unit %s.\n"),
      newgaussian.score, newgaussian.power, newgaussian.chisq,
      newgaussian.chirprate, (const char*)newgaussian.wu_name,
      Record.gaussian.score, Record.gaussian.power, Record.gaussian.chisq,
      Record.gaussian.chirprate, (const char*)Record.gaussian.wu_name);
    title = i18n("New Record Gaussian for %1").arg(description());			
    QMessageBox* mb = new QMessageBox(title, msgbuf, QMessageBox::Information,
                    QMessageBox::Ok|QMessageBox::Default, 0, 0, 0, 0, false,
                    WStyle_DialogBorder|WDestructiveClose);
    mb->show();
    }
  }

// Display a message if a new high signal is found. Checking mymax for -1
// ensures that the message is not shown when starting Ksetiwatch.
if(NEWPEAK_CURRENTWU && mymax.gaussian.score != -1.0)
  {			
  if(newgaussian.score > mymax.gaussian.score &&
     newgaussian.score <= Record.gaussian.score)
	  {
    QString msgbuf, title;
    msgbuf.sprintf(i18n("You have found a new high gaussian in the current WU:\n"
      "score = %.4lf, power = %.2lf, fit = %.2lf at %.4lf Hz/s chirp rate.\n\n"
      "Previous gaussian:\n"
      "score = %.4lf, power = %.2lf, fit = %.2lf at %.4lf Hz/s chirp rate."),
      newgaussian.score, newgaussian.power, newgaussian.chisq,
      newgaussian.chirprate,
      mymax.gaussian.score, mymax.gaussian.power, mymax.gaussian.chisq,
      mymax.gaussian.chirprate);
    title = i18n("New High Gaussian for %1").arg(description());
    QMessageBox* mb = new QMessageBox(title, msgbuf, QMessageBox::Information,
                    QMessageBox::Ok|QMessageBox::Default, 0, 0, 0, 0, false,
                    WStyle_DialogBorder|WDestructiveClose);
    mb->show();
	  }
	}

// update highscores
if(newgaussian.score > mymax.gaussian.score || mymax.gaussian.score == -1.0)
  {
	mymax.gaussian = newgaussian;
	}
if(newgaussian.score > Record.gaussian.score)
  {
	Record.gaussian = newgaussian;
	}

if(listitem[AnalysisList])
  {
  // display the strongest gaussian according to its score value
  QString val;
  if(mymax.gaussian.score > 0.0)
    val.sprintf(" scr=%.4f, pwr=%.2f, fit=%.2f", mymax.gaussian.score,
                mymax.gaussian.power, mymax.gaussian.chisq);
  else
    val = "";
  listitem[AnalysisList]->setText(9, val);    	
  }
}

/*------------------------------------------------------------------------ */
void SetiLoc::slotUpdateAnalysisPulseData(PulseScore newpulse)
{
// Display a message if a new record is found.
if(NEWPEAK_RECORD)
	{
	if(newpulse.score > Record.pulse.score)
		{
    QString msgbuf, title;
    msgbuf.sprintf(i18n("You have found a new record pulse:\n"
      "score = %.4lf, power = %.2lf, period = %.4lf at %.4lf Hz/s chirp rate,\n"
      "found in work unit %s.\n\n"
      "Previous pulse:\n"
      "score = %.4lf, power = %.2lf, period = %.4lf at %.4lf Hz/s chirp rate,\n"
      "found in work unit %s.\n"),
      newpulse.score, newpulse.power, newpulse.period,
      newpulse.chirprate, (const char*)newpulse.wu_name,
      Record.pulse.score, Record.pulse.power, Record.pulse.period,
      Record.pulse.chirprate, (const char*)Record.pulse.wu_name);
      title = i18n("New Record Pulse for %1").arg(description());			
    QMessageBox* mb = new QMessageBox(title, msgbuf, QMessageBox::Information,
                        QMessageBox::Ok|QMessageBox::Default, 0, 0, 0, 0, false,
                        WStyle_DialogBorder|WDestructiveClose);
    mb->show();
    }
  }

// Display a message if a new high signal is found. Checking mymax for -1
// ensures that the message is not shown when starting Ksetiwatch.
if(NEWPEAK_CURRENTWU && mymax.pulse.score != -1.0)
  {			
  if(newpulse.score > mymax.pulse.score &&
     newpulse.score <= Record.pulse.score)
	  {
    QString msgbuf, title;
    msgbuf.sprintf(i18n("You have found a new high pulse in the current WU:\n"
      "score = %.4lf, power = %.2lf, period = %.4lf at %.4lf Hz/s chirp rate.\n\n"
      "Previous pulse:\n"
      "score = %.4lf, power = %.2lf, period = %.4lf at %.4lf Hz/s chirp rate."),
      newpulse.score, newpulse.power, newpulse.period,
      newpulse.chirprate,
      mymax.pulse.score, mymax.pulse.power, mymax.pulse.period,
      mymax.pulse.chirprate);
    title = i18n("New High Pulse for %1").arg(description());
    QMessageBox* mb = new QMessageBox(title, msgbuf, QMessageBox::Information,
                    QMessageBox::Ok|QMessageBox::Default, 0, 0, 0, 0, false,
                    WStyle_DialogBorder|WDestructiveClose);
    mb->show();
	  }
	}

// update highscores
if(newpulse.score > mymax.pulse.score || mymax.pulse.score == -1.0)
  {
	mymax.pulse = newpulse;
	}
if(newpulse.score > Record.pulse.score)
  {
	Record.pulse = newpulse;
	}

if(listitem[AnalysisList])
  {
  // display the strongest pulse according to its score value
  QString val;
  if(mymax.pulse.score > 0.0)
    val.sprintf(" scr=%.4f, pwr=%.2f, prd=%.4f", mymax.pulse.score,
                mymax.pulse.power, mymax.pulse.period);
  else
    val = "";
  listitem[AnalysisList]->setText(10, val);    	
  }
}

/*------------------------------------------------------------------------ */
void SetiLoc::slotUpdateAnalysisTripletData(TripletScore newtriplet)
{
// Display a message if a new record is found.
if(NEWPEAK_RECORD)
	{
	if(newtriplet.score > Record.triplet.score)
		{
    QString msgbuf, title;
    msgbuf.sprintf(i18n("You have found a new record triplet:\n"
      "score = %.4lf, period = %.4lf at %.4lf Hz/s chirp rate,\n"
      "found in work unit %s.\n\n"
      "Previous triplet:\n"
      "score = %.4lf, period = %.4lf at %.4lf Hz/s chirp rate,\n"
      "found in work unit %s.\n"),
      newtriplet.score, newtriplet.period, newtriplet.chirprate,
      (const char*)newtriplet.wu_name,
      Record.triplet.score, Record.triplet.period, Record.triplet.chirprate,
      (const char*)Record.triplet.wu_name);
    title = i18n("New Record Triplet for %1").arg(description());			
    QMessageBox* mb = new QMessageBox(title, msgbuf, QMessageBox::Information,
                  QMessageBox::Ok|QMessageBox::Default, 0, 0, 0, 0, false,
                  WStyle_DialogBorder|WDestructiveClose);
    mb->show();
    }
  }

// Display a message if a new high signal is found. Checking mymax for -1
// ensures that the message is not shown when starting Ksetiwatch.
if(NEWPEAK_CURRENTWU && mymax.triplet.score != -1.0)
  {			
  if(newtriplet.score > mymax.triplet.score &&
     newtriplet.score <= Record.triplet.score)
	  {
    QString msgbuf, title;
    msgbuf.sprintf(i18n("You have found a new high triplet in the current WU:\n"
      "score = %.4lf, period = %.4lf at %.4lf Hz/s chirp rate.\n\n"
      "Previous triplet:\n"
      "score = %.4lf, period = %.4lf at %.4lf Hz/s chirp rate."),
      newtriplet.score, newtriplet.period, newtriplet.chirprate,
      mymax.triplet.score, mymax.triplet.period, mymax.triplet.chirprate);
    title = i18n("New High Triplet for %1").arg(description());
    QMessageBox* mb = new QMessageBox(title, msgbuf, QMessageBox::Information,
                    QMessageBox::Ok|QMessageBox::Default, 0, 0, 0, 0, false,
                    WStyle_DialogBorder|WDestructiveClose);
    mb->show();
	  }
	}

// update highscores
if(newtriplet.score > mymax.triplet.score || mymax.triplet.score == -1.0)
  {
	mymax.triplet = newtriplet;
	}
if(newtriplet.score > Record.triplet.score)
  {
	Record.triplet = newtriplet;
	}

if(listitem[AnalysisList])
  {
  // display the strongest triplet according to its score value
  QString val;
  if(mymax.triplet.score > 0.0)
    val.sprintf(" scr=%.4f, prd=%.4f", mymax.triplet.score, mymax.triplet.period);
  else
    val = "";
  listitem[AnalysisList]->setText(11, val);    	
  }
}

/*------------------------------------------------------------------------ */
void SetiLoc::slotUpdateWorkUnitData()
{
if(listitem[DataInfoList])
  {
  QString val;
  int h, m, s;

  listitem[DataInfoList]->setText(0, description());

  if(exists(SC_WorkUnitFile))
    {
    double ra  = startRA();
    double dec = startDec();
    h = static_cast<int>(ra);
    m = static_cast<int>((ra - h)*60);
    s = static_cast<int>((ra - h)*3600 - m*60);
    val = (QString("%1 hr %2 min %3 sec RA ")).arg(h).arg(m).arg(s);
   	if(dec >= 0.0) val += '+'; else val += '-';
    dec = fabs(dec);
    h = static_cast<int>(dec);
    m = static_cast<int>((dec - h)*60);
    s = static_cast<int>((dec - h)*3600 - m*60);
    val += (QString(" %1 deg %2' %3'' Dec")).arg(h).arg(m).arg(s);
    listitem[DataInfoList]->setText(1, val);

    listitem[DataInfoList]->setText(2, timeRecordedString());

    val = receiver();
    if(val == "ao1420")
      listitem[DataInfoList]->setText(3, i18n("Arecibo Telescope "));
    else
      listitem[DataInfoList]->setText(3, 0);

    double freq = baseFrequency();
  	val = (QString("  %1 GHz")).arg(freq, 0, 'f', 9);
    listitem[DataInfoList]->setText(4, val);    	

    listitem[DataInfoList]->setText(5, val.setNum(angleRange()));
    listitem[DataInfoList]->setText(6,
                             val.setNum(teraFlops(angleRange()), 'f', 4));
    }
  else
    for(int i=1;i<=6;i++) listitem[DataInfoList]->setText(i, 0);
  }
}

/*------------------------------------------------------------------------ */
void SetiLoc::slotUpdateUserInfoData()
{
if(listitem[UserInfoList])
  {
  QString val;

  listitem[UserInfoList]->setText(0, description());

  if(exists(SC_UserInfoFile))
    {
    listitem[UserInfoList]->setText(1, userName());

    listitem[UserInfoList]->setText(2, val.setNum(numberOfResults()));

    val = convertTime(totalCPUTime(), HMS);
    listitem[UserInfoList]->setText(3, val);

    val = convertTime(averageCPUTime(), HMS);
    listitem[UserInfoList]->setText(4, val);

    listitem[UserInfoList]->setText(5, "  " + lastResultTimeString());
    listitem[UserInfoList]->setText(6, "  " + registerTimeString());
    }
  else
    for(int i=1;i<=6;i++) listitem[UserInfoList]->setText(i, 0);
  }
}

/*------------------------------------------------------------------------ */
void SetiLoc::slotInitMaxScore(WorkUnitData wud)
{
initWUScore(&mymax);

// update work unit and user info
slotUpdateAnalysisTimeData();
slotUpdateWorkUnitData();
slotUpdateUserInfoData();

// Hide the old graph widgets if present; implement a more elegant way
// later.
for(int i=0;i<3;i++)
  {
  if(graphvsbl[i])
    {
    graphwidget[i]->hide();
	  graphvsbl[i] = false;
	  }
	}

//...and notify graph widgets with extended information
emit newWorkUnit(wud, description(), color());
}

/*------------------------------------------------------------------------ */
void SetiLoc::slotShowNewState(int state, int loadsize)
{
int cw(28);

// log the WU if necessary
if(state == Finished)
  {
  checkWUStatus();
  }

if(listitem[AnalysisList])
	{
	// first change the icon
  // center the pixmap in the column (why can't Qt handle this?)
  cw = listitem[AnalysisList]->listView()->columnWidth(1);
  if(cw < 28) cw = 28;

  // For unknown reasons, I have to subtract 5 pixels to prevent the status
  // column width from increasing steadily when switching between the
  // combobox items.
  cw -= 5;

  QPixmap stpix(cw, 16);
  QPainter p(&stpix);
  p.fillRect(0, 0, cw, 16, white);
  p.drawPixmap((cw-status_icon[state].width())/2, 0, status_icon[state]);
  p.end();
  		
  stpix.setMask( stpix.createHeuristicMask() );
  listitem[AnalysisList]->setPixmap(1, stpix);

  // modify the columns in accordance to the new state
  switch(state)
    {
    case Finished:
      {
      listitem[AnalysisList]->setText(2, 0);
      listitem[AnalysisList]->setText(3, "100.000");
      listitem[AnalysisList]->setText(5, 0);
      break;
      }
    case Loading:
      {
      for(int i=2;i<12;i++) listitem[AnalysisList]->setText(i, 0);

      QString buf;
      buf.sprintf("%d %%", loadsize);
      listitem[AnalysisList]->setText(3, buf);  	
      break;
      }
    }
  }
}

/*------------------------------------------------------------------------ */
/** Checks for a completed work unit */
void SetiLoc::checkWUStatus()
{
if(sets.log == true)
  {
  if(exists(SC_WorkUnitFile) == false)
    {
    // Work unit file doesn't exist; this is an indication of a completed
    // work unit.
    if(exists(SC_ResultFile) && exists(SC_ResultHeaderFile) == false)
      {
      // Result file exists, while result_header is gone; we're getting
      // closer. In the following, we're using the timestamp of the result
      // file to prevent multiple logging.
      QFileInfo fi(rsfFileName);
      QDateTime dt = QDateTime(fi.lastModified());
      QString res_dt = dt.toString();
      if(res_dt != timestamp())
        {
        setTimestamp(res_dt);
        logResults();
        logWorkUnit();
        // Save the config to make sure that the timestamp is written to
        // disk.
        emit workUnitLogged(this);
        }
      }
    }
  }
}

/*------------------------------------------------------------------------ */
/** Returns the log directory in a string. */
QString SetiLoc::logDirectory()
{
QString logdir;

// determine the right log directory
if(sets.redirectLog)
  {
  if(sets.logDirectory.isEmpty())
    logdir = directory();
  else
    logdir = sets.logDirectory;
  }
else
  {
  logdir = directory();
  }

return(logdir);
}

/*------------------------------------------------------------------------ */
/** Writes all important work unit results to a file named SETILog.csv */
void SetiLoc::logWorkUnit()
{
QString buffer, entry;
int i;
int error(0);
int warn(0);
QString setilog_file;

setilog_file = logDirectory() + "/SETILog.csv";
QFile f_sel(setilog_file);

// check if SETILog.csv needs to be upgraded
CSVDataBase csv(setilog_file);
bool upgrade = false;
if(f_sel.exists())
	{
	if(csv.open(IO_ReadOnly))
		{
		int cols = csv.columnCount();
		if(cols < CSV_USERINFO_COLUMNS+CSV_DATAINFO_COLUMNS+CSV_STATE_COLUMNS)
			{
			// rename SETILog.csv to SETILog.old
			if(0 != rename((const char*)setilog_file,
			               (const char*)(logDirectory()+"/SETILog.old")))
				{
				error = 3;
				debug("Renaming SETILog.csv failed.");
				}
			else
				{
				upgrade = true;
				}
			}
		csv.close();
  	}
  }

// create new SETILog.csv if necessary
if(f_sel.exists() == false && error == 0)
	{
	if(f_sel.open(IO_ReadWrite))
		{
		// write the definition line
	  for(i=0;i<CSV_USERINFO_COLUMNS;i++)
	  	{
	  	f_sel.putch('"');
	  	f_sel.writeBlock(csv_userinfo[i].text, strlen(csv_userinfo[i].text));
	  	f_sel.putch('"');	f_sel.putch(',');
	  	}
	  for(i=0;i<CSV_DATAINFO_COLUMNS;i++)
	  	{
	  	f_sel.putch('"');
	  	f_sel.writeBlock(csv_datainfo[i].text, strlen(csv_datainfo[i].text));
	  	f_sel.putch('"');	f_sel.putch(',');
	  	}
	  for(i=0;i<CSV_STATE_COLUMNS;i++)
	  	{
	  	f_sel.putch('"');
	  	f_sel.writeBlock(csv_state[i].text, strlen(csv_state[i].text));
			f_sel.putch('"');
	  	if(i < CSV_STATE_COLUMNS-1) f_sel.putch(',');
	  	}
	  // If the upgrade flag is set copy the data from SETILog.old to the new
	  // file.
	  if(upgrade)
	  	{
			QFile oldf_sel(logDirectory()+"/SETILog.old");
			if(oldf_sel.open(IO_ReadWrite))
				{
				QTextStream tout(&oldf_sel);
				QTextStream tin(&f_sel);
				QString s;
				i = 0;
				while(!oldf_sel.atEnd())
					{
					s = tout.readLine();
					// discard the first line
					if(i > 0) tin << endl << s;
					i++;
					}
				oldf_sel.close();
				}
			else
				{
				error = 4;
				debug("Could not create new SETILog.csv.");
				}
			}
	  f_sel.close();
	  }
	}	

QFileInfo finfo(rsfFileName);
QDateTime dt = finfo.lastModified();
if(dt.isValid())
	{
	QDate d = dt.date();
	buffer.sprintf("\"%d.%02d.%02d ", d.year(), d.month(), d.day() );
	QTime t = dt.time();
	buffer += t.toString() + "\",";
	}
else
	{
	QDate d = QDate::currentDate();
	buffer.sprintf("\"%d.%02d.%02d ", d.year(), d.month(), d.day() );
	QTime t = QTime::currentTime();
	buffer += t.toString() + "\",";
	}

// Read the user info information.
if(exists(SC_UserInfoFile) && error == 0)
	{
	for(i=1;i<CSV_USERINFO_COLUMNS;i++)
		{
  	if(i == 4) // instead of "user_name" take "name" to retrieve the users's name
  		entry = readEntry(SC_UserInfoFile, "name");
  	else
  		entry = readEntry(SC_UserInfoFile, csv_userinfo[i].text);
    buffer += constructEntry(entry, csv_userinfo[i].type);
		}
  }
else
	error = 1;

// Read the result information.
if(exists(SC_ResultFile) && error == 0)
	{
	for(i=0;i<CSV_DATAINFO_COLUMNS;i++)
		{
		if(i == 1) // version
			{
			entry.sprintf("%.2f,", (double)clientVersion()/100.0);
			buffer += entry;
			}
		else
		  {
		  entry = readEntry(SC_ResultFile, csv_datainfo[i].text);
			buffer += constructEntry(entry, csv_datainfo[i].type);
			}
		}
	}
else
  if(error == 0) error = 2;

// Read the state file information. If state file could not be found
// issue a warning, but try it anyway (maybe we have some data in memory).
if(exists(SC_StateFile) == false) warn = 1;
if(error == 0)
  {
  for(i=0;i<CSV_STATE_COLUMNS;i++)
    {
    if(strcmp(csv_state[i].text, "cpu") == 0 && warn == 1)
      {
      // try to get at least cpu time from result.sah
      entry = readEntry(rsfFileName, "cpu_time");
      }
    else if(strcmp(csv_state[i].text, "prog") == 0 && warn == 1)
      {
      // assume progress is 100 %
      entry = "0.99994457";
      }
    else
      entry = readEntry(SC_StateFile, csv_state[i].text);
    buffer += constructEntry(entry, csv_state[i].type);
    }
	buffer = buffer.left(buffer.length()-1);
	}

if(error == 0)
	{			
  if(f_sel.open(IO_Append | IO_WriteOnly))
  	{
    QTextStream tlog(&f_sel);
  	tlog << endl << buffer;
  	f_sel.close();
  	
  	if(warn)
  		{
  		QString buf;
  		buf = i18n("Ksetiwatch has logged the work unit for \"%1\" without\n"
            "existing 'state.sah' file. Some values might therefore be wrong\n"
            "or out of date. Consider adding the '-stop_after_process' \n"
            "command line option when starting the SETI@home client.").arg(description());
 			QMessageBox* mb = new QMessageBox(i18n("Warning"), buf,
                                        QMessageBox::Information,
                                        QMessageBox::Ok|QMessageBox::Default,
                                        0, 0, 0, 0, false,
                                        WStyle_DialogBorder|WDestructiveClose);
      mb->show();
  		}
  	}
  }
else
	{
	buffer = i18n("Missing file(s). Logging of work unit failed.\n"
	              "Error code: %1").arg(error);
	KMessageBox::error(0, buffer);
	}
}

/*------------------------------------------------------------------------ */
/** constructs a string from readEntry  */
QString SetiLoc::constructEntry(const QString& entry, int type)
{
QString val = entry;

if(type == 0)
	{
	val.append(",");
	}
if(type == 1)
	{
	if(!val.isEmpty())
		{
		val.insert(0, '\"');
		val.append("\",");
		}
	else
		{
		val.append(",");
		}
	}

return(val);
}

/*------------------------------------------------------------------------ */
void SetiLoc::logResults()
{
QFile resFile(rsfFileName);
if(resFile.open(IO_ReadOnly))
  {
  QString reslog_file;

  reslog_file = logDirectory() + "/SETIResult.log";
  QFile resLog(reslog_file);

  // used to prevent empty line at the beginning
  bool logEx = resLog.exists();
  if(resLog.open(IO_Append | IO_WriteOnly))
    {
    QTextStream tin(&resLog);
    QTextStream tout(&resFile);
    QString line;
    // take the WU name as group name
    if(logEx) tin << endl;
    tin << "[" << readEntry(SC_ResultFile, "name") << "]";
    // skip past the header
    while(line.find("end_seti_header") != 0 && !resFile.atEnd())
      line = tout.readLine();
    // now scan for signal lines
    int sCnt = 1, gCnt = 1, pCnt = 1, tCnt = 1;
    int idlen = strlen("gaussian:");
    QString id;
    QString key;
    while(!resFile.atEnd())
      {
      line = tout.readLine();
      // take the left part as identifier
      id = line.left(idlen);
      if(id.find("spike:") == 0)
        {
        key.sprintf("spike%02d=", sCnt);
        tin << endl << key << line.mid(strlen("spike:"));
        sCnt++;
        }
      else if(id.find("gaussian:") == 0)
        {
        key.sprintf("gaussian%02d=", gCnt);
        tin << endl << key << line.mid(strlen("gaussian:"));
        gCnt++;
        }
      else if(id.find("pulse:") == 0)
        {
        key.sprintf("pulse%02d=", pCnt);
        tin << endl << key << line.mid(strlen("pulse:"));
        pCnt++;
        }
      else if(id.find("triplet:") == 0)
        {
        key.sprintf("triplet%02d=", tCnt);
        tin << endl << key << line.mid(strlen("triplet:"));
        tCnt++;
        }
      }
    resLog.close();
    }
  resFile.close();
  }
}

/*------------------------------------------------------------------------ */
bool SetiLoc::startClient()
{
bool ret;

// take care of the -stop_after_send option
QFile sas(directory() + "/stop_after_send.txt");
if(sets.stopAfterSend)
  {
  if(sas.exists() == false)
    {
    sas.open(IO_WriteOnly);
    sas.close();
    }
  }
else
  {
  if(sas.exists())
    {
    sas.remove();
    }
  }

client.clearArguments();
client << (directory() + "/setiathome");

// check the settings of the check boxes
if(sets.stopAfterProcess) client << "-stop_after_process";
if(sets.email) client << "-email";
if(sets.graphics) client << "-graphics";
// and now any text in the arguments field
if(sets.arguments.isEmpty() == false)
	{
	int p1(0), p2;
	do
		{
		p2 = sets.arguments.find(' ', p1);
		if(p2 >= 0)
			{
			client << sets.arguments.mid(p1, p2-p1);
			p1 = p2+1;
			}
		} while(p2 != -1);
	client << sets.arguments.right(sets.arguments.length()-p1);
	}
// we have to change to the setiathome directory to properly start the client
QString curdir = QDir::currentDirPath();
QDir::setCurrent(directory());
client.start(KProcess::DontCare, KProcess::NoCommunication);
QDir::setCurrent(curdir);
// wait a tenth of a sec
usleep(100000);
// and check if the process is running
ret = isClientRunning(client.getPid());
if(ret == true)
	{
	cltPid = (pid_t)client.getPid();
	}

return(ret);
}

/*------------------------------------------------------------------------ */
int SetiLoc::stopClient()
{
int ret(-1);

if(cltPid > 0)
	{
	ret = ::kill(cltPid, SIGTERM);
	}

return(ret);
}

#include "setiloc.moc"

