/***************************************************************************
                          seticontainer.cpp  -  version 0.3 - 2001.02.25
                             -------------------
    begin                : Fri Nov 17 2000
    copyright            : (C) 2000 by Gordon Machel
    email                : gmachel@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include <math.h>

#include <qfile.h>
#include <qfileinfo.h>
#include <qdatetime.h>
#include <qtextstream.h>

#include "seticontainer.h"

// needed to prevent rounding errors when comparing doubles
#define DIGITS (double)1.0e-6

// set this to 1 for debug output
#define SC_DEBUG 0

/*------------------------------------------------------------------------ */
SetiContainer::SetiContainer(const QString& dir, int refresh)
{
// store refresh interval in ms
refreshIval   = refresh*1000;

setDirectory(dir);

// client state is undefined
cltState = 0;
cltPid   = 0;

// initialize oldProgress variable for passive monitoring
oldProgress = stfData.prog;

// start the timer in order to periodically refresh the data
refTimer = startTimer(refreshIval);

// start timers that check periodically the state of the client
sttTimer1 = startTimer(500);   // every 500 ms
sttTimer2 = startTimer(120000); // every 2 minutes
}

/*------------------------------------------------------------------------ */
SetiContainer::~SetiContainer()
{
}

/*------------------------------------------------------------------------ */
void SetiContainer::setDirectory(const QString& dir)
{
// make a back-up
setiDirectory = dir;

// read the version number of the client
sahVersion = readClientVersion();

stfFileName = dir + "/state.sah";
wufFileName = dir + "/work_unit.sah";
uifFileName = dir + "/user_info.sah";
rsfFileName = dir + "/result.sah";
rhdFileName = dir + "/result_header.sah";
wtpFileName = dir + "/wtemp.sah";

initAllData();
}

/*------------------------------------------------------------------------ */
void SetiContainer::initAllData()
{
// fill all file data members with zeros
initStateFileData();
initWorkUnitData();
initUserInfoData();

// read all data
updateSahData(false);
}

/*------------------------------------------------------------------------ */
void SetiContainer::setRefreshInterval(int refresh)
{
refreshIval = refresh*1000; // in ms

// restart the timer with the new refresh interval
killTimer(refTimer);
refTimer = startTimer(refreshIval);
}

/*------------------------------------------------------------------------ */
void SetiContainer::timerEvent(QTimerEvent* e)
{
if(e->timerId() == refTimer)
  updateSahData();
else if(e->timerId() == sttTimer1)
  checkClientState();
else if(e->timerId() == sttTimer2)
  checkClientStatePassive();
}

/*------------------------------------------------------------------------ */
int SetiContainer::updateSahData(bool sig)
{
int err;

err = readUserInfoFile();
if(err == 0)
  scanUserInfoData();
else
  {
  #if SC_DEBUG
    debug("Read Error: user_info.sah");
  #endif
  }

err = readWorkUnitFile();
if(err == 0)
  scanWorkUnitData();
else
  {
  #if SC_DEBUG
    debug("Read Error: work_unit.sah");
  #endif
  }

err = readStateFile();
if(err == 0)
  scanStateFile(sig);
else
  {
  #if SC_DEBUG
    debug("Read Error: state.sah");
  #endif
  }

return(err);
}

/*------------------------------------------------------------------------ */
int SetiContainer::readSahFile(const QString& file, QStringList& list, const QString& endId)
{
int ret(0);
QFile sah(file);

// clear the list first
list.clear();

if(sah.open(IO_ReadOnly))
  {
  QTextStream t(&sah);
  QString s;
  while( !t.atEnd() )
    {
    s = t.readLine();
    // if reading a line fails or the eof identifier is found stop here
    if (s.isEmpty() || (endId && endId == s)) break;
    list.append(s);
    }
  sah.close();
  }
else
  {
  ret = 1;
  }

return(ret);
}

/*------------------------------------------------------------------------ */
int SetiContainer::readStateFile()
{
int ret(0);

// Return with error if file couldn't be found.
if(QFile::exists(stfFileName) == false) return(1);

// Read the time stamp of the file...
QFileInfo fi(stfFileName);
QDateTime dt = QDateTime(fi.lastModified());
QString   ts = dt.toString();
// ...and compare it with the stored one.
if(ts != stfTimeStamp)
  {
  stfTimeStamp = ts;
  ret = readSahFile(stfFileName, stateFile);
  }

return(ret);
}

/*------------------------------------------------------------------------ */
int SetiContainer::readWorkUnitFile()
{
int ret(0);

// Return with error if file couldn't be found.
if(QFile::exists(wufFileName) == false) return(1);

// Read the time stamp of the file...
QFileInfo fi(wufFileName);
QDateTime dt = QDateTime(fi.lastModified());
QString   ts = dt.toString();
// ...and compare it with the stored one.
if(ts != wufTimeStamp)
  {
  wufTimeStamp = ts;
  ret = readSahFile(wufFileName, workunitFile, "end_seti_header");
  }

return(ret);
}

/*------------------------------------------------------------------------ */
int SetiContainer::readUserInfoFile()
{
int ret(0);

// Return with error if file couldn't be found.
if(QFile::exists(uifFileName) == false) return(1);

// Read the time stamp of the file...
QFileInfo fi(uifFileName);
QDateTime dt = QDateTime(fi.lastModified());
QString   ts = dt.toString();
// ...and compare it with the stored one.
if(ts != uifTimeStamp)
  {
  uifTimeStamp = ts;
  ret = readSahFile(uifFileName, userinfoFile);
  }

return(ret);
}

/*------------------------------------------------------------------------ */
QString SetiContainer::readEntry(int sah, const QString& e)
{
QString val;

switch(sah)
  {
  case SC_StateFile:
    val = readEntry(stfFileName, e);
    break;
  case SC_UserInfoFile:
    val = readEntry(uifFileName, e);
    break;
  case SC_WorkUnitFile:
    val = readEntry(wufFileName, e);
    break;
  case SC_ResultFile:
    val = readEntry(rsfFileName, e);
    break;
  default:
    val = "";
    break;
  }

return(val);
}


/*------------------------------------------------------------------------ */
QString SetiContainer::readEntry(const QString& fn, const QString& e)
{
QString l,s;
QString v("");
int eq(-1);
bool found(false);
QFile sah(fn);

if(sah.open(IO_ReadOnly))
  {
  QTextStream t(&sah);
  while( !t.eof() )
    {
    s = t.readLine();
    // the next line is to prevent hangs when monitoring NFS directories
    // (suggested by Frode Tenneboe)
    if (s.isEmpty()) break;
    eq = s.find('=');
    if(eq != -1)
      {
      l = s.left(eq);
      if(l == e) {found = true; break;}
      }
    }
  if(found) v = s.right( s.length()-eq-1 );
  sah.close();
  }
	
return( v );
}
						
/*------------------------------------------------------------------------ */
QString SetiContainer::readEntry(QStringList& list, const QString& e)
{
QString v("");
QString s;
int pos;
bool found(false);

QStringList::Iterator it;
for(it=list.begin(); it!=list.end(); ++it)
  {
  // copy the contents of the iterator to a string
  s = (*it);
  pos = s.find(e);
  // accept the found entry only if it is at position 0
  if(pos == 0)
    {
    found = true;
    break;
    }
  }

if(found)
  {
  // find the equal char
  pos = s.find('=');
  // and take the part to the right
  if(pos > 0) v = s.right(s.length()-pos-1);
  }

return(v);
}

/*------------------------------------------------------------------------ */
void SetiContainer::initWUScore(WUScore* score)
{
score->spike.power     = 0.0;
score->spike.score     = 0.0;
score->spike.bin       = 0;
score->spike.fft_index = 0;
score->spike.fft_len   = 0;
score->spike.chirprate = 0.0;
score->spike.ra        = 0.0;
score->spike.dec       = 0.0;
score->spike.wu_name   = "";

score->gaussian.score     = 0.0;
score->gaussian.power     = 0.0;
score->gaussian.bin       = 0;
score->gaussian.chisq     = 0.0;
score->gaussian.true_mean = 0.0;
score->gaussian.sigma     = 0.0;
score->gaussian.fft_index = 0;
score->gaussian.fft_len   = 0;
score->gaussian.chirprate = 0.0;
score->gaussian.ra        = 0.0;
score->gaussian.dec       = 0.0;
score->gaussian.wu_name   = "";
//for(int i=0;i<64;i++) score->gaussian.data[i] = 0.0;

score->pulse.score     = 0.0;
score->pulse.power     = 0.0;
score->pulse.mean      = 0.0;
score->pulse.period    = 0.0;
score->pulse.freq_bin  = 0;
score->pulse.time_bin  = 0;
score->pulse.fft_len   = 0;
score->pulse.chirprate = 0.0;
score->pulse.ra        = 0.0;
score->pulse.dec       = 0.0;
score->pulse.wu_name   = "";

score->triplet.score      = 0.0;
score->triplet.power      = 0.0;
score->triplet.mean       = 0.0;
score->triplet.period     = 0.0;
score->triplet.bperiod    = 0.0;
score->triplet.tpotind0_0 = 0;
score->triplet.tpotind0_1 = 0;
score->triplet.tpotind1_0 = 0;
score->triplet.tpotind1_1 = 0;
score->triplet.tpotind2_0 = 0;
score->triplet.tpotind2_1 = 0;
score->triplet.freq_bin   = 0;
score->triplet.time_bin   = 0.0;
score->triplet.scale      = 0.0;
score->triplet.fft_len    = 0;
score->triplet.chirprate  = 0.0;
score->triplet.ra         = 0.0;
score->triplet.dec        = 0.0;
score->triplet.wu_name    = "";
}

/*------------------------------------------------------------------------ */
void SetiContainer::initStateFileData()
{
stfData.ncfft       = 0;
stfData.cr          = 0.0;
stfData.fl          = 0;
stfData.cpu         = 0.0;
stfData.prog        = 0.0;
stfData.potfreq     = 0;
stfData.potactivity = 0;
stfData.outfilepos  = 0;
initWUScore(&stfData.max);
}

/*------------------------------------------------------------------------ */
void SetiContainer::initWorkUnitData()
{
wufData.task                = "";
wufData.version             = 0;;
wufData.name                = "";
wufData.data_type           = "";
wufData.data_class          = 0;
wufData.splitter_version    = "";
wufData.start_ra            = 0.0;
wufData.start_dec           = 0.0;
wufData.end_ra              = 0.0;
wufData.end_dec             = 0.0;
wufData.angle_range         = 0.0;
wufData.time_recorded       = "";
wufData.subband_center      = 0.0;
wufData.subband_base        = 0.0;
wufData.subband_sample_rate = 0.0;
wufData.fft_len             = 0;
wufData.ifft_len            = 0;
wufData.subband_number      = 0;
wufData.receiver            = "";
wufData.nsamples            = 0;
wufData.tape_version        = "";
wufData.num_positions       = 0;
wufData.coordinates.clear();
}

/*------------------------------------------------------------------------ */
void SetiContainer::initUserInfoData()
{
uifData.id               = 0;
uifData.key              = 0;
uifData.email_addr       = "";
uifData.name             = "";
uifData.url              = "";
uifData.country          = "";
uifData.postal_code      = 0;
uifData.show_name        = false;
uifData.show_email       = false;
uifData.venue            = 0;
uifData.register_time    = "";
uifData.last_wu_time     = "";
uifData.last_result_time = "";
uifData.nwus             = 0;
uifData.nresults         = 0;
uifData.total_cpu        = 0.0;
uifData.params_index     = 0;
}

/*------------------------------------------------------------------------ */
void SetiContainer::scanStateFile(bool sig)
{
QString val;

val = readEntry(stateFile, "ncfft");
stfData.ncfft = val.toInt();

val = readEntry(stateFile, "cr");
stfData.cr = val.toDouble();

val = readEntry(stateFile, "fl");
stfData.fl = val.toInt();

val = readEntry(stateFile, "cpu");
stfData.cpu = val.toDouble();

double oldprog = stfData.prog;
val = readEntry(stateFile, "prog");
stfData.prog = val.toDouble();
double diff = fmod(stfData.prog-oldprog, DIGITS);
if(diff > 0.0 && sig)
  {
  emit progressIncreased();
  if(cltState != Running)
    {
    cltState = Running;
    emit stateChanged(cltState);
    }
  }
if(diff < 0.0 && sig) emit progressDecreased();

val = readEntry(stateFile, "potfreq");
stfData.potfreq = val.toInt();

val = readEntry(stateFile, "potactivity");
stfData.potactivity = val.toInt();

val = readEntry(stateFile, "outfilepos");
stfData.outfilepos = val.toInt();

scanSpikeData(sig);
scanGaussianData(sig);
scanPulseData(sig);
scanTripletData(sig);

}

/*------------------------------------------------------------------------ */
void SetiContainer::scanSpikeData(bool sig)
{
QString val;

// read the highest peak so far    				
double oldspike = stfData.max.spike.power;
val = readEntry(stateFile, "bs_power");
stfData.max.spike.power = val.toDouble();

double diff = fmod(stfData.max.spike.power-oldspike, DIGITS);
if(diff > 0.0)
  { 		
  // read the binary value of the highest peak
  val = readEntry(stateFile, "bs_bin");
  stfData.max.spike.bin = val.toInt();

  // read the corresponding chirp rate of the highest peak    				
  val = readEntry(stateFile, "bs_chirp_rate");
  stfData.max.spike.chirprate = val.toDouble();

  // read the fft index of the highest peak
  val = readEntry(stateFile, "bs_fft_ind");
  stfData.max.spike.fft_index = val.toInt();
		
  // read the fft length of the highest peak
  val = readEntry(stateFile, "bs_fft_len");
  stfData.max.spike.fft_len = val.toInt();
		
  // read the score of the highest peak
  val = readEntry(stateFile, "bs_score");
  stfData.max.spike.score = val.toDouble();

  stfData.max.spike.wu_name = wuName();

  if(sig) emit newSpike(stfData.max.spike);
  }
}		

/*------------------------------------------------------------------------ */
void SetiContainer::scanGaussianData(bool sig)
{
QString val;
		
// read the strongest gaussian according to its score value
double oldgaussian = stfData.max.gaussian.score;
val = readEntry(stateFile, "bg_score");
stfData.max.gaussian.score = val.toDouble();

double diff = fmod(stfData.max.gaussian.score-oldgaussian, DIGITS);
if(diff > 0.0)
  {
  // read the binary value of the highest gaussian
  val = readEntry(stateFile, "bg_bin");
  stfData.max.gaussian.bin = val.toInt();

  // read the gaussian power
  val = readEntry(stateFile, "bg_power");
  stfData.max.gaussian.power = val.toDouble();

  // read the gaussian chi square
  val = readEntry(stateFile, "bg_chisq");
  stfData.max.gaussian.chisq = val.toDouble();

  // read the fft length of the highest gaussian
  val = readEntry(stateFile, "bg_fft_len");
  stfData.max.gaussian.fft_len = val.toInt();
						
  // read the corresponding chirp rate of the highest gaussian    				
  val = readEntry(stateFile, "bg_chirp_rate");
  stfData.max.gaussian.chirprate = val.toDouble();

  // read the sigma value of the top Gaussian
  val = readEntry(stateFile, "bg_sigma");
  stfData.max.gaussian.sigma = val.toDouble();

  // read the mean value of the top Gaussian
  val = readEntry(stateFile, "bg_true_mean");
  stfData.max.gaussian.true_mean = val.toDouble();

  // read the fft index of the top Gaussian
  val = readEntry(stateFile, "bg_fft_ind");
  stfData.max.gaussian.fft_index = val.toInt();

  // read the data for the Gaussian graph
  scanGaussianGraphData();

  stfData.max.gaussian.wu_name = wuName();

  if(sig) emit newGaussian(stfData.max.gaussian);
  }
}

/*------------------------------------------------------------------------ */
void SetiContainer::scanGaussianGraphData()
{
QString e;
QString val;

for(int i=0;i<=63;i++)
  {
  val = readEntry(stateFile, (e = "bg_pot %1").arg(i));
  stfData.max.gaussian.data[i] = val.toDouble();
  }
}

/*------------------------------------------------------------------------ */
void SetiContainer::scanPulseData(bool sig)
{
QString val;

// read the score of the top pulse
double oldpulse = stfData.max.pulse.score;
val = readEntry(stateFile, "bp_score");
stfData.max.pulse.score = val.toDouble();
				
double diff = fmod(stfData.max.pulse.score-oldpulse, DIGITS);
if(diff > 0.0)
  {
  // read the power of the top pulse
  val = readEntry(stateFile, "bp_power");
  stfData.max.pulse.power = val.toDouble();
			
  // read the mean value of the top pulse
  val = readEntry(stateFile, "bp_mean");
  stfData.max.pulse.mean = val.toDouble();

  // read the period of the top pulse
  val = readEntry(stateFile, "bp_period");
  stfData.max.pulse.period = val.toDouble();
			
  // read the chirp rate of the top pulse
  val = readEntry(stateFile, "bp_chirp_rate");
  stfData.max.pulse.chirprate = val.toDouble();

  // read the freq_bin value of the top pulse
  val = readEntry(stateFile, "bp_freq_bin");
  stfData.max.pulse.freq_bin = val.toInt();

  // read the time_bin value of the top pulse
  val = readEntry(stateFile, "bp_time_bin");
  stfData.max.pulse.time_bin = val.toInt();

  // read the fft length of the top pulse
  val = readEntry(stateFile, "bp_fft_len");
  stfData.max.pulse.time_bin = val.toInt();

  // read the graph data
  if(scanPulseGraphData() == false)
    debug("Pulse data could not be read.");

  stfData.max.pulse.wu_name = wuName();

  if(sig) emit newPulse(stfData.max.pulse);
  }
}

/*------------------------------------------------------------------------ */
bool SetiContainer::scanPulseGraphData()
{
bool status = false;
char byte[2];        // stores a byte value

QString buf = readEntry(stateFile, "bp_pot");
if(!buf.isEmpty())
  {
  QTextStream ts(&buf, IO_ReadOnly);
  int val;
  int i = 0;
  while(!ts.atEnd() && i < 512)
    {
    ts >> byte[0] >> byte[1];
    sscanf(byte, "%x", &val);
    stfData.max.pulse.data[i] = (unsigned short int)val;
    i++;
    }
  status = true;
  }

return(status);
}

/*------------------------------------------------------------------------ */
void SetiContainer::scanTripletData(bool sig)
{
QString val;

// read the score of the top triplet
double oldtriplet = stfData.max.triplet.score;
val = readEntry(stateFile, "bt_score");
stfData.max.triplet.score = val.toDouble();

double diff = fmod(stfData.max.triplet.score-oldtriplet, DIGITS);
if(diff > 0.0)
  {
  // read the power of the top triplet
  val = readEntry(stateFile, "bt_power");
  stfData.max.triplet.power = val.toDouble();

  // read the mean value of the top pulse
  val = readEntry(stateFile, "bt_mean");
  stfData.max.triplet.mean = val.toDouble();

  // read the period of the top pulse
  val = readEntry(stateFile, "bt_period");
  stfData.max.triplet.period = val.toDouble();
			
  // read the chirp rate of the top pulse
  val = readEntry(stateFile, "bt_chirp_rate");
  stfData.max.triplet.chirprate = val.toDouble();
			
  // read the index values
  val = readEntry(stateFile, "bt_tpotind0_0");
  stfData.max.triplet.tpotind0_0 = val.toInt();
  val = readEntry(stateFile, "bt_tpotind0_1");
  stfData.max.triplet.tpotind0_1 = val.toInt();
  val = readEntry(stateFile, "bt_tpotind1_0");
  stfData.max.triplet.tpotind1_0 = val.toInt();
  val = readEntry(stateFile, "bt_tpotind1_1");
  stfData.max.triplet.tpotind1_1 = val.toInt();
  val = readEntry(stateFile, "bt_tpotind2_0");
  stfData.max.triplet.tpotind2_0 = val.toInt();
  val = readEntry(stateFile, "bt_tpotind2_1");
  stfData.max.triplet.tpotind2_1 = val.toInt();

  // read the bperiod value of the top pulse
  val = readEntry(stateFile, "bt_bperiod");
  stfData.max.triplet.bperiod = val.toDouble();

  // read the freq_bin value of the top pulse
  val = readEntry(stateFile, "bt_freq_bin");
  stfData.max.triplet.freq_bin = val.toInt();

  // read the time_bin value of the top pulse
  val = readEntry(stateFile, "bt_time_bin");
  stfData.max.triplet.time_bin = val.toDouble();

  // read the scale of the top pulse
  val = readEntry(stateFile, "bt_scale");
  stfData.max.triplet.scale = val.toDouble();

  // read the fft length of the top pulse
  val = readEntry(stateFile, "bt_fft_len");
  stfData.max.triplet.fft_len = val.toInt();

  // read the graph data
  if(scanTripletGraphData() == false)
    debug("Triplet data could not be read.");

  stfData.max.triplet.wu_name = wuName();

  if(sig) emit newTriplet(stfData.max.triplet);
  }			
}

/*------------------------------------------------------------------------ */
bool SetiContainer::scanTripletGraphData()
{
bool status = false;
QString buf;
char byte[2];        // stores a byte value

buf = readEntry(stateFile, "bt_pot");
if(!buf.isEmpty())
  {
  QTextStream ts(&buf, IO_ReadOnly);
  int val;
  int i = 0;
  while(!ts.atEnd() && i < 512)
    {
    ts >> byte[0] >> byte[1];
    sscanf(byte, "%x", &val);
    stfData.max.triplet.data[i] = (unsigned short int)val;
    i++;
    }
  status = true;
  }

return(status);
}

/*------------------------------------------------------------------------ */
void SetiContainer::scanUserInfoData()
{
QString val;

// read user id
val = readEntry(userinfoFile, "id");
uifData.id = val.toInt();

// read the key
val = readEntry(userinfoFile, "key");
uifData.key = val.toInt();

// read the user's email address
uifData.email_addr = readEntry(userinfoFile, "email_addr");

// read the user name
uifData.name = readEntry(userinfoFile, "name");

// read the user's url
uifData.url = readEntry(userinfoFile, "url");

// read the user's country
uifData.country = readEntry(userinfoFile, "country");

// read the postal code
val = readEntry(userinfoFile, "postal_code");
uifData.postal_code = val.toInt();

// show the name?
val = readEntry(userinfoFile, "show_name");
if(val == "yes")
  uifData.show_name = true;
else
  uifData.show_name = false;

// show the email address?
val = readEntry(userinfoFile, "show_email");
if(val == "yes")
  uifData.show_email = true;
else
  uifData.show_email = false;

// read the venue parameter (whatever it is good for)
val = readEntry(userinfoFile, "venue");
uifData.venue = val.toInt();

// read the register time
uifData.register_time = readEntry(userinfoFile, "register_time");

// read last wu time
uifData.last_wu_time = readEntry(userinfoFile, "last_wu_time");

// read last result time
uifData.last_result_time = readEntry(userinfoFile, "last_result_time");

// read number of sent wus (not used any more)
val = readEntry(userinfoFile, "nwus");
uifData.nwus = val.toInt();

// read number of completed wus
val = readEntry(userinfoFile, "nresults");
uifData.nresults = val.toInt();

// read the total cpu time
val = readEntry(userinfoFile, "total_cpu");
uifData.total_cpu = val.toDouble();

// read the parameter index value (what's that?)
val = readEntry(userinfoFile, "params_index");
uifData.params_index = val.toInt();
}

/*------------------------------------------------------------------------ */
void SetiContainer::scanWorkUnitData()
{
QString val;

// read the task (seti, of course)
wufData.task = readEntry(workunitFile, "task");

// read the version (of what?)
val = readEntry(workunitFile, "version");
wufData.version = val.toInt();

// read the name of the wu
wufData.name = readEntry(workunitFile, "name");

// read the data type
wufData.data_type = readEntry(workunitFile, "data_type");

// read the data class
val = readEntry(workunitFile, "data_class");
wufData.data_class = val.toInt();

// read the splitter version
wufData.splitter_version = readEntry(workunitFile, "splitter_version");

// read the start ra value
val = readEntry(workunitFile, "start_ra");
wufData.start_ra = val.toDouble();

// read the start dec value
val = readEntry(workunitFile, "start_dec");
wufData.start_dec = 0.0F;
wufData.start_dec = val.toDouble();

// read the end ra value
val = readEntry(workunitFile, "end_ra");
wufData.end_ra = val.toDouble();

// read the end dec value
val = readEntry(workunitFile, "end_dec");
wufData.end_dec = val.toDouble();

// read the angle range
val = readEntry(workunitFile, "angle_range");
wufData.angle_range = val.toDouble();

// read the time recorded
wufData.time_recorded = readEntry(workunitFile, "time_recorded");

// read the subband center frequency
val = readEntry(workunitFile, "subband_center");
wufData.subband_center = val.toDouble();

// read the subband base frequency
val = readEntry(workunitFile, "subband_base");
wufData.subband_base = val.toDouble();

// read the subband sample rate
val = readEntry(workunitFile, "subband_sample_rate");
wufData.subband_sample_rate = val.toDouble();

// read the fft length
val = readEntry(workunitFile, "fft_len");
wufData.fft_len = val.toInt();

// read the ifft (?) length
val = readEntry(workunitFile, "ifft_len");
wufData.ifft_len = val.toInt();

// read the subband number
val = readEntry(workunitFile, "subband_number");
wufData.subband_number = val.toInt();

// read the receiver's id
wufData.receiver = readEntry(workunitFile, "receiver");

// read the number of samples
val = readEntry(workunitFile, "nsamples");
wufData.nsamples = val.toInt();

// read the tape version
wufData.tape_version = readEntry(workunitFile, "tape_version");

// read the number of sky positions
val = readEntry(workunitFile, "num_positions");
wufData.num_positions = val.toInt();

// read the coordinates
wufData.coordinates.clear();
if(wufData.num_positions > 0)
  {
  QString e;
  for(int i=0; i<wufData.num_positions; i++)
    {
    val = readEntry(workunitFile, (e = "coord%1").arg(i));
    if(!val.isEmpty())
      wufData.coordinates.append(val);
    }
  }
}

/*------------------------------------------------------------------------ */
int SetiContainer::readClientVersion()
{
int version(0);

QString versionFile(setiDirectory + "/version.sah");

if(QFile::exists(versionFile))
	{
	QString entry = readEntry(versionFile, "major_version");
	version = 100*entry.toInt();
	entry = readEntry(versionFile, "minor_version");
	version += entry.toInt();
	}
else if(QFile::exists(setiDirectory + "/version.txt"))
	version = 100;

return(version);
}


/*------------------------------------------------------------------------ */
double SetiContainer::progressRate()
{
double prograte = (cpuTime() > 0.0) ? (3600.0/cpuTime())*progress() : 0.0;
return(prograte);
}

/*------------------------------------------------------------------------ */
double SetiContainer::remainingTime()
{
double timeleft = (progressRate() > 0.0) ?
                    3600.0*(100.0 - progress())/progressRate() : 0.0;
return(timeleft);
}

/*------------------------------------------------------------------------ */
QString SetiContainer::remainingTimeAsString()
{
double tl = remainingTime();
return(convertTime(tl, true));
}

/*------------------------------------------------------------------------ */
QString SetiContainer::convertTime(double time, bool hms)
{
int d, h, m, s;
QString t;

d = static_cast<int>(time) / 86400;
h = static_cast<int>(time) % 86400 / 3600;
m = static_cast<int>(time) % 3600 / 60;
s = static_cast<int>(time) % 60;

// display with days?
if( hms || (hms == false && d == 0))
  {
  h += d*24;
  t.sprintf("%d:%02d:%02d", h, m, s);
  }
else
  t.sprintf("%dd:%02d:%02d:%02d", d, h, m, s);

return(t);
}

/*------------------------------------------------------------------------ */
QString SetiContainer::timeRecordedString()
{
QString tr = timeRecorded();

int k = tr.find('(');
int l = tr.find(')');
if(k > -1 && l > -1 && l > k)
  tr = tr.mid(k+1, l-k-1);
else
  tr = "";

return(tr);
}

/*------------------------------------------------------------------------ */
double SetiContainer::averageCPUTime()
{
double avgcpu = (numberOfResults() > 0) ? totalCPUTime()/numberOfResults() : 0.0;
return(avgcpu);
}

/*------------------------------------------------------------------------ */
QString SetiContainer::lastResultTimeString()
{
QString lrt = lastResultTime();

int k = lrt.find('(');
int l = lrt.find(')');
if(k > -1 && l > -1 && l > k)
  lrt = lrt.mid(k+1, l-k-1);
else
  lrt = "";

return(lrt);
}

/*------------------------------------------------------------------------ */
QString SetiContainer::registerTimeString()
{
QString rt = registerTime();

int k = rt.find('(');
int l = rt.find(')');
if(k > -1 && l > -1 && l > k)
  rt = rt.mid(k+1, l-k-1);
else
  rt = "";

return(rt);
}

/*------------------------------------------------------------------------ */
bool SetiContainer::exists(int sah)
{
bool ex;

switch(sah)
  {
  case SC_StateFile:
    ex = QFile::exists(stfFileName);
    break;
  case SC_UserInfoFile:
    ex = QFile::exists(uifFileName);
    break;
  case SC_WorkUnitFile:
    ex = QFile::exists(wufFileName);
    break;
  case SC_ResultFile:
    ex = QFile::exists(rsfFileName);
    break;
  case SC_WtempFile:
    ex = QFile::exists(wtpFileName);
    break;
  case SC_ResultHeaderFile:
    ex = QFile::exists(rhdFileName);
    break;
  default:
    ex = false;
    break;
  }

return(ex);
}

/*------------------------------------------------------------------------ */
double SetiContainer::teraFlops(double ar, int version)
{
double tf(0.0);

if(version == 0) version = clientVersion();

if(version >= 200 && version <= 204)
  {
  if(ar >= 0.2255 && ar <= 1.1274)
    tf = 1.45*pow(ar, -0.0826);
  else
    tf = 1.34;
  }
else if(version >= 300 && version < 303)
  {
  if(ar < 0.2255)
    tf = 2.37;

  if(ar >= 0.2255 && ar <= 1.1274)
    tf = 2.58*pow(ar, -0.1531);

  if(ar > 1.1274)
    tf = 2.23*pow(ar, -0.0115);
  }
else if(version >= 303)
  {
  if(ar < 0.2255)
    tf = 3.54*exp(0.0327*ar);

  if(ar >= 0.2255 && ar <= 1.1274)
    tf = 3.74*pow(ar, -0.1075);

  if(ar > 1.1274)
    tf = 3.37*pow(ar, -0.0065);
  }
else
  tf = 1.5;

return(tf);
}

/*------------------------------------------------------------------------ */
double SetiContainer::megaFlopsPerSecond()
{
return((10.0*teraFlops(angleRange())*progressRate())/3.6);
}

/*------------------------------------------------------------------------ */
int SetiContainer::checkClientState()
{
// define this for efficiency reasons
bool workunitfileExists(exists(SC_WorkUnitFile));

// make a back-up of the old state
int oldState = cltState;

if(exists(SC_WtempFile))
  {
  if(exists(SC_ResultHeaderFile) == false && workunitfileExists == false)
    {
    if(exists(SC_ResultFile))
      cltState = Finished;
    else
      {
      cltState = Loading;

      // get the file size of wtemp.sah
    	QFileInfo fi(wtpFileName);
    	// approximate file size of WUs is 360000
    	int loaded = (int)( 100*((double)fi.size()/360000.0) );
    	if(loaded > 100) loaded = 100;
    	
    	// emit a signal every time to keep others up-to-date with the
    	// actual size of wtemp.sah
    	emit stateChanged(cltState, loaded);
    	}
    }
  }
else // wtemp.sah doesn't exist
  {
  switch(cltState)
    {
    case Loading:
      // set all entries to zero and read the new data
      initAllData();
      // make sure that the timing data are read the first time
      stfData.prog = -1.0;
      cltState     = Running;
      // notify others that a new WU has arrived
      emit newWorkUnit(wufData);
      break;
    case Running:
      if(cltPid > 0)
        {
        if(isClientRunning(cltPid) == false)
          {
          cltState = Stopped;
          cltPid   = (pid_t)0;
          }
        }
      break;
    default:
      if(isClientRunning() == true)
        {
        /* This indicates that we can control the client locally, so
           store the pid. */
        cltPid = (pid_t)getClientPid();
        cltState = Running;
        }
      else
        {
        cltState = Stopped;
        cltPid   = (pid_t)0;
        }
      break;
    }
  }

// if state is 'Loading', a signal is emitted every time (see above)
if((cltState != oldState) && cltState != Loading)
  emit stateChanged(cltState);

return(cltState);
}

/*------------------------------------------------------------------------ */
int SetiContainer::getClientPid()
{
int pid(-1);
QFile pidfile((directory() + "/pid.sah"));

if(pidfile.open(IO_ReadOnly))
	{
	QTextStream t(&pidfile);
	t >> pid;
	pidfile.close();
	}
return(pid);
}

/*------------------------------------------------------------------------ */
bool SetiContainer::isClientRunning(pid_t pid, bool checkCmdLine)
{
int err(-1);

// read from pid.sah if pid is zero
if(pid == 0) pid = getClientPid();

// check the command line to see whether it is really the S@h client
if(checkCmdLine)
  {
  QString procstr = QString("/proc/%1/cmdline").arg(pid);
  QFile procfile(procstr);
  if(procfile.open(IO_ReadOnly))
    {
    QString cmd;
  	QTextStream t(&procfile);
  	t >> cmd;
  	procfile.close();
  	// We assume the if there's somewhere the expression "seti" in the
  	// cmd line, it is a S@h client.
  	// If "seti" is not found return false.
    if(cmd.find("seti") == -1) return(false);
    }
  }

// don't send a signal but only check for error;
// if zero, process is alive
if(pid > 0) err = ::kill(pid, 0);
if(err == 0)
	return(true);
else
	return(false);
}

/*------------------------------------------------------------------------ */
int SetiContainer::checkClientStatePassive()
{
/* Evaluate the status of this location (either running or stopped)
   by checking every 2 minutes if the progress value has increased. */
int oldState = cltState;
if(cltPid == 0 && cltState == Running)
  {
  // this is to avoid a stopped process as being indicated as running
  // during the first check
  if(oldProgress > -1.0)
    {
    // now compare
    double diff = fmod(stfData.prog-oldProgress, DIGITS);
    if(diff == 0.0)
      cltState = Stopped;
    }
  oldProgress = stfData.prog;
  }

if(cltState != oldState) emit stateChanged(cltState);

return(cltState);
}
