/***************************************************************************
                          dockwidget.cpp  -  description
                             -------------------
    begin                : Sun Nov 28 1999
    copyright            : (C) 1999 by Gordon Machel
    email                : gmachel@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include <kapp.h>
#include <klocale.h>
#include <kpopupmenu.h>

#include <qtooltip.h>
#include <qpainter.h>

#include "dockwidget.h"

/*------------------------------------------------------------------------ */
DockWidget::DockWidget(QWidget* parent, const char* name) : KSystemTray(parent,name)
{
location = 0;
docked = false;

dockIcons = new KIconLoader();
if(dockIcons)
	{
	mini_seti         = dockIcons->loadIcon("mini-seti", KIcon::User);
	mini_setigrey     = dockIcons->loadIcon("mini-setidockgrey", KIcon::User);
	mini_setifinished = dockIcons->loadIcon("mini-setidockfinished", KIcon::User);
	mini_setistopped  = dockIcons->loadIcon("mini-setidockstopped", KIcon::User);
	mini_setiloading  = dockIcons->loadIcon("mini-setidockloading", KIcon::User);
	picsmall_pixmap   = dockIcons->loadIcon("ksetiwatch", KIcon::Small);
	}
delete dockIcons;

contextMenu()->changeTitle(contextMenu()->idAt(0), "Ksetiwatch");	
contextMenu()->insertItem(i18n("Start Client"), StartClient);
contextMenu()->insertItem(i18n("Stop Client"), StopClient);
connect(contextMenu(), SIGNAL(aboutToShow()), SLOT(preparePopupMenu()));
connect(contextMenu(), SIGNAL(activated(int)), SLOT(handlePopupCommand(int)));

QToolTip::add(this, "Ksetiwatch");

// initialize and start the timer for a blinking icon
showIcon = true;
// blinking interval is 1 sec
startTimer(1000);
}

/*------------------------------------------------------------------------ */
DockWidget::~DockWidget()
{
}

/*------------------------------------------------------------------------ */
void DockWidget::timerEvent(QTimerEvent* e)
{
slotBlinkIcon();
}

/*------------------------------------------------------------------------ */
void DockWidget::dock(SetiLoc* loc)
{
if (!docked)
 	{
	// copy the pointer to the SetiLoc location
	location = loc;
  docked = true;
 	this->show();
  }
}

/*------------------------------------------------------------------------ */
void DockWidget::undock()
{
if (docked)
 	{
  docked = false;
  this->hide();
  }
}

/*------------------------------------------------------------------------ */
void DockWidget::paintEvent(QPaintEvent *e)
{
paintIcon(0, 0);
}

/*------------------------------------------------------------------------ */
void DockWidget::updateProgress()
{
if(location)
  {
  if(location->clientState() == SetiLoc::Running)
    updateState(location->clientState(), 0);
  }
}

/*------------------------------------------------------------------------ */
void DockWidget::updateState(int st, int ls)
{
// determine the text to be shown
if(location)
  {
  QString info;
  int state;
  if(st == 0)
    state = location->clientState();
  else
    state = st;

  switch(state)
  	{
   	case SetiLoc::Stopped:
   		if(location->progress() >= 0.0)
  			info = i18n("stopped");
  		else
  			info = "";
   		break;
   	case SetiLoc::Running:
  		info = "-" + location->remainingTimeAsString();
   		break;
   	case SetiLoc::Loading:
   		info = i18n("loading");
   		break;
   	case SetiLoc::Finished:
   		info = i18n("finished");
   		break;
   	default:
   		info = "";
   		break;
    }
	// update tooltip with "remaining time" info
	QString tip("Ksetiwatch - ");
	tip += location->description();
	if(!info.isEmpty()) tip += " (" + info + ")";
 	QToolTip::remove(this);
 	QToolTip::add(this, tip);
  }
else
  {
  QToolTip::remove(this);
  QToolTip::add(this, "Ksetiwatch");
  }
paintIcon(st, ls);
}

/*------------------------------------------------------------------------ */
void DockWidget::paintIcon(int st, int ls)
{
int x(0),y(0);
		
//clear the widget first
erase();
if(location == 0) // no location selected
	{	
	QSize s(picsmall_pixmap.size());
	if(s.width() < 24) x = (24 - s.width())/2;
	if(s.height() < 24) y = (24 - s.height())/2;
	bitBlt(this, x, y, &picsmall_pixmap);
	}
else //location selected, show details
	{
	QPixmap* icon;
	QString prog;
	
  int state;
  if(st == 0)
    state = location->clientState();
  else
    state = st;

	if(showIcon)
		{
		// determine the icon to be shown
  	switch(state)
  		{
  		case SetiLoc::Stopped:
  			icon = &mini_setistopped;
  			break;
  		case SetiLoc::Running:
  			icon = &mini_seti;
  			break;
  		case SetiLoc::Loading:
  			icon = &mini_setiloading;
  			break;
  		case SetiLoc::Finished:
  			icon = &mini_setifinished;
  			break;
  		default:
  			icon = &mini_setigrey;
  			break;
  		}
  	}
  else
  	icon = &mini_setigrey;
  // determine the text to be shown
  QString info;
  switch(state)
  	{
 		case SetiLoc::Stopped:
 			if(location->progress() >= 0.0)
				prog.sprintf("%.0f%%", location->progress());
			else
				prog = "";
 			break;
 		case SetiLoc::Running:
			prog.sprintf("%.0f%%", location->progress());
 			break;
 		case SetiLoc::Loading:
 			prog = "";
 			break;
 		case SetiLoc::Finished:
 			prog = "100%";
 			break;
 		default:
 			// don't show any text
 			prog = "";
 			break;
  		}
	QSize s(icon->size());
	if(s.width() < 24) x = (24 - s.width())/2;
	bitBlt(this, x, 2, icon);
	// draw the progress bar
	QRect pbar(0, 14, 23, 8);
	QPainter p(this);
	p.fillRect(pbar, darkGray);
	p.setPen(QColor(200, 200, 200));
	p.moveTo(0, 21);
	p.lineTo(23, 21);
	p.lineTo(23, 14);
	p.fillRect(0, 15, (int)(location->progress()*24/100.0), 6, QColor(180,100,0));
	p.setPen(white);
	p.setFont(QFont("system", 8));
	p.drawText( 0, 13, 23, 8, AlignCenter, prog);
	}
}

/*------------------------------------------------------------------------ */
void DockWidget::handlePopupCommand(int id)
{
switch(id)
	{
	case StartClient:
		if(location) location->startClient();
		break;
	case StopClient:
		if(location) location->stopClient();
		break;
	}
}

/*------------------------------------------------------------------------ */
void DockWidget::preparePopupMenu()
{
// disable start/stop client first
contextMenu()->setItemEnabled(StartClient, false);
contextMenu()->setItemEnabled(StopClient, false);
if(location)
	{
	switch(location->clientState())
		{
		case SetiLoc::Running:
		case SetiLoc::Loading:
			{
			// enable stop
	  	contextMenu()->setItemEnabled(StopClient, true);
	  	break;
	  	}
	  case SetiLoc::Stopped:
	  	{
			// enable start
	  	contextMenu()->setItemEnabled(StartClient, true);
	  	break;
	  	}
	  case SetiLoc::Finished:
	  	{
	  	if(location->isClientRunning() == true)
		  	contextMenu()->setItemEnabled(StopClient, true);
		  else
		  	contextMenu()->setItemEnabled(StartClient, true);			  		
	  	break;
	  	}
		}
	}
}

/*------------------------------------------------------------------------ */
void DockWidget::updateDockWidget(SetiLoc* loc)
{
// disconnect any previous connection
if(location != loc && location != 0)
  {
  disconnect(location, SIGNAL(progressIncreased()),
             this,     SLOT(updateProgress()));
  disconnect(location, SIGNAL(stateChanged(int,int)),
             this,     SLOT(updateState(int,int)));
  }
// react on changes of the status
if(location != loc)
  {
  location = loc;
  if(location)
    {
    connect(location, SIGNAL(progressIncreased()),
            this,     SLOT(updateProgress()));
    connect(location, SIGNAL(stateChanged(int,int)),
            this,     SLOT(updateState(int,int)));
    }
  }
updateState(0, 0);
}

/*------------------------------------------------------------------------ */
void DockWidget::slotBlinkIcon()
{
if(location)
	{
	if(location->clientState() != SetiLoc::Running)
		{
		showIcon ^= true;
		paintIcon(0, 0);
		}
	else
		{
		if(showIcon == false)
			{
			showIcon = true;
			paintIcon(0, 0);
			}
		}
	}
}

#include "dockwidget.moc"

