/***************************************************************************
                          analysis.cpp  -  description
                             -------------------
    begin                : Sun Oct 24 1999
    copyright            : (C) 1999 by Gordon Machel
    email                : gmachel@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include <kapp.h>
#include <klocale.h>

#include <qcursor.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qheader.h>
#include <qlist.h>
#include <qtimer.h>
#include <qdir.h>

#include "main.h"
#include "ksetiwatch.h"
#include "analysis.h"
#include "addsetilocdlg.h"
#include "gaussianplot.h"
#include "pulseplot.h"
#include "tripletplot.h"

// extern globals
extern QList<SetiLoc> SetiList;
extern int StatusFilterValue;
extern QTimer* RefreshTimer;
extern int RefreshInterval;
extern bool DRAWGRID;

AnalysisListViewItem::AnalysisListViewItem(QListView* parent):QListViewItem(parent)
{
}

AnalysisListViewItem::~AnalysisListViewItem()
{
}

void AnalysisListViewItem::paintCell(QPainter* p, const QColorGroup& cg, int column, int width, int align)
{
QListViewItem::paintCell(p, cg, column, width, align);
if(DRAWGRID)
	{
	p->setPen( cg.midlight() );
	p->moveTo(0, height()-1);
	p->lineTo(width-1, height()-1);
	p->lineTo(width-1, 0);
  }
}

/** Returns a string which is used for sorting the items. */
QString AnalysisListViewItem::key(int column, bool ascending) const
{
// return value must be static
static QString ret;
QString val;
double tmp;

if(column == 2 || column == 3 || (column >= 6 && column <= 8))
	{
	val = text(column);
	tmp = val.toDouble();
	val.sprintf("%f", tmp);
	ret = val.rightJustify(32, '0');
	}
else if(column == 4 || column == 5)
	{
	int colon;
	int d(0), h(0), m(0), s(0);
	
	val = text(column);
	colon = val.contains(':');
	if(colon == 2)
		{
		sscanf(text(column), "%d:%d:%d", &h, &m, &s);
		}
	else if(colon == 3)
		{
		sscanf(text(column), "%dd:%d:%d:%d", &d, &h, &m, &s);
		}
	tmp = 86400*d + 3600*h + 60*m + s;
	ret.sprintf("%032d", (int)tmp);
	}
else if(column == 9)
	{
	if(text(column).isEmpty())
		tmp = 0.0;
	else
		sscanf(text(column), " scr=%lf", &tmp);
	val.sprintf("%f", tmp);
	ret = val.rightJustify(32, '0');
	}
else if(column == 1)
	{
	SetiLoc* loc = Ksetiwatch::getLocation(SetiLoc::AnalysisList, (QListViewItem*)this);
	ret.sprintf("%d", loc->state());
	}
else
	{
	ret = text(column);
	}
	
return(ret);
}

/***************************************************************************/									
Analysis::Analysis(QWidget *parent, const char *name) : QWidget(parent,name)
	{
	TableColumn antc[] =  {
												{i18n("Location"), -1, AlignLeft},
												{i18n("Status"), -1, AlignCenter},
												{i18n("Chirp Rate"), -1, AlignRight},
												{i18n("% Done"), -1, AlignRight},
												{i18n("CPU Time"), 65, AlignRight},
												{i18n("Time Left"), 60, AlignRight},
												{i18n("% / CPU Hour"), -1, AlignRight},
												{i18n("MFlops"), -1, AlignRight},
												{i18n("Peak"), 55, AlignRight},
												{i18n("Strongest Gaussian"), -1, AlignLeft},
												{i18n("Strongest Pulse"), -1, AlignLeft},
												{i18n("Strongest Triplet"), -1, AlignLeft}
												};
	const int nrOfCols = 12;
	
	const QStringList statusfilterList = QStringList() << i18n("All")
																										 << i18n("Stopped")
									 																	 << i18n("Running")
																										 << i18n("Finished")
																										 << i18n("Loading")
																										 << i18n("Running+Finished");															
																						 	
  QListView_1 = new popListView(this);
  QListView_1->popupSetTitle(i18n("Location"));
  QListView_1->insertPopupItem(i18n("Add...   "), PopupAdd);
  QListView_1->insertPopupItem(i18n("Edit..."), PopupEdit);
  QListView_1->insertPopupItem(i18n("Delete"), PopupDelete);
  QListView_1->insertSeparator();
  QListView_1->insertPopupItem(i18n("Start Client"), PopupStart);
  QListView_1->insertPopupItem(i18n("Stop Client"), PopupStop);
  QListView_1->insertSeparator();
  showOptions = new QPopupMenu();
  showOptions->insertItem(i18n("Gaussian"), SetiLoc::GaussianGraph);
  showOptions->insertItem(i18n("Pulse"), SetiLoc::PulseGraph);
  showOptions->insertItem(i18n("Triplet"), SetiLoc::TripletGraph);
  QListView_1->insertPopupItem(i18n("Show..."), showOptions, PopupShow);
  QListView_1->setPopupStyle(popListView::ShowAnywhere);
  QListView_1->move(5,1);
  connect(showOptions, SIGNAL(activated(int)), this, SLOT(handleShowPopupCommand(int)));
  connect(QListView_1->popupMenu(), SIGNAL(activated(int)), this, SLOT(handlePopupCommand(int)));
  connect(QListView_1->popupMenu(), SIGNAL(aboutToShow()), this, SLOT(checkPopupStatus()));
  connect(QListView_1->header(), SIGNAL(sectionClicked(int)), SLOT(toggleSorting(int)));
  sortingorder = true;
  connect(QListView_1, SIGNAL(selectionChanged(QListViewItem*)), SLOT(handleNewSelection(QListViewItem*)));

  for(int i=0;i<nrOfCols;i++)
  	{
  	QListView_1->addColumn(antc[i].text,-1);
  	if(antc[i].width != -1) QListView_1->setColumnWidth(i, antc[i].width);  	
	  QListView_1->setColumnAlignment(i, antc[i].alignment);
  	}

  QPushButton_1= new QPushButton(this);
  QPushButton_1->setGeometry(10,100,80,25);
  QPushButton_1->setText(i18n("Refresh"));
  connect( QPushButton_1, SIGNAL(clicked()), SLOT(refreshList()) );

  Help = new QPushButton(this);
  Help->setGeometry(200,100,80,25);
  Help->setText(i18n("Help"));
  connect(Help, SIGNAL(clicked()), SLOT(slotShowHelp()) );

  /* this is a very hackish attempt to implement a Pushbutton with a pixmap;
   * sooner or later it should be replaced with a properly defined, new
   * Button class. */
  icons = new KIconLoader();
  Homepage = new QPushButton(i18n("Homepage"), this);
  Homepage->setGeometry(300, 100, 100, 25);
  if(icons) Homepage->setIconSet(QIconSet(icons->loadIcon("ksetiwatch", KIcon::Small)));
  connect( Homepage, SIGNAL(clicked()), SLOT(openBrowser()) );
	
	StatusFilter = new QComboBox(false, this);
	StatusFilter->setGeometry(150, 80, 140, 25);
	StatusFilter->insertStringList(statusfilterList);
	ComboLabel = new QLabel(this);
	ComboLabel->setAutoResize(true);
	ComboLabel->setAlignment(AlignRight);
	ComboLabel->setText(i18n("Show:"));
	connect(StatusFilter, SIGNAL(activated(int)), SLOT(modifyStatusFilter(int)));	
	}

Analysis::~Analysis()
{
}

void Analysis::resizeEvent(QResizeEvent *e)
	{
	QListView_1->resize(this->width()-10, this->height()-QPushButton_1->height()-5);
	QPushButton_1->move(5, QListView_1->y()+QListView_1->height()+2);
	StatusFilter->move(150, QListView_1->y()+QListView_1->height()+2);
	ComboLabel->move(StatusFilter->x()-ComboLabel->width()-5, QListView_1->y()+QListView_1->height()+8);
	Homepage->move(width()-Homepage->width()-5, QListView_1->y()+QListView_1->height()+2);
	Help->move(width()-Homepage->width()-Help->width()-15, QListView_1->y()+QListView_1->height()+2);
	}

void Analysis::handlePopupCommand(int id)
	{
	QListViewItem *it = selectedItem();	
	SetiLoc* loc = Ksetiwatch::getLocation(SetiLoc::AnalysisList, it);
	
	switch(id)
		{
		case PopupAdd: // add a new SETI location
			{
			AddSetiLocDlg dlg(topLevelWidget());
			dlg.setCaption(i18n("Add SETI@home Location..."));
			dlg.setLogwu(true);
			if(dlg.exec() == QDialog::Accepted)
				{
				SetiLoc* loc = new SetiLoc(dlg.directory(), dlg.location(), dlg.color());
				loc->setCmdLineArgs(dlg.cmdLineArgs());
				loc->setStartup(dlg.startup());
				loc->setLogwu(dlg.logwu());
				loc->addItem(SetiLoc::AnalysisList, QListView_1);
				SetiList.append(loc);
				loc->refresh(SetiLoc::AnalysisList);
				emit listModified();
				}
			break;
			}
		case PopupEdit: // edit an existing SETI location
		  {
			if(loc)
				{
				// create the dialog and fill it with the corresponding data
			  AddSetiLocDlg dlg(topLevelWidget());
			  dlg.setCaption(i18n("Edit SETI@home Location..."));
			  dlg.setLocation((const char*)loc->location());
			  dlg.setDirectory((const char*)loc->directory());
			  dlg.setColor(loc->color());
			  dlg.setCmdLineArgs((const char*)loc->cmdLineArgs());
			  dlg.setStartup(loc->startup());
			  dlg.setLogwu(loc->logwu());
				if(dlg.exec() == QDialog::Accepted)
					{
					loc->setLocation(dlg.location());
					loc->setDirectory(dlg.directory());
					loc->setColor(dlg.color());
					loc->setCmdLineArgs(dlg.cmdLineArgs());
					loc->setStartup(dlg.startup());
					loc->setLogwu(dlg.logwu());
					loc->refresh(SetiLoc::AnalysisList);
					emit listModified();
					}
				}
			break;
			}
		case PopupDelete: // remove a SETI location
			{
			if(loc)
				{
				emit selectionChanged(0);
				loc->removeAllItems();
		  	SetiList.remove(loc);
				emit listModified();
			  }
			break;
			}
		case PopupStart: // start the client
			{
			if(loc)
				{
				loc->startClient();
				}
			break;
			}
		case PopupStop: // stop the client
			{
			if(loc)
				{
				loc->stopClient();
				}
			break;
			}
		}
	}
	
void Analysis::checkPopupStatus()
	{
	//convert global coordinates to local widget coords and subtract header height
	QPoint p = QListView_1->mapFromGlobal(QCursor::pos());
	QPoint q( 0, QListView_1->header()->height() );
	p -= q;
  if(QListView_1->itemAt(p))
  	{
  	// is an item below the mouse pointer? If so, enable Edit and Delete
  	QListView_1->popupMenu()->setItemEnabled(PopupEdit, true);
  	QListView_1->popupMenu()->setItemEnabled(PopupDelete, true);
  	// get the status of the clients and enable or disable the Start/Stop entries
  	QListView_1->popupMenu()->setItemEnabled(PopupStart, false);
  	QListView_1->popupMenu()->setItemEnabled(PopupStop, false);
		QListViewItem* it = QListView_1->currentItem();	
		SetiLoc* loc = Ksetiwatch::getLocation(SetiLoc::AnalysisList, it);
		if(loc)
			{
			switch(loc->state())
				{
				case SetiLoc::Running:
				case SetiLoc::Loading:
					{
					// enable stop
			  	QListView_1->popupMenu()->setItemEnabled(PopupStop, true);
			  	break;
			  	}
			  case SetiLoc::Stopped:
			  	{
					// enable start
			  	QListView_1->popupMenu()->setItemEnabled(PopupStart, true);
			  	break;
			  	}
			  case SetiLoc::Finished:
			  	{
			  	if(loc->isClientRunning() == true)
				  	QListView_1->popupMenu()->setItemEnabled(PopupStop, true);
				  else
				  	QListView_1->popupMenu()->setItemEnabled(PopupStart, true);			  		
			  	break;
			  	}
				}
		  // enable Gaussian display only when a Gaussian has been found
		  WUScore* max = loc->topScore();
		  if(max->gaussian > 0.0 && loc->checkVersion() > 1)
		  	showOptions->setItemEnabled(soGaussian, true);
		  else
		  	showOptions->setItemEnabled(soGaussian, false);		
		  // enable pulse display only when a pulse has been found
		  if(max->pulse > 0.0 && loc->checkVersion() > 2)
		  	showOptions->setItemEnabled(soPulse, true);
		  else
		  	showOptions->setItemEnabled(soPulse, false);		
		  // enable triplet display only when a triplet has been found
		  if(max->triplet > 0.0 && loc->checkVersion() > 2)
		  	showOptions->setItemEnabled(soTriplet, true);
		  else
		  	showOptions->setItemEnabled(soTriplet, false);		
		  }
		}
	else
  	{
  	// Disable everything except Add
  	QListView_1->popupMenu()->setItemEnabled(PopupEdit, false);
  	QListView_1->popupMenu()->setItemEnabled(PopupDelete, false);
  	QListView_1->popupMenu()->setItemEnabled(PopupStart, false);
  	QListView_1->popupMenu()->setItemEnabled(PopupStop, false);
  	QListView_1->popupMenu()->setItemEnabled(PopupShow, false);
		}
	}

/**  */
void Analysis::handleShowPopupCommand(int id)
{
QListViewItem *it = selectedItem();	
SetiLoc* loc = Ksetiwatch::getLocation(SetiLoc::AnalysisList, it);


if(loc)
  {
  if(loc->isGraphVisible(id))
   	{
 	  if(loc->graphWidget(id)->isActiveWindow() == false)
 		  {
   		loc->graphWidget(id)->show();
   		loc->graphWidget(id)->raise();
   		}					
   	}
  else
 	  {
   	QWidget* gp;
   	switch(id)
   	  {
   	  case SetiLoc::GaussianGraph:
   			gp = new GaussianPlot(loc);
   			break;
   		case SetiLoc::PulseGraph:
   		  gp = new PulsePlot(loc);
   		  break;
   		case SetiLoc::TripletGraph:
   		  gp = new TripletPlot(loc);
   		  break;
   		default:
   		  gp = 0;
   		  break;
   		}
   	if(gp)
   		{
   		gp->show();
   		loc->setGraphVisible(id, true, (QWidget*)gp);
   		}					
   	}
  }
}
	
void Analysis::refreshList()
{
SetiLoc *loc;

// stop the timer during refresh
RefreshTimer->stop();
for( loc=SetiList.first(); loc != 0; loc=SetiList.next() )
	{
	loc->refresh(SetiLoc::AnalysisList);
	}
// restart the timer
RefreshTimer->start(RefreshInterval, FALSE);
}

/**  */
void Analysis::openBrowser()
{
kapp->invokeBrowser("http://ksetiwatch.sourceforge.net");
}

/** handles the pressed state of the pixmap button */
void Analysis::modifyButton()
{
  QPixmap pix("mini-ksetiwatch.xpm");
  QPainter p;
  QFont f(Homepage->font());
  QColorGroup cg = Homepage->colorGroup();
  QPixmap bp(75, 20);

  f.setPointSize(10);
  p.begin(&bp);
  if(Homepage->isDown())
  	p.fillRect(0, 0, 75, 20, cg.mid());
  else
  	p.fillRect(0, 0, 75, 20, cg.background());
  p.drawPixmap(0, 0, pix);
  p.setFont(f);
  p.drawText(24, 14, i18n("Homepage"));
  p.end();
  Homepage->setPixmap(bp);
}

/** adds or removes ListView items depending on their state */
void Analysis::updateAnalysisList(SetiLoc *loc)
{
bool refreshlist(false);

switch(StatusFilterValue)
	{
	case 0: // All
		{
  	if(loc->item(SetiLoc::AnalysisList) == 0)
  		{
			loc->addItem(SetiLoc::AnalysisList, QListView_1);
			loc->showStatusIcon();
			refreshlist = true;
			}
		break;
		}
	case 5: // Running + Finished
		{
		if(loc->item(SetiLoc::AnalysisList) && (loc->state() != SetiLoc::Running && loc->state() != SetiLoc::Finished))
			{
			loc->removeItem(SetiLoc::AnalysisList);
			refreshlist = true;
			}
		else if(loc->item(SetiLoc::AnalysisList) == 0 && (loc->state() == SetiLoc::Running || loc->state() == SetiLoc::Finished))
			{
			loc->addItem(SetiLoc::AnalysisList, QListView_1);
			loc->showStatusIcon();
			refreshlist = true;
			}
		break;
		}
	default: // Stopped, Running, Finished, Loading
		{
		if(loc->item(SetiLoc::AnalysisList) && loc->state() != StatusFilterValue-1)
			{
			loc->removeItem(SetiLoc::AnalysisList);
			refreshlist = true;
			}
		else if(loc->item(SetiLoc::AnalysisList) == 0 && loc->state() == StatusFilterValue-1)
			{
			loc->addItem(SetiLoc::AnalysisList, QListView_1);
			loc->showStatusIcon();
			refreshlist = true;
			}
		break;
		}		
	}

if(refreshlist == true) loc->refresh(SetiLoc::AnalysisList);
}

/** Changes the StatusFilterValue to the selected one. */
void Analysis::modifyStatusFilter(int sel)
{
StatusFilterValue = sel;
}

/** Switches between ascending and descending sorting order. */
void Analysis::toggleSorting(int column)
{
sortingorder = !sortingorder;
QListView_1->setSorting(column, sortingorder);
}

/** Invokes the Help window. */
void Analysis::slotShowHelp()
{
kapp->invokeHTMLHelp("", "");
}

/** Returns the currently selected entry in the Analysis list. */
QListViewItem* Analysis::selectedItem()
{
QListViewItem* it(QListView_1->currentItem());

if(QListView_1->isSelected(it))
	return(it);
else
	return(0);
}

void Analysis::handleNewSelection(QListViewItem* it)
{
SetiLoc* loc(0);

if(it) loc = Ksetiwatch::getLocation(SetiLoc::AnalysisList, it);
emit selectionChanged(loc);
}

