#include <stdio.h> //for debugging

#include <math.h>

#include <qdialog.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qcheckbox.h>
#include <qlabel.h>

#include <klined.h>
#include <kputil.h>

#include "kps2dmap.moc"
#include "kpview.h"

// # of patchs to draw per reDraw() -- This is system/situation dependent.  
//  Can we find a way to adapt?
const int NPATCHES=1000;
  



KPS2DMap::KPS2DMap (KPView *_view, const KPCoords &_coords,
		    KPMatrix *_matrix, 
		    const KPColorTable &_colortable,
		    double _x, double _y,
		    double _w, double _h, int _z) 
  : KPS2D (_view, _coords, _matrix, _x, _y, _w, _h, _z)
{

  colortable = _colortable; 
  //setInterpolation (BilinearVertex);
  setInterpolation (None);
  drawGrid(false);
  gridsize0=.0025;
  computeSizes();
  
  propwid=0;

  //  ctR=0;
  //  ctD=0;
  //  ctRD=0;
  //  ctR=new QColor (0,0,0);
  //  ctD=new QColor (0,0,0);
  //  ctRD=new QColor (0,0,0);
}


KPS2DMap::KPS2DMap (KPView *_view, const KPCoords &_coords,
		    const KPColorTable &_colortable,
		    double _x, double _y,
		    double _w, double _h, int _z) 
{
  KPS2DMap (_view, _coords, 0, _colortable, _x, _y, _w, _h, _z);
}

void
KPS2DMap::drawPatch (QPainter *p,
		     int x, int y,
		     int width, int height, 
		     unsigned int row,  unsigned int col,
		     unsigned int nrow)
{
  QColor c= colortable.color(matrix()->matrix() [nrow+col]);

  
  switch (interp)
    {
    case None:
      p->setBrush (c);

      p->setPen (NoPen);
      p->drawRect (x, y, width, height);
      rdpatches++;
      break;

    case BilinearVertex: //Upper-left corner takes on val
      {
	QColor f00=c;
	QColor f01 = getD (row, col, nrow);
	QColor f10 = getR (row, col, nrow);
	QColor f11 = getRD (row, col, nrow);

	int xend = x+width;
	int yend = y+height;
	double d_x=0, d_y=1;
	double oow=1./width, ooh=1./height;
	//printf ("%f %f %f %f %d %d\n",		d_x, d_y, oow, ooh, width, height);
	int iy;
	int cnt=0;
	for (int ix=x; ix<xend; ix++, d_x+=oow)
	  for (iy=y, d_y=1; iy>yend; iy--, d_y+=ooh)
	    {
	      p->setPen (bilinInterp (d_x, d_y,
				      f00, f01, f10, f11));
	      p->drawPoint (ix, iy);
	      cnt++;
	      if (cnt>50)
		{
		  cnt=0;
		  rdpatches++;
		}
	    }
      }
      break;
    }

  if (bdrawgrid)
    {
      p->setBrush (black);
      p->setPen (NoPen);
      p->drawRect (x, y, width, -gridsize);
      p->drawRect (x, y, gridsize, height);

      if (row==0)
	p->drawRect (x, y+height+2,
		     width, +gridsize);
      if (col==matrix()->nCols()-1)
	p->drawRect (x+width-1, y,
		     -gridsize, height);
    }

}

void
KPS2DMap::reDraw (QPainter *painter)
{

  //	  for (j=0, ndx=brect.x(); j<matrix()->nCols(); j++, ndx+=dx)
  
  double ndy, ndx;
  int i,j;
  int ni=rdni;
  if (matrix()) 
    {
      QPainter * p = painter;
      
      //      double dx = coords()->owidth()/(double)matrix()->nCols();
      //      double dy = coords()->oheight()/(double)matrix()->nRows();
      double dx = (double)coords()->owidth()/
	(coords()->cxmax()-coords()->cxmin());
      double dy = (double) coords()->oheight()/
	(coords()->cymin()-coords()->cymax());

      if (needRedraw ()) //fresh start
	{
	  rdi=(int)coords()->cymin()-1; //rows, ymin>ymax (yes, inverted)
	  rdj=(int)coords()->cxmin(); //cols
	  rdndy = coords()->oY0() + coords()->oheight()-1;
	  rdndx = coords()->oX0();
	  rdni=rdi*matrix()->nCols();
	}
      rdpatches = 0;
	  
      j=rdj;
      ndx = rdndx;
      int xmax = (int)coords()->cxmax();
      int ymax = (int)coords()->cymax();

      for (i=rdi, ni=rdni, ndy=rdndy; //ROWS
	   i>=ymax;)
	{
	  for (; j<xmax;)
	    {
	      // printf ("MAP: %d %d\n",i,j);
	      drawPatch (p, ndx, ndy, dx+1, -dy-1, i, j, ni);
	      rdpatches ++;
	      j++;
	      ndx+=dx;
	      if (rdpatches>NPATCHES)
		break;
	    }
	  if (j>=xmax)
	    {
	      j=(int)coords()->cxmin();
	      ndx = coords()->oX0();
	      i--;
	      ni-=matrix()->nCols();
	      ndy-=dy;
	    }
	  if (rdpatches>NPATCHES)
	    break;
	}
      // when, say, the window is resized
      if (i>=ymax) //<matrix()->nRows())
	{
	  rdi=i;
	  rdni=ni;
	  rdj=j;
	  rdndx = ndx;
	  rdndy = ndy;
	  rdpatches = 0;
	}
      else
	doneRedrawing();  //done redrawing the plot
      
    }
}


void
KPS2DMap::setInterpolation (Interpolation i)
{
  interp = i;
  if (interp==BilinearVertex)
    {
      if (coords()->cxmax()==matrix()->nRows())
	coords()->setCxmax (matrix()->nRows()-1);
      if (coords()->cymin()==matrix()->nCols())
	coords()->setCymin (matrix()->nCols()-1);
    }
}


void
KPS2DMap::setCoords (const KPCoords &cc)
{
  KPCoords c (cc);

  if (interp==BilinearVertex)
    {
      if (c.cxmax()==matrix()->nRows())
	c.setCxmax (matrix()->nRows()-1);
      if (c.cymin()==matrix()->nCols())
	c.setCymin (matrix()->nCols()-1);
    }

  KPS2D::setCoords (c);
}

void
KPS2DMap::computeSizes (void)
{
  gridsize=kpmax ((int) (gridsize0*view()->width()), 1);
}

void
KPS2DMap::resizeEvent (QResizeEvent *re)
{
  computeSizes();

  KPS2D::resizeEvent (re);
}

void
KPS2DMap::installDefaultRMBMenu (void)
{
  rmbmenu = new QPopupMenu;
  rmbmenu->insertItem ("&Properties", this, SLOT(slotProperties()));
}

QWidget *
KPS2DMap::propWidget (QWidget *parent, const char *name)
{
  if (propwid!=0)
    delete propwid;

  propwid = new KPS2DMapProp (this, parent, name);
  return propwid;
}

void
KPS2DMap::slotProperties ()
{
  QDialog d (0, 0, TRUE);
  QGridLayout *layout = new QGridLayout (&d, 2, 3, 10);

  KPS2DMapProp *p= (KPS2DMapProp *)propWidget (&d);
  layout->addMultiCellWidget (p, 0, 0, 0, 2);
  QPushButton *b = new QPushButton ("OK", &d);
  b->setMinimumSize (b->sizeHint());
  b->setMaximumSize (b->sizeHint());
  b->setDefault(true);
  connect ( b, SIGNAL (clicked()), 
	    &d, SLOT (accept()) );
  layout->addWidget (b, 1, 1);
  b = new QPushButton ("Cancel", &d);
  b->setMinimumSize (b->sizeHint());
  b->setMaximumSize (b->sizeHint());
  connect ( b, SIGNAL (clicked()), 
	    &d, SLOT (reject()) );
  layout->addWidget (b, 1, 2);
  layout->activate();

  if (d.exec())
    {
      setInterpolation (p->interpolation());
      drawGrid (p->grid());
      setColorTable (p->colorTable());
      setGridSize (p->gridSize());
      bneedredraw=true;
      update();
    }

  delete propwid;
  propwid=0;
}


void
KPS2DMap::setGridSize (double size)
{
  gridsize0 = size; 
  computeSizes();
}

/////////Properties
KPS2DMapProp::KPS2DMapProp (KPS2DMap *kp2m, QWidget *parent, 
			    const char *name, WFlags f) : 
  QWidget (parent, name, f)
{
  QGridLayout *layout = new QGridLayout (this, 3, 3, 10);
  
  bg = new QButtonGroup ("Interpolation", this);
  QRadioButton *rb = new QRadioButton ("&None", bg);
  rb->setGeometry (10, 15, rb->sizeHint().width(),
		    rb->sizeHint().height());
  rb = new QRadioButton ("&Bilinear, value at vertex", bg);
  rb->setGeometry (10, 35, rb->sizeHint().width(),		    
		   rb->sizeHint().height());

  bg->setMinimumSize (QSize (20 + rb->sizeHint().width(),
			     45 + rb->sizeHint().height()));
  connect ( bg, SIGNAL (clicked (int)),
	    this, SLOT (slotInterpolation (int)) );
  layout->addMultiCellWidget (bg, 0, 0, 0, 1);

  cbgrid = new QCheckBox ("&Draw grid", this);
  cbgrid->setMinimumSize (cbgrid->size());
  connect ( cbgrid, SIGNAL (clicked()),
	    this, SLOT (slotDisable()) );
  layout->addWidget (cbgrid, 1, 0);
  
  QPushButton *pb = new QPushButton ("&Colortable", this);
  pb->setEnabled (false);
  pb->setMinimumSize (pb->sizeHint());
  layout->addWidget (pb, 1, 1);

  lesize = new KLineEdit (this);
  QString qs;
  qs.setNum (100.*kp2m->gridSize());
  lesize->setText (qs);
  lesize->setMinimumSize (lesize->sizeHint());
  layout->addWidget (lesize, 2, 1);

  gridsize = new QLabel (lesize, "&Grid line thickness (%):", this);
  gridsize->setMinimumSize (gridsize->sizeHint());
  layout->addWidget (gridsize, 2, 0);

  layout->activate();

  slotDisable();

  kpct=kp2m->colorTable ();
  cbgrid->setChecked (kp2m->grid());
  setInterpolation (kp2m->interpolation());
}

void
KPS2DMapProp::slotDisable()
{
  if (cbgrid->isChecked())
    {
      gridsize->setEnabled (true);
      lesize->setEnabled (true);
    }
  else
    {
      gridsize->setEnabled (false);
      lesize->setEnabled (false);
    }

}

void
KPS2DMapProp::setInterpolation (KPS2DMap::Interpolation i)
{
  interp = i;
  
  switch (interp)
    {
    case KPS2DMap::None:
      bg->setButton (0);
      break;
    case KPS2DMap::BilinearVertex:
      bg->setButton (1);
      break;
    }
}
void
KPS2DMapProp::slotInterpolation (int id)
{
  switch (id)
    {
    case 0:
      setInterpolation (KPS2DMap::None);
      break;
    case 1:
      setInterpolation (KPS2DMap::BilinearVertex);
      break;
    };
}



QColor
KPS2DMap::getR (unsigned int i, unsigned int j, unsigned int ni)
{

  int r,g,b;

  if (j<matrix()->nCols()-1)
      return colortable.color(matrix()->matrix() [ni+j+1]);
  else
    {
      QColor ct (0,0,0);
      QColor c = colortable.color(matrix()->matrix() [ni+j]);
      QColor cl = colortable.color(matrix()->matrix() [ni+j-1]);
      //QColor *cl = colortable.color(matrix()->matrix() [ni+j]);
      r=2*c.red()-cl.red();
      g=2*c.green()-cl.green();
      b=2*c.blue()-cl.blue();

      r=kpmax(kpmin(r,255),0);
      g=kpmax(kpmin(g,255),0);
      b=kpmax(kpmin(b,255),0);
      ct.setRgb (r,g,b);
      return ct;
    }
}

QColor 
KPS2DMap::getD (unsigned int i, unsigned int j, unsigned int ni)
{
  int r,g,b;

  if (i<matrix()->nRows()-1)
      return colortable.color(matrix()->matrix() [ni+matrix()->nCols()+j]);
  else
    {
      QColor ct (0,0,0);
      QColor c = colortable.color(matrix()->matrix() [ni+j]);
      QColor cu = colortable.color(matrix()->matrix() [ni+j-matrix()->nCols()]);
      //QColor *cu = colortable.color(matrix()->matrix() [ni+j]);
      r=2*c.red()-cu.red();
      g=2*c.green()-cu.green();
      b=2*c.blue()-cu.blue();

      r=kpmax(kpmin(r,255),0);
      g=kpmax(kpmin(g,255),0);
      b=kpmax(kpmin(b,255),0);
      ct.setRgb (r,g,b);

      return ct;
    } 


}

QColor
KPS2DMap::getRD (unsigned int i, unsigned int j, unsigned int ni)
{
  if (i<matrix()->nRows()-1 && j<matrix()->nCols()-1)
    return colortable.color(matrix()->
			    matrix() [ni+matrix()->nCols()+j+1]);
  else
    {
      int r,g,b;

      QColor ct (0,0,0);
      
      if (i>=matrix()->nRows()-1 && j<matrix()->nCols()-1)
	{
	  QColor c = colortable.color(matrix()->matrix() [ni+j]);
	  QColor cu = colortable.color(matrix()->matrix() [ni+j-matrix()->nCols()]);
	  //QColor *cu = colortable.color(matrix()->matrix() [ni+j]);
	  r=2*c.red()-cu.red();
	  g=2*c.green()-cu.green();
	  b=2*c.blue()-cu.blue();

	  r=kpmax(kpmin(r,255),0);
	  g=kpmax(kpmin(g,255),0);
	  b=kpmax(kpmin(b,255),0);
	  ct.setRgb (r,g,b);
	  
	}
      else if (i<matrix()->nRows()-1 && j>=matrix()->nCols()-1)
	{
	  QColor c = colortable.color(matrix()->matrix() [ni+j]);
	  QColor cl = colortable.color(matrix()->matrix() [ni+j-1]);
	  //QColor *cl = colortable.color(matrix()->matrix() [ni+j]);
	  r=2*c.red()-cl.red();
	  g=2*c.green()-cl.green();
	  b=2*c.blue()-cl.blue();

	  r=kpmax(kpmin(r,255),0);
	  g=kpmax(kpmin(g,255),0);
	  b=kpmax(kpmin(b,255),0);
	  ct.setRgb (r,g,b);
	}
      else 
	{
	  QColor c = colortable.color(matrix()->matrix() [ni+j]);
	  QColor cu = colortable.color(matrix()->matrix() [ni+j-matrix()->nCols()]);
	  QColor cl = colortable.color(matrix()->matrix() [ni+j-1]);
	  
	  r=(4*c.red() - cl.red() - cu.red())/2;
	  g=(4*c.green() - cl.green() - cu.green())/2;
	  b=(4*c.blue() - cl.blue() - cu.blue())/2;
	  
	  r=kpmax(kpmin(r,255),0);
	  g=kpmax(kpmin(g,255),0);
	  b=kpmax(kpmin(b,255),0);
	  ct.setRgb (r,g,b);
	}
      return ct;
    }
}

inline QColor
KPS2DMap::bilinInterp (double dx, double dy,
		       QColor f00, QColor f01,
		       QColor f10, QColor f11)
{
  int r, g, b;
  double omdx=1-dx, omdy=1-dy;

  r =(int) (.25*(omdx*omdy*f00.red() +
		 omdx*dy*f01.red() +
		 dx*omdy*f10.red() +
		 dx*dy*f11.red()));

  g =(int) (.25*(omdx*omdy*f00.green() +
		 omdx*dy*f01.green() +
		 dx*omdy*f10.green() +
		 dx*dy*f11.green()));

  b = (int) (.25*(omdx*omdy*f00.blue() +
		  omdx*dy*f01.blue() +
		  dx*omdy*f10.blue() +
		  dx*dy*f11.blue()));
  //  printf ("RGB = %d %d %d,  %f %f\n",r,g,b,dx,dy);
  return QColor (r, g, b);
}

KPCoords
KPS2DMap::autoCoords (const KPCoords *c = 0)
{
  // if (c==0)  //ignore c for now
  {
    if (matrix()==0)
      return KPCoords (0,0,0,0);
    KPCoords coords (0, matrix()->nCols(),
		     matrix()->nRows(), 0);
    return coords;
  }
}

