/****************************************************************************
**
** Implementation of QwPainterSpriteField
**
** Author  : David Sweet <dsweet@chaos.umd.edu>
** Created : 19990622
**
** Based on code by Warwick Allison <warwick@troll.no>
**
*****************************************************************************/


#include "QwPrinterSpriteFieldView.h"
#include <qpainter.h>
#include <qpaintdevicemetrics.h>

/*!
\class QwPrinterSpriteFieldView QwSpriteField.h
\brief Used for printing a QwSpriteField
*/ 



 
/*!
We'll record all of the drawing commands in a QPicture
 and then print everything when print() is called.
*/           

void QwPrinterSpriteFieldView::beginPainter(QPainter& painter)
{

  QPaintDeviceMetrics qpdm (this);

  painter.begin(pic);

  painter.setWindow (0, 0, viewing->width(), viewing->height());
  painter.setViewport ( (qpdm.width() - w)/2,
			(qpdm.height() - h)/2,
			w, h);
}


/*!
Construct a QwPrinterSpriteFieldView which views the given QwSpriteField.
Nothing will be drawn initially. (viewArea() will return zero.)  You
must call cleanPage() before printing.

If you plan to use printAll (QwSpriteField *), then you can
 pass zero for viewing.
*/
QwPrinterSpriteFieldView::QwPrinterSpriteFieldView
 (QwSpriteField* v) :
    QwAbsSpriteFieldView(v)
{
  pic = 0;
}

/*!
 Sets up a clean page (clears all previous print commands).  You should
  set the size of your field after constructing, call the setup()
  method (optionally), call this method, then call print().
*/
void
QwPrinterSpriteFieldView::cleanPage (void)
{
  if (!viewing)
    return;

  if (pic)
    delete pic;
  pic = new QPicture;

  //Note that this calculation is independent of the size of
  // the field. 

  double aspectratio = viewing->height()/(double)viewing->width();
  QPaintDeviceMetrics qpdm (this);

  if ( (h=(int) (qpdm.width()*aspectratio)) <= qpdm.height())
    w = qpdm.width();
  else
    {
      w = (int) (qpdm.height()/aspectratio);
      h = qpdm.height();
    }

}



/*!
Deconstruct the QwPrinterSpriteFieldView
*/
QwPrinterSpriteFieldView::~QwPrinterSpriteFieldView()
{
  if (pic)
    delete pic;
}

/*!
Returns the printable area of
the page.
*/
QRect QwPrinterSpriteFieldView::viewArea() const
{
  if (!pic || !viewing)
    {
      QRect area (0,0,0,0);
      return area;
    }

  QRect area (0, 0, viewing->width(), viewing->height());
  
  return area;
}

/*!
Returns FALSE: We will not be animating on the printer, so we don't need
 this.
*/
bool QwPrinterSpriteFieldView::preferDoubleBuffering() const
{
    return FALSE;
}

void QwPrinterSpriteFieldView::flush ( const QRect & area )
{
  //Nothing
}

void QwPrinterSpriteFieldView::print ()
{
  if (pic==0 || viewing==0)
    return;

  QPainter p;
  p.begin (this);
  p.drawPicture ( *pic);
  p.end();
}


/*!
 Use this for the simplest printing case.  Your sprites should
  reformat themselves in response to a resizing of the field
  appropriately.
*/
void QwPrinterSpriteFieldView::printAll (QwSpriteField *field)
{

  if (field==0)
    {
      field = viewing;
    }

  view (field);
  cleanPage();
  field->updateInView (this, 
		       QRect (0, 0, viewing->width(), viewing->height()));
  print();

  view (viewing);

  return;
}
