/*************************************************/
/* methods for class MainWidget                  */
/*                                               */
/* main window of this application               */
/*   widget management                           */
/*                                               */
/* Andreas Rostin                                */
/* 27.10.97                                      */
/*************************************************/
#include <kapp.h>
#include <kfiledialog.h>
#include <kmainwindow.h>
#include <kstddirs.h>
#include <kmenubar.h>

#include <qpainter.h>
#include <qkeycode.h>
#include <qprinter.h>
#include <qprintdialog.h>
#include <qfileinfo.h>
#include <qmessagebox.h>
#include <qstringlist.h>
#include <qslider.h>

#include <mainw.h>
#include <netw.h>
#include <klogic.h>
#include <version.h>
#include <xdevice.h>
#include <clientw.h>
#include <simw.h>
#include <propGlob.h>

#include "mainw.moc"

/***************************************************/
/* static methods of class MainWidget              */
/***************************************************/
QString MainWidget::PATH;
QString MainWidget::TBPATH;

// private static
void MainWidget::setPATH(KApplication *app)
{	QPixmap pix;

	QString pixpath(getenv("KLOGIC_PATH"));
	QString tbpath(getenv("KLOGIC_PATH"));

	// path comes from environment
	if (!pixpath.isNull()) {
		fprintf(stderr, "\npath was set to <%s>\n", (const char *)pixpath);
		warning("using local directories: [remove KLOGIC_PATH] try \"make install\"");
		pixpath.append("/pics/");
		tbpath.append("/toolbar/");
	}

	// path comes from kde
	if (pixpath.isNull() || tbpath.isNull()) {
		int found = 0;
		KStandardDirs *stdDir = app->dirs();
		QStringList sl = stdDir->findDirs("data", "klogic");
		QStringList::ConstIterator it;
		for( it = sl.begin(); !found && it != sl.end(); ++it ) {
			pixpath = (*it) + "pics/";
			tbpath = (*it) + "toolbar/";
			if (TRUE == pix.load(pixpath + "AND.xpm")) found = 1;
		}
	}

	// check pixpath
	if (FALSE == pix.load(pixpath + "AND.xpm")) {
		fatal("***** aborting ***** : cant find my icons ...
	- either try \"make install\"
	- or set the environment variable KLOGIC_PATH to the klogic base directory");
	}

	// set static variables
	PATH = pixpath;
	TBPATH = tbpath;

}

/***************************************************/
/* methods of class MainWidget                     */
/***************************************************/
MainWidget::MainWidget(KApplication *app, char *file=(char *)NULL)
	:KMainWindow(NULL)
{	int i;

	setPATH(app);

	setBackgroundMode( PaletteBase );    // set widgets background
	resize(SCREEN_SIZE_X, SCREEN_SIZE_Y);                  // set default size in pixels
	setCaption(ME);

	// read config
	kconf = app->sessionConfig();
	readConfig();

	// load device pixmaps
	XDevice::loadPix();

	// initial net
	clientW[0] = new ClientW(this, file);
	if (!(main_netwidget = clientW[0]->net())) fatal("unable to create inital circuit..\n");
	NetWidget::setDrawMode(NetWidget::MODE_DRAW);
	clientW[0]->show();
	connect(main_netwidget, SIGNAL(netChanged(int)), this, SLOT(netChange(int)));
	connect(main_netwidget, SIGNAL(createNewWidget(XDeviceNet *)), this, SLOT(newWidget(XDeviceNet *)));
	connect(main_netwidget, SIGNAL(netDeleted(XDeviceNet *)), this, SLOT(netDelete(XDeviceNet *)));
	connect(main_netwidget, SIGNAL(netChanged(int)), clientW[0], SLOT(netChange(int)));

	initToolbar();
	initMenubar();
	setCentralWidget(clientW[0]);

	simW = new SimW(main_netwidget->getActive());
	connect(simW, SIGNAL(shidden()), this, SLOT(simWidgetHidden()));
	connect(main_netwidget, SIGNAL(graphChanged()), simW, SLOT(graphChange()));
	connect(main_netwidget, SIGNAL(simStepped()), simW, SLOT(simStep()));
	connect(main_netwidget, SIGNAL(showGraph(XDeviceNet *)), simW, SLOT(changeNet(XDeviceNet *)));
	simW->setSimMode(main_netwidget->getSimMode());
	simW->hide();
	simw = 0;

	// construct client widgets
	for(i=1;i<MAXCLIENTS;i++) {
		clientW[i] = new ClientW(this, i);
		clientW[i]->hide();
		connect(clientW[i], SIGNAL(chidden(ClientW *)), this, SLOT(clientHidden(ClientW *)));
	}
}

MainWidget::~MainWidget()
{
}

void MainWidget::mousePressEvent(QMouseEvent *)
{
}

void MainWidget::mouseReleaseEvent(QMouseEvent *)
{
}

void MainWidget::mouseMoveEvent(QMouseEvent *)
{
}

void MainWidget::paintEvent(QPaintEvent *)
{	int i;

	for(i=1;i<MAXCLIENTS;i++) {
		if (clientW[i]->net() && clientW[i]->isActiveWindow()) clientW[i]->repaint(FALSE);
	}
}

void MainWidget::initToolbar()
{	QPixmap px;

	toolbar = new KToolBar(this);
	px.load(TBPATH + "tbDraw.xpm");
	toolbar->insertButton(px, ID_DRAW, SIGNAL(clicked()),
			this, SLOT(modeDraw()), TRUE, klocale->translate("draw wires"), -1);
	toolbar->setToggle(ID_DRAW, TRUE);
	toolbar->setButton(ID_DRAW, TRUE);

	px.load(TBPATH + "tbSel.xpm");
	toolbar->insertButton(px, ID_SEL, SIGNAL(clicked()),
			this, SLOT(modeSel()), TRUE, klocale->translate("select groups"), -1);
	toolbar->setToggle(ID_SEL, TRUE);

	toolbar->insertSeparator();
	toolbar->insertSeparator();
	toolbar->insertSeparator();
	toolbar->insertSeparator();
	toolbar->insertSeparator();
	toolbar->insertSeparator();

	px.load(TBPATH + "tbNET.xpm");
	toolbar->insertButton(px, XDevice::fNET, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, klocale->translate("new sub-circuit"), -1);
	toolbar->setToggle(XDevice::fNET, TRUE);

	px.load(TBPATH + "tbFx.xpm");
	toolbar->insertButton(px, XDevice::fEQU, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, klocale->translate("new equation"), -1);
	toolbar->setToggle(XDevice::fEQU, TRUE);

	px.load(TBPATH + "tbAND.xpm");
	toolbar->insertButton(px, XDevice::fAND, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, klocale->translate("AND gate"), -1);
	toolbar->setToggle(XDevice::fAND, TRUE);

	px.load(TBPATH + "tbOR.xpm");
	toolbar->insertButton(px, XDevice::fOR, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, klocale->translate("OR gate"), -1);
	toolbar->setToggle(XDevice::fOR, TRUE);

	px.load(TBPATH + "tbINV.xpm");
	toolbar->insertButton(px, XDevice::fONE, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, klocale->translate("inverter"), -1);
	toolbar->setToggle(XDevice::fONE, TRUE);

	px.load(TBPATH + "tbXOR.xpm");
	toolbar->insertButton(px, XDevice::fXOR, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, klocale->translate("XOR gate"), -1);
	toolbar->setToggle(XDevice::fXOR, TRUE);

	px.load(TBPATH + "tbRS.xpm");
	toolbar->insertButton(px, XDevice::fRS, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, klocale->translate("RS flipflop"), -1);
	toolbar->setToggle(XDevice::fRS, TRUE);

	px.load(TBPATH + "tbD.xpm");
	toolbar->insertButton(px, XDevice::fDFF, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, klocale->translate("D flipflop"), -1);
	toolbar->setToggle(XDevice::fDFF, TRUE);

	px.load(TBPATH + "tbJK.xpm");
	toolbar->insertButton(px, XDevice::fJK, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, klocale->translate("JK flipflop"), -1);
	toolbar->setToggle(XDevice::fJK, TRUE);

	px.load(TBPATH + "tbOSZ.xpm");
	toolbar->insertButton(px, XDevice::fOSZ, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, klocale->translate("oscillator"), -1);
	toolbar->setToggle(XDevice::fOSZ, TRUE);

	px.load(TBPATH + "tbSWI.xpm");
	toolbar->insertButton(px, XDevice::fSWI, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, klocale->translate("switch"), -1);
	toolbar->setToggle(XDevice::fSWI, TRUE);

	px.load(TBPATH + "tbSS.xpm");
	toolbar->insertButton(px, XDevice::fSS, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, klocale->translate("7 segment"), -1);
	toolbar->setToggle(XDevice::fSS, TRUE);

	px.load(TBPATH + "tbLED.xpm");
	toolbar->insertButton(px, XDevice::fLEDgreen, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, klocale->translate("LED"), -1);
	toolbar->setToggle(XDevice::fLEDgreen, TRUE);

	px.load(TBPATH + "tbPWR.xpm");
	toolbar->insertButton(px, XDevice::fPWR, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, klocale->translate("Static"), -1);
	toolbar->setToggle(XDevice::fPWR, TRUE);

	px.load(TBPATH + "tbTXT.xpm");
	toolbar->insertButton(px, XDevice::fTXT, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, klocale->translate("Text"), -1);
	toolbar->setToggle(XDevice::fTXT, TRUE);

	px.load(TBPATH + "tbIN.xpm");
	toolbar->insertButton(px, XDevice::fIN, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, klocale->translate("input"), -1);
	toolbar->setToggle(XDevice::fIN, TRUE);
	toolbar->hideItem(XDevice::fIN);

	px.load(TBPATH + "tbOUT.xpm");
	toolbar->insertButton(px, XDevice::fOUT, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, klocale->translate("output"), -1);
	toolbar->setToggle(XDevice::fOUT, TRUE);
	toolbar->hideItem(XDevice::fOUT);

	toolbar->insertSeparator();
	toolbar->insertSeparator();
	toolbar->insertSeparator();
	toolbar->insertSeparator();
	toolbar->insertSeparator();
	toolbar->insertSeparator();

	px.load(TBPATH + "tbStep.xpm");
	toolbar->insertButton(px, NetWidget::MODE_SIM_STEP, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, klocale->translate("single step simulation"), -1);
	toolbar->setToggle(NetWidget::MODE_SIM_STEP, TRUE);
	toolbar->alignItemRight(NetWidget::MODE_SIM_STEP);

	px.load(TBPATH + "tbSim.xpm");
	toolbar->insertButton(px, NetWidget::MODE_SIM_MULT, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, klocale->translate("simulate"), -1);
	toolbar->setToggle(NetWidget::MODE_SIM_MULT, TRUE);
	toolbar->alignItemRight(NetWidget::MODE_SIM_MULT);
	toolbar->setButton(NetWidget::MODE_SIM_MULT, TRUE);

	QLabel *speed_text = new QLabel(toolbar);
	speed_text->setFixedSize(50, 20);
	speed_text->setText(klocale->translate("speed"));

	speed_slider = new QSlider(QSlider::Horizontal, toolbar);
	speed_slider->setFixedSize(100, 20);
	speed_slider->setRange(NetWidget::MIN_SIMSTEPTIME, NetWidget::MAX_SIMSTEPTIME);
	speed_slider->setValue(NetWidget::MAX_SIMSTEPTIME - NetWidget::simTime());
	connect(speed_slider, SIGNAL(sliderMoved(int)), this, SLOT(simTimeSliderChanged(int)));

	px.load(TBPATH + "tbShow.xpm");
	toolbar->insertButton(px, ID_SIM_SHOW, SIGNAL(clicked(int)),
			this, SLOT(menuCallback(int)), TRUE, klocale->translate("simulation graph"), -1);
	toolbar->setToggle(ID_SIM_SHOW, TRUE);
	toolbar->alignItemRight(ID_SIM_SHOW);

	toolbar->insertSeparator();
	toolbar->insertSeparator();

	toolbar->setBarPos(KToolBar::Top);
	toolbar->enable(KToolBar::Show);
	addToolBar(toolbar);
	toolbar->show();
	tool_visible = 1;
}

void MainWidget::initMenubar()
{
	menubar = menuBar();
	CHECK_PTR(menubar);

	pm_file = new QPopupMenu;
	pm_file->insertItem(klocale->translate("&New..."), ID_NEW);
	pm_file->insertItem(klocale->translate("&Open..."), ID_OPEN);
	pm_file->setAccel(Key_F4, ID_OPEN);
	pm_file->insertSeparator();
	pm_file->insertItem(klocale->translate("&Save"), ID_SAVE);
	pm_file->setAccel(Key_F5, ID_SAVE);
	pm_file->insertItem(klocale->translate("Save &As..."), ID_SAVE_AS);
	pm_file->setAccel(Key_F6, ID_SAVE_AS);
	pm_file->insertSeparator();
	pm_file->insertItem(klocale->translate("&Print..."), ID_PRINT);
	pm_file->setAccel(Key_F8, ID_PRINT);
	pm_file->insertSeparator();
	pm_file->insertItem(klocale->translate("&Exit"), ID_EXIT);

	pm_edit = new QPopupMenu;
	pm_edit->insertItem(klocale->translate("&Copy"), ID_COPY);
	pm_edit->insertItem(klocale->translate("&Paste"), ID_PASTE);
	pm_edit->insertItem(klocale->translate("C&ut"), ID_CUT);

	pm_logic = new QPopupMenu;
	pm_logic->insertItem(klocale->translate("&Connection Mode"), ID_DRAW);
	pm_logic->setAccel(Key_C + CTRL, ID_DRAW);
	pm_logic->insertItem(klocale->translate("&Selection Mode"), ID_SEL);
	pm_logic->setAccel(Key_G + CTRL, ID_SEL);
	pm_logic->insertSeparator();
	pm_logic->insertItem(klocale->translate("&Equation"), XDevice::fEQU);
	pm_logic->setAccel(Key_E + CTRL, XDevice::fEQU);
	pm_logic->insertItem(klocale->translate("&AND"), XDevice::fAND);
	pm_logic->setAccel(Key_A + CTRL, XDevice::fAND);
	pm_logic->insertItem(klocale->translate("&OR"), XDevice::fOR);
	pm_logic->setAccel(Key_O + CTRL, XDevice::fOR);
	pm_logic->insertItem(klocale->translate("&XOR"), XDevice::fXOR);
	pm_logic->setAccel(Key_X + CTRL, XDevice::fXOR);
	pm_logic->insertItem(klocale->translate("&Inverter"), XDevice::fONE);
	pm_logic->setAccel(Key_I+ CTRL, XDevice::fONE);
	pm_logic->insertItem(klocale->translate("&RS flipflop"), XDevice::fRS);
	pm_logic->setAccel(Key_R+ CTRL, XDevice::fRS);
	pm_logic->insertItem(klocale->translate("&D flipflop"), XDevice::fDFF);
	pm_logic->setAccel(Key_D+ CTRL, XDevice::fDFF);
	pm_logic->insertItem(klocale->translate("&JK flipflop"), XDevice::fJK);
	pm_logic->setAccel(Key_J+ CTRL, XDevice::fJK);
	pm_logic->insertItem(klocale->translate("Os&cillator"), XDevice::fOSZ);
	pm_logic->setAccel(Key_Z+ CTRL, XDevice::fOSZ);
	pm_logic->insertItem(klocale->translate("&Switch"), XDevice::fSWI);
	pm_logic->setAccel(Key_S+ CTRL, XDevice::fSWI);
	pm_logic->insertItem(klocale->translate("&7-Segment"), XDevice::fSS);
	pm_logic->setAccel(Key_7+ CTRL, XDevice::fSS);
	pm_logic->insertItem(klocale->translate("&LED"), XDevice::fLEDgreen);
	pm_logic->setAccel(Key_L+ CTRL, XDevice::fLEDgreen);
	pm_logic->insertItem(klocale->translate("&Static"), XDevice::fPWR);
	pm_logic->setAccel(Key_P+ CTRL, XDevice::fPWR);
	pm_logic->insertItem(klocale->translate("&Text"), XDevice::fTXT);
	pm_logic->setAccel(Key_T+ CTRL, XDevice::fTXT);
	pm_logic->insertSeparator();
	pm_logic->insertItem(klocale->translate("&new sub-circuit"), XDevice::fNET);
	pm_logic->insertItem(klocale->translate("&input"), XDevice::fIN);
	pm_logic->insertItem(klocale->translate("&output"), XDevice::fOUT);
	pm_logic->setItemEnabled(XDevice::fIN, FALSE);
	pm_logic->setItemEnabled(XDevice::fOUT, FALSE);
	pm_logic->setCheckable(TRUE);
	pm_logic->setItemChecked(ID_DRAW, TRUE);

	pm_sim = new QPopupMenu;
	pm_sim->setCheckable(TRUE);
	pm_sim->insertItem(klocale->translate("Si&mulate"), NetWidget::MODE_SIM_MULT);
	pm_sim->setAccel(Key_M, NetWidget::MODE_SIM_MULT);
	pm_sim->setItemChecked(NetWidget::MODE_SIM_MULT, TRUE);
	pm_sim->insertItem(klocale->translate("&Step through"), NetWidget::MODE_SIM_STEP);
	pm_sim->setAccel(Key_S, NetWidget::MODE_SIM_STEP);
	pm_sim->insertSeparator();
	pm_sim->insertItem(klocale->translate("Show &Graph"), ID_SIM_SHOW);
	pm_sim->setAccel(Key_G, ID_SIM_SHOW);
	pm_sim->insertSeparator();
	pm_sim->insertItem(klocale->translate("Show &Equations"), ID_SIM_EQU);
	pm_sim->setAccel(Key_E, ID_SIM_EQU);

	pm_options = new QPopupMenu;
	pm_options->insertItem(klocale->translate("&Global Settings"), ID_GLOBAL);
	pm_options->setAccel(Key_G + ALT, ID_GLOBAL);
	pm_options->insertItem(klocale->translate("&Hide Toolbar"), ID_TOOL_VISIBLE);
	pm_options->setCheckable(FALSE);

	//pm_help = new QPopupMenu;
	QString version = " KLogic "KLVERSION"\n";
	version += klocale->translate("\n Digital Circuit Simulator\n\n \
	by Andreas Rostin\n \
	www.a-rostin.de/klogic/index.html\n\n \
	please report bugs and suggestions");
	pm_help=helpMenu(version, FALSE);
	CHECK_PTR(pm_help);

	menubar->insertItem(klocale->translate("&File"), pm_file );
	menubar->insertSeparator();
	menubar->insertItem(klocale->translate("&Edit"), pm_edit );
	menubar->insertSeparator();
	menubar->insertItem(klocale->translate("&Logic"), pm_logic );
	menubar->insertSeparator();
	menubar->insertItem(klocale->translate("&Simulation"), pm_sim);
	menubar->insertSeparator();
	menubar->insertItem(klocale->translate("&Options"), pm_options);
	menubar->insertSeparator();
	menubar->insertItem(klocale->translate("&Help"), pm_help);		
	menubar->show();

	connect (pm_file, SIGNAL (activated (int)), SLOT (menuCallback(int)));
	connect (pm_edit, SIGNAL (activated (int)), SLOT (menuCallback(int)));
	connect (pm_logic, SIGNAL (activated (int)), SLOT (menuCallback(int)));
	connect (pm_sim, SIGNAL (activated (int)), SLOT (menuCallback(int)));
	connect (pm_options, SIGNAL (activated (int)), SLOT (menuCallback(int)));
	connect (pm_help, SIGNAL (activated (int)), SLOT (menuCallback(int)));
}

//************* slots **************

void MainWidget::menuCallback(int id)
{	QPrinter prt;
	QPainter p;
	QFileInfo fi;
	QString new_filename;
	int i;

	switch(id) {
		case XDevice::fEQU:
		case XDevice::fAND:
		case XDevice::fOR:
		case XDevice::fXOR:
		case XDevice::fONE:
		case XDevice::fRS:
		case XDevice::fDFF:
		case XDevice::fJK:
		case XDevice::fSS:
		case XDevice::fOSZ:
		case XDevice::fSWI:
		case XDevice::fLEDgreen:
		case XDevice::fIN:
		case XDevice::fOUT:
		case XDevice::fNET:
		case XDevice::fPWR:
		case XDevice::fTXT:
			devChange(id);
			break;
		case ID_OPEN:
			filename = KFileDialog::getOpenFileName(filename, "*.circuit");
			fi.setFile(filename);
			if (fi.exists()) {
				killClients();
				main_netwidget->openNet(filename);
			}
			break;
		case ID_SAVE:
			fi.setFile(filename);
			if (fi.exists())
				main_netwidget->saveNet(filename);
			else
				menuCallback(ID_SAVE_AS);
			break;
		case ID_SAVE_AS:
			fi.setFile(filename);
			new_filename = KFileDialog::getSaveFileName(fi.dirPath(), "*.circuit");
			if (!new_filename.isEmpty()) {
				main_netwidget->saveNet(new_filename);
				filename = new_filename;
			}
			break;
		case ID_PRINT:
			if (TRUE == QPrintDialog::getPrinterSetup(&prt)) {
				p.begin(&prt);
				main_netwidget->printNet(&p);
				p.end();
			}
			break;
		case ID_NEW:
			killClients();
			main_netwidget->newNet();
			fi.setFile(filename);
			filename = fi.dirPath();
			break;
		case ID_EXIT:
			exit(0);
			break;
		case ID_COPY:
			getActive()->rmenuSelCallback(RPOPSEL_COPY);
			break;
		case ID_PASTE:
			getActive()->rmenuSelCallback(RPOPSEL_PASTE);
			break;
		case ID_CUT:
			getActive()->rmenuSelCallback(RPOPSEL_CUT);
			break;
		case ID_GLOBAL:
			dialogGlobal();
			break;
		case ID_TOOL_VISIBLE:
			setToolbar();
			break;
		case ID_DRAW:
			modeDraw();
			break;
		case ID_SEL:
			modeSel();
			break;
		case NetWidget::MODE_SIM_STEP:
			if (NetWidget::MODE_SIM_STEP != main_netwidget->getSimMode()) {
				simMode(id);
				speed_slider->setEnabled(false);
			} else {
				for(i=0;i<MAXCLIENTS;i++)
					if (clientW[i]->net()) clientW[i]->net()->simStep();
			}
			break;
		case NetWidget::MODE_SIM_MULT:
			if (NetWidget::MODE_SIM_MULT != main_netwidget->getSimMode()) {
				simMode(id);
				speed_slider->setEnabled(true);
			}
			break;
		case ID_SIM_SHOW:
			if (simw) {
				simW->hide();	// calls simWidgetHidden() via signal
			} else {
				pm_sim->changeItem(klocale->translate("Hide &Graph"),ID_SIM_SHOW);
				toolbar->setButton(ID_SIM_SHOW, TRUE);
				// display the circuit shown in the main widget
				simW->changeNet(main_netwidget->getActive());
				simW->show();
				simw = 1;
			}
			break;
		case ID_SIM_EQU:
			main_netwidget->showCircuitEquations();
			break;
		default:
			break;
	}
}

void MainWidget::turnAllCheckButton(int id)
{
	// set all buttons off
	toolbar->setButton(ID_DRAW, FALSE);
	menubar->setItemChecked(ID_DRAW, FALSE);
	toolbar->setButton(ID_SEL, FALSE);
	menubar->setItemChecked(ID_SEL, FALSE);
	toolbar->setButton(XDevice::fEQU, FALSE);
	menubar->setItemChecked(XDevice::fEQU, FALSE);
	toolbar->setButton(XDevice::fAND, FALSE);
	menubar->setItemChecked(XDevice::fAND, FALSE);
	toolbar->setButton(XDevice::fOR, FALSE);
	menubar->setItemChecked(XDevice::fOR, FALSE);
	toolbar->setButton(XDevice::fXOR, FALSE);
	menubar->setItemChecked(XDevice::fXOR, FALSE);
	toolbar->setButton(XDevice::fONE, FALSE);
	menubar->setItemChecked(XDevice::fONE, FALSE);
	toolbar->setButton(XDevice::fSS, FALSE);
	menubar->setItemChecked(XDevice::fSS, FALSE);
	toolbar->setButton(XDevice::fRS, FALSE);
	menubar->setItemChecked(XDevice::fDFF, FALSE);
	toolbar->setButton(XDevice::fDFF, FALSE);
	menubar->setItemChecked(XDevice::fRS, FALSE);
	toolbar->setButton(XDevice::fJK, FALSE);
	menubar->setItemChecked(XDevice::fJK, FALSE);
	toolbar->setButton(XDevice::fOSZ, FALSE);
	menubar->setItemChecked(XDevice::fOSZ, FALSE);
	toolbar->setButton(XDevice::fSWI, FALSE);
	menubar->setItemChecked(XDevice::fSWI, FALSE);
	toolbar->setButton(XDevice::fLEDgreen, FALSE);
	menubar->setItemChecked(XDevice::fLEDgreen, FALSE);
	toolbar->setButton(XDevice::fIN, FALSE);
	menubar->setItemChecked(XDevice::fIN, FALSE);
	toolbar->setButton(XDevice::fOUT, FALSE);
	menubar->setItemChecked(XDevice::fOUT, FALSE);
	toolbar->setButton(XDevice::fNET, FALSE);
	menubar->setItemChecked(XDevice::fNET, FALSE);
	toolbar->setButton(XDevice::fPWR, FALSE);
	menubar->setItemChecked(XDevice::fPWR, FALSE);
	toolbar->setButton(XDevice::fTXT, FALSE);
	menubar->setItemChecked(XDevice::fTXT, FALSE);
	toolbar->setButton(id, TRUE);
	menubar->setItemChecked(id, TRUE);
}

void MainWidget::dialogGlobal()
{	int ret;

	GlobProp *dlg = new GlobProp(this, klocale->translate("global settings"), main_netwidget);
	dlg->move(pos().x() + 50, pos().y() + 50);
	dlg->show();
	ret = dlg->result();
	if (ret == QDialog::Accepted)
		writeConfig();
	speed_slider->setValue(NetWidget::MAX_SIMSTEPTIME - NetWidget::simTime());
	delete dlg;
}

void MainWidget::readConfig()
{	int ret;

	kconf->setGroup("device");
	ret = kconf->readNumEntry("size", XDevice::defSize());
	XDevice::setDefSize(ret);
	ret = kconf->readNumEntry("delay", Device::defDelay());
	Device::setDefDelay(ret);
	ret = kconf->readNumEntry("undefined", Device::defUndefined());
	Device::setDefUndefined(ret);
	ret = kconf->readNumEntry("invert", XDevice::isDefInverted());
	XDevice::setDefInverted(ret);
	ret = kconf->readNumEntry("nameDisplayed", XDevice::textDisplayedGlobal());
	XDevice::displayTextGlobal(ret);

	kconf->setGroup("simulation");
	ret = kconf->readNumEntry("time", NetWidget::simTime());
	NetWidget::setSimTime(ret);
	ret = kconf->readNumEntry("burst", NetWidget::simBurst());
	NetWidget::setSimBurst(ret);

	kconf->setGroup("pulsed");
	ret = kconf->readNumEntry("clock", Device::defClock());
	Device::setDefClock(ret);
	ret = kconf->readNumEntry("reverse_trigger", Device::triggerInverted());
	Device::invertTrigger(ret);
}

void MainWidget::writeConfig()
{
	kconf->setGroup("device");
	kconf->writeEntry("size", XDevice::defSize());
	kconf->writeEntry("delay", Device::defDelay());
	kconf->writeEntry("undefined", Device::defUndefined());
	kconf->writeEntry("invert", XDevice::isDefInverted());
	kconf->writeEntry("nameDisplayed", XDevice::textDisplayedGlobal());

	kconf->setGroup("simulation");
	kconf->writeEntry("time", NetWidget::simTime());
	kconf->writeEntry("burst", NetWidget::simBurst());

	kconf->setGroup("pulsed");
	kconf->writeEntry("clock", Device::defClock());
	kconf->writeEntry("reverse_trigger", Device::triggerInverted());
	kconf->sync();
}


void MainWidget::setToolbar()
{
	if (tool_visible == 1) {
		toolbar->hide();
		menubar->changeItem(klocale->translate("Show &Toolbar"),ID_TOOL_VISIBLE);
		tool_visible = 0;
	} else {
		toolbar->show();
		menubar->changeItem(klocale->translate("Hide &Toolbar"),ID_TOOL_VISIBLE);
		tool_visible = 1;
	}
	//repaint(FALSE);
}

// change mode of drawing (drop devices OR draw wires / move objects)
void MainWidget::devChange(int i)
{
	NetWidget::setDrawMode(NetWidget::MODE_DROP);
	NetWidget::setDevice(i);
	turnAllCheckButton(i);
	for (i=0;i<MAXCLIENTS;i++)
		if (clientW[i]->net()) clientW[i]->net()->removeSelection();
}

// slot
// change mode of drawing (drop objects/wire or move objects)
void MainWidget::modeDraw()
{	int i;

	NetWidget::setDrawMode(NetWidget::MODE_DRAW);
	turnAllCheckButton(ID_DRAW);
	for (i=0;i<MAXCLIENTS;i++)
		if (clientW[i]->net()) clientW[i]->net()->removeSelection();
}

// slot
// change mode of drawing (make selections)
void MainWidget::modeSel()
{
	NetWidget::setDrawMode(NetWidget::MODE_SEL);
	turnAllCheckButton(ID_SEL);
}

// slot
// set toolbar buttons/menu entries in dependence of the new net
void MainWidget::netChange(int hasparent)
{	int i = 0;

	// check if there are open sub circuits left
	while (!hasparent && (i < MAXCLIENTS)) {
		if ((clientW[i]->net()) && clientW[i]->net()->getActive()->parent()) hasparent = 1;
		i++;
	}

	// if there are no sub-circuits anymore, lock the devices input and output
	if (hasparent) {
		pm_logic->setItemEnabled(XDevice::fIN, TRUE);
		pm_logic->setItemEnabled(XDevice::fOUT, TRUE);
		toolbar->showItem(XDevice::fIN);
		toolbar->showItem(XDevice::fOUT);
	} else {
		pm_logic->setItemEnabled(XDevice::fIN, FALSE);
		pm_logic->setItemEnabled(XDevice::fOUT, FALSE);
		toolbar->hideItem(XDevice::fIN);
		toolbar->hideItem(XDevice::fOUT);
		if (NetWidget::getDrawMode() == NetWidget::MODE_DROP) {
			NetWidget::setDrawMode(NetWidget::MODE_DRAW);
			turnAllCheckButton(ID_DRAW);
		}
	}
}

// slot
// display circuit in a new window
void MainWidget::newWidget(XDeviceNet *devnet)
{	int idx = 0;
	int i=0;
	NetWidget *newnet;

	while(!idx && (i++ < MAXCLIENTS)) {
		if (!(clientW[i]->net())) idx = i;
	}
	if (!idx) {
		QMessageBox::information(this, klocale->translate("sorry.."), klocale->translate("maximum number of windows reached.."));
		return;
	}
	clientW[idx]->show(devnet);
	newnet = clientW[idx]->net();
	connect(this, SIGNAL(idChange(int, int)), clientW[idx], SLOT(idChanged(int, int)));
	connect(newnet, SIGNAL(createNewWidget(XDeviceNet *)), this, SLOT(newWidget(XDeviceNet *)));
	connect(newnet, SIGNAL(netDeleted(XDeviceNet *)), this, SLOT(netDelete(XDeviceNet *)));
	// NetWidget reports to clientw and this
	connect(newnet, SIGNAL(netChanged(int)), clientW[idx], SLOT(netChange(int)));
	connect(newnet, SIGNAL(netChanged(int)), this, SLOT(netChange(int)));

	// device deleted which was shown in the simulation graph
        connect(newnet, SIGNAL(graphChanged()), simW, SLOT(graphChange()));
	// NetWidget shown in simulation graph has changed
        connect(newnet, SIGNAL(showGraph(XDeviceNet *)), simW, SLOT(changeNet(XDeviceNet *)));
	// perform a simulation stepin the simulation graph
        connect(newnet, SIGNAL(simStepped()), simW, SLOT(simStep()));

	if (newnet->getActive()->parent()) netChange(1);
	else netChange(0);
}

// slot
// change the simulation step time via the toolbar slider
void MainWidget::simTimeSliderChanged(int value)
{
	NetWidget::setSimTime(NetWidget::MAX_SIMSTEPTIME - value);
}

// slot: signal netContentChanged coming from NetWidget
// find active window, redraw it and all client widgets with the same content
void MainWidget::repaintClients()
{	NetWidget *activeNetWidget = getActive();
	XDeviceNet *activeDeviceNet;
	int j;

	if (activeNetWidget) {
		activeDeviceNet = activeNetWidget->getActive();

		// find inactive widgets containing the same net
		for(j=0; j<MAXCLIENTS; j++) {
			if ((clientW[j]->net()) && (activeNetWidget != clientW[j]->net()) && (clientW[j]->net()->getActive() == activeDeviceNet))
				clientW[j]->net()->repaint(FALSE);
		}
	}
}

// find the currently active netwidget
NetWidget * MainWidget::getActive()
{	int i;

	for(i=0; i <MAXCLIENTS; i++) {
		if (clientW[i]->net() && clientW[i]->isActiveWindow()) return clientW[i]->net();
	}
	return (NetWidget *)NULL;
}

// slot
// sub circuit deleted
void MainWidget::netDelete(XDeviceNet *delnet)
{	int idx;
	NetWidget *oldnet;

	// close all client widgets containing the deleted circuit
	for(idx=0; idx < MAXCLIENTS; idx++) {
		// check if *delnet or a parent of *delnet is displayed in the widget
		if ((clientW[idx]->net()) && (clientW[idx]->net()->contained(delnet))) {
			oldnet = clientW[idx]->net();
			if (oldnet) {
				disconnect(this, SIGNAL(idChange(int, int)), clientW[idx], SLOT(idChanged(int, int)));
				disconnect(oldnet, SIGNAL(createNewWidget(XDeviceNet *)), this, SLOT(newWidget(XDeviceNet *)));
				disconnect(oldnet, SIGNAL(netDeleted(XDeviceNet *)), this, SLOT(netDelete(XDeviceNet *)));
				disconnect(oldnet, SIGNAL(netChanged(int)), clientW[idx], SLOT(netChange(int)));
				disconnect(oldnet, SIGNAL(netChanged(int)), this, SLOT(netChange(int)));
				// signals for the widget showing simulation graphs
				disconnect(oldnet, SIGNAL(graphChanged()), simW, SLOT(graphChange()));
				disconnect(oldnet, SIGNAL(showGraph(XDeviceNet *)), simW, SLOT(changeNet(XDeviceNet *)));
				disconnect(oldnet, SIGNAL(simStepped()), simW, SLOT(simStep()));
			}
			clientW[idx]->hide();
		}
	}

	// check if deleted circuit is shown in the sim.graph
	if (simW->currentNet() == delnet) simW->changeNet(main_netwidget->getActive());
	netChange(0);
}

// remove all client windows
void MainWidget::killClients()
{
	for (int i=1;i < MAXCLIENTS; i++)
		clientW[i]->hide();
	netChange(0);
}

// slot
// client widget goes hidden
void MainWidget::clientHidden(ClientW *client)
{	int idx = 0;
	NetWidget *oldnet;

	while ((idx++ < MAXCLIENTS) && (clientW[idx] != client));
	if ((idx < MAXCLIENTS) && (clientW[idx] == client)) {
		oldnet = clientW[idx]->net();
		if (oldnet) {
			disconnect(this, SIGNAL(idChange(int, int)), clientW[idx], SLOT(idChanged(int, int)));
			disconnect(oldnet, SIGNAL(createNewWidget(XDeviceNet *)), this, SLOT(newWidget(XDeviceNet *)));
			disconnect(oldnet, SIGNAL(netDeleted(XDeviceNet *)), this, SLOT(netDelete(XDeviceNet *)));
			disconnect(oldnet, SIGNAL(netChanged(int)), clientW[idx], SLOT(netChange(int)));
			disconnect(oldnet, SIGNAL(netChanged(int)), this, SLOT(netChange(int)));
			// signals for the widget showing simulation graphs
			disconnect(oldnet, SIGNAL(graphChanged()), simW, SLOT(graphChange()));
			disconnect(oldnet, SIGNAL(showGraph(XDeviceNet *)), simW, SLOT(changeNet(XDeviceNet *)));
			disconnect(oldnet, SIGNAL(simStepped()), simW, SLOT(simStep()));
		}
		clientW[idx]->hide();
		netChange(0);
	}
}

// slot
// simulation widget goes hidden
void MainWidget::simWidgetHidden()
{
	simw = 0;
	toolbar->setButton(ID_SIM_SHOW, FALSE);
	pm_sim->changeItem(klocale->translate("Show &Graph"),ID_SIM_SHOW);
}

// change mode of simulation
void MainWidget::simMode(int i)
{
	NetWidget::setSimMode(i);
	simW->setSimMode(i);

	toolbar->setButton(NetWidget::MODE_SIM_STEP, FALSE);
	pm_sim->setItemChecked(NetWidget::MODE_SIM_STEP, FALSE);
	toolbar->setButton(NetWidget::MODE_SIM_MULT, FALSE);
	pm_sim->setItemChecked(NetWidget::MODE_SIM_MULT, FALSE);
	toolbar->setButton(i, TRUE);
	pm_sim->setItemChecked(i, TRUE);
}

