/*
 *  KSeg
 *  Copyright (C) 1999 Ilya Baran
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Send comments and/or bug reports to:
 *                 ibaran@mit.edu
 */


#include <kapp.h>
#include <kfiledialog.h>
#include <qmessagebox.h>
#include <qcolordialog.h>
#include <qfontdialog.h>
#include <qsplitter.h>
#include "KSegConstructionList.H"
#include "KSegWindow.H"
#include "KSegView.H"
#include "KSegDocument.H"
#include "KSegConstruction.H"
#include "G_ref.H"
#include "defs.H"


KSegWindow::KSegWindow(KSegDocument *doc)
  : QMainWindow(0, 0, WDestructiveClose | WType_TopLevel)
{
  if(doc == 0) {
    doc = new KSegDocument();
  }
  else {
    if(doc->getFilename().isEmpty()) setCaption("kseg: Untitled");
    else setCaption(QString("kseg: ") + doc->getFilename());
  }

  if(doc->isConstruction() == false) {
    view = new KSegView(doc, this);

    setCentralWidget(view);

    resize(640, 480);
  }
  else {
    QSplitter *sp = new QSplitter(this);

    view = new KSegView(doc, sp);
    KSegConstructionList *lb = new KSegConstructionList(doc, sp);

    sp->setOrientation(Horizontal);
    sp->setOpaqueResize(true);

    setCentralWidget(sp);

    resize(640, 480);
    lb->resize(50, 480);
  }

  connect(view, SIGNAL(statusBarMessage(const QString &)), statusBar(), SLOT(message(const QString &)));
  connect(view, SIGNAL(updateMenus()), this, SLOT(updateMenus()));

  fileMenu = new QPopupMenu(this);
  newMenu = new QPopupMenu(this);
  editMenu = new QPopupMenu(this);
  colorMenu = new QPopupMenu(this);
  pointstyleMenu = new QPopupMenu(this);
  linestyleMenu = new QPopupMenu(this);
  fontMenu = new QPopupMenu(this);
  measureMenu = new QPopupMenu(this);
  transformMenu = new QPopupMenu(this);
  playMenu = new QPopupMenu(this);
  if(doc->isConstruction())
    constructionMenu = new QPopupMenu(this);

  QPopupMenu *helpMenu = new QPopupMenu(this);;

  connect(newMenu, SIGNAL(aboutToShow()), this, SLOT(updateNewMenu()));
  connect(editMenu, SIGNAL(aboutToShow()), this, SLOT(updateEditMenu()));
  connect(measureMenu, SIGNAL(aboutToShow()), this, SLOT(updateMeasureMenu()));
  connect(transformMenu, SIGNAL(aboutToShow()), this, SLOT(updateTransformMenu()));
  connect(playMenu, SIGNAL(aboutToShow()), this, SLOT(updatePlayMenu()));
  if(doc->isConstruction())
    connect(constructionMenu, SIGNAL(aboutToShow()), this, SLOT(updateConstructionMenu()));

  menuBar()->insertItem("&File", fileMenu);
  menuBar()->insertItem("&Edit", editMenu);
  menuBar()->insertItem("&New", newMenu);
  menuBar()->insertItem("Meas&ure", measureMenu);
  menuBar()->insertItem("&Transform", transformMenu);
  menuBar()->insertItem("Pla&y", playMenu);
  if(doc->isConstruction())
    menuBar()->insertItem("Const&ruction", constructionMenu);
  menuBar()->insertSeparator();
  menuBar()->insertItem("&Help", helpMenu);

  //file menu init:
  fileMenu->insertItem("&New Sketch", this, SLOT(newSketch()), CTRL+Key_N);
  fileMenu->insertItem("Ne&w Construction", this, SLOT(newConstruction()));
  fileMenu->insertItem("New &View", this, SLOT(newView()));
  fileMenu->insertItem("&Open...", this, SLOT(fileOpen()), CTRL+Key_O);
  fileMenu->insertItem("Save &As...", this, SLOT(fileSaveAs()));
  fileMenu->insertItem("&Save", this, SLOT(fileSave()), CTRL+Key_S);

  fileMenu->insertSeparator();
  
  fileMenu->insertItem("&Print...", this, SLOT(filePrint()), CTRL+Key_P);

  fileMenu->insertSeparator();

  fileMenu->insertItem("&Close", this, SLOT(close()), CTRL+Key_W);
  fileMenu->insertItem("&Quit", kapp, SLOT(closeAllWindows()), CTRL+Key_Q);

  //edit menu init:
  editMenu->insertItem("&Undo", view->getDocument(), SLOT(editUndo()),
		       CTRL+Key_Z, ID_EDIT_UNDO);
  editMenu->insertItem("&Redo", view->getDocument(), SLOT(editRedo()),
		       CTRL+Key_R, ID_EDIT_REDO);

  editMenu->insertSeparator();

  editMenu->insertItem("&Delete Objects", view->getDocument(), SLOT(editDelete()),
		       CTRL+Key_Delete, ID_EDIT_DELETE);
  editMenu->insertSeparator();
  editMenu->insertItem("Toggle &Label", view->getDocument(),
		       SLOT(editToggleLabels()), CTRL+Key_L, ID_EDIT_TOGGLELABELS);
  editMenu->insertItem("Change La&bel...", view->getDocument(),
		       SLOT(editChangeLabel()), 0, ID_EDIT_CHANGELABEL);
  editMenu->insertItem("&Hide Objects", view->getDocument(), SLOT(editHide()),
		       CTRL+Key_H, ID_EDIT_HIDE);
  editMenu->insertItem("&Unhide All", view->getDocument(), SLOT(editShowHidden()),
		       CTRL+Key_U, ID_EDIT_SHOWHIDDEN);

  editMenu->insertSeparator();

  editMenu->insertItem("&Color", colorMenu, ID_EDIT_COLOR);
  editMenu->insertItem("Line &Style", linestyleMenu, ID_EDIT_LINESTYLE);
  editMenu->insertItem("&Point Style", pointstyleMenu, ID_EDIT_POINTSTYLE);
  editMenu->insertItem("&Font", fontMenu, ID_EDIT_FONT);

  editMenu->insertSeparator();

  editMenu->insertItem("Change &Number of Samples...", view->getDocument(),
		       SLOT(editChangeNumberOfSamples()), 0,
		       ID_EDIT_CHANGE_NUMBER_OF_SAMPLES);
  

  //edit color menu init:
  QPixmap p(100, 20);

  p.fill(black);
  colorMenu->insertItem(p, this, SLOT(editColorBlack()), 0, ID_EDIT_COLOR_BLACK);
  p.fill(gray);
  colorMenu->insertItem(p, this, SLOT(editColorGray()), 0, ID_EDIT_COLOR_GRAY);
  p.fill(red);
  colorMenu->insertItem(p, this, SLOT(editColorRed()), 0, ID_EDIT_COLOR_RED);
  p.fill(green);
  colorMenu->insertItem(p, this, SLOT(editColorGreen()), 0, ID_EDIT_COLOR_GREEN);
  p.fill(blue);
  colorMenu->insertItem(p, this, SLOT(editColorBlue()), 0, ID_EDIT_COLOR_BLUE);
  p.fill(yellow);
  colorMenu->insertItem(p, this, SLOT(editColorYellow()), 0, ID_EDIT_COLOR_YELLOW);
  p.fill(magenta);
  colorMenu->insertItem(p, this, SLOT(editColorPurple()), 0, ID_EDIT_COLOR_PURPLE);
  p.fill(cyan);
  colorMenu->insertItem(p, this, SLOT(editColorCyan()), 0, ID_EDIT_COLOR_CYAN);

  colorMenu->insertSeparator();

  colorMenu->insertItem("&Other...", this, SLOT(editColorOther()), 0,
			ID_EDIT_COLOR_OTHER);

  //edit pointstyle menu init:
  QPainter pa;
  pa.begin(&p);
  G_drawstyle *d;
  G_point pt(50, 10);

  p.fill(lightGray);
  d = G_drawstyle::match(LARGE_CIRCLE);
  pt.draw(pa, *d);
  d->deleteReference();
  pointstyleMenu->insertItem(p, this, SLOT(editPointstyleLargecircle()), 0,
			    ID_EDIT_POINTSTYLE_LARGECIRCLE);

  p.fill(lightGray);
  d = G_drawstyle::match(MEDIUM_CIRCLE);
  pt.draw(pa, *d);
  d->deleteReference();
  pointstyleMenu->insertItem(p, this, SLOT(editPointstyleMediumcircle()), 0,
			    ID_EDIT_POINTSTYLE_MEDIUMCIRCLE);

  p.fill(lightGray);
  d = G_drawstyle::match(SMALL_CIRCLE);
  pt.draw(pa, *d);
  d->deleteReference();
  pointstyleMenu->insertItem(p, this, SLOT(editPointstyleSmallcircle()), 0,
			    ID_EDIT_POINTSTYLE_SMALLCIRCLE);


  //edit linestyle menu init:
  QPen solid(SolidLine), dashed(DashLine), dotted(DotLine);
  QPen thin(black, 0), normal(black, 2), thick(black, 3);

  p.fill(lightGray);  
  pa.setPen(solid);
  pa.drawLine(0, 10, 100, 10);
  linestyleMenu->insertItem(p, this, SLOT(editLinestyleSolid()), 0,
			    ID_EDIT_LINESTYLE_SOLID);

  p.fill(lightGray);  
  pa.setPen(dashed);
  pa.drawLine(0, 10, 100, 10);
  linestyleMenu->insertItem(p, this, SLOT(editLinestyleDashed()), 0,
			    ID_EDIT_LINESTYLE_DASHED);

  p.fill(lightGray);  
  pa.setPen(dotted);
  pa.drawLine(0, 10, 100, 10);
  linestyleMenu->insertItem(p, this, SLOT(editLinestyleDotted()), 0,
			    ID_EDIT_LINESTYLE_DOTTED);

  linestyleMenu->insertSeparator();

  p.fill(lightGray);  
  pa.setPen(thin);
  pa.drawLine(0, 10, 100, 10);
  linestyleMenu->insertItem(p, this, SLOT(editLinestyleThin()), 0,
			    ID_EDIT_LINESTYLE_THIN);

  p.fill(lightGray);  
  pa.setPen(normal);
  pa.drawLine(0, 10, 100, 10);
  linestyleMenu->insertItem(p, this, SLOT(editLinestyleNormal()), 0,
			    ID_EDIT_LINESTYLE_NORMAL);

  p.fill(lightGray);  
  pa.setPen(thick);
  pa.drawLine(0, 10, 100, 10);
  linestyleMenu->insertItem(p, this, SLOT(editLinestyleThick()), 0,
			    ID_EDIT_LINESTYLE_THICK);


  //edit font menu init:
  fontMenu->insertItem("10", this, SLOT(editFont10()), 0, ID_EDIT_FONT_10);
  fontMenu->insertItem("12", this, SLOT(editFont12()), 0, ID_EDIT_FONT_12);
  fontMenu->insertItem("14", this, SLOT(editFont14()), 0, ID_EDIT_FONT_14);
  fontMenu->insertItem("24", this, SLOT(editFont20()), 0, ID_EDIT_FONT_20);
  fontMenu->insertItem("30", this, SLOT(editFont30()), 0, ID_EDIT_FONT_30);

  fontMenu->insertSeparator();
  
  fontMenu->insertItem("&Font...", this, SLOT(editFontFont()), 0, ID_EDIT_FONT_FONT);

  //new menu init:
  newMenu->insertItem("&Intersection Point", view->getDocument(),
		      SLOT(newIntersection()), ALT+Key_I, ID_NEW_INTERSECTION);
  newMenu->insertItem("&Segment", view->getDocument(), SLOT(newSegment()),
		      ALT+Key_S, ID_NEW_SEGMENT);
  newMenu->insertItem("&Midpoint", view->getDocument(), SLOT(newMidpoint()),
		      ALT+Key_M, ID_NEW_MIDPOINT);
  newMenu->insertItem("&Ray", view->getDocument(), SLOT(newRay()), 0, ID_NEW_RAY);
  newMenu->insertItem("&Line", view->getDocument(), SLOT(newLine()), ALT+Key_L,
		      ID_NEW_LINE);
  newMenu->insertItem("&Perpendicular Line", view->getDocument(),
		      SLOT(newPerpendicular()), ALT+Key_P, ID_NEW_PERPENDICULAR);
  newMenu->insertItem("&Circle By Center And Point", view->getDocument(),
		      SLOT(newCircle()), ALT+Key_C, ID_NEW_CIRCLE);
  newMenu->insertItem("&Arc By Three Points", view->getDocument(),
		      SLOT(newArc()), ALT+Key_A, ID_NEW_ARC);
  newMenu->insertItem("Angle &Bisector", view->getDocument(),
		      SLOT(newBisector()), ALT+Key_B, ID_NEW_BISECTOR);
  newMenu->insertItem("L&ocus", view->getDocument(),
		      SLOT(newLocus()), ALT+Key_O, ID_NEW_LOCUS);

  //measure menu init:
  measureMenu->insertItem("&Distance", view->getDocument(),
			  SLOT(measureDistance()), 0, ID_MEASURE_DISTANCE);
  measureMenu->insertItem("&Length", view->getDocument(),
			  SLOT(measureLength()), 0, ID_MEASURE_LENGTH);
  measureMenu->insertItem("&Radius", view->getDocument(),
			  SLOT(measureRadius()), 0, ID_MEASURE_RADIUS);
  measureMenu->insertItem("&Angle", view->getDocument(),
			  SLOT(measureAngle()), 0, ID_MEASURE_ANGLE);
  measureMenu->insertItem("Ra&tio", view->getDocument(),
			  SLOT(measureRatio()), 0, ID_MEASURE_RATIO);
  measureMenu->insertItem("&Slope", view->getDocument(),
			  SLOT(measureSlope()), 0, ID_MEASURE_SLOPE);

  measureMenu->insertSeparator();

  measureMenu->insertItem("&Calculate...", view->getDocument(),
			  SLOT(measureCalculate()), 0, ID_MEASURE_CALCULATE);

  //transform menu init:
  transformMenu->insertItem("Choose &Vector", view->getDocument(),
			    SLOT(transformChooseVector()), 0,
			    ID_TRANSFORM_CHOOSE_VECTOR);
  transformMenu->insertItem("Choose &Mirror", view->getDocument(),
			    SLOT(transformChooseMirror()), 0,
			    ID_TRANSFORM_CHOOSE_MIRROR);
  transformMenu->insertItem("Choose &Center", view->getDocument(),
			    SLOT(transformChooseCenter()), 0,
			    ID_TRANSFORM_CHOOSE_CENTER);
  transformMenu->insertItem("Choose &Angle", view->getDocument(),
			    SLOT(transformChooseAngle()), 0,
			    ID_TRANSFORM_CHOOSE_ANGLE);
  transformMenu->insertItem("Choose Rati&o", view->getDocument(),
			    SLOT(transformChooseRatio()), 0,
			    ID_TRANSFORM_CHOOSE_RATIO);

  transformMenu->insertSeparator();

  transformMenu->insertItem("&Translate", view->getDocument(),
			    SLOT(transformTranslate()), 0, ID_TRANSFORM_TRANSLATE);
  transformMenu->insertItem("R&eflect", view->getDocument(),
			    SLOT(transformReflect()), 0, ID_TRANSFORM_REFLECT);
  transformMenu->insertItem("&Scale", view->getDocument(),
			    SLOT(transformScale()), 0, ID_TRANSFORM_SCALE);
  transformMenu->insertItem("&Rotate", view->getDocument(),
			    SLOT(transformRotate()), 0, ID_TRANSFORM_ROTATE);


  //construction menu init
  if(doc->isConstruction()) {
    constructionMenu->insertItem("Make &Normal", view->getDocument(),
				 SLOT(constructionMakeNormal()), 0,
				 ID_CONSTRUCTION_MAKE_NORMAL);
 
    constructionMenu->insertItem("Make &Given", view->getDocument(),
				 SLOT(constructionMakeGiven()), 0,
				 ID_CONSTRUCTION_MAKE_GIVEN);

    constructionMenu->insertItem("Make &Final", view->getDocument(),
				 SLOT(constructionMakeFinal()), 0,
				 ID_CONSTRUCTION_MAKE_FINAL);

    constructionMenu->insertItem("Make &Initial", view->getDocument(),
				 SLOT(constructionMakeInitial()), 0,
				 ID_CONSTRUCTION_MAKE_INITIAL);

    constructionMenu->insertSeparator();

    constructionMenu->insertItem("&Recurse", view->getDocument(),
				 SLOT(constructionRecurse()), 0,
				 ID_CONSTRUCTION_RECURSE);
  }


  //help menu init
  helpMenu->insertItem("&About...", this, SLOT(helpAbout()));

  statusBar()->show();
}


void KSegWindow::statusBarMessage(const QString &s)
{
  statusBar()->message(s);
}


void KSegWindow::updateMenus()
{
  if(view->getDrag() != KSegView::NO_DRAG || view->getMenusEnabled() == false) {
    menuBar()->setEnabled(false);

    int i;

    for(i = 0; i < (int)editMenu->count(); i++) {
      editMenu->setItemEnabled(editMenu->idAt(i), false);
    }

    for(i = 0; i < (int)newMenu->count(); i++) {
      newMenu->setItemEnabled(newMenu->idAt(i), false);
    }

    for(i = 0; i < (int)measureMenu->count(); i++) {
      measureMenu->setItemEnabled(measureMenu->idAt(i), false);
    }

    for(i = 0; i < (int)transformMenu->count(); i++) {
      transformMenu->setItemEnabled(transformMenu->idAt(i), false);
    }
  }
  else {
    menuBar()->setEnabled(true);
    updateNewMenu();
    updateEditMenu();
    updateMeasureMenu();
    updateTransformMenu();
    updatePlayMenu();
    if(view->getDocument()->isConstruction()) updateConstructionMenu();
  }
}


void KSegWindow::updateTransformMenu()
{
  transformMenu->setItemEnabled(ID_TRANSFORM_CHOOSE_VECTOR,
			   view->getDocument()->canTransformChooseVector());
  transformMenu->setItemEnabled(ID_TRANSFORM_CHOOSE_ANGLE,
			   view->getDocument()->canTransformChooseAngle());
  transformMenu->setItemEnabled(ID_TRANSFORM_CHOOSE_RATIO,
			   view->getDocument()->canTransformChooseRatio());
  transformMenu->setItemEnabled(ID_TRANSFORM_CHOOSE_CENTER,
			   view->getDocument()->canTransformChooseCenter());
  transformMenu->setItemEnabled(ID_TRANSFORM_CHOOSE_MIRROR,
			   view->getDocument()->canTransformChooseMirror());

  transformMenu->setItemEnabled(ID_TRANSFORM_TRANSLATE,
			   view->getDocument()->canTransformTranslate());
  transformMenu->setItemEnabled(ID_TRANSFORM_REFLECT,
			   view->getDocument()->canTransformReflect());
  transformMenu->setItemEnabled(ID_TRANSFORM_ROTATE,
			   view->getDocument()->canTransformRotate());
  transformMenu->setItemEnabled(ID_TRANSFORM_SCALE,
			   view->getDocument()->canTransformScale());

}


void KSegWindow::updateMeasureMenu()
{
  bool tmp;

  measureMenu->setItemEnabled(ID_MEASURE_DISTANCE,
			   view->getDocument()->canMeasureDistance());

  tmp = view->getDocument()->canMeasureLength(); 
  if(!tmp || view->getDocument()->getSelected()[0]->getType() == G_SEGMENT) {
    measureMenu->changeItem(ID_MEASURE_LENGTH, "&Length");
  }
  else if(view->getDocument()->getSelected()[0]->getType() == G_CIRCLE) {
    measureMenu->changeItem(ID_MEASURE_LENGTH, "&Circumference");
  }
  else { //the object is an arc
    measureMenu->changeItem(ID_MEASURE_LENGTH, "Arc &Length");
  }
  measureMenu->setItemEnabled(ID_MEASURE_LENGTH, tmp);


  measureMenu->setItemEnabled(ID_MEASURE_RADIUS,
			   view->getDocument()->canMeasureRadius());
  measureMenu->setItemEnabled(ID_MEASURE_ANGLE,
			   view->getDocument()->canMeasureAngle());
  measureMenu->setItemEnabled(ID_MEASURE_RATIO,
			   view->getDocument()->canMeasureRatio());
  measureMenu->setItemEnabled(ID_MEASURE_SLOPE,
			   view->getDocument()->canMeasureSlope());
  measureMenu->setItemEnabled(ID_MEASURE_CALCULATE,
			   view->getDocument()->canMeasureCalculate());
}

void KSegWindow::updateEditMenu()
{
  editMenu->setItemEnabled(ID_EDIT_UNDO, view->getDocument()->canEditUndo());
  editMenu->setItemEnabled(ID_EDIT_REDO, view->getDocument()->canEditRedo());
  editMenu->setItemEnabled(ID_EDIT_DELETE, view->getDocument()->canEditDelete());

  bool tmp;
  tmp = view->getDocument()->canEditToggleLabels();
  if(tmp && view->getDocument()->selectedCount() > 1) {
    newMenu->changeItem(ID_EDIT_TOGGLELABELS, "Toggle &Labels");
  }
  else newMenu->changeItem(ID_EDIT_TOGGLELABELS, "Toggle &Label");
  editMenu->setItemEnabled(ID_EDIT_TOGGLELABELS, tmp);

  editMenu->setItemEnabled(ID_EDIT_CHANGELABEL,
			   view->getDocument()->canEditChangeLabel());
  editMenu->setItemEnabled(ID_EDIT_HIDE, view->getDocument()->canEditHide());
  editMenu->setItemEnabled(ID_EDIT_SHOWHIDDEN,
			   view->getDocument()->canEditShowHidden());

  tmp = view->getDocument()->canEditChangeColor();
  editMenu->setItemEnabled(ID_EDIT_COLOR, tmp);
  if(tmp) updateEditColorMenu();

  tmp = view->getDocument()->canEditChangePointstyle();
  editMenu->setItemEnabled(ID_EDIT_POINTSTYLE, tmp);
  if(tmp) updateEditPointstyleMenu();

  tmp = view->getDocument()->canEditChangeLinestyle();
  editMenu->setItemEnabled(ID_EDIT_LINESTYLE, tmp);
  if(tmp) updateEditLinestyleMenu();

  tmp = view->getDocument()->canEditChangeFont();
  editMenu->setItemEnabled(ID_EDIT_FONT, tmp);
  if(tmp) updateEditFontMenu();

  editMenu->setItemEnabled(ID_EDIT_CHANGE_NUMBER_OF_SAMPLES,
			   view->getDocument()->canEditChangeNumberOfSamples());
}


void KSegWindow::updateEditColorMenu()
{
  QColor c = view->getDocument()->getCurrentColor();

  colorMenu->setItemChecked(ID_EDIT_COLOR_BLACK, c == black);
  colorMenu->setItemChecked(ID_EDIT_COLOR_GRAY, c == gray);
  colorMenu->setItemChecked(ID_EDIT_COLOR_RED, c == red);
  colorMenu->setItemChecked(ID_EDIT_COLOR_GREEN, c == green);
  colorMenu->setItemChecked(ID_EDIT_COLOR_BLUE, c == blue);
  colorMenu->setItemChecked(ID_EDIT_COLOR_YELLOW, c == yellow);
  colorMenu->setItemChecked(ID_EDIT_COLOR_PURPLE, c == magenta);
  colorMenu->setItemChecked(ID_EDIT_COLOR_CYAN, c == cyan);

  if(c.isValid() && c != black && c != gray && c != red && c != green &&
     c != blue && c != yellow && c != magenta && c != cyan) {
    colorMenu->setItemChecked(ID_EDIT_COLOR_OTHER, true);
  }
  else {
    colorMenu->setItemChecked(ID_EDIT_COLOR_OTHER, false);
  }
}


void KSegWindow::updateEditPointstyleMenu()
{
  PointStyle style = view->getDocument()->getCurrentPointstyle();

  pointstyleMenu->setItemChecked(ID_EDIT_POINTSTYLE_LARGECIRCLE, style == LARGE_CIRCLE);
  pointstyleMenu->setItemChecked(ID_EDIT_POINTSTYLE_MEDIUMCIRCLE,
				 style == MEDIUM_CIRCLE);
  pointstyleMenu->setItemChecked(ID_EDIT_POINTSTYLE_SMALLCIRCLE, style == SMALL_CIRCLE);
}


void KSegWindow::updateEditLinestyleMenu()
{
  PenStyle style = view->getDocument()->getCurrentPenstyle();
  int width = view->getDocument()->getCurrentPenWidth();

  linestyleMenu->setItemChecked(ID_EDIT_LINESTYLE_SOLID, style == SolidLine);
  linestyleMenu->setItemChecked(ID_EDIT_LINESTYLE_DASHED, style == DashLine);
  linestyleMenu->setItemChecked(ID_EDIT_LINESTYLE_DOTTED, style == DotLine);

  linestyleMenu->setItemChecked(ID_EDIT_LINESTYLE_THIN, width == 0);
  linestyleMenu->setItemChecked(ID_EDIT_LINESTYLE_NORMAL, width == 2);
  linestyleMenu->setItemChecked(ID_EDIT_LINESTYLE_THICK, width == 3);
}


void KSegWindow::updateEditFontMenu()
{
  int size = view->getDocument()->getCurrentFontsize();

  fontMenu->setItemChecked(ID_EDIT_FONT_10, size == 10);
  fontMenu->setItemChecked(ID_EDIT_FONT_12, size == 12);
  fontMenu->setItemChecked(ID_EDIT_FONT_14, size == 14);
  fontMenu->setItemChecked(ID_EDIT_FONT_20, size == 20);
  fontMenu->setItemChecked(ID_EDIT_FONT_30, size == 30);
}


void KSegWindow::updateNewMenu()
{
  bool tmp;
  int i;

  //segment
  tmp = view->getDocument()->canNewSegment();
  newMenu->setItemEnabled(ID_NEW_SEGMENT, tmp);
  if(tmp && view->getDocument()->selectedCount() > 2)
    newMenu->changeItem(ID_NEW_SEGMENT, "&Segments");
  else newMenu->changeItem(ID_NEW_SEGMENT, "&Segment");

  //ray
  tmp = view->getDocument()->canNewRay();
  newMenu->setItemEnabled(ID_NEW_RAY, tmp);
  if(tmp && view->getDocument()->selectedCount() > 2)
    newMenu->changeItem(ID_NEW_RAY, "&Rays");
  else newMenu->changeItem(ID_NEW_RAY, "&Ray");

  //bisector
  newMenu->setItemEnabled(ID_NEW_BISECTOR, view->getDocument()->canNewBisector());

  //arc
  newMenu->setItemEnabled(ID_NEW_ARC, view->getDocument()->canNewArc());

  //locus
  newMenu->setItemEnabled(ID_NEW_LOCUS, view->getDocument()->canNewLocus());
      
  //midpoint
  tmp = view->getDocument()->canNewMidpoint();
  newMenu->setItemEnabled(ID_NEW_MIDPOINT, tmp);
  if(tmp && view->getDocument()->selectedCount() > 1)
    newMenu->changeItem(ID_NEW_MIDPOINT, "&Midpoints");
  else newMenu->changeItem(ID_NEW_MIDPOINT, "&Midpoint");

  //intersection point
  tmp = view->getDocument()->canNewIntersection();
  newMenu->setItemEnabled(ID_NEW_INTERSECTION, tmp);
  if(tmp && !((view->getDocument()->getSelected()[0]->getType() & G_STRAIGHT) &&
     (view->getDocument()->getSelected()[1]->getType() & G_STRAIGHT)))
    newMenu->changeItem(ID_NEW_INTERSECTION, "&Intersection Points");
  else newMenu->changeItem(ID_NEW_INTERSECTION, "&Intersection Point");

  //circle
  tmp = view->getDocument()->canNewCircle();
  newMenu->setItemEnabled(ID_NEW_CIRCLE, tmp);
  if(tmp && view->getDocument()->getSelected()[0]->getType() !=
     view->getDocument()->getSelected()[1]->getType())
    newMenu->changeItem(ID_NEW_CIRCLE, "&Circle By Center And Radius");
  else newMenu->changeItem(ID_NEW_CIRCLE, "&Circle By Center And Point");

  // line/parallel
  tmp = view->getDocument()->canNewLine();
  newMenu->setItemEnabled(ID_NEW_LINE, tmp);
  if(tmp) {
    G_refs x = view->getDocument()->getSelected();

    for(i = 0; i < (int)x.count(); i++) if(x[i]->getType() & G_STRAIGHT) break;

    if(i == (int)x.count()) {
      if(x.count() > 2) newMenu->changeItem(ID_NEW_LINE, "&Lines");
      else newMenu->changeItem(ID_NEW_LINE, "&Line");
    }
    else {
      if(x.count() > 2) newMenu->changeItem(ID_NEW_LINE, "Parallel &Lines");
      else newMenu->changeItem(ID_NEW_LINE, "Parallel &Line");
    }
  }
  else newMenu->changeItem(ID_NEW_LINE, "&Line");

  //perpendicular
  tmp = view->getDocument()->canNewPerpendicular();
  newMenu->setItemEnabled(ID_NEW_PERPENDICULAR, tmp);
  if(tmp && view->getDocument()->selectedCount() > 2)
    newMenu->changeItem(ID_NEW_PERPENDICULAR, "&Perpendicular Lines");
  else newMenu->changeItem(ID_NEW_PERPENDICULAR, "&Perpendicular Line");
    
  
}


void KSegWindow::updatePlayMenu()
{
  playMenu->clear();

  std::vector<KSegConstruction *> c = KSegConstruction::getAllConstructions();

  if(c.size() == 0 || (c.size() == 1 && c[0] == view->getDocument())) {
    playMenu->insertItem("No constructions", this, SLOT(fileOpen()));//dummy item
    playMenu->setItemEnabled(playMenu->idAt(0), false);
    return;
  }
  else {
    int i;
    for(i = 0; i < (int)c.size(); ++i) {
      if(c[i] == view->getDocument()) continue;

      QString f = c[i]->getFilename();
      if(f.isEmpty()) f = QString("Untitled");
      else {
	f = QFileInfo(f).fileName();
      }

      playMenu->insertItem(f, view, SLOT(playConstruction(int)));

      int id = playMenu->idAt(playMenu->count() - 1);

      playMenu->setItemParameter(id, i);
      playMenu->setItemEnabled(id, view->getDocument()->canPlay(i));
    }
  }
}


void KSegWindow::updateConstructionMenu()
{
  ASSERT(view->getDocument()->isConstruction());

  KSegConstruction *c = (KSegConstruction *)(view->getDocument());

  constructionMenu->setItemEnabled(ID_CONSTRUCTION_MAKE_NORMAL,
				   c->canConstructionMakeNormal());
  constructionMenu->setItemEnabled(ID_CONSTRUCTION_MAKE_GIVEN,
				   c->canConstructionMakeGiven());
  constructionMenu->setItemEnabled(ID_CONSTRUCTION_MAKE_FINAL,
				   c->canConstructionMakeFinal());
  constructionMenu->setItemEnabled(ID_CONSTRUCTION_MAKE_INITIAL,
				   c->canConstructionMakeInitial());
  constructionMenu->setItemEnabled(ID_CONSTRUCTION_RECURSE,
				   c->canConstructionRecurse());
}



void KSegWindow::keyPressEvent(QKeyEvent *ev)
{
  updateMenus();
  
  QMainWindow::keyPressEvent(ev);

  view->updateStatusBar(ev->stateAfter());
}

//this is a workaround for a seeming bug...
void KSegWindow::keyReleaseEvent(QKeyEvent *ev)
{
  updateMenus();
  
  QMainWindow::keyReleaseEvent(ev);

  view->updateStatusBar(ev->stateAfter());
}


void KSegWindow::closeEvent(QCloseEvent *ev)
{
  if(view->getDocument()->isModified() == false) {
    ev->accept();
    return;
  }

  int ans = QMessageBox::warning(this, "Save?",
				 "Would you like to save your changes?",
				 QMessageBox::Yes, QMessageBox::No,
				 QMessageBox::Cancel);

  if(ans == QMessageBox::Cancel) {
    ev->ignore();
    return;
  }
  if(ans == QMessageBox::Yes) {
    fileSave();
    if(view->getDocument()->isModified() == true) { //check if the guy hit cancel
      ev->ignore();
      return;
    }
  }

  ev->accept();
}


void KSegWindow::newSketch()
{
  (new KSegWindow())->show();
}

void KSegWindow::newConstruction()
{
  KSegConstruction *c = new KSegConstruction;

  (new KSegWindow(c))->show();
}


void KSegWindow::newView()
{
  (new KSegWindow(view->getDocument()))->show();
}

void KSegWindow::fileSaveAs()
{
  QString fileName;

  if(view->getDocument()->isConstruction()) {
    fileName = KFileDialog::
      getSaveFileName(view->getDocument()->getFilename(), "*.sec");
  }
  else {
    fileName = KFileDialog::
      getSaveFileName(view->getDocument()->getFilename(), "*.seg");
  }

  if(fileName.isNull()) return;

  QFile f(fileName);

  if(f.open(IO_WriteOnly)) {
    QDataStream stream(&f);

    stream << *(view->getDocument());
    f.close();
  }

  view->getDocument()->emitDocumentSaved(fileName);
}

void KSegWindow::fileSave()
{
  if(view->getDocument()->getFilename().isEmpty()) {
    fileSaveAs();
    return;
  }

  QString fileName = view->getDocument()->getFilename();

  QFile f(fileName);

  if(f.open(IO_WriteOnly)) {
    QDataStream stream(&f);

    stream << *(view->getDocument());
    f.close();
  }

  view->getDocument()->emitDocumentSaved(fileName);
}

void KSegWindow::fileOpen()
{
  QString fileName = KFileDialog::getOpenFileName(QString::null, "*.seg *.sec");

  if(fileName.isNull()) return;

  QFile f(fileName);

  KSegWindow *win = 0;

  if(f.open(IO_ReadOnly)) {
    QDataStream stream(&f);

    QString tmp;
    
    stream >> tmp;
    
    if(tmp == QString("KSeg Document Version 0.0")) {
      win = new KSegWindow;
    }
    else if(tmp == QString("KSeg Construction Version 0.0")) {
      win = new KSegWindow(new KSegConstruction);
    }
    else {
      QMessageBox::warning(this, "Invalid file",
			   "The file does not look like a valid KSeg file.",
			   QMessageBox::Ok, 0);
      f.close();
      return;
    }

    stream >> *(win->view->getDocument());
    f.close();
  }

  win->view->getDocument()->emitDocumentSaved(fileName);
  
  win->show();
}


void KSegWindow::filePrint()
{
  QPrinter *pr = view->getDocument()->getPrinter();

  if(pr->setup(this)) {
    pr->setFullPage(false);

    QPainter p(pr);

    QRect docSize = view->getDocument()->getSize();

    //make sure the document is at least 400 by 400
    QPoint oldCenter = docSize.center();
    if(docSize.width() < 400) docSize.setWidth(400);
    if(docSize.height() < 400) docSize.setHeight(400);
    docSize.moveCenter(oldCenter);

    QRect viewPort = p.viewport();

    if(docSize.width() * viewPort.height() >
       docSize.height() * viewPort.width()) {
      //the docSize needs to be made taller
      int oldHeight = docSize.height();
      int h = (docSize.width() * viewPort.height()) / viewPort.width();
      docSize.setHeight(h);
      docSize.moveBy(0, -(h - oldHeight) / 2);
    }
    else {
      //the docSize needs to be made wider
      int oldWidth = docSize.width();
      int w = (docSize.height() * viewPort.width()) / viewPort.height();
      docSize.setWidth(w);
      docSize.moveBy(-(w - oldWidth) / 2, 0);

    }

    p.setWindow(docSize);
    view->getDocument()->print(p);
  }
}


void KSegWindow::editColorBlack()
{
  view->getDocument()->setCurrentColor(black);
}

void KSegWindow::editColorGray()
{
  view->getDocument()->setCurrentColor(gray);
}

void KSegWindow::editColorRed()
{
  view->getDocument()->setCurrentColor(red);
}

void KSegWindow::editColorGreen()
{
  view->getDocument()->setCurrentColor(green);
}

void KSegWindow::editColorBlue()
{
  view->getDocument()->setCurrentColor(blue);
}

void KSegWindow::editColorYellow()
{
  view->getDocument()->setCurrentColor(yellow);
}

void KSegWindow::editColorPurple()
{
  view->getDocument()->setCurrentColor(magenta);
}

void KSegWindow::editColorCyan()
{
  view->getDocument()->setCurrentColor(cyan);
}

void KSegWindow::editColorOther()
{
  QColor newColor;
  
  newColor = QColorDialog::getColor(view->getDocument()->getCurrentColor());

  if(newColor.isValid()) {
    view->getDocument()->setCurrentColor(newColor);
  }
}

//----------

void KSegWindow::editPointstyleLargecircle()
{
  view->getDocument()->setCurrentPointstyle(LARGE_CIRCLE);
}


void KSegWindow::editPointstyleMediumcircle()
{
  view->getDocument()->setCurrentPointstyle(MEDIUM_CIRCLE);
}


void KSegWindow::editPointstyleSmallcircle()
{
  view->getDocument()->setCurrentPointstyle(SMALL_CIRCLE);
}

//---------

void KSegWindow::editLinestyleSolid()
{
  view->getDocument()->setCurrentPenstyle(SolidLine);
}


void KSegWindow::editLinestyleDashed()
{
  view->getDocument()->setCurrentPenstyle(DashLine);
}


void KSegWindow::editLinestyleDotted()
{
  view->getDocument()->setCurrentPenstyle(DotLine);
}


void KSegWindow::editLinestyleThin()
{
  view->getDocument()->setCurrentPenWidth(0);
}


void KSegWindow::editLinestyleNormal()
{
  view->getDocument()->setCurrentPenWidth(2);
}


void KSegWindow::editLinestyleThick()
{
  view->getDocument()->setCurrentPenWidth(3);
}

//---------

void KSegWindow::editFont10()
{
  view->getDocument()->setCurrentFontsize(10);
}
void KSegWindow::editFont12()
{
  view->getDocument()->setCurrentFontsize(12);
}
void KSegWindow::editFont14()
{
  view->getDocument()->setCurrentFontsize(14);
}
void KSegWindow::editFont20()
{
  view->getDocument()->setCurrentFontsize(20);
}
void KSegWindow::editFont30()
{
  view->getDocument()->setCurrentFontsize(30);
}

void KSegWindow::editFontFont()
{
  bool ok;
  QFont f = QFontDialog::getFont(&ok, view->getDocument()->getCurrentFont());
  if(ok) {
    view->getDocument()->setCurrentFont(f);
  }
}

void KSegWindow::helpAbout()
{
  QMessageBox::about(this, "About KSEG",
		     "KSEG v0.2\n"
		     "Copyright (C) 1999 Ilya Baran\n"
		     "http://www.mit.edu/~ibaran\n"
		     "ibaran@mit.edu");
}



#include "KSegWindow.moc"
