/*
 *  KSeg
 *  Copyright (C) 1999 Ilya Baran
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Send comments and/or bug reports to:
 *                 ibaran@mit.edu
 */


#include "KSegView.H"
#include "G_point.H"
#include "G_line.H"
#include "G_segment.H"
#include "G_drawstyle.H"
#include "G_label.H"
#include "KSegDocument.H"
#include "KSegConstruction.H"
#include "KSegCalculateEditor.H"
#include "G_object.H"
#include <qscrollbar.h>
#include <qapplication.h>
#include "KSegConstructionList.H"
#include "G_refSearcher.H"


//the following class and the function after are for determining if A can be
//reconstrained to B.

class ATypeFinder : public G_refSearcher
{
public:
  enum AType {
    Normal,
    Given,
    Initial
  };

  ATypeFinder(G_ref *inB) { B = inB; type = Normal; }

  bool is_found(const G_ref *ref)
  {
    if(ref == B) return true;
    if(ref->getGiven()) type = Given;
    if(ref->getInitial() && type != Given) type = Initial;
    return false;
  }

  G_refs search_next(const G_ref *ref) { return ref->getChildrenConst(); }

  AType type;
  G_ref *B;
};

//----------------------------------------------------------------------------------


bool canReconstrain(G_ref *A, G_ref *B)
{
  if(B->getType() & (G_POINT | G_CURVE) == 0) return false;

  if(B->getDocument()->isConstruction() && B->getType() == G_POINT && B->getGiven() &&
     ((KSegConstruction *)(B->getDocument()))->hasLoops()) return false;

  ATypeFinder atf(B);
  if(atf.search(A) == true) return false;
  
  if(atf.type == ATypeFinder::Normal || B->getDocument()->isConstruction() == false) return true;
  
  if(atf.type == ATypeFinder::Given) {
    CanMakeGivenSearcher cmgs;
    if(cmgs.search(B) == true) return false;
  }

  if(atf.type == ATypeFinder::Initial) {
    CanMakeInitialAncSearcher cmias;
    if(cmias.search(B) == true) return false;
  }

  return true;
}

//----------------------------------------------------------------------------------
//----------------------------------------------------------------------------------


KSegView::SelectType KSegView::selectType;
QList<KSegView> KSegView::allViews;
QPainter *KSegView::constructionPainter;

const int KSegView::ScrollBarThickness = 20;

KSegView::KSegView(KSegDocument *inDoc, QWidget *parent)
  : QWidget(parent)
{
  allViews.append(this);

  menusEnabled = true;

  doc = inDoc;

  doc->addView();

  connect(doc, SIGNAL(documentChanged()), this, SLOT(redrawBuffer()));
  connect(doc, SIGNAL(documentModified()), this, SLOT(documentChanged()));
  connect(doc, SIGNAL(documentSaved(const QString &)),
	  this, SLOT(documentSaved(const QString &)));

  setMinimumSize(ScrollBarThickness * 4, ScrollBarThickness * 4);
  buffer = QPixmap(width(), height());

  buffer.setOptimization(QPixmap::BestOptim);

  forcePaint = true;
  selectType = BORDER_SELECT;

  hBar = new QScrollBar(QScrollBar::Horizontal, this);
  vBar = new QScrollBar(QScrollBar::Vertical, this);

  setMouseTracking(true);

  drag = NO_DRAG;

  offsetX = offsetY = 0;

  connect(hBar, SIGNAL(valueChanged(int)), this, SLOT(moved(int)));
  connect(vBar, SIGNAL(valueChanged(int)), this, SLOT(moved(int)));

  setBackgroundMode(NoBackground);

  connect(&selectTimer, SIGNAL(timeout()), this, SLOT(selectTimeout()));

}


KSegView::~KSegView()
{
  doc->delView();
  allViews.removeRef(this);
}


void KSegView::moved(int)
{
  offsetX = hBar->value();
  offsetY = vBar->value();
  redrawBuffer();
}

void KSegView::resizeEvent(QResizeEvent *e)
{
  hBar->move(0, height() - ScrollBarThickness);
  hBar->resize(width() - ScrollBarThickness, ScrollBarThickness);
  vBar->move(width() - ScrollBarThickness, 0);
  vBar->resize(ScrollBarThickness, height());

  buffer.resize(e->size() - QSize(ScrollBarThickness, ScrollBarThickness));

  redrawBuffer();
}

void KSegView::redrawBuffer()
{
  QPainter p;

  emit updateMenus();

  if(selectType == BLINKING_SELECT) {
    if(doc->getSelected().count() > 0) {
      if(!selectTimer.isActive()) selectTimer.start(319);
    }
    else {
      if(selectTimer.isActive()) selectTimer.stop();
    }
  }

  buffer.fill();

  p.begin(&buffer);
  QRect window = p.viewport();
  window.moveBy(offsetX, offsetY);
  p.setWindow(window);

  doc->draw(p);

  p.end();

  if(forcePaint) repaint(FALSE);
}

void KSegView::paintEvent(QPaintEvent *)
{
  QRect tmp = doc->getSize();

  if(tmp.isEmpty() || tmp.isNull()) tmp = QRect(offsetX + 1, offsetY + 1, 2, 2);

  hBar->setRange(QMIN(offsetX, tmp.left()), QMAX(offsetX, tmp.right() - width() + ScrollBarThickness));
  hBar->setSteps(10, width() - ScrollBarThickness);
  vBar->setRange(QMIN(offsetY, tmp.top()), QMAX(offsetY, tmp.bottom() - height() + ScrollBarThickness));
  vBar->setSteps(10, height() - ScrollBarThickness);

  if(drag == RECTANGLE_DRAG) {
    QPainter p(&buffer);

    //quickly draw a flicker-free focus rectangle by doing a xor draw before
    //and after blitting.
    p.drawWinFocusRect(QRect(dragStartX, dragStartY,
			     dragCurX - dragStartX, dragCurY - dragStartY).normalize());
    bitBlt(this, 0, 0, &buffer);
    p.drawWinFocusRect(QRect(dragStartX, dragStartY,
			     dragCurX - dragStartX, dragCurY - dragStartY).normalize());

  }
  else bitBlt(this, 0, 0, &buffer);

}


void KSegView::documentSaved(const QString &filename)
{
  parentWidget()->setCaption(QString("kseg: ") + filename);
}

void KSegView::documentChanged()
{
  QString caption = "kseg: ";
  if(getDocument()->getFilename().isEmpty()) caption += QString("Untitled");
  else caption += getDocument()->getFilename();
  caption += '*';
  
  if(parentWidget()->caption() != caption) parentWidget()->setCaption(caption);
}

//---------------------MOUSE PRESS EVENT-----------------------------
void KSegView::mousePressEvent(QMouseEvent *e)
{
  dragCurX = dragStartX = lastMouseX = e->x();
  dragCurY = dragStartY = lastMouseY = e->y();

  if(e->button() == LeftButton) {//----------------LEFT BUTTON PRESSED
    G_refs tmp = doc->whatAmIOn(e->x() + offsetX, e->y() + offsetY);

    if(e->state() & ControlButton) { //CTRL pressed--reconstrain
      if(tmp.count() == 0 || tmp[0]->getType() != G_POINT) return;

      if(!doc->isPointFreeable(tmp[0])) return;

      reconstrainUndo = new G_undoReconstrain(tmp[0]);

      tmp[0]->reconstrain(G_FREE_POINT, G_refs());

      drag = RECONSTRAIN_DRAG;
      objectsDragged.append(tmp[0]);
      doc->clearSel();
      doc->addSel(tmp[0]);

      setCursor(pointingHandCursor);

      doc->emitDocumentChanged();

      updateStatusBar(e->state());

      return;
    }

    if(tmp.count() == 0 || (e->state() & AltButton)) {
      tmp = doc->whatLabelsAmIOn(e->x() + offsetX, e->y() + offsetY);
      
      if(tmp.count() == 0) {
	if(!(e->state() & ShiftButton)) doc->clearSel();

	drag = RECTANGLE_DRAG;
	doc->emitDocumentChanged();
	return;
      }

      setCursor(pointingHandCursor);

      drag = LABEL_DRAG;
      objectsDragged.append(tmp[0]);
      dragOffset = tmp[0]->getLabelPos() - G_point(e->x() + offsetX, e->y() + offsetY);
      return;
    }


    G_ref *r = doc->getNextSel(tmp, e->state() & ShiftButton);

    if(r) {
      if(!(e->state() & ShiftButton)) doc->clearSel();

      if(r->getSelected()) doc->delSel(r); else doc->addSel(r);
    }

    //for a constrained point, grab the offset
    if(doc->getSelected().count() == 1 && doc->getSelected()[0]->getType() == G_POINT) {
      dragOffset = doc->getSelected()[0]->getObject()->getPoint() -
	G_point(e->x() + offsetX, e->y() + offsetY);
    }

    setCursor(pointingHandCursor);

    if(!(e->state() & ShiftButton)) drag = MOVE_DRAG;
    else drag = NO_DRAG;
  }
  else if(e->button() == RightButton) { //-----------RIGHT BUTTON PRESSED
    G_refs tmp = doc->whatAmIOn(e->x() + offsetX, e->y() + offsetY);

    int i;
    if(tmp.count() > 2) return;

    for(i = 0; i < (int)tmp.count(); i++) {
      if(!(tmp[i]->getType() & G_CURVE)) {
	return;
      }
    }


    G_ref *p;

    if(tmp.count() == 2) { //create intersection point
      if((G_point(e->x() + offsetX, e->y() + offsetY) -
	  tmp[0]->getObject()->getCurveRef()->
	  getIntersection(tmp[1]->getObject()->getCurveRef())).length() < 5) {
	if(!(e->state() & ShiftButton)) doc->clearSel();

	p = new G_ref;
	p->create(G_POINT, G_INTERSECTION_POINT, tmp, doc);
	p->update();
	doc->addSel(p);

      }
      else if((G_point(e->x() + offsetX, e->y() + offsetY) -
	       tmp[0]->getObject()->getCurveRef()->
	       getIntersection(tmp[1]->getObject()->getCurveRef(), 1)).length() < 5) {
	if(!(e->state() & ShiftButton)) doc->clearSel();

	p = new G_ref;
	p->create(G_POINT, G_INTERSECTION2_POINT, tmp, doc);
	p->update();
	doc->addSel(p);

      }
      else return;
    }

    if(tmp.count() == 1) { //create constrained point
      if(!(e->state() & ShiftButton)) doc->clearSel();

      p = new G_ref;
      p->create(G_POINT, G_CONSTRAINED_POINT, tmp, doc);
      ((G_pointObject*)(p->getObject()))->setPoint(G_point(e->x() + offsetX, e->y() + offsetY));
      p->update();
      doc->addSel(p);
    }

    if(tmp.count() == 0) { //create free point
      if(!(e->state() & ShiftButton)) doc->clearSel();
	
      p = new G_ref;
      p->create(G_POINT, G_FREE_POINT, G_refs(), doc);
      ((G_pointObject*)(p->getObject()))->setPoint(G_point(e->x() + offsetX, e->y() + offsetY));
      doc->addSel(p);
    }
  }

  doc->emitDocumentChanged();
}

//--------------------------MOUSE MOVE EVENT-----------------------------
void KSegView::mouseMoveEvent(QMouseEvent *e)
{
  if(drag == RECONSTRAIN_DRAG || drag == LABEL_DRAG || drag == MOVE_DRAG) {
    KSegConstructionList::disableRedraws();
  }

  if(drag == NO_DRAG) { //display status message
    updateStatusBar(e->state());
  }

  if(e->x() == lastMouseX && e->y() == lastMouseY) return;

  if(drag == RECTANGLE_DRAG) {
    dragCurX = e->x();
    dragCurY = e->y();

    repaint(false);
  }

  if(drag == LABEL_DRAG) {
    objectsDragged[0]->setLabelPos(dragOffset + G_point(e->x() + offsetX, e->y() + offsetY));
    doc->emitDocumentChanged();
    doc->emitDocumentModified();
  }

  if(drag == RECONSTRAIN_DRAG) {
    if(objectsUpdating.count() == 0) {
      objectsUpdating.topologicalSort(objectsDragged);
    }
    
    int i;

    for(i = 0; i < (int)objectsDragged.count(); i++) {
      objectsDragged[i]->getObject()->translate(G_point(e->x() - lastMouseX, e->y() - lastMouseY));
    }
    objectsUpdating.update();
    
    doc->emitDocumentChanged();
    doc->emitDocumentModified();
    updateStatusBar(e->state());
  }

  if(drag == MOVE_DRAG) {
    if(objectsDragged.count() == 0 && doc->getSelected().count()) {
      objectsDragged.buildReverse(doc->getSelected());
      objectsUpdating.topologicalSort(objectsDragged);
    }

    if(objectsDragged.count() == 1 && objectsDragged[0]->getType() == G_POINT &&
       objectsDragged[0]->getDescendType() == G_CONSTRAINED_POINT) { // a constrained point is moving.
      ((G_pointObject *)(objectsDragged[0]->getObject()))->
	setPoint(dragOffset + G_point(e->x() + offsetX, e->y() + offsetY));
      
      objectsUpdating.update();
      lastMouseX = e->x(); lastMouseY = e->y();
      
      doc->emitDocumentChanged();
      doc->emitDocumentModified();
    }
    else {
      int i;
      for(i = 0; i < (int)objectsDragged.count(); i++) {
	objectsDragged[i]->getObject()->translate(G_point(e->x() - lastMouseX, e->y() - lastMouseY));
      }
      objectsUpdating.update();
      
      doc->emitDocumentChanged();
      doc->emitDocumentModified();
    }
  }

  lastMouseX = e->x(); lastMouseY = e->y();
}

//--------------------------MOUSE RELEASE EVENT-----------------------------
void KSegView::mouseReleaseEvent(QMouseEvent *e)
{
  KSegConstructionList::enableRedraws();
  if(drag == MOVE_DRAG && doc->isConstruction()) doc->emitDocumentChanged();

  if(drag == RECTANGLE_DRAG) {
    doc->addSelect(QRect(dragStartX + offsetX, dragStartY + offsetY,
			 dragCurX - dragStartX, dragCurY - dragStartY).normalize());

    drag = NO_DRAG;
    doc->emitDocumentChanged();
    return;
  }
  if(drag == RECONSTRAIN_DRAG) {
    if(objectsUpdating.count() == 0) {
      objectsUpdating.topologicalSort(objectsDragged);
    }

    G_point releasePt = objectsDragged[0]->getObject()->getPoint();

    G_refs tmp = doc->whatAmIOn(releasePt.getX(), releasePt.getY(), false);

    int i;

    //remove objects to which it cannot be reconstrained
    //here do check for various scripting constraints.
    tmp.removeRef(objectsDragged[0]);

    for(i = 0; i < (int)tmp.count(); i++) {
      if(canReconstrain(objectsDragged[0], tmp[i]) == false) {
	tmp.remove(i);
	i--;
      }
    }

    if(tmp.count() != 0 && tmp[0]->getType() == G_POINT) {
      while(objectsDragged[0]->getChildren().count()) {
	G_refs oldpars;
	G_ref *curChild;

	curChild = objectsDragged[0]->getChildren()[0];
	oldpars = curChild->getParents();

	for(i = 0; i < (int)oldpars.count(); i++) {
	  if(oldpars[i] == objectsDragged[0]) {
	    oldpars.insert(i, tmp[0]);
	    oldpars.remove(i + 1);
	  }
	}

	doc->addUndo(new G_undoReconstrain(curChild));

	curChild->reconstrain(curChild->getDescendType(), oldpars);
      }

      if(reconstrainUndo->isChanged(objectsDragged[0])) doc->addUndo(reconstrainUndo);
      else {
	((G_pointObject *)(objectsDragged[0]->getObject()))->
	  setPoint(reconstrainUndo->getOldPos());
      }
      reconstrainUndo = 0;
      
      objectsUpdating.update();
      objectsDragged[0]->remove();
      doc->emitDocumentChanged();
    }

    if(tmp.count() == 2 && tmp[0]->getType() & G_CURVE && tmp[1]->getType() & G_CURVE) {
      //make it an intersection point
      if((G_point(releasePt.getX(), releasePt.getY()) -
	  tmp[0]->getObject()->getCurveRef()->
	  getIntersection(tmp[1]->getObject()->getCurveRef())).length() < 5) {

	objectsDragged[0]->reconstrain(G_INTERSECTION_POINT, tmp);
	objectsUpdating.update();
	doc->emitDocumentChanged();
      }
      else if((G_point(releasePt.getX(), releasePt.getY()) -
	       tmp[0]->getObject()->getCurveRef()->
	       getIntersection(tmp[1]->getObject()->getCurveRef(), 1)).length() < 5) {

	objectsDragged[0]->reconstrain(G_INTERSECTION2_POINT, tmp);
	objectsUpdating.update();
	doc->emitDocumentChanged();

      }
    }

    if(tmp.count() == 1 && tmp[0]->getType() & G_CURVE) { //create constrained point

      objectsDragged[0]->reconstrain(G_CONSTRAINED_POINT, tmp);
      ((G_pointObject *)(objectsDragged[0]->getObject()))->setP(BIG);
      objectsUpdating.update();
      doc->emitDocumentChanged();
    }

    if(reconstrainUndo &&
       reconstrainUndo->isChanged(objectsDragged[0])) doc->addUndo(reconstrainUndo);

  }

  drag = NO_DRAG;
  objectsDragged.clear();
  objectsUpdating.clear();
  updateStatusBar(e->state());
  emit updateMenus();
}

//-------------------------MOUSE DOUBLE CLICK EVENT-----------------------------
void KSegView::mouseDoubleClickEvent(QMouseEvent *e)
{
  if(e->button() == LeftButton) {
    G_refs tmp = doc->whatAmIOn(e->x() + offsetX, e->y() + offsetY);

    if(tmp.count() == 0 || (e->state() & AltButton)) {
      tmp = doc->whatLabelsAmIOn(e->x() + offsetX, e->y() + offsetY);

      if(tmp.count() == 0) return;

      doc->clearSel();
      doc->addSel(tmp[0]);

      doc->emitDocumentChanged();

      if(doc->canEditChangeLabel()) doc->editChangeLabel();
      return;
    }
    if(tmp[0]->getType() & G_VALUE) { //start or edit a calculation
      if(tmp[0]->getType() == G_CALCULATE) { //edit
	KSegCalculateEditor *ed;
	G_calculateObject *formula = (G_calculateObject *)(tmp[0]->getObject());
	ed = new KSegCalculateEditor(this, formula->getFormulaString(),
				     tmp[0]->getParents(), tmp[0]);
	ed->run();
	if(ed->result() == QDialog::Rejected) { delete ed; return; }
	formula->changeFormula(ed->getOutputParents(), ed->getOutputString());
	G_refs dependents; dependents.topologicalSort(tmp[0]); dependents.update();
	doc->emitDocumentChanged();
	delete ed;
	return;
      }
      else { //measure
	doc->clearSel();
	doc->addSel(tmp[0]);
	doc->measureCalculate();
      }
    }
    if(tmp[0]->getType() & G_LOCUS) { //change number of samples in locus
      if(tmp[0]->getDescendType() == G_OBJECT_LOCUS) {
	doc->clearSel();
	doc->addSel(tmp[0]);
	doc->editChangeNumberOfSamples();
      }
    }
  }
}

//-------------------------UPDATE STATUS BAR-----------------------------
void KSegView::updateStatusBar(ButtonState state)
{
  QPoint e = mapFromGlobal(QCursor::pos());

  if(drag == NO_DRAG) {
    G_refs tmp = doc->whatAmIOn(e.x() + offsetX, e.y() + offsetY);

    if(tmp.count() == 0) {
      tmp = doc->whatLabelsAmIOn(e.x() + offsetX, e.y() + offsetY);

      if(tmp.count() == 0) {
	unsetCursor();
	emit statusBarMessage("Ready");
	return;
      }

      setCursor(upArrowCursor);

      QString message;

      message = G_ref::getNameFromType(tmp[0]->getType()) + " " +
	KFormula::toUgly(tmp[0]->getLabel().getText());

      message = QString("Move Label of ") + message;

      emit statusBarMessage(message);
      return;
    }

    setCursor(upArrowCursor);

    QString message;


    G_ref *r = doc->getNextSel(tmp, state & ShiftButton);

    if(r) {
      message = G_ref::getNameFromType(r->getType()) + " " +
	KFormula::toUgly(r->getLabel().getText());

      if(!(state & ShiftButton)) {
	message = QString("Select ") + message;
      }
      else {
	if(r->getSelected()) {
	  message = QString("Deselect ") + message; 
	}
	else {
	  message = QString("Select ") + message;
	  if(doc->selectedCount() > 0) message = QString("Also ") + message;
	}
      }
    }
    else message = QString("Move Selected Objects");

    emit statusBarMessage(message);

  }

  if(drag == RECONSTRAIN_DRAG) {
    QString message("Reconstrain");

    message += G_ref::getNameFromType(objectsDragged[0]->getType()) + " " +
      KFormula::toUgly(objectsDragged[0]->getLabel().getText());

    message += " to ";
    
    G_point releasePt = objectsDragged[0]->getObject()->getPoint();

    G_refs tmp = doc->whatAmIOn(releasePt.getX(), releasePt.getY(), false);

    int i;

    //remove objects to which it cannot be reconstrained
    //here do check for various scripting constraints.
    tmp.removeRef(objectsDragged[0]);

    for(i = 0; i < (int)tmp.count(); i++) {
      if(canReconstrain(objectsDragged[0], tmp[i]) == false) {
	tmp.remove(i);
	i--;
      }
    }

    if(tmp.count() != 0 && tmp[0]->getType() == G_POINT) {
      message += G_ref::getNameFromType(tmp[0]->getType()) + " " +
	KFormula::toUgly(tmp[0]->getLabel().getText()) + ".";
    }
    else if(tmp.count() == 1 && tmp[0]->getType() & G_CURVE) {
      message += G_ref::getNameFromType(tmp[0]->getType()) + " " +
	KFormula::toUgly(tmp[0]->getLabel().getText()) + ".";
    }
    else if(tmp.count() == 2 && tmp[0]->getType() & G_CURVE && tmp[1]->getType() & G_CURVE) {
      //that's a long if statement--checks if it's really near the intersection or just
      //near both curves
      if((G_point(releasePt.getX(), releasePt.getY()) -
	  tmp[0]->getObject()->getCurveRef()->
	  getIntersection(tmp[1]->getObject()->getCurveRef())).length() < 5 ||
	 (G_point(releasePt.getX(), releasePt.getY()) -
	  tmp[0]->getObject()->getCurveRef()->
	  getIntersection(tmp[1]->getObject()->getCurveRef(), 1)).length() < 5) {

	message += "the intersection of ";
	message += G_ref::getNameFromType(tmp[0]->getType()) + " " +
	  KFormula::toUgly(tmp[0]->getLabel().getText()) + " and ";
	message += G_ref::getNameFromType(tmp[1]->getType()) + " " +
	  KFormula::toUgly(tmp[1]->getLabel().getText()) + ".";
      }
      else message += "be free.";
    }
    else {
      message += "be free.";
    }

    emit statusBarMessage(message);
  }
}


void KSegView::playConstruction(int which)
{
  QPainter p;

  constructionPainter = &p;

  p.begin(this);
  p.setClipRect(buffer.rect());
  QRect window = p.viewport();
  window.moveBy(offsetX, offsetY);
  p.setWindow(window);
  
  doc->doPlay(which);

  p.end();
}

#include "KSegView.moc"







