/*
 *  KSeg
 *  Copyright (C) 1999 Ilya Baran
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Send comments and/or bug reports to:
 *                 ibaran@mit.edu
 */


#include "G_segment.H"
#include "G_line.H"
#include "G_drawstyle.H"
#include "KSegView.H"

//drawing:
void G_segment::draw(QPainter &p, const G_drawstyle &d, bool selected)
{
  QRect r = p.window();

  if(!inRect(r)) return;

  G_point tmp1, tmp2;
  
  if(fabs(p1.getX()) < DRAW_MAX && fabs(p1.getY()) < DRAW_MAX) {
    tmp1 = p1;
  }
  else {
    tmp1 = getNearestPoint(r.center());
    tmp1 += (p1 - tmp1) * (r.width() + r.height()) / (p1 - tmp1).length();
  }

  if(fabs(p2.getX()) < DRAW_MAX && fabs(p2.getY()) < DRAW_MAX) {
    tmp2 = p2;
  }
  else {
    tmp2 = getNearestPoint(r.center());
    tmp2 += (p2 - tmp2) * double(r.width() + r.height()) / (p2 - tmp2).length();
  }

  //Now Draw!

  if(selected && KSegView::getSelectType() == KSegView::BORDER_SELECT) {
    int width = d.getPen().width() ? d.getPen().width() + 3 : 4;

    if(d.getPen().color() != Qt::black) p.setPen(QPen(Qt::black, width)); // set a normal pen
    else p.setPen(QPen(Qt::red, width));

    p.drawLine(tmp1.toQPoint(), tmp2.toQPoint());
    
  }

  p.setPen(d.getPen());
  if(selected && KSegView::getSelectType() == KSegView::BLINKING_SELECT) {
    QColor c(QTime::currentTime().msec() * 17, 255, 255, QColor::Hsv);

    p.setPen(QPen(c, d.getPen().width(), d.getPen().style()));
  }

  p.drawLine(tmp1.toQPoint(), tmp2.toQPoint());

  return;
	     
}

G_point G_segment::getPointOnCurve(double p) const
{
  return p1 + (p2 - p1) * p;
}

double G_segment::getParamFromPoint(const G_point &p) const
{ 
  return sqrt(((p - p1) * (p - p1)) / ((p2 - p1) * (p2 - p1)));
}


G_point G_segment::getNearestPoint(const G_point &p) const
{
  double x = (p - p1) * (p2 - p1) / ((p1 - p2) * (p1 - p2));
  if(x <= 0) return p1;
  if(x >= 1) return p2;
  return p1 + (p2 - p1) * x;
}

bool G_segment::inRect(const QRect &r) const
{
  if(r.contains(p1.toQPoint()) || r.contains(p2.toQPoint())) return true;

  if(p1.getX() < r.left() && p2.getX() < r.left()) return false;
  if(p1.getX() > r.right() && p2.getX() > r.right()) return false;

  if(p1.getY() < r.top() && p2.getY() < r.top()) return false;
  if(p1.getY() > r.bottom() && p2.getY() > r.bottom()) return false;

  //the following could be optimized
  G_segment tmp;

  tmp = G_segment(r.topLeft(), r.bottomLeft());
  if(getIntersection(&tmp).isValid()) return true;

  tmp = G_segment(r.topRight(), r.bottomRight());
  if(getIntersection(&tmp).isValid()) return true;

  tmp = G_segment(r.bottomLeft(), r.bottomRight());
  if(getIntersection(&tmp).isValid()) return true;

  tmp = G_segment(r.topRight(), r.topLeft());
  if(getIntersection(&tmp).isValid()) return true;

  return false;
}

G_point G_segment::getIntersection(const G_curve *c, int which) const
{
  if(c->getType() == G_SEGMENT) {
    G_segment *s = (G_segment *)c;

    double r, t, tmp;

    tmp = (p2 - p1) % (s->p2 - s->p1);

    if(fabs(tmp) < SMALL) return G_point::inValid();

    r = (s->p2 - s->p1) % (p1 - s->p1) / tmp;

    t = (p2 - p1) % (p1 - s->p1) / tmp;

    if( r < -SMALL || r > 1 + SMALL || t < -SMALL || t > 1 + SMALL)
      return G_point::inValid();
    
    if(r < 0) r = 0; if(r > 1) r = 1;
    
    return p1 + r * (p2 - p1);

  }

  if(c->getType() == G_LINE) {
    G_line *l = (G_line *)c;

    double r, tmp;

    tmp = (p2 - p1) % l->getDirection();

    if(fabs(tmp) < SMALL) return G_point::inValid();

    r = l->getDirection() % (p1 - l->getP1()) / tmp;

    if(r < -SMALL || r > 1 + SMALL)
      return G_point::inValid();

    if(r < 0) r = 0; if(r > 1) r = 1;

    return p1 + r * (p2 - p1);

  }

  return c->getIntersection(this, which);
}
