// -*- Mode: C++; -*-
/*
 * $Id: smart_pointer.h,v 2.6 2000/10/22 23:44:45 ral Exp $
 */

#ifndef __smart_pointer_h
#define __smart_pointer_h

#ifdef _WIN32
#include <windows.h>
#endif
//! reference counter class
/*!
 * This class is required from smart_pointer. All classes 
 * are used as a template for smart_pointer has to be a child
 * of this class.
 */
class refCounter
{
protected:
    //! the internal counter itself
    long ref_counter;
public:
    //! simple constructor
    /*!
     * has to be called from all classes depending on it
     * else the refcounter will not work correct.
     */
    refCounter(){ref_counter = 0;}
    //! destructor (does nothing)
    virtual ~refCounter(){};
    //! increment
    /*!
     * increments the internal reference count by one
     */
    virtual void Inc()
    {
#ifdef _WIN32
        InterlockedIncrement(&ref_counter);
#else
        ++ref_counter;
#endif
    }
    //! decrement
    /*!
     * decrement the internal reference counter by one
     */
    virtual void Dec()
    {
#ifdef _WIN32
        InterlockedDecrement(&ref_counter);
#else
        --ref_counter;
#endif
    }
    //! is this pointer used
    /*!
     * \return true if the internal counter isn't 0, else false
     */
    virtual bool used(){return (ref_counter > 0);}
};

/** Smart pointer class
 *
 * it is used to avoid memory leaks. All classes to be used
 * within this helper has to be a child class of refCounter
 * You can act as the same like the stored pointer itself.
 */
template<class T> class smart_pointer
{
protected:
    //! the pointer to this class itself
    T * ptr;
public:
    //! empty constructor
    smart_pointer(){ptr = 0;}
    //! constructor
    /*!
     * \param t the new pointer
     */
    smart_pointer(T* t){if (ptr = t) ptr->Inc();}
    //! copy constructor
    /*!
     * \param old the smartpointer to be copied
     */
    smart_pointer(const smart_pointer<T>&old){if (ptr = old.ptr) ptr->Inc();}
    //! assign operator
    /*!
     * \param old a reference to a smartpointer to be copied
     * \return reference to this object
     */
    smart_pointer<T>& operator=(const smart_pointer<T>&old){if (ptr = old.ptr) ptr->Inc();return *this;}
    //! assign operator
    /*!
     * \param old a pointer of type T
     * \return reference to this object
     */
    smart_pointer<T>& operator=(T*old);
    //! destructor
    /*!
     * Releases intern the object. If it were the last reference, ptr will be destroyed.
     */
    ~smart_pointer();
    //! bool operator
    /*!
     * for using statements like if (a) 
     */
    operator bool() {return (ptr != 0);}
    //! negate operator
    /*!
     * for using statements like if (!a)
     */
    bool operator!(){return ptr == 0;}
    //! pointer operator
    /*!
     * for using like a->
     */
    T*operator->(){return ptr;}
    //! reference operator
    /*!
     * for using like (*a). If the internal ptr == 0, this methode fails!
     */
    T&operator*(){return *ptr;}
    //! cast to conventional pointer
    operator T*(){return ptr;}
};

template<class T> inline smart_pointer<T>::~smart_pointer()
{
    if (!ptr) return;
    ptr->Dec();
    if (!ptr->used()) delete ptr;
}

template<class T> inline smart_pointer<T>& smart_pointer<T>::operator=(T*old)
{
    if (ptr == old) return *this;
    if (ptr) {
	ptr->Dec();
	if (!ptr->used()) delete ptr;
    }
    ptr = old;
    if (ptr) ptr->Inc();
    return *this;
}

#endif
