/*
 * kpvm.cpp
 *
 * Copyright (c) 1998 Michael Kropfberger <michael.kropfberger@gmx.net>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
 

#include <qapp.h>
#include <qdir.h>
#include <qtimer.h>
#include <qstrlist.h>

#include "kpvm.h"
#include "kpvm.moc"


/*****************************************************************************/
/*************************    CLASS   KPvm    ********************************/
/*****************************************************************************/
KPvm::KPvm(QWidget *parent, const char *name)
  :QWidget(parent,name)
{
  _exit=FALSE;

  if (this->hasPvmParent()) {  // is newly spawned
    int l;
    char *cs=0;
    //send back the hostname to parent
    if (pvm_initsend(PvmDataDefault)<0)
       fatal("pvm_initsend error!");
    l=this->hostName().length()+1;
    if  (pvm_pkint(&l,1,1)<0) // send length of hostName
          fatal("pvm_pkint hostname-length error!");
    cs =(char *)realloc(cs,l);
    strcpy(cs,this->hostName().data());
    if (pvm_pkstr(cs)<0)  //send running on hostName
       fatal("pvm_pkstr hostname error!");
    if (pvm_send(pvm_parent(),53) < 0) 
        fatal("pvm_send for initial stats ERROR!");

    //receive necessary data from parent
    pvm_recv(pvm_parent(),53);
    pvm_upkint(&l,1,1);
    cs = (char *)realloc(cs,l);  //host of parent
    pvm_upkstr(cs);

    _parent = new KPvmParent(pvm_parent(),cs); 
    this->insertNewEntity(_parent);

    delete cs;
  }//if hasPvmParent

  _interval=20;  //initial msec value

  QTimer::singleShot(_interval,this,SLOT(checkRecv()) );
}

/*****************************************************************************/
void KPvm::changeCheckInterval(int msecs)
{
  _interval=msecs;
};


/*****************************************************************************/
QList<KPvmChild> KPvm::spawn(QString fname, char **argv
                               , int flag
                               , QString where
                               , int numProc )
{
  if (fname.find('/',0)<0) { // not a path whith slashes in it
    fname=QDir::currentDirPath()+"/"+fname;
  }
  //debug("spawning [%s] ...",fname.data());
  QList<KPvmChild> childs;
  KPvmChild * child;
  int joiner_tid;
  for (int i=0;i<numProc;i++) {
      if (pvm_spawn(fname.data(),argv,flag,where.data(),1, &joiner_tid)!=1)
         fatal("could not spawn [%s]!",fname.data());
      int l;
      char* cs=0;
      //receive all necessary data from child
      pvm_recv(joiner_tid,53);
      pvm_upkint(&l,1,1);
      cs = (char *)realloc(cs,l);
      pvm_upkstr(cs);
      child = new KPvmChild(joiner_tid,cs); CHECK_PTR(child);
      this->insertNewEntity(child);
      childs.append(child);

      //send the hostname to child
      if (pvm_initsend(PvmDataDefault)<0)
         fatal("pvm_initsend error!");
      l=this->hostName().length()+1;
      if  (pvm_pkint(&l,1,1)<0) // send length of hostName
         fatal("pvm_pkint hostname-length error!");
      cs =(char *)realloc(cs,l);
      strcpy(cs,this->hostName().data());
      if (pvm_pkstr(cs)<0)  //send running on hostName
         fatal("pvm_pkstr hostname error!");
      if (pvm_send(joiner_tid,53) < 0) 
          fatal("pvm_send for initial stats ERROR!");
      delete cs;
  }
  return childs;
} //spawn

/*****************************************************************************/
void KPvm::send(KPvmEntity* entity, Serialize* data
                , int msgtag, bool sendNetEfficient)
{
  //debug("KPvm::send a [%s] to t%x",data->className(),entity->tid());
  if (pvm_initsend(PvmDataDefault)<0)
     fatal("pvm_initsend error!");
  QList<VarTuple> state=data->objState();
  //debug("got %i varTuples in objState",state.count() );
  int i, type;
  //VarTuple::VarType type;
  char *cs;
  i=strlen(data->className())+1;  // add zero-byte
  if (pvm_pkint(&i,1,1)<0) // send length of className
          fatal("pvm_pkint classname-length error!");

  cs=(char *)malloc(i); //get memory for className
  strcpy(cs,data->className());
  if (pvm_pkstr(cs)<0)  //send identifying className
     fatal("pvm_pkstr classname error!");

  int nrVars=state.count();     //send number of following VariableTuples
  if (pvm_pkint(&nrVars,1,1) <0)
       fatal("pvm_pkint nrVars-length error!");

  //need to store it for packing... casting is just on stack
   short sendNetEfficientShort=sendNetEfficient;
   if (pvm_pkshort(&sendNetEfficientShort,1,1) <0)
      fatal("pvm_pkshort sendNetEfficient error!");
   

  if (sendNetEfficient) {    
     float f; double d; long l;
     for(VarTuple* vt=state.first();vt!=0;vt=state.next()) {
        type=vt->type();
        if (pvm_pkint(&type,1,1)<0)  // send varType
           fatal("pvm_pkint varType error!");
        switch ( type ) {
	  case VarTuple::INT:
                          vt->storeValueTo(i);
                          pvm_pkint(&i,1,1);
                          break;
	  case VarTuple::LONG:
                          vt->storeValueTo(l);
                          pvm_pklong(&l,1,1);
                          break;
	  case VarTuple::STRING:
                          // length and add zero-byte
                          i=strlen(vt->valueToString().data())+1;
                          if (pvm_pkint(&i,1,1)<0)  // send length
                             fatal("pvm_pkint varlength error!");
                         //set VarEntry
                          if (pvm_pkstr(vt->valueToString().data())<0)
                             fatal("pvm_pkstr var error!");
                          break;
	  case VarTuple::FLOAT:
                          vt->storeValueTo(f);
                          pvm_pkfloat(&f,1,1);
                          break;
	  case VarTuple::DOUBLE:
                          vt->storeValueTo(d);
                          pvm_pkdouble(&d,1,1);
                          break;
          default: 
                fatal("Tried to pvm_pack an unknown type for [%s]"
                      ,vt->data().data());
        }//switch Type
     }//for
     if (pvm_send(entity->tid(),msgtag) < 0) 
        fatal("pvm_send error!");
     
  } else { //send incl. VarTags as string
    //debug("send Net Inefficient, but all Tags");
     for(VarTuple *vt=state.first();vt != 0;vt=state.next()) {
        i=strlen(vt->data().data() )+1; // add zero-byte
        if (pvm_pkint(&i,1,1)<0)  // send length
           fatal("pvm_pkint varlength error!");
        if (pvm_pkstr(vt->data().data())<0) //set VarEntry
           fatal("pvm_pkstr var error!");
	//debug("endgueltig gepackt: [%s]\n",(const char *)sl.current() );
     }//for
     if (pvm_send(entity->tid(),msgtag) < 0) 
        fatal("pvm_send error!");
  }// if sendNetEfficient
 
  delete cs;
};//send

/*****************************************************************************/
void KPvm::checkRecv()
{
  //debug("KPvm::checkRecv");
  if (_exit) {  //qApp is now securely running...
    //debug("bye, bye requested");
    pvm_exit();
    qApp->exit();
  }

  int bufid=pvm_nrecv(-1,-1);

  if (bufid > 0) { //got data
    int bytes, msgtag, tid;
    if (pvm_bufinfo(bufid, &bytes, &msgtag, &tid)<0)
       fatal("Error reading bufinfo");
    QString s;
    s.sprintf("%i",tid);
    KPvmEntity *ent;
    if ((ent=_knownEntities[s])==0)
       fatal("Got data from an unknown sender t%x!",tid);
    int i;
    char *cs;
    if (pvm_upkint(&i,1,1)<0) //get length of className
       fatal("pvm_upkint classname-length error");
    cs=(char *)malloc(i); //get memory for className
    if (pvm_upkstr(cs)<0)
       fatal("pvm_upkint classname error");
    QString cName(cs);
    int nrVars;
    if (pvm_upkint(&nrVars,1,1)<0)
       fatal("pvm_upkint nrVars-length error");

    short sentNetEfficientShort;
    if (pvm_upkshort(&sentNetEfficientShort,1,1)<0)
       fatal("pvm_upkshort sentNetEfficient error");
    bool sentNetEfficient=(bool)sentNetEfficientShort;

    QList<VarTuple> state;
    if (sentNetEfficient) { 
      int type;
      float f; double d; long l;
      for (int j=0; j<nrVars; j++) {
         if (pvm_upkint(&type,1,1)<0)
            fatal("pvm_upkint varType error");
         switch ( type ) {
	    case VarTuple::INT:
                          pvm_upkint(&i,1,1);
                          state.append( new VarTuple("",i) );
                          break;
	    case VarTuple::LONG:
                          pvm_upklong(&l,1,1);
                          state.append( new VarTuple("",l) );
                          break;
	    case VarTuple::STRING:
                          if (pvm_upkint(&i,1,1)<0) //get length
                              fatal("pvm_upkint var-length %d",j);
                          cs=(char *)realloc(cs,i); //get memory
                          if (pvm_upkstr(cs)<0)
                             fatal("pvm_upkint var %d",j);
                          state.append( new VarTuple("",cs) );
                          break;
	    case VarTuple::FLOAT:
                          pvm_upkfloat(&f,1,1);
                          state.append( new VarTuple("",f) );
                          break;
	    case VarTuple::DOUBLE:
                          pvm_upkdouble(&d,1,1);
                          state.append( new VarTuple("",d) );
                          break;
            default: 
                fatal("Tried to pvm_unpack an unknown type for var nr. %d",j);
         }//switch Type         
      }//for nrVars
    } else {
      VarTuple* vt;
      for (int j=0; j<nrVars; j++) {
        if (pvm_upkint(&i,1,1)<0) //get length of varstring incl. zero-byte
           fatal("pvm_upkint var-length %d",j);
        cs=(char *)realloc(cs,i); //get memory for varstring incl. zero-byte
        if (pvm_upkstr(cs)<0)
          fatal("pvm_upkint var %d",j);
        //debug("received %d char [%s]",i,cs);
        vt = new VarTuple;
        vt->setData(cs);
        state.append(vt);
      }//for nrVars
    }// if sentNetEfficient
    emit dataReceived(cName, state, msgtag, ent);

    //ugly, but cant emit foreign protected signals directly
    ent->dataNotify(msgtag);
    delete cs;
  } //if got data
  else if (bufid <0)  //errror
    fatal("pvm_nrecv error !");

  QTimer::singleShot(_interval,this,SLOT(checkRecv()) );
  //debug("end of checkRecv.. new singleShot in %i msecs initiated",_interval);

}; // checkRecv

/*****************************************************************************/
bool KPvm::hasPvmParent()
{ 
  return (pvm_parent() != PvmNoParent);
};

/*****************************************************************************/
QString KPvm::hostName() const 
{
  char buf[65];
  if (gethostname(buf,64) <0)
     fatal("could not relove hostName of running KPvm");
  QString hn = QString(buf);
  return hn;
};

/*****************************************************************************/
void KPvm::exit()
{
  //debug("KPvm::exit");
  _exit=TRUE;
};


/*****************************************************************************/
void KPvm::insertNewEntity(KPvmEntity *ent)
{
  QString s;
  s.sprintf("%i",ent->tid());
  _knownEntities.replace(s,ent);
};
