/* This file is part of the KDE libraries
    Copyright (c) 1998 Emmeran Seehuber (the_emmy@hotmail.com)
 
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#include "kltabgroup.h"


KLTabGroup::KLTabGroup(KLPageGroup::CalcMode calcMode) : QWidget((QWidget *)0,0)
{
  setKLWidget(this);
  a_tabBar = NULL;
  setCalcMode(calcMode);
  a_group.parentChild = this;
}

void KLTabGroup::setupGrap()
{
  KLChild::setupGrap();
  a_group.setupGrap();
}

void KLTabGroup::cleanupGrap()
{
  a_group.cleanupGrap();
  KLChild::cleanupGrap();
}

bool KLTabGroup::klAskMinMax(KLMinMaxSizes *minMaxSizes)
{
  if( !KLWidgetBase::klAskMinMax(minMaxSizes) )
    return false;
  ulong x = a_tabBar->sizeHint().height() + TopBorder + BottomBorder;
  ulong y = LeftBorder + RightBorder;
  minMaxSizes->minX += x;
  minMaxSizes->minY += y;
  minMaxSizes->defX += x;
  minMaxSizes->defY += y;
  minMaxSizes->maxX += x;
  minMaxSizes->maxY += y;
  KLMinMaxSizes &mms = a_group.minMaxSizes();
  mms.init();
  a_group.klDoAskMinMax(&mms);
  *minMaxSizes += mms;
  minMaxSizes->validate();
  return true;
}

void KLTabGroup::refreshTabs()
{
  if( a_tabBar ) 
    delete a_tabBar;
  a_tabBar = new KTabBar(this,0);
  connect(a_tabBar,SIGNAL(selected(int)),this,SLOT(pageSelected(int)));
  ulong i;
  for( i = 0; i < a_tabTitles.count(); i++ ) {
    QTab *tab = new QTab;
    tab->id = i;
    tab->label = a_tabTitles.at(i);
    tab->enabled = true;
    a_tabBar->addTab(tab);
  }
}


bool KLTabGroup::klSetup( KLSetupInfo *setupInfo )
{
  if( !KLWidgetBase::klSetup(setupInfo) )
    return false;
  refreshTabs();
  if( !a_group.klSetup(setupInfo) ) {
    KLWidgetBase::klCleanup();
    return false;
  }
  return true;
}


bool KLTabGroup::klShow(KLShowInfo *showInfo)
{
  if( !KLWidgetBase::klShow(showInfo) ) 
    return false;
  a_tabBar->setGeometry(2,1,a_tabBar->sizeHint().width(),a_tabBar->sizeHint().height());
  ulong x = a_tabBar->sizeHint().height() + TopBorder;
  ulong y = LeftBorder;
  KLShowInfo &si = a_group.showInfo();
  si = *showInfo;
  si.x += x;
  si.y += y;
  si.xSize -= x + BottomBorder;
  si.ySize -= y + RightBorder;

  if( !a_group.klShow(&si) ) {
    KLWidgetBase::klHide();
    return false;
  }
  return true;
}

void KLTabGroup::klHide()
{
  if( a_group.a_state == SI_Showed )
    a_group.klHide();
  KLWidgetBase::klHide();
}


KLChild *KLTabGroup::findChild( ulong x, ulong y ) const
{
  if( !KLWidgetBase::findChild(x,y) )
    return 0;
  KLChild *child = a_group.findChild(x,y);
  if( child == 0 )
    child = (KLChild *)this; // Explizied discard const
  HASSERT(child);
  return child;
}


void KLTabGroup::klCleanup()
{
  if( a_group.a_state == SI_SetupDone )
    a_group.klCleanup();
  KLWidgetBase::klCleanup();
}

KLTabGroup &KLTabGroup::operator<<(KLChild *child)
{
  a_group.addChild(child);
  return *this;
}

KLTabGroup &KLTabGroup::operator<<(KLChild &child)
{
  a_group.addChild(&child);
  return *this;
}

KLTabGroup &KLTabGroup::operator<<(KLGroup::_grp_change_func func)
{
  func(a_group);
  return *this;
}

KLTabGroup &KLTabGroup::operator<<(KLChild::_child_change_func func)
{
  func(*this);
  return *this;
}

void KLTabGroup::pageSelected(int i)
{
  a_group.setActPage(i);
}

KLTabGroup &KLTabGroup::operator<<(const char *text)
{
  a_tabTitles.append(text);
  return *this;
}

KLChild::KLStreamHelp &KLTabGroup::operator<<(KLChild::_child_change_p1_func func)
{
  static KLStreamHelp sh;
  sh.child = this;
  sh.func = func;
  return sh;
}

void KLTabGroup::setActPage(int n)
{
  if( a_tabBar )
    a_tabBar->setCurrentTab(n);
  a_group.setActPage(n);
}

ulong KLTabGroup::actPage()
{
  return a_group.actPage();
}

QRect KLTabGroup::getChildRect() const
{
  return QRect(2, a_tabBar->height() + 1, width() - 4,
   height() - a_tabBar->height() - 4);
}

KLPageGroup::CalcMode KLTabGroup::calcMode()
{
  return a_group.calcMode();
}

void KLTabGroup::setCalcMode( KLPageGroup::CalcMode calcMode )
{
  a_group.setCalcMode(calcMode);
}

void KLTabGroup::addDropMarks( KLDropMarkList * dml ) const
{
  a_group.addDropMarks(dml);
}

bool KLTabGroup::isAChild( KLChild * _child ) const
{
  if( KLChild::isAChild(_child) )
    return true;
  return a_group.isAChild(_child);
}

void KLTabGroup::paintEvent(QPaintEvent *)
{
  QPainter p;
  p.begin(this);

  HASSERT(a_state == SI_Showed);
  int y0 = getChildRect().top() - 1;
  int y1 = getChildRect().bottom() + 2;
  int x1 = getChildRect().right() + 2;
  int x0 = getChildRect().left() - 1;

  p.setPen( colorGroup().light() );
  p.drawLine(x0, y0, x1 - 1, y0);      /* top line */
  p.drawLine(x0, y0 + 1, x0, y1);      /* left line */
  p.setPen(black);
  p.drawLine(x1, y1, x0, y1);          /* bottom line */
  p.drawLine(x1, y1 - 1, x1, y0);
  p.setPen(colorGroup().dark());
  p.drawLine(x0 + 1, y1 - 1, x1 - 1, y1 - 1);  /* bottom */
  p.drawLine(x1 - 1, y1 - 2, x1 - 1, y0 + 1);
  p.end();
}

#include "kltabgroup.moc"
