/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * vim: set ts=8 sts=4 et sw=4 tw=99:
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#ifndef jit_x86_SharedICHelpers_x86_h
#define jit_x86_SharedICHelpers_x86_h

#include "jit/BaselineFrame.h"
#include "jit/BaselineIC.h"
#include "jit/MacroAssembler.h"
#include "jit/SharedICRegisters.h"

namespace js {
namespace jit {

// Distance from stack top to the top Value inside an IC stub (this is the
// return address).
static const size_t ICStackValueOffset = sizeof(void*);

inline void EmitRestoreTailCallReg(MacroAssembler& masm) {
  masm.Pop(ICTailCallReg);
}

inline void EmitRepushTailCallReg(MacroAssembler& masm) {
  masm.Push(ICTailCallReg);
}

inline void EmitCallIC(CodeOffset* patchOffset, MacroAssembler& masm) {
  // Move ICEntry offset into ICStubReg
  CodeOffset offset = masm.movWithPatch(ImmWord(-1), ICStubReg);
  *patchOffset = offset;

  // Load stub pointer into ICStubReg
  masm.loadPtr(Address(ICStubReg, (int32_t)ICEntry::offsetOfFirstStub()),
               ICStubReg);

  // Load stubcode pointer from BaselineStubEntry into ICTailCallReg
  // ICTailCallReg will always be unused in the contexts where ICs are called.
  masm.call(Address(ICStubReg, ICStub::offsetOfStubCode()));
}

inline void EmitEnterTypeMonitorIC(
    MacroAssembler& masm,
    size_t monitorStubOffset = ICMonitoredStub::offsetOfFirstMonitorStub()) {
  // This is expected to be called from within an IC, when ICStubReg
  // is properly initialized to point to the stub.
  masm.loadPtr(Address(ICStubReg, (int32_t)monitorStubOffset), ICStubReg);

  // Jump to the stubcode.
  masm.jmp(Operand(ICStubReg, (int32_t)ICStub::offsetOfStubCode()));
}

inline void EmitReturnFromIC(MacroAssembler& masm) { masm.ret(); }

inline void EmitChangeICReturnAddress(MacroAssembler& masm, Register reg) {
  masm.storePtr(reg, Address(StackPointer, 0));
}

inline void EmitBaselineLeaveStubFrame(MacroAssembler& masm,
                                       bool calledIntoIon = false) {
  // Ion frames do not save and restore the frame pointer. If we called
  // into Ion, we have to restore the stack pointer from the frame descriptor.
  // If we performed a VM call, the descriptor has been popped already so
  // in that case we use the frame pointer.
  if (calledIntoIon) {
    Register scratch = ICStubReg;
    masm.Pop(scratch);
    masm.shrl(Imm32(FRAMESIZE_SHIFT), scratch);
    masm.addl(scratch, BaselineStackReg);
  } else {
    masm.mov(BaselineFrameReg, BaselineStackReg);
  }

  masm.Pop(BaselineFrameReg);
  masm.Pop(ICStubReg);

  // The return address is on top of the stack, followed by the frame
  // descriptor. Use a pop instruction to overwrite the frame descriptor
  // with the return address. Note that pop increments the stack pointer
  // before computing the address.
  masm.Pop(Operand(BaselineStackReg, 0));
}

template <typename AddrType>
inline void EmitPreBarrier(MacroAssembler& masm, const AddrType& addr,
                           MIRType type) {
  masm.guardedCallPreBarrier(addr, type);
}

inline void EmitStubGuardFailure(MacroAssembler& masm) {
  // Load next stub into ICStubReg
  masm.loadPtr(Address(ICStubReg, ICStub::offsetOfNext()), ICStubReg);

  // Return address is already loaded, just jump to the next stubcode.
  masm.jmp(Operand(ICStubReg, ICStub::offsetOfStubCode()));
}

}  // namespace jit
}  // namespace js

#endif /* jit_x86_SharedICHelpers_x86_h */
