/*
 *	Interface GTK+ Theme Engine
 * 
 *  interface_theme_color.c:
 *		Routines for modifying colors and remapping the colors in XPM data
 *		to the colors in the GtkStyle.
 *
 *	Copyright  2000 Johan Hanson <johan@tiq.com>.
 *	
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Library General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *	
 *	This library is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *	Library General Public License for more details.
 *	
 *	You should have received a copy of the GNU Library General Public
 *	License along with this library; if not, write to the 
 *	Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
 *	Boston, MA  02111-1307  USA.
 */

#include "Interface_theme.h"


/*** Conversion routines ***/

void interface_color_to_rgb (const GdkColor *src, InterfaceRGB *dst) {
	const gdouble f = 1.0/65535.0;
	dst->r = f * src->red;
	dst->g = f * src->green;
	dst->b = f * src->blue;
}

void interface_rgb_to_color (const InterfaceRGB *src, GdkColor *dst) {
	dst->red   = src->r * 65535.0;
	dst->green = src->g * 65535.0;
	dst->blue  = src->b * 65535.0;
}

void interface_rgb_to_hsl (const InterfaceRGB *src, InterfaceHSL *dst) {
	gdouble max, min, d, r,g,b;
	r = src->r;
	g = src->g;
	b = src->b;
	if (r > g) {
		max = MAX(r, b);
		min = MIN(g, b);
	} else {
		max = MAX(g, b);
		min = MIN(r, b);
	}
	dst->l = (max + min) / 2;
	dst->s = 0.0;
	dst->h = 0.0;

	if (max != min) {
		d = max - min;
		dst->s = d / (dst->l <= 0.5 ? max + min : 2.0 - max - min);
		
		if (max==r)	dst->h = (g - b)/d;
		else if (max==g) dst->h = 2.0 + (b - r)/d;
		else if (max==b) dst->h = 4.0 + (r - g)/d;
		
		dst->h *= 60;
		if (dst->h<0.0) dst->h += 360.0;
	}
}

void interface_hsl_to_rgb (const InterfaceHSL *src, InterfaceRGB *dst) {
	gdouble h,l,s,  m1,m2, r,g,b;
	
	l = src->l; l = CLAMP(l, 0.0, 1.0);
	s = src->s; s = CLAMP(s, 0.0, 1.0);
	
	m2 = l <= 0.5 ? l*(1.0+s) : l+s - l*s;
	m1 = 2.0*l - m2;
	
	if (s == 0) {
		dst->r = dst->g = dst->b = l;
    } else {
		h = src->h + 120.0;
		while (h > 360.0) h -= 360.0;
		while (h < 0.0) h += 360.0;
		
		if (h < 60.0)	r = m1 + (m2 - m1) * h / 60.0;
		else if (h < 180.0) r = m2;
		else if (h < 240.0) r = m1 + (m2 - m1) * (240.0 - h) / 60.0;
		else r = m1;
		
		h = src->h;
		while (h > 360.0) h -= 360.0;
		while (h < 0.0) h += 360.0;
		
		if (h < 60.0) g = m1 + (m2 - m1) * h / 60.0;
		else if (h < 180.0) g = m2;
		else if (h < 240.0) g = m1 + (m2 - m1) * (240.0 - h) / 60.0;
		else g = m1;
		
		h = src->h - 120.0;
		while (h > 360.0) h -= 360.0;
		while (h < 0.0) h += 360.0;
		
		if (h < 60.0) b = m1 + (m2 - m1) * h / 60.0;
		else if (h < 180.0) b = m2;
		else if (h < 240.0) b = m1 + (m2 - m1) * (240.0 - h) / 60.0;
		else b = m1;
		
		dst->r = r;
		dst->g = g;
		dst->b = b;
	}
}


void interface_color_to_hsl (const GdkColor *color, InterfaceHSL *hsl) {
	InterfaceRGB rgb;
	interface_color_to_rgb (color, &rgb);
	interface_rgb_to_hsl (&rgb, hsl);
}

void interface_hsl_to_color (const InterfaceHSL *hsl, GdkColor *color) {
	InterfaceRGB rgb;
	interface_hsl_to_rgb (hsl, &rgb);
	interface_rgb_to_color (&rgb, color);
}


/*** Blending routines ***/

void interface_shade_color (const GdkColor *src, GdkColor *dst, gdouble k) {
	InterfaceRGB rgb;
	InterfaceHSL hsl;

	interface_color_to_rgb (src, &rgb);
	interface_rgb_to_hsl (&rgb, &hsl);

	if (k>2.0) k=2.0;
	else if (k<0.0) k=0.0;

	if (k > 1.0) {
		hsl.l = (k - 1.0) * hsl.l + (k - 1.0);
		hsl.s = MIN(hsl.s, (k - 1.0) * hsl.s);
	} else {
		hsl.l = k * hsl.l;
	}

	interface_hsl_to_rgb(&hsl, &rgb);
	interface_rgb_to_color(&rgb, dst);
}

static void interface_blend_hsl (const InterfaceHSL *a, const InterfaceHSL *b, gdouble k, InterfaceHSL *dst) {
	dst->l = a->l + k * (b->l - a->l);
	dst->s = a->s + k * (b->s - a->s);

	dst->h = b->h;
  /* cheat it
	if (ABS(a->h - b->h) < 180) {
		dst->h = a->h + k * (b->h - a->h);
	} else {
		dst->h = (a->h > b->h) ? a->h + k * (360.0+b->h - a->h)
							   : b->h + (1.0-k) * (360.0+a->h - b->h);
		if (dst->h >= 360.0) dst->h -= 360.0;
	}
  */
}

static void
interface_blend_rgb_to_hsl (const InterfaceRGB *a_rgb, const InterfaceRGB *b_rgb, gdouble k, InterfaceHSL *dst)
{
	InterfaceHSL a_hsl, b_hsl;
	interface_rgb_to_hsl (a_rgb, &a_hsl);
	interface_rgb_to_hsl (b_rgb, &b_hsl);
	interface_blend_hsl (&a_hsl, &b_hsl, k, dst);
}

void
interface_blend_color (const GdkColor *a, const GdkColor *b, gdouble k, GdkColor *dst) {
	InterfaceHSL hsla, hslb;
	interface_color_to_hsl (a, &hsla);
	interface_color_to_hsl (b, &hslb);

	interface_blend_hsl (&hsla, &hslb, k, &hsla);

	interface_hsl_to_color (&hsla, dst);
}


/*** Pixmap remapping functions ***/
static const guchar * interface_read_rgb (const guchar *stream, const GtkStyle *style,
									 const GtkStyle *style2, GtkStateType state_type,
									 InterfaceRGB *dst)
{
	guchar		c;
	GtkStateType state_use;

	c = *(stream++);
	if (c < INTERFACE_WHITE) {
		/* assume the pen is a legal Gtk pen */
		state_use = (GtkStateType)(*(stream++));
		if (state_use & INTERFACE_PARENT) {
			style = style2;
			state_use &= 0x0f;
		}
		if (state_use == (GtkStateType)INTERFACE_CURRENT)
			state_use = state_type;
		interface_color_to_rgb(&style->fg[5*c + state_use], dst);
		if (c == INTERFACE_BG || c == INTERFACE_FG)
			return stream;
	} else if (c == INTERFACE_BLACK) {
		dst->r = dst->g = dst->b = 0.0;
	} else if (c == INTERFACE_WHITE) {
		dst->r = dst->g = dst->b = 1.0;
	} else if (c == INTERFACE_USER) {
		dst->r = *(stream++) / 255.0;
		dst->g = *(stream++) / 255.0;
		dst->b = *(stream++) / 255.0;
	} else if (c == INTERFACE_THIN) {
		/* conditional expression, for thin mode */
		InterfaceThemeData *data;
		data = style->engine_data;
		stream = interface_read_rgb (stream, style, style2, state_type, dst);
		if (data && data->thin) {
			InterfaceRGB dummy;
			return interface_read_rgb (stream, style, style2, state_type, &dummy);
		} else {
			return interface_read_rgb (stream, style, style2, state_type, dst);
		}
	} else {
		InterfaceRGB	rgba, rgbb;
		InterfaceHSL hsla, hslb, hsld;
		gdouble	k, k2;
	
		/* blend operations */
		if (c == INTERFACE_BI) {
			k = *(stream++) / 255.0;
			stream = interface_read_rgb(stream, style, style2, state_type, &rgba);
			stream = interface_read_rgb(stream, style, style2, state_type, &rgbb);
			interface_blend_rgb_to_hsl (&rgba, &rgbb, k, &hsld);
		} else if (c == INTERFACE_HUE) {
			stream = interface_read_rgb (stream, style, style2, state_type, &rgba);
			interface_rgb_to_hsl (&rgba, &hsld);
			k = *(stream++) / 255.0;
			if (k < hsld.l) hsld.l = k;
			hsld.l = *(stream++) / 255.0;
		} else {
			k  = *(stream++) / 255.0;
			k2 = *(stream++) / 255.0;
			stream = interface_read_rgb(stream, style, style2, state_type, &rgba);
			stream = interface_read_rgb(stream, style, style2, state_type, &rgbb);

			if (c == INTERFACE_TRI) {
				interface_blend_rgb_to_hsl (&rgba, &rgbb, k, &hsla);
				stream = interface_read_rgb(stream, style, style2, state_type, &rgba);
				interface_rgb_to_hsl (&rgba, &hslb);
				interface_blend_hsl (&hsla, &hslb, k2, &hsld);
			} else if (c == INTERFACE_QUAD) {
				interface_blend_rgb_to_hsl (&rgba, &rgbb, k, &hsla);
				stream = interface_read_rgb(stream, style, style2, state_type, &rgba);
				stream = interface_read_rgb(stream, style, style2, state_type, &rgbb);
				interface_blend_rgb_to_hsl (&rgba, &rgbb, k, &hslb);
				interface_blend_hsl (&hsla, &hslb, k2, &hsld);
			} else {
				g_error("error in remapping stream\n");
			}
		}
		interface_hsl_to_rgb (&hsld, dst);
	}
	
	return stream;
}


char ** interface_xpm_copy (const char **xpm) {
	gint w, h, nc, cpp, i;
	char **ret;
	sscanf(xpm[0], "%d %d %d %d", &w, &h, &nc, &cpp);
	ret = g_malloc(sizeof(gpointer) * (h + nc + 1));
	for (i=0; i<h+nc+1; i++)
		ret[i] = g_strdup(xpm[i]);

	return ret;
}

void interface_xpm_free (char **xpm) {
	gint w, h, nc, cpp, i;
	const gchar **ret;
	sscanf(xpm[0], "%d %d %d %d", &w, &h, &nc, &cpp);
	for (i=0; i<h+nc+1; i++)
		g_free(xpm[i]);
	g_free(xpm);
}


/* The main remapping function */
void interface_xpm_remap (const GtkStyle *style, const GtkStyle *style2, GtkStateType state_type,
					 const guchar *stream, char **xpm)
{
	static char hextab[] = "0123456789ABCDEF";
	InterfaceRGB	rgb, bg;
	guint	r, g, b;
	guint	i = 1;
	guchar	c;

	if (   state_type == GTK_STATE_INSENSITIVE
		&& !(style->engine_data && ((InterfaceThemeData *)style->engine_data)->thin))
	{
		interface_color_to_rgb(&style2->bg[GTK_STATE_INSENSITIVE], &bg);
	}
	
	while (c = *(stream++)) {
		if (xpm[i][0] != c) {
			g_error("XPM and remapping stream do not match at color i==%d, remap==%d\n", i-1, c);
			return;
		}
		
		/* parse color entry */	
		stream = interface_read_rgb(stream, style, style2, state_type, &rgb);
		if (   state_type == GTK_STATE_INSENSITIVE
			&& !(style->engine_data && ((InterfaceThemeData *)style->engine_data)->thin))
		{
			rgb.r = 0.5 * (rgb.r + bg.r);
			rgb.g = 0.5 * (rgb.g + bg.g);
			rgb.b = 0.5 * (rgb.b + bg.b);
		}

		r = CLAMP(rgb.r, 0.0, 1.0) * 255;
		g = CLAMP(rgb.g, 0.0, 1.0) * 255;
		b = CLAMP(rgb.b, 0.0, 1.0) * 255;

		(xpm[i])[5]  = hextab[((r&0x00f0)>>4) & 0x0000000f];
		(xpm[i])[6]  = hextab[  r				& 0x0000000f];
		(xpm[i])[7]  = hextab[((g&0x00f0)>>4) & 0x0000000f];
		(xpm[i])[8]  = hextab[  g				& 0x0000000f];
		(xpm[i])[9]  = hextab[((b&0x00f0)>>4) & 0x0000000f];
		(xpm[i])[10] = hextab[  b				& 0x0000000f];
		(xpm[i])[11] = 0;
		i++;
	}
}


/* Compare two GdkColors */
gboolean interface_color_equal (const GdkColor *a, const GdkColor *b) {
	return (   ((a->red   & 0xfe00) == (b->red   & 0xfe00))
			&& ((a->green & 0xfe00) == (b->green & 0xfe00))
			&& ((a->blue  & 0xfe00) == (b->blue  & 0xfe00)));
}

