/*
 * src/properties.c - control panel applet for gnome-volume-manager
 *
 * Robert Love <rml@ximian>
 *
 * (C) Copyright 2004 Novell, Inc.
 *
 * Licensed under the GNU GPL v2.  See COPYING.
 */

#include "config.h"

#include <gnome.h>
#include <gdk/gdkx.h>
#include <gtk/gtkwindow.h>
#include <glade/glade.h>
#include <gconf/gconf-client.h>

#include "gvm.h"

#ifdef ENABLE_NLS
#  include <libintl.h>
#  define _(String) gettext (String)
#  ifdef gettext_noop
#    define N_(String) gettext_noop (String)
#  else
#    define N_(String) (String)
#  endif
#else
#  define _(String)
#  define N_(String) (String)
#endif

GtkWidget *dialog;
GtkWidget *automount_drives_cb;
GtkWidget *automount_media_cb;
GtkWidget *autorun_cb;
GtkWidget *autoplay_cda_cb;
GtkWidget *command_cda_hbox;
GtkWidget *command_cda_entry;
GtkWidget *command_cda_fileentry;
GtkWidget *autoburn_cdr_cb;
GtkWidget *command_cdr_fileentry;
GtkWidget *command_cdr_entry;
GtkWidget *command_cdr_hbox;
GtkWidget *autoplay_dvd_cb;
GtkWidget *command_dvd_hbox;
GtkWidget *command_dvd_fileentry;
GtkWidget *command_dvd_entry;
GtkWidget *autophoto_cb;
GtkWidget *autophoto_command;
GtkWidget *command_photo_fileentry;
GtkWidget *command_photo_entry;
GtkWidget *command_photo_hbox;

gboolean lock_config = FALSE;

struct gvm_configuration config;

/*
 * set_sensitivity - set the widget sensitivity based on the configuration
 */
static void
set_sensitivity (void)
{
	gtk_widget_set_sensitive (autorun_cb,
			config.automount_media || config.automount_drives);
	gtk_widget_set_sensitive (command_cda_hbox, config.autoplay_cda);
	gtk_widget_set_sensitive (command_cdr_hbox, config.autoburn_cdr);
	gtk_widget_set_sensitive (command_dvd_hbox, config.autoplay_dvd);
	gtk_widget_set_sensitive (command_photo_hbox, config.autophoto);
}

/*
 * write_config - sync configure structure => gconf keys
 */
static void
write_config (void)
{
	gconf_client_set_bool (config.client, GCONF_ROOT "automount_drives",
			       config.automount_drives, NULL);
	gconf_client_set_bool (config.client, GCONF_ROOT "automount_media",
			       config.automount_media, NULL);
	gconf_client_set_bool (config.client, GCONF_ROOT "autorun",
			       config.autorun, NULL);
	gconf_client_set_bool (config.client, GCONF_ROOT "autoburn_cdr",
			       config.autoburn_cdr, NULL);
	gconf_client_set_string (config.client,
				 GCONF_ROOT "autoburn_cdr_command",
				 config.autoburn_cdr_command, NULL);	
	gconf_client_set_bool (config.client, GCONF_ROOT "autoplay_cda",
			       config.autoplay_cda, NULL);
	gconf_client_set_string (config.client,
				 GCONF_ROOT "autoplay_cda_command",
				 config.autoplay_cda_command, NULL);
	gconf_client_set_bool (config.client, GCONF_ROOT "autoplay_dvd",
			config.autoplay_dvd, NULL);
	gconf_client_set_string (config.client,
				 GCONF_ROOT "autoplay_dvd_command",
				 config.autoplay_dvd_command, NULL);
	gconf_client_set_bool (config.client, GCONF_ROOT "autophoto",
			       config.autophoto, NULL);
	gconf_client_set_string (config.client, GCONF_ROOT "autophoto_command",
				 config.autophoto_command, NULL);
}

/*
 * signal_response - respond to GTK signal by sync'ing widgets => config struct
 */
static void
signal_response (void)
{
	config.automount_drives = GTK_TOGGLE_BUTTON
			(automount_drives_cb)->active;
	config.automount_media = GTK_TOGGLE_BUTTON
			(automount_media_cb)->active;
	config.autorun = GTK_TOGGLE_BUTTON (autorun_cb)->active;
	config.autoburn_cdr = GTK_TOGGLE_BUTTON (autoburn_cdr_cb)->active;
	config.autoplay_cda = GTK_TOGGLE_BUTTON (autoplay_cda_cb)->active;
	config.autoplay_dvd = GTK_TOGGLE_BUTTON (autoplay_dvd_cb)->active;
	config.autophoto = GTK_TOGGLE_BUTTON (autophoto_cb)->active;
	g_free (config.autoplay_cda_command);
	config.autoplay_cda_command = gtk_editable_get_chars
	    (GTK_EDITABLE (command_cda_entry), 0, -1);
	g_free (config.autoplay_dvd_command);
	config.autoplay_dvd_command = gtk_editable_get_chars
	    (GTK_EDITABLE (command_dvd_entry), 0, -1);
	g_free (config.autoburn_cdr_command);
	config.autoburn_cdr_command = gtk_editable_get_chars
		(GTK_EDITABLE (command_cdr_entry), 0, -1);
	g_free (config.autophoto_command);
	config.autophoto_command = gtk_editable_get_chars
		(GTK_EDITABLE (command_photo_entry), 0, -1);

	set_sensitivity ();

	if (lock_config == FALSE)
		write_config ();
}

/*
 * check_clipboard - check if the CLIPBOARD_NAME clipboard is available
 *
 * Returns TRUE if the CLIPBOARD_NAME clipboard is out there and FALSE
 * otherwise
 */
static gboolean
check_clipboard (void)
{
	Atom clipboard_atom = gdk_x11_get_xatom_by_name (CLIPBOARD_NAME);

	return XGetSelectionOwner (GDK_DISPLAY (), clipboard_atom) != None;
}

/*
 * check_daemon - check if the daemon itself is running. if not, and if we
 * have configuration options set that would make it worthwhile to run, run
 * it.
 */
static void
check_daemon (void)
{
	char *argv[2];
	GError *error = NULL;

	if (!(config.automount_drives || config.automount_media
			|| config.autorun || config.autoplay_cda
			|| config.autoplay_dvd || config.autophoto)
			|| check_clipboard ()) 
		return;

	argv[0] = BINDIR "/gnome-volume-manager";
	argv[1] = NULL;
	g_spawn_async (g_get_home_dir (), argv, NULL, 0, NULL,
			NULL, NULL, &error);

	if (error) {
		GtkWidget *message = gtk_message_dialog_new
			(GTK_WINDOW (dialog), GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
			_("Error starting gnome-volume-manager daemon:\n%s"),
			error->message);

		gtk_dialog_run (GTK_DIALOG (message));

		g_error_free (error);

		exit (EXIT_FAILURE);
	}
}

/*
 * update_widgets - sync configure structure -> widgets
 */
static void
update_widgets (void)
{
	lock_config = TRUE;

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (automount_drives_cb),
				      config.automount_drives);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (automount_media_cb),
				      config.automount_media);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (autorun_cb),
				      config.autorun);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (autoplay_cda_cb),
				      config.autoplay_cda);

	if (config.autoplay_cda_command != NULL)
		gtk_entry_set_text (GTK_ENTRY (command_cda_entry),
				    config.autoplay_cda_command);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (autoburn_cdr_cb),
				      config.autoburn_cdr);
	if (config.autoburn_cdr_command != NULL)
		gtk_entry_set_text (GTK_ENTRY (command_cdr_entry),
				    config.autoburn_cdr_command);	
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (autoplay_dvd_cb),
				      config.autoplay_dvd);
	if (config.autoplay_dvd_command != NULL)
		gtk_entry_set_text (GTK_ENTRY (command_dvd_entry),
				    config.autoplay_dvd_command);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (autophoto_cb),
				      config.autophoto);
	if (config.autophoto_command != NULL)
		gtk_entry_set_text (GTK_ENTRY (command_photo_entry),
				    config.autophoto_command);

	set_sensitivity ();

	check_daemon ();

	lock_config = FALSE;
}

/*
 * read_config - sync gconf keys -> configure structure
 */
static void
read_config (void)
{
	config.automount_drives = gconf_client_get_bool (config.client,
			GCONF_ROOT "automount_drives", NULL);
	config.automount_media = gconf_client_get_bool (config.client,
			GCONF_ROOT "automount_media", NULL);
	config.autorun = gconf_client_get_bool (config.client,
			GCONF_ROOT "autorun", NULL);
	config.autoplay_cda = gconf_client_get_bool (config.client,
			GCONF_ROOT "autoplay_cda", NULL);
	config.autoplay_cda_command =
			gconf_client_get_string (config.client,
			GCONF_ROOT "autoplay_cda_command", NULL);
	config.autoburn_cdr =
			gconf_client_get_bool (config.client,
			GCONF_ROOT "autoburn_cdr", NULL);
	config.autoburn_cdr_command =
			gconf_client_get_string (config.client,
			GCONF_ROOT "autoburn_cdr_command", NULL);
	config.autoplay_dvd = gconf_client_get_bool (config.client,
			GCONF_ROOT "autoplay_dvd", NULL);
	config.autoplay_dvd_command =
			gconf_client_get_string (config.client,
			GCONF_ROOT "autoplay_dvd_command", NULL);
	config.autophoto = gconf_client_get_bool (config.client,
			GCONF_ROOT "autophoto", NULL);
	config.autophoto_command =
			gconf_client_get_string (config.client,
			GCONF_ROOT "autophoto_command", NULL);

	update_widgets ();
}

/*
 * config_change_cb - gconf call-back in response to gconf key change.  causes
 * us to reread our configuration.
 */
static void
config_change_cb (GConfClient *client, guint id, GConfEntry *entry,
		   gpointer data)
{
	(void) client;
	(void) id;
	(void) entry;
	(void) data;

	read_config ();
}

/*
 * close_cb - gtk call-back on the "Close" button
 */
static void
close_cb (void)
{
	gtk_widget_destroy (dialog);
	gtk_main_quit ();
}

/*
 * set_icon - helper function to bind an icon from a given GnomeIconTheme to
 * a given GtkImage.
 */
static void
set_icon (GtkImage *image, GnomeIconTheme *theme, const char *name)
{
	char *file;

	file = gnome_icon_theme_lookup_icon (theme, name, 48, NULL, NULL);
	if (file) {
		gtk_image_set_from_file (image, file);
		g_free (file);
	}
}

#define GLADE_XML_FILE	"gnome-volume-properties.glade"

/*
 * init_capplet - initialize this little beast.
 */
static void
init_capplet (void)
{
	GnomeIconTheme *icon_theme;
	GtkWidget *icon;
	GladeXML *xml;
	gchar *glade_file;
	char *file;

	config.client = gconf_client_get_default ();
	gconf_client_add_dir (config.client, GCONF_ROOT_SANS_SLASH,
			      GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
	gconf_client_notify_add (config.client, GCONF_ROOT_SANS_SLASH,
				 config_change_cb, NULL, NULL, NULL);

	glade_file = g_concat_dir_and_file (GLADEDIR, GLADE_XML_FILE);
	xml = glade_xml_new (glade_file, NULL, PACKAGE);
	g_free (glade_file);

	if (xml == NULL) {
		dialog = gtk_message_dialog_new (NULL,
						 0, GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_OK,
						 "<b>%s</b>\n%s.",
						 _("Could not load the main "
						   "interface"),
						 _("Please make sure that the "
						   "volume manager is properly "
						   "installed"));
		gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
		gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);
		gtk_label_set_use_markup (GTK_LABEL
					  (GTK_MESSAGE_DIALOG (dialog)->label),
					  TRUE);
		gtk_dialog_set_default_response (GTK_DIALOG (dialog),
						 GTK_RESPONSE_OK);
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);
		exit (EXIT_FAILURE);
	}

	dialog = glade_xml_get_widget (xml, "dialog1");

	icon_theme = gnome_icon_theme_new ();

	icon = glade_xml_get_widget (xml, "removable_drives_image");
	set_icon (GTK_IMAGE (icon), icon_theme, "gnome-dev-zipdisk");
	icon = glade_xml_get_widget (xml, "audio_cd_image");
	set_icon (GTK_IMAGE (icon), icon_theme, "gnome-dev-cdrom-audio");
	icon = glade_xml_get_widget (xml, "blank_cd_image");
	set_icon (GTK_IMAGE (icon), icon_theme, "gnome-dev-cdrom");
	icon = glade_xml_get_widget (xml, "dvd_video_image");
	set_icon (GTK_IMAGE (icon), icon_theme, "gnome-dev-dvd");
	icon = glade_xml_get_widget (xml, "digital_camera_image");
	set_icon (GTK_IMAGE (icon), icon_theme, "applets-screenshooter");

	file = gnome_icon_theme_lookup_icon (icon_theme,
			"gnome-dev-cdrom", 48, NULL, NULL);
	if (file) {
		gtk_window_set_default_icon_from_file (file, NULL);
		g_free (file);
	}

	g_object_unref (icon_theme);

	automount_drives_cb = glade_xml_get_widget (xml, "automount_drives_cb");
	automount_media_cb = glade_xml_get_widget (xml, "automount_media_cb");
	autorun_cb = glade_xml_get_widget (xml, "autorun_cb");

	autoplay_cda_cb = glade_xml_get_widget (xml, "autoplay_cda_cb");
	command_cda_entry = glade_xml_get_widget (xml, "command_cda_entry");
	command_cda_fileentry = gnome_file_entry_gnome_entry (GNOME_FILE_ENTRY
			(glade_xml_get_widget (xml, "command_cda_fileentry")));
	command_cda_hbox = glade_xml_get_widget (xml, "command_cda_hbox");

	gnome_entry_set_history_id (GNOME_ENTRY (command_cda_fileentry),
				    "CD_CAPPLET_ID");
	gtk_combo_set_case_sensitive (GTK_COMBO (command_cda_fileentry),
				      FALSE);

	autoplay_dvd_cb = glade_xml_get_widget (xml, "autoplay_dvd_cb");
	command_dvd_entry = glade_xml_get_widget (xml, "command_dvd_entry");
	command_dvd_fileentry = gnome_file_entry_gnome_entry (GNOME_FILE_ENTRY
			(glade_xml_get_widget (xml, "command_dvd_fileentry")));
	command_dvd_hbox = glade_xml_get_widget (xml, "command_dvd_hbox");
	gnome_entry_set_history_id (GNOME_ENTRY (command_dvd_fileentry),
			"CD_CAPPLET_ID");
	gtk_combo_set_case_sensitive (GTK_COMBO (command_dvd_fileentry),
				      FALSE);

	autoburn_cdr_cb = glade_xml_get_widget (xml, "autoburn_cdr_cb");
	command_cdr_entry = glade_xml_get_widget (xml, "command_cdr_entry");
	command_cdr_fileentry = gnome_file_entry_gnome_entry (GNOME_FILE_ENTRY
			(glade_xml_get_widget (xml, "command_cdr_fileentry")));
	command_cdr_hbox = glade_xml_get_widget (xml, "command_cdr_hbox");
	gnome_entry_set_history_id (GNOME_ENTRY (command_cdr_fileentry),
			"CD_CAPPLET_ID");
	gtk_combo_set_case_sensitive (GTK_COMBO (command_cdr_fileentry),
			FALSE);
	autophoto_cb = glade_xml_get_widget (xml, "autophoto_cb");
	command_photo_entry = glade_xml_get_widget (xml, "command_photo_entry");
	command_photo_fileentry = gnome_file_entry_gnome_entry
			(GNOME_FILE_ENTRY (glade_xml_get_widget (xml,
			"command_photo_fileentry")));
	command_photo_hbox = glade_xml_get_widget (xml, "command_photo_hbox");
	gnome_entry_set_history_id (GNOME_ENTRY (command_photo_fileentry),
			"CD_CAPPLET_ID");
	gtk_combo_set_case_sensitive (GTK_COMBO (command_photo_fileentry),
			FALSE);

	read_config ();
	update_widgets ();

	gtk_signal_connect (GTK_OBJECT (automount_drives_cb), "toggled",
			    GTK_SIGNAL_FUNC (signal_response), NULL);
	gtk_signal_connect (GTK_OBJECT (automount_media_cb), "toggled",
			    GTK_SIGNAL_FUNC (signal_response), NULL);
	gtk_signal_connect (GTK_OBJECT (autorun_cb), "toggled",
			    GTK_SIGNAL_FUNC (signal_response), NULL);

	gtk_signal_connect (GTK_OBJECT (autoplay_cda_cb), "toggled",
			    GTK_SIGNAL_FUNC (signal_response), NULL);
	gtk_signal_connect (GTK_OBJECT (command_cda_entry), "changed",
			    GTK_SIGNAL_FUNC (signal_response), NULL);

	gtk_signal_connect (GTK_OBJECT (autoburn_cdr_cb), "toggled",
			    GTK_SIGNAL_FUNC (signal_response), NULL);
	gtk_signal_connect (GTK_OBJECT (command_cdr_entry), "changed",
			    GTK_SIGNAL_FUNC (signal_response), NULL);

	gtk_signal_connect (GTK_OBJECT (autoplay_dvd_cb), "toggled",
			    GTK_SIGNAL_FUNC (signal_response), NULL);
	gtk_signal_connect (GTK_OBJECT (command_dvd_entry), "changed",
			    GTK_SIGNAL_FUNC (signal_response), NULL);

	gtk_signal_connect (GTK_OBJECT (autophoto_cb), "toggled",
			    GTK_SIGNAL_FUNC (signal_response), NULL);
	gtk_signal_connect (GTK_OBJECT (command_photo_entry), "changed",
			    GTK_SIGNAL_FUNC (signal_response), NULL);	

	gtk_widget_show_all (dialog);
	gtk_signal_connect (GTK_OBJECT (dialog), "response",
			    GTK_SIGNAL_FUNC (close_cb), NULL);

	check_daemon ();
}

int
main (int argc, char *argv[])
{

	gnome_program_init (PACKAGE, VERSION, LIBGNOMEUI_MODULE, argc, argv,
			    GNOME_PARAM_NONE);

	glade_gnome_init ();

	bindtextdomain(PACKAGE, GNOMELOCALEDIR);
	bind_textdomain_codeset(PACKAGE, "UTF-8");
	textdomain(PACKAGE);

	init_capplet ();

	gtk_main ();

	return 0;
}
