from DataTarget import DataTarget
from utils.datatypes import *
from urllib import unquote # for filename conversion
from main import UNSET_COORD
import gtk



#
# Abstract class for DisplayTargets.
#
class DisplayTarget(gtk.HBox, DataTarget):

    # observer actions
    OBS_MOVE = 0

    # user actions
    ACTION_CLICK = 0
    ACTION_DOUBLECLICK = 1
    ACTION_PRESS = 2
    ACTION_RELEASE = 3
    ACTION_MENU = 4
    ACTION_SCROLL = 5
    ACTION_ENTER = 6
    ACTION_LEAVE = 7
    ACTION_MOTION = 8
    ACTION_FILE_DROP = 9
    ACTION_LINK = 10

    # placement anchors
    ANCHOR_NW = "nw"
    ANCHOR_N = "n"
    ANCHOR_NE = "ne"
    ANCHOR_E = "e"
    ANCHOR_SE = "se"
    ANCHOR_S = "s"
    ANCHOR_SW = "sw"
    ANCHOR_W = "w"
    ANCHOR_CENTER = "center"


    # what we accept for drag & drop
    __DND_FILE = [('text/uri-list', 0, 0)]


    def __init__(self, parent, display):

        # coordinates and size of this widget
        self.__geometry = (0, 0, 0, 0)

        # the widget to which this one is positioned relatively
        self.__relative = None

        # relative settings (is_rel_x, is_rel_y)
        self.__is_relative = (0, 0)

        # whether the coord values are percents
        self.__is_percent_coords = (0, 0)

        # whether the size values are percents
        self.__is_percent_size = (0, 0)

        # the coordinates set by the user
        self.__user_coords = (0, 0)

        # the size set by the user
        self.__user_size = (-1, -1)

        # the placement anchor
        self.__anchor = self.ANCHOR_NW


        DataTarget.__init__(self, parent, display)
        gtk.HBox.__init__(self)
        self.connect("expose-event", self.__on_expose)

        self._set_property_type("x", TYPE_COORD_SIZE)
        self._set_property_type("y", TYPE_COORD_SIZE)
        self._set_property_type("width", TYPE_COORD_SIZE)
        self._set_property_type("height", TYPE_COORD_SIZE)
        self._set_property_type("relative-to", TYPE_LIST)

        self._set_property_type("visible", TYPE_BOOL)

        self._set_property_type("on-enter", TYPE_STRING)
        self._set_property_type("on-leave", TYPE_STRING)
        self._set_property_type("on-click", TYPE_STRING)
        self._set_property_type("on-press", TYPE_STRING)
        self._set_property_type("on-release", TYPE_STRING)
        self._set_property_type("on-scroll", TYPE_STRING)
        self._set_property_type("on-file-drop", TYPE_STRING)
        self._set_property_type("on-doublelick", TYPE_STRING)
        self._set_property_type("on-motion", TYPE_STRING)
        self._set_property_type("on-menu", TYPE_STRING)

        if (parent):
            parent.add_observer(self.__on_observe_parent)

        self.show()



    #
    # Reacts on expose events of this widget.
    #
    def __on_expose(self, src, event):
        
        x, y, ow, oh = self.__geometry
        w, h = src.size_request()

        if ((w, h) != (ow, oh)):
            self.__geometry = (x, y, w, h)
            self.update_observer(self.OBS_MOVE, x, y, w, h)
            if (w < ow or h < oh):
                # see if siblings can collapse
                parent = self._get_parent()
                if (parent): parent.collapse_siblings(self, w < ow, h < oh)
        #end if



    #
    # Reacts on notifications by the relative of this widget.
    #
    def __on_observe_relative(self, src, cmd, *args):

        if (cmd == src.OBS_MOVE):
            x, y = self.__user_coords
            self.__relative = src
            self.set_position(x, y)



    #
    # Reacts on notifications by the parent.
    #
    def __on_observe_parent(self, src, cmd, *args):

        if (cmd == src.OBS_MOVE):
            x, y, w, h = args
            px, py = self.__user_coords
            pw, ph = self.__user_size
            is_px, is_py = self.__is_percent_coords
            is_pw, is_ph = self.__is_percent_size

            if (is_px or is_py):
                self.set_position(px, py)

            if (is_pw or is_ph):
                self.set_size(pw, ph)

            return



    def __on_file_drop( self, widget, context, x, y, data, info, time ):
        ''' catch DND events and process them to send them to them
        main display, which forwards them to the sensor '''

        # get the display
        display = self._get_display()

        # tell the main display to send files and coordinates to the sensor
        files = [unquote(uri) for uri in data.data.split("\r\n") if uri != '']
        x, y = self._get_display().get_pointer()
        display.file_drop(files, x, y)
        


    def set_config(self, key, value):

        if (key == "x"):
            v, is_percent = value
            x, y = self.__user_coords
            is_px, is_py = self.__is_percent_coords

            if (is_percent): self.__is_percent_coords = (1, is_py)
            self.set_position(v, y)

        elif (key == "y"):
            v, is_percent = value
            x, y = self.__user_coords
            is_px, is_py = self.__is_percent_coords

            if (is_percent): self.__is_percent_coords = (is_px, 1)
            self.set_position(x, v)

        elif (key == "width"):
            v, is_percent = value
            w, h = self.__user_size
            is_pw, is_ph = self.__is_percent_size

            if (is_percent): self.__is_percent_size = (1, is_ph)
            self.__user_size = (v, h)
            self.set_size(v, h)

        elif (key == "height"):
            v, is_percent = value
            w, h = self.__user_size
            is_pw, is_ph = self.__is_percent_size

            if (is_percent): self.__is_percent_size = (is_pw, 1)
            self.__user_size = (w, v)
            self.set_size(w, v)

        elif (key == "relative-to"):
            name, mode = value
            if (mode == "x"): self.__is_relative = (1, 0)
            elif (mode == "y"): self.__is_relative = (0, 1)
            elif (mode == "xy"): self.__is_relative = (1, 1)

            # no longer watch the old relative
            if (self.__relative):
                self.__relative.remove_observer(self.__on_observe_relative)

            target = self._get_parent().get_child_by_id(name)
            target.add_observer(self.__on_observe_relative)

        elif (key == "anchor"):
            x, y = self.__user_coords
            self.__anchor = value
            self.set_position(x, y)

        elif (key == "visible"):
            if (value): self.show()
            else: self.hide()

        elif (key == "link"):
            self.set_action_call(self.ACTION_LINK, value)

        elif (key == "on-enter"):
            self.set_action_call(self.ACTION_ENTER, value)
        elif (key == "on-leave"):
            self.set_action_call(self.ACTION_LEAVE, value)
        elif (key == "on-click"):
            self.set_action_call(self.ACTION_CLICK, value)
        elif (key == "on-file-drop"):
            self.set_action_call(self.ACTION_FILE_DROP, value)
            self.drag_dest_set(gtk.DEST_DEFAULT_ALL, self.__DND_FILE,
                               gtk.gdk.ACTION_COPY)
            self.connect("drag_data_received", self.__on_file_drop)
        elif (key == "on-press"):
            self.set_action_call(self.ACTION_PRESS, value)
        elif (key == "on-release"):
            self.set_action_call(self.ACTION_RELEASE, value)
        elif (key == "on-doubleclick"):
            self.set_action_call(self.ACTION_DOUBLECLICK, value)
        elif (key == "on-motion"):
            self.set_action_call(self.ACTION_MOTION, value)
        elif (key == "on-scroll"):
            self.set_action_call(self.ACTION_SCROLL, value)
        elif (key == "on-menu"):
            self.set_action_call(self.ACTION_MENU, value)

        else:
            DataTarget.set_config(self, key, value)



    #
    # Returns the true coordinates of this target when the given coordinates
    # are the hotspot.
    #
    def get_anchored_coords(self, x, y, w, h):

        if (x == y == UNSET_COORD): return (x, y)

        anchor = self.__anchor
        if (anchor in [self.ANCHOR_NW, self.ANCHOR_W, self.ANCHOR_SW]):
            ax = x
        elif (anchor in [self.ANCHOR_N, self.ANCHOR_CENTER, self.ANCHOR_S]):
            ax = x - (w / 2)
        else:
            ax = x - w

        if (anchor in [self.ANCHOR_NW, self.ANCHOR_N, self.ANCHOR_NE]):
            ay = y
        elif (anchor in [self.ANCHOR_W, self.ANCHOR_CENTER, self.ANCHOR_E]):
            ay = y - (h / 2)
        else:
            ay = y - h

        return (ax, ay)



    #
    # Returns the target ID to which this target is placed relatively.
    #
    def get_relative_to(self):

        return self.__relative



    #
    # Sets the position of this target. The coordinates are not anchored.
    #
    def set_position(self, x, y, update = 1):

        self.__user_coords = (x, y)

        # handle relative positioning
        if (self.__relative):
            is_rel_x, is_rel_y = self.__is_relative
            # get enter coords of relative
            arx, ary, rw, rh = self.__relative.get_geometry()
            # get real coordinates of relative
            rx, ry = self.__relative.get_anchored_coords(arx, ary, rw, rh)
            tw = (rx + rw) - arx
            th = (ry + rh) - ary

            x += arx
            y += ary
            if (is_rel_x): x += tw
            if (is_rel_y): y += th
            #print x, y, tw, th
        #end if

        # handle percentual positioning
        parent = self._get_parent()
        is_px, is_py = self.__is_percent_coords
        if (parent):
            nil, nil, pw, ph = parent.get_geometry()
        else:
            pw, ph = gtk.gdk.screen_width(), gtk.gdk.screen_height()

        if (is_px):
            x = int(x * (pw / 100.0))
        if (is_py):
            y = int(y * (ph / 100.0))
        #end if

        nil, nil, w, h = self.get_geometry()

        # don't let children stretch their parents
        ax, ay = self.get_anchored_coords(x, y, w, h)
        if (is_px and ax + w > pw - 1):
            x -= ((ax + w) - pw)
        if (is_py and ay + h > ph - 1):
            y -= ((ay + h) - ph)

        nil, nil, w, h = self.__geometry
        self.__geometry = (x, y, w, h)

        if (update):
            self.update_observer(self.OBS_MOVE, x, y, w, h)



    #
    # Sets the size of this target.
    #
    def set_size(self, width, height):

        parent = self._get_parent()
        is_pw, is_ph = self.__is_percent_size
        if (parent):
            nil, nil, w, h = parent.get_geometry()
        else:
            w, h = gtk.gdk.screen_width(), gtk.gdk.screen_height()
        cx, cy, cw, ch = self.get_geometry()

        if (is_pw):
            width = int(width * (w / 100.0))
        if (is_ph):
            height = int(height * (h / 100.0))

        ax, ay = self.get_anchored_coords(cx, cy, cw, ch)

        # prevent the child from growing over its parent
        if (is_pw and width >= ax and ax + width >= w): width = w - ax
        if (is_ph and height >= ay and ay + height >= h): height = h - ay

        self.set_size_request(width, height)



    #
    # Returns the geometry (coordinates and size) of this target.
    #
    def get_geometry(self):

        x, y, nil, nil = self.__geometry
        #nil, nil, w, h = self.get_allocation()
        w, h = self.size_request()
        return (x, y, w, h)



    #
    # Returns the geometry from the user's point of view.
    #
    def get_user_geometry(self):

        x, y = self.__user_coords
        w, h = self.__user_size
        return (x, y, w, h)



    #
    # Returns the target at the given position and its path.
    #
    def get_target_at(self, px, py, path):

        x, y, w, h = self.get_geometry()
        x, y = self.get_anchored_coords(x, y, w, h)
        
        if (x <= px <= x + w and y <= py <= y + h):
            return [(self, path)]
        else:
            return []



    #
    # Returns the percentual state of this target, i.e. whether coordinates
    # or size values are percentual.
    #
    def get_percentual_state(self):

        is_px, is_py = self.__is_percent_coords
        is_pw, is_ph = self.__is_percent_size
        return (is_px, is_py, is_pw, is_ph)
